/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_GAMECLOCK
#include "GameClock.h"
#endif

#include <allegro.h>

#include <time.h>

//----------------------------------------------------------------------------

GameClock::GameClock()
: msec_per_game_hour_(1000)
, game_time(0)
, min_speed(10)
, max_speed(600)
, normal_fps(30)
, speed_percent(100)
, is_pause_(false)
, is_stopped_(false)
{
}

//----------------------------------------------------------------------------

static int the_months[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

int GameClock::get_minute_of_hour() const
{
  return (((int)game_time)%ticks_per_game_hour) * ( 60.0 / ticks_per_game_hour );
}

int GameClock::get_hour_of_day() const
{
  return ((int)(game_time/ticks_per_game_hour))%24;
}

int GameClock::get_hour_since0() const
{
  return (int)(game_time/ticks_per_game_hour);
}

int GameClock::get_days_since0() const
{
  return ((int)(game_time/ticks_per_game_hour))/24;
}

int GameClock::get_day_of_month() const
{
  int d0 = get_days_since0() % 365;
  for( int i=0; i<12; ++i )
  {
    if( d0 < the_months[i] )
      break;
    d0 -= the_months[i];
  }
  return 1 + d0;
}

int GameClock::get_month_of_year() const
{
  int d0 = get_days_since0() % 365;
  for( int i=0; i<12; ++i )
  {
    if( d0 < the_months[i] )
      return 1 + i;
    d0 -= the_months[i];
  }
  assertCondition(false);
  return 0;
}

int GameClock::get_years_since0() const
{
  return 1 + get_days_since0() / 365;
}

//----------------------------------------------------------------------------

void GameClock::init( const char*ini_fn )
{
  assertCondition( ini_fn );

  push_config_state();
  set_config_file(ini_fn);
  msec_per_game_hour_ = get_config_int( "timer", "msec_per_game_hour", msec_per_game_hour_ );
  normal_fps = get_config_int( "timer", "normal_fps", normal_fps );
  min_speed = get_config_int( "timer", "min_speed", min_speed );
  max_speed = get_config_int( "timer", "max_speed", max_speed );
  pop_config_state();
  assertCondition( normal_fps > 0 );
}

void GameClock::reset_game_time( int new_time, int new_speed )
{
  speed_percent = new_speed;
  is_pause_ = false;
  is_stopped_ = false;
  game_time = new_time;
}

void GameClock::advance_game_time()
{
  if( !is_pause_ && !is_stopped_ )
    game_time += ( speed_percent / (double)normal_fps ) * ( ticks_per_game_hour / 100.0 ) * ( 1000.0 / (double)msec_per_game_hour_ );
}

void GameClock::handle_kbd( int scan_code, int shift_flags )
{
  if( scan_code == KEY_X )
    if( speed_percent < max_speed )
      speed_percent += speed_percent<40 ? 2 : speed_percent<200 ? 10 : speed_percent < 400 ? 20 : 40;
  if( scan_code == KEY_Z )
    if( speed_percent > min_speed )
      speed_percent -= speed_percent<40 ? 2 : speed_percent<200 ? 10 : speed_percent < 400 ? 20 : 40;
  if( scan_code == KEY_P )
    is_pause_ = !is_pause_;
}

//----------------------------------------------------------------------------

int GameClock::get_realtime() const
{
  return ( clock() * (double)ticks_per_second_realtime / CLOCKS_PER_SEC );
}

//----------------------------------------------------------------------------
