/*

mtrand.cpp - Uses the Mersenne Twister RNG to generate random numbers.

Mersenne Twister RNG made by Makoto Matsumoto and Takuji Nishimura,

More information about MT can be found at:
http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

3. The names of its contributors may not be used to endorse or promote 
products derived from this software without specific prior written 
permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS   
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR   
A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef INC_MTRAND
#include "mtrand.h"
#endif

namespace
{
  const unsigned long MATRIX_A        = 0x9908b0dfUL;     // Constant vector a
  const unsigned long UPPER_MASK      = 0x80000000UL;     // Most significant w-r bits
  const unsigned long LOWER_MASK      = 0x7fffffffUL;     // Least significant r bits
  const unsigned long DEFAULT_SEED    = 4357UL;
}

// Default CTOR
MTRand::MTRand()
{
  Seed(DEFAULT_SEED);
}

// CTOR which takes a seed
MTRand::MTRand(unsigned long seed)
{
  Seed(seed);
}

// Initializes mt[N] with a seed
void MTRand::Seed(unsigned long seed)
{
  mt[0]= seed & 0xffffffffUL;
  for (mti = 1; mti < N; mti++) {
    mt[mti] = (1812433253UL * (mt[mti-1] ^ (mt[mti-1] >> 30)) + mti); 
    /* 
    See Knuth TAOCP Vol2. 3rd Ed. P.106 for multiplier.
    In the previous versions, MSBs of the seed affect
    only MSBs of the array mt[].
    2002/01/09 modified by Makoto Matsumoto
    */
    mt[mti] &= 0xffffffffUL;
    // For >32 bit machines
  }
}

// Generates a random number on [0,0xffffffff]-interval
unsigned long MTRand::Rand()
{
  unsigned long y;
  static unsigned long mag01[2] = { 0x0UL, MATRIX_A };
  // mag01[x] = x * MATRIX_A for x = 0,1

  // Generate N words at one time
  if (mti >= N) {
    int kk;

    for (kk = 0; kk < (N - M); kk++) {
      y = (mt[kk] & UPPER_MASK) | (mt[kk + 1] & LOWER_MASK);
      mt[kk] = mt[kk + M] ^ (y >> 1) ^ mag01[y & 0x1UL];
    }

    for (; kk < (N - 1); kk++) {
      y = (mt[kk] & UPPER_MASK) | (mt[kk + 1] & LOWER_MASK);
      mt[kk] = mt[kk + (M - N)] ^ (y >> 1) ^ mag01[y & 0x1UL];
    }

    y = (mt[N - 1] & UPPER_MASK) | (mt[0] & LOWER_MASK);
    mt[N - 1] = mt[M - 1] ^ (y >> 1) ^ mag01[y & 0x1UL];

    mti = 0;
  }

  y = mt[mti++];

  // Tempering
  y ^= (y >> 11);
  y ^= (y << 7) & 0x9d2c5680UL;
  y ^= (y << 15) & 0xefc60000UL;
  y ^= (y >> 18);

  return y;
}

// Generates a random number on [min, max[-interval
int MTRand::IRand(int max, int min)
{    
  return (int)(Rand() % ((max) - min) + min);
}

// Generates a random number on [0, max[-interval
int MTRand::IRand(int max)
{
  return (int)(Rand() % (max));
}

double MTRand::DRand()
{
  return (double)(Rand()) / (double)(0xffffffffUL);
}

double MTRand::DRand(double max)
{
  return (double)(Rand()*max) / (double)(0xffffffffUL);
}


MTRand rng;

