/* 
Copyright (c) 2005 - 2007 Tobias Scheuer

The zlib/libpng License

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from the 
use of this software.

Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim 
that you wrote the original software. If you use this software in a product, 
an acknowledgment in the product documentation would be appreciated but is 
not required.

2. Altered source versions must be plainly marked as such, and must not be 
misrepresented as being the original software.

3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INC_DISPLAYENGINE
#include "DisplayEngine.h"
#endif

#include <allegro.h>

//----------------------------------------------------------------------------

DisplayEngine::DisplayEngine()
: update_method_(UPDATE_NONE)
, active_buffer(0)
, system_buffer(0)
, video_buffer1(0)
, video_buffer2(0)
{
}

DisplayEngine::~DisplayEngine()
{
  destroy_all_buffers();
}

void DisplayEngine::destroy_all_buffers()
{
  active_buffer = 0;
  if( system_buffer )
    destroy_bitmap( system_buffer );
  system_buffer = 0;
  if( video_buffer1 )
    destroy_bitmap( video_buffer1 );
  video_buffer1 = 0;
  if( video_buffer2 )
    destroy_bitmap( video_buffer2 );
  video_buffer2 = 0;
}

void DisplayEngine::allocate_and_clear_buffers( UpdateMethod update_method )
{
  // SVV  uses sys vid1 vid2
  //  VV  uses     vid1 vid2
  //  SV  uses sys           screen
  // NONE uses               screen
  if( update_method == UPDATE_SVV_TRIPLE || update_method == UPDATE_VV_DOUBLE )
  {
    if( !video_buffer1 )
      video_buffer1 = create_video_bitmap(SCREEN_W, SCREEN_H);
    if( !video_buffer2 )
      video_buffer2 = create_video_bitmap(SCREEN_W, SCREEN_H);
  }
  else
  {
    if( video_buffer1 )
      destroy_bitmap( video_buffer1 );
    video_buffer1 = 0;
    if( video_buffer2 )
      destroy_bitmap( video_buffer2 );
    video_buffer2 = 0;
  }
  if( update_method == UPDATE_SVV_TRIPLE || update_method == UPDATE_SV_DOUBLE )
  {
    if( !system_buffer )
      system_buffer = create_system_bitmap(SCREEN_W, SCREEN_H);
  }
  else
  {
    if( system_buffer )
      destroy_bitmap( system_buffer );
    system_buffer = 0;
  }
  if( system_buffer )
    clear_bitmap( system_buffer );
  if( video_buffer1 )
    clear_bitmap( video_buffer1 );
  if( video_buffer2 )
    clear_bitmap( video_buffer2 );
}

//----------------------------------------------------------------------------

void DisplayEngine::init_gfx()
{
  if( update_method_ == UPDATE_SVV_TRIPLE )
  {
    if( !(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER) )
      enable_triple_buffer();
    if( (gfx_capabilities & GFX_CAN_TRIPLE_BUFFER) )
    {
      allocate_and_clear_buffers( UPDATE_SVV_TRIPLE );
      if( video_buffer1 && video_buffer2 && system_buffer )
      {
        active_buffer = video_buffer1;
        show_video_bitmap(video_buffer2);
        return;
      }
      else if( video_buffer1 && video_buffer2 )
        update_method_ = UPDATE_VV_DOUBLE;
      else if( video_buffer1 && system_buffer )
        update_method_ = UPDATE_SV_DOUBLE;
    }
    else
      update_method_ = UPDATE_SV_DOUBLE;
  }

  if( update_method_ == UPDATE_VV_DOUBLE )
  {
    allocate_and_clear_buffers( UPDATE_VV_DOUBLE );
    if( video_buffer1 && video_buffer2 )
    {
      active_buffer = video_buffer1;
      show_video_bitmap(video_buffer2);
      return;
    }
    if( video_buffer1 )
      update_method_ = UPDATE_SV_DOUBLE;
    else
      update_method_ = UPDATE_NONE;
  }

  if( update_method_ == UPDATE_SV_DOUBLE )
  {
    allocate_and_clear_buffers( UPDATE_SV_DOUBLE );
    if( system_buffer )
    {
      active_buffer = system_buffer;
      return;
    }
    else
      update_method_ = UPDATE_NONE;
  }

  if( update_method_ == UPDATE_NONE )
  {
    allocate_and_clear_buffers( UPDATE_NONE );
    active_buffer = screen;
    update_method_ = UPDATE_NONE;
    return;
  }

  fatalError( "could not create any update method" );
}


void DisplayEngine::init( const char*ini_fn )
{
  assertCondition( ini_fn );

  push_config_state();
  set_config_file( ini_fn );

  int w = 0;
  int h = 0;
  int bpp = 16;

  int count = 0;
  const char*const *const data = get_config_argv("graphics", "mode", &count);
  if( count == 3 )
  {
    w = atoi(data[0]);
    h = atoi(data[1]);
    bpp = atoi(data[2]);
  }
  else if( count == 2 )
  {
    w = atoi(data[0]);
    h = atoi(data[1]);
  }
  else
  {
    allegro_message( "Found %i parameters in graphics.mode instead of 3 expected: w x h x bpp.\n", count );
    fatalError("can't get screen resolution");
  }

  int gfx_mode = GFX_AUTODETECT_FULLSCREEN;
  if( get_config_int( "graphics", "windowed", 0 ) != 0 )
    gfx_mode = GFX_AUTODETECT_WINDOWED;
  update_method_ = (DisplayEngine::UpdateMethod)get_config_int( "graphics", "update_method", UPDATE_SV_DOUBLE );

  assertCondition( bpp>=16 );
  set_color_depth( bpp );
  if( set_gfx_mode( gfx_mode, w, h, 0, 0 ) < 0 )
  {
    if( gfx_mode == GFX_AUTODETECT_WINDOWED )
      gfx_mode = GFX_AUTODETECT_FULLSCREEN;
    if( set_gfx_mode( gfx_mode, w, h, 0, 0 ) < 0 )
      fatalError("Unable to set a graphics mode");
  }
  set_display_switch_mode( SWITCH_BACKAMNESIA );

  init_gfx();

  pop_config_state();
  enable_hardware_cursor();
  show_mouse(screen);
}

void DisplayEngine::prepare_draw_buffer()
{
  if( system_buffer )
  {
    if( !is_system_bitmap(system_buffer) )
      acquire_bitmap( system_buffer );
    clear_bitmap( system_buffer );
  }
  else
  {
    if( !is_video_bitmap(active_buffer) )
      acquire_bitmap( active_buffer );
    clear_bitmap( active_buffer );
  }
}

void DisplayEngine::display_draw_buffer()
{
  if( system_buffer )
  {
    if( !is_system_bitmap(system_buffer) )
      release_bitmap( system_buffer );
  }
  else
  {
    if( !is_video_bitmap(active_buffer) )
      release_bitmap( active_buffer );
  }
  scare_mouse();
  switch( update_method_ )
  {
  case UPDATE_SVV_TRIPLE:
    blit( system_buffer, active_buffer, 0, 0, 0, 0, SCREEN_W, SCREEN_H );
    while( poll_scroll() );
    request_video_bitmap( active_buffer );
    active_buffer = (active_buffer==video_buffer1) ? video_buffer2 : video_buffer1;
    break;
  case UPDATE_VV_DOUBLE:
    show_video_bitmap( active_buffer );
    active_buffer = (active_buffer==video_buffer1) ? video_buffer2 : video_buffer1;
    break;
  case UPDATE_SV_DOUBLE:
    blit( system_buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H );
    break;
  case UPDATE_NONE:
    break;
  }
  unscare_mouse();
}

//----------------------------------------------------------------------------

ViewPortBase::ViewPortBase( const ScreenRect&asr )
: area(asr)
{
}

ViewPortBase::~ViewPortBase()
{
}

void ViewPortBase::notify_chg_size()
{
}

void ViewPortBase::chg_width_left( int w )
{
  const int dw = w - area.w;
  area.x -= dw;
  area.w = w;
  notify_chg_size();
}

void ViewPortBase::chg_width_right( int w )
{
  area.w = w;
  notify_chg_size();
}

void ViewPortBase::chg_height_top( int h )
{
  const int dh = h - area.h;
  area.y -= dh;
  area.h = h;
  notify_chg_size();
}

void ViewPortBase::chg_height_bottom( int h )
{
  area.h = h;
  notify_chg_size();
}

//----------------------------------------------------------------------------

