#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

#include "allegro.h"
#include "game.h"                   // header for my data file

#define GAME_SETTINGS     "game.cfg" // game configuration gets stored in this file

#define SHOW_BLANK        0
#define SHOW_ONE          1
#define SHOW_TWO          2
#define SHOW_THREE        3
#define SHOW_FOUR         4
#define SHOW_FIVE         5
#define SHOW_SIX          6
#define SHOW_SEVEN        7
#define SHOW_EIGHT        8
#define HIDDEN_MINE       9      // left or middle mouse buton is pressed down, see function check_mask()
#define MOUSE_B1_DOWN     10
#define FLAG_SET          11
#define QUESTION_SET      12
#define SHOW_X            13
#define TRIPPED_MINE      14
#define SHOW_BUTTON       15
#define STOP_WALKING_LOOP 16
#define GFX_SPPED_LIMIT   30
// functions

int  init_gfx          (void);
void game_loop         (void);
void gfx_update        (void);
void key_input         (void);
void setup_screen      (void);
int  mouse_events      (void);
void menu_events       (void);
void select_bar        (int x1,int y1, int x2, int y2);
int  game_menu_select  (void);
int  help_menu_select  (void);
void update_menu_checks(void);
void execute_menu_click(int menu_id,int item_clicked);
void init_map(void);
int minesweeper (int x, int y);
void board_clicks(void);
void check_mask(void);
void press_group(int x, int y);
void uncover_blanks(int x, int y);
int flag_sweeper(int x, int y);
void uncover_around_flags(int x, int y);
void lose_game (void);
void update_timer(void);
void led_print(int num,int x,int y);
int custom_menu(void);
void load_settings(void);
void save_settings(void);
int show_score(void);
int add_score(void);
void show_help (void);
void update_gfx_speed_control(void);

// global varables
DATAFILE *gamedata;
int x_rev      = 800 ;
int y_rev      = 600 ;
int old_x_rev  = 0   ;
int old_y_rev  = 0   ;
int rows       = 16  ;
int columns    = 30  ;
int mine_num   = 15  ;
int game_level = 1   ;
int exit_game  = 0   ;
BITMAP *gfx_buf      ;
int **board_map;
int **board_mask;
int lose=FALSE;
int game_timer = 0;
int game_start=FALSE;
int flag_cnt;
int menu_open = FALSE;
int pause_timer = FALSE;
int best_score_begin  = 999;
int best_score_medium = 999;
int best_score_expert = 999;
char champ_name_begin[11] = {"Anonymous"};
char champ_name_medium[11]= {"Anonymous"};
char champ_name_expert[11]= {"Anonymous"};
int gfx_speed_control = 0;

int main(int argc, char *argv[])
{
  char data_file_name[256];       // data file name
  allegro_init();
  install_keyboard();
  install_mouse();
  install_timer();
  set_color_conversion(COLORCONV_NONE);

  // load game data file
  replace_filename(data_file_name, argv[0], "game.dat", sizeof(data_file_name));
  gamedata = load_datafile(data_file_name);
  if (!gamedata)   {
    allegro_message("Error loading %s!\n", data_file_name);
    return 1;
   }

  load_settings();          //load game setting from file

  if (init_gfx()!=0)        // Check if there where any errors installing
      return 1;             // check video

    gfx_buf=create_bitmap(x_rev, y_rev);  // creat video buffer
    clear(gfx_buf);
    if (!gfx_buf) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Not enough video memory\n");
      return -1;
    }
  install_int(update_gfx_speed_control,GFX_SPPED_LIMIT);
  setup_screen();
  game_loop();
  free(board_map);
  free(board_mask);
  allegro_exit();
	return 0;
}

END_OF_MAIN()

int init_gfx(void)
{
  if (!old_x_rev){
     if (set_gfx_mode(GFX_AUTODETECT, x_rev, y_rev, 0, 0) !=0)
       {
           set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
           allegro_message("Unable to set any graphic mode\n%s\n", allegro_error);
           return -1;
       }
  }
  else {
    if (set_gfx_mode(GFX_AUTODETECT, x_rev, y_rev, 0, 0) !=0){
       x_rev = old_x_rev;
       y_rev = old_y_rev;
       set_gfx_mode(GFX_AUTODETECT, old_x_rev, old_y_rev, 0, 0);
       set_palette(gamedata[MY_PAL].dat);
       return 0;
    }
    destroy_bitmap(gfx_buf);
    gfx_buf = create_bitmap(x_rev, y_rev);
    clear(gfx_buf);
  }

  set_palette(gamedata[MY_PAL].dat);
  return 0;
}

void game_loop(void)
{
  flag_cnt = mine_num;
  init_map();
  show_mouse(screen);
  while (!exit_game)
  {
     if (keypressed())
       key_input();
     mouse_events ();
     gfx_update();
  }
  save_settings();
}

void gfx_update(void)
{
  if (keypressed())
    key_input();
  update_timer();
  if (gfx_speed_control){
    blit(gfx_buf, screen,0,0,0,0,x_rev,y_rev);
    gfx_speed_control = 0;
  }
}

void key_input(void)
{
   int key_pressed;
   while(keypressed())
   {
    key_pressed = readkey() >> 8;
    switch(key_pressed)
     {
      case KEY_ESC: exit_game=TRUE;
           break;
      case KEY_C: save_bmp("screen.bmp",gfx_buf,gamedata[MY_PAL].dat);
           break;
     }
   }
}

void setup_screen(void)
{
   int x,y;
   int button_size = 20;     // buttons are 20x20
   int game_width ;          // game width in pixels
   int game_hieght;

    switch(game_level) {
      case 1: rows = 8;
              columns = 8;
              mine_num = 10;
              break;
      case 2: rows = 16;
              columns = 16;
              mine_num = 40;
              break;
      case 3: rows = 16;
              columns = 30;
              mine_num = 99;
              break;
   }
   flag_cnt = mine_num;
   game_width = columns * button_size + 10;
   game_hieght = rows * button_size + 40+15;
   
   show_mouse(NULL);
   rectfill(gfx_buf,0,0,columns*20+10,14,27);  // Display menubar
   blit(gamedata[MENU].dat,gfx_buf,0,0,0,0,90,15);

   rect(gfx_buf,0,15,game_width,16,21);
   rect(gfx_buf,0,15,1,game_hieght,21);
   rect(gfx_buf,0,game_hieght-1,game_width,game_hieght,16);
   rect(gfx_buf,game_width-1,game_hieght,game_width,15,16);
   rectfill(gfx_buf,2,17,game_width-2,game_hieght-2,27);

   // blit Smiley button
   blit(gamedata[SMILEY].dat,gfx_buf,0,0,game_width/2-15,19,30,30);
   for(y=51;y<=rows*20+51-20;y+=20)
      for(x=5;x<=columns*20+5-20;x+=20)
         blit(gamedata[BUTTON].dat,gfx_buf,0,0,x,y,20,20);
   show_mouse(screen);
   gfx_update();
}

int mouse_events(void)
{
  int game_width = columns * 20 + 10;
  int x_smiley =game_width/2-15;
  int y_smiley = 19;
  if (mouse_x >=0 && mouse_x <=80 && mouse_y >= 0 && mouse_y <=15){
     menu_events();
     return 0; 
	}
	if (mouse_x >= 6 && mouse_x <= columns*20+6-1 && mouse_y >= 52 && mouse_y <= rows*20+52-1 && mouse_b > 0 && !lose){
     board_clicks();
     return 0;
  }  
	if (mouse_x >=x_smiley && mouse_x <=x_smiley +30 && mouse_y > 19 && mouse_y <=19+30 && mouse_b ==1){
    show_mouse(NULL);
    blit(gamedata[SMILEY_PRESSED].dat,gfx_buf,0,0,game_width/2-15,19,30,30);
    show_mouse(screen);
    while (mouse_x >=x_smiley && mouse_x <=x_smiley +30 && mouse_y > 19 && mouse_y <=19+30 && mouse_b==1){
      
      gfx_update();
    }
    if (mouse_x >=x_smiley && mouse_x <=x_smiley +30 && mouse_y > 19 && mouse_y <=19+30 ){
      lose = FALSE;
      pause_timer = FALSE;
      game_start = FALSE;
      init_map();
      setup_screen();
    }
    else if (!pause_timer && !lose) {
      show_mouse(NULL);
      blit(gamedata[SMILEY].dat,gfx_buf,0,0,game_width/2-15,19,30,30);
      show_mouse(screen);
    }
    else if(pause_timer){
      show_mouse(NULL);
      blit(gamedata[SMILEY_WIN].dat,gfx_buf,0,0,game_width/2-15,19,30,30);
      show_mouse(screen);
    }
    else if(lose){
      show_mouse(NULL);
      blit(gamedata[SMILEY_SAD].dat,gfx_buf,0,0,game_width/2-15,19,30,30);
      show_mouse(screen);
    }
  }
  return 0;
}

void menu_events(void)
{
  int menulock;
  int game_menu_click=0;
  int help_menu_click=0;
  if (mouse_x >=0 && mouse_x <=42 && mouse_y >= 0 && mouse_y <=15)
   {
     show_mouse(NULL);
     blit(gamedata[MOUSE_ON_GAME].dat,gfx_buf,0,0,0,0,90,15);
     show_mouse(screen);
     //GAME MENU
     while (mouse_x >=0 && mouse_x <=42 && mouse_y >= 0 && mouse_y <=15)
       {
        
        if (mouse_b == 1)
          {
            menu_open = TRUE;
            show_mouse(NULL);
            masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
            update_menu_checks();
            show_mouse(screen);
            while (mouse_b == 1)
              {      // mouse over Game menu box
                if (mouse_x >=0 && mouse_x <=89 && mouse_y >=15 && mouse_y <=174)
                   menulock = FALSE;
                     // mouse over Game menu button
                else if (mouse_x >=0 && mouse_x <=42 && mouse_y >=0 && mouse_y <=15)
                   menulock = TRUE;
                     // mouse over Help menu button
                else if (mouse_x >=43 && mouse_x <=80 && mouse_y >=0 && mouse_y <=15)
                  {
                   menulock =FALSE;
                   break;
                  }
                    // mouse over anything else
                else
                   menulock = FALSE;
                // select menu items if mouse is over them
                game_menu_click=game_menu_select();
                gfx_update();
              }
            if (!menulock){
              menu_open = FALSE;
              check_mask();
            }
            else
             {
              while(1)
               {
                game_menu_select();
                gfx_update();
                // mouse over game menu box
                if (mouse_x >=0 && mouse_x <=89 && mouse_y >=15 && mouse_y <=174 && mouse_b == 1)
                  {
                    while (mouse_b ==1){
                      game_menu_click=game_menu_select();
                      gfx_update();
                    }
                    menu_open = FALSE;
                    check_mask();
                    break;
                  }
                 else if (mouse_b ==1)
                  {
                   if (mouse_x >=0 && mouse_x <=42 && mouse_y >=0 && mouse_y <=15)
                     while(mouse_b)
                       gfx_update();
                   menu_open = FALSE;
                   check_mask();
                   break;
                  }
               }
             }
          }
        gfx_update();
       }

    show_mouse(NULL);
    blit(gamedata[MENU].dat,gfx_buf,0,0,0,0,90,15);
    show_mouse(screen);
  }

       // HELP menu
 if (mouse_x >=43 && mouse_x <=80 && mouse_y >= 0 && mouse_y <=15)
   {
     show_mouse(NULL);
     blit(gamedata[MOUSE_ON_HELP].dat,gfx_buf,0,0,0,0,90,15);
     show_mouse(screen);

      while (mouse_x >42 && mouse_x <=80 && mouse_y >= 0 && mouse_y <=15)
       {
        if (mouse_b == 1)
          {
            menu_open = TRUE;
            show_mouse(NULL);
            masked_blit(gamedata[HELP_MENU].dat,gfx_buf,0,0,0,0,94,54);
            show_mouse(screen);
            while (mouse_b == 1)
              {      //mouse over Help Menu box
                if (mouse_x >42 && mouse_x <=94 && mouse_y >=15 && mouse_y <=54)
                   menulock = FALSE;
                    // mouse over Help menu button
                else if (mouse_x >42 && mouse_x <=80 && mouse_y >=0 && mouse_y <=15)
                   menulock = TRUE;
                     // mouse over Game menu box
                else if (mouse_x >=0 && mouse_x <=42 && mouse_y >=0 && mouse_y <=15)
                  {
                    menulock =FALSE;
                    break;
                  }
                    // anywhere else
                else
                   menulock = FALSE;
                help_menu_click = help_menu_select();
                gfx_update();
              }
            if (!menulock){
               menu_open = FALSE;
               check_mask();
            }
            else
             {
              while(1)
               {
                help_menu_select();
                gfx_update();
                if (mouse_x >42 && mouse_x <=94 && mouse_y >=15 && mouse_y <=54 && mouse_b == 1)
                  {
                    while(mouse_b ==1){
                      help_menu_click=help_menu_select();
                      gfx_update();
                    }
                    menu_open = FALSE;
                    check_mask();
                    break;
                  }
                 else if (mouse_b ==1)
                  {
                   if (mouse_x >42 && mouse_x <=80 && mouse_y >=0 && mouse_y <=15)
                     while(mouse_b)
                       gfx_update();
                   menu_open = FALSE;
                   check_mask();
                   break;
                  }
               }
             }
          }
        gfx_update();
      }
    show_mouse(NULL);
    blit(gamedata[MENU].dat,gfx_buf,0,0,0,0,90,15);
    show_mouse(screen);
  }

 // prevent you from clicking on the mine field when menu is open
 if (mouse_x >= 6 && mouse_x <= columns*20+6-1 && mouse_y >= 17 && mouse_y <= rows*20+52-1 && mouse_b > 0 ) {
   while(mouse_b)
     gfx_update();
 }
 menu_open = FALSE;
 gfx_update();
 // process the menu clicks
 if (game_menu_click != 0)
   execute_menu_click(1,game_menu_click);
 else if(help_menu_click !=0)
   execute_menu_click(2,help_menu_click);
}


void select_bar(int x1,int y1, int x2, int y2)
{
int y,x;
for (y=y1;y<=y2;y++)
  for (x=x1;x<=x2;x++)
   {
     if ((getpixel(gfx_buf,x,y))==27)
         putpixel(gfx_buf,x,y,11);
     else if((getpixel(gfx_buf,x,y))==10)
         putpixel(gfx_buf,x,y,21);
   }
}

int game_menu_select(void)
{
  int menu=0;
   // new button selected
  if (mouse_x >=0 && mouse_x<=86 && mouse_y >=17 && mouse_y <= 29)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,17,86,29);
    show_mouse(screen);
    menu=1;
   }
   // beginner button selected
  else if (mouse_x >=0 && mouse_x<=89 && mouse_y >=32 && mouse_y <= 46)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,32,86,46);
    show_mouse(screen);
    menu=2;
   }
   // intermediate button selected
  else if (mouse_x >=0 && mouse_x<=89 && mouse_y >=47 && mouse_y <= 61)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,47,86,61);
    show_mouse(screen);
    menu=3;
   }
   // expert button selected
  else if (mouse_x >=0 && mouse_x<=89 && mouse_y >=62 && mouse_y <= 76)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,62,86,76);
    show_mouse(screen);
    menu=4;
   }
   // custom button selected
  else if (mouse_x >=0 && mouse_x<=89 && mouse_y >=77 && mouse_y <= 89)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,77,86,89);
    show_mouse(screen);
    menu=5;
   }
  // 640x480 button selected
  else if (mouse_x >=0 && mouse_x<=89 && mouse_y >=92 && mouse_y <= 105)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,92,86,105);
    show_mouse(screen);
    menu=6;
   }
  // 800x600 button selected
  else if (mouse_x >=0 && mouse_x<=89 && mouse_y >=106 && mouse_y <= 120)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,106,86,120);
    show_mouse(screen);
    menu=7;
   }
  // 1024x768 button selected
  else if (mouse_x >=0 && mouse_x<=89 && mouse_y >=121 && mouse_y <= 135)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,121,86,135);
    show_mouse(screen);
    menu=8;
   }
  // scores button selected
  else if (mouse_x >=0 && mouse_x<=89 && mouse_y >=138 && mouse_y <= 150)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,138,86,150);
    show_mouse(screen);
    menu=9;
   }
  // exit button selected
  else if (mouse_x >=0 && mouse_x<=89 && mouse_y >=153 && mouse_y <= 165)
   {
    show_mouse(NULL);
    masked_blit(gamedata[GAME_MENU].dat,gfx_buf,0,0,0,0,91,170);
    update_menu_checks();
    select_bar(0,153,86,165);
    show_mouse(screen);
    menu=10;
   }
   // nothing selected
  else
    menu=0;

  return menu;
}

int help_menu_select(void)
{

 int menu = 0;
   // help button selected
  if (mouse_x >=45 && mouse_x<=87 && mouse_y >=17 && mouse_y <= 29)
   {
    show_mouse(NULL);
    masked_blit(gamedata[HELP_MENU].dat,gfx_buf,0,0,0,0,91,170);
    select_bar(45,17,87,29);
    show_mouse(screen);
    menu=1;
   }
  // about button selected
  else if (mouse_x >=45 && mouse_x<=87 && mouse_y >=32 && mouse_y <= 45)
   {
    show_mouse(NULL);
    masked_blit(gamedata[HELP_MENU].dat,gfx_buf,0,0,0,0,91,170);
    select_bar(45,32,87,45);
    show_mouse(screen);
    menu=2;
   }
  else
    menu=0;

  return menu;
}

void update_menu_checks(void)
{
 switch (x_rev)
  {
   case 640: blit(gamedata[CHECKED].dat,gfx_buf,0,0,1,94,10,10);
             break;
   case 800: blit(gamedata[CHECKED].dat,gfx_buf,0,0,1,109,10,10);
             break;
   case 1024:blit(gamedata[CHECKED].dat,gfx_buf,0,0,1,125,10,10);
             break;
  }
  switch (game_level)
  {
   case 0: blit(gamedata[CHECKED].dat,gfx_buf,0,0,1,79,10,10);
             break;
   case 1: blit(gamedata[CHECKED].dat,gfx_buf,0,0,1,34,10,10);
             break;
   case 2: blit(gamedata[CHECKED].dat,gfx_buf,0,0,1,49,10,10);
             break;
   case 3: blit(gamedata[CHECKED].dat,gfx_buf,0,0,1,64,10,10);
             break;
  }
}

void execute_menu_click(int menu_id,int item_clicked)
{
 // game menu
if (menu_id == 1)
 switch(item_clicked)
 {
  case 1: lose = FALSE;
          pause_timer = FALSE;
          game_start = FALSE;
          init_map();
          setup_screen();
          break;
  case 2: if (game_level !=1){
            lose = FALSE;
            pause_timer = FALSE;
            game_start = FALSE;
            game_level = 1;
            clear(gfx_buf);
            setup_screen();
            init_map();
          }
          break;
  case 3: if (game_level !=2){
            lose = FALSE;
            pause_timer = FALSE;
            game_start = FALSE;
            game_level = 2;
            clear(gfx_buf);
            setup_screen();
            init_map();
          }
          break;
  case 4: if (game_level !=3){
            lose = FALSE;
            pause_timer = FALSE;
            game_start = FALSE;
            game_level = 3;
            clear(gfx_buf);
            setup_screen();
            init_map();
          }
          break;
  case 5: if (custom_menu()){
             lose = FALSE;
             pause_timer = FALSE;
             game_start = FALSE;
             game_level = 0;
             show_mouse(NULL);
             clear(gfx_buf);
             show_mouse(screen);
             setup_screen();
             init_map();
          }
          break;
   case 6: if (x_rev !=640){
            old_x_rev = x_rev;
            old_y_rev = y_rev;
            x_rev =640;
            y_rev =480;
            init_gfx();
            check_mask();
            show_mouse(screen);
          }
          break;
  case 7: if (x_rev !=800){
            old_x_rev = x_rev;
            old_y_rev = y_rev;
            x_rev =800;
            y_rev =600;
            init_gfx();
            check_mask();
            show_mouse(screen);
          }
          break;
  case 8: if (x_rev !=1024){
            old_x_rev = x_rev;
            old_y_rev = y_rev;
            x_rev =1024;
            y_rev =768;
            init_gfx();
            check_mask();
            show_mouse(screen);
          }
          break;
  case 9:show_score();
         break;
  case 10: exit_game=TRUE;
         break;
 }

 // help menu
else if (menu_id ==2)
  switch(item_clicked){
   // help button
   case 1:show_help ();
          break;
   // about button
   case 2:
          break;
  }
}

void init_map(void){
 int x,y;
 int cnt=0;
 // allocate memory for arrays
 board_map = malloc(columns * sizeof(int*));
    if(!board_map){
        fputs("board_map allocation failure.\n", stderr);
        exit(EXIT_FAILURE);
    }
 board_mask = malloc(columns * sizeof(int*));
    if(!board_mask){
        fputs("board_mask allocation failure.\n", stderr);
        exit(EXIT_FAILURE);
    }


 for (x=0;x<columns;x++){
    board_map[x] = malloc(rows * sizeof(int));
        if(!board_map[x]){
            fprintf(stderr, "board_map[%i] allocation failure.\n", x);
            exit(EXIT_FAILURE);
        }

 }

 for (x=0;x<columns;x++){
    board_mask[x] = malloc(rows * sizeof(int));
        if(!board_mask[x]){
            fprintf(stderr, "board_mask[%i] allocation failure.\n", x);
            exit(EXIT_FAILURE);
        }
 }
for(y=0;y<rows;y++)
  for(x=0;x<columns;x++){
    board_map[x][y] = 0;
    board_mask[x][y] = SHOW_BUTTON;
  }

 srand(time(0));
  // put mines in random positions
 do {
   x = rand() % columns;
   y = rand() % rows;
   if (board_map[x][y] != 9){
      board_map[x][y] = 9;
      cnt++;
   }
 }while (cnt != mine_num);

 for (y=0;y<rows;y++)
  for (x=0;x<columns;x++) {
   if (board_map[x][y] != 9)
     board_map[x][y] = minesweeper(x, y);
  }
}

int minesweeper(int x, int y)
{
  int cnt = 0;
  if (board_map[x][y] == 0){
    if (x-1 >= 0 && y - 1 >= 0)
      if (board_map[x-1][y-1] == 9) cnt++;
    if (x >= 0 && y - 1 >= 0)
      if (board_map[x][y-1] == 9) cnt++;
    if (x+1 <= columns-1 && y-1 >= 0)
      if (board_map[x+1][y-1] == 9) cnt++;
    if (x+1 <= columns-1 && y <= rows-1)
      if (board_map[x+1][y] == 9) cnt++;
    if (x+1 <= columns-1 && y+1 <= rows-1)
      if (board_map[x+1][y+1] == 9) cnt++;
    if (x <= columns-1 && y+1 <= rows-1)
      if (board_map[x][y+1] == 9) cnt ++;
    if (x-1 >= 0 && y+1 <= rows-1)
      if (board_map[x-1][y+1] == 9) cnt ++;
    if (x-1 >= 0 && y <= rows-1)
      if (board_map[x-1][y] == 9) cnt ++;
  }
  return cnt;
}


void board_clicks(void)
{
 int x_index,y_index,x_blit,y_blit;
 int mouse_b_lock =FALSE;
 int old_mouse_x=0;
 int old_mouse_y=0;
 int old_mouse_b=0;
 while (mouse_x >= 6 && mouse_x <= columns*20+6-1 && mouse_y >= 52 && mouse_y <= rows*20+52-1 && mouse_b > 0){
   if (old_mouse_x != mouse_x || old_mouse_y != mouse_y || old_mouse_b != mouse_b){
     old_mouse_x =mouse_x;
     old_mouse_y =mouse_y;
     x_index = abs((mouse_x-6)/20);
     y_index = abs((mouse_y-52)/20);
     x_blit = (x_index*20+6)-20-1;
     y_blit = (y_index*20+52)-20-1;
     if (mouse_x >= 6 && mouse_x <= columns*20+6-1 && mouse_y >= 52 && mouse_y <= rows*20+52-1 && !pause_timer){
       if (mouse_b ==1 && board_mask[x_index][y_index] == SHOW_BUTTON )
          board_mask[x_index][y_index]=MOUSE_B1_DOWN;
       if (mouse_b ==3 || mouse_b ==4){
          old_mouse_b = 4;
          mouse_b_lock = TRUE;
          press_group(x_index,y_index);
       }
     }

     if (!mouse_b_lock && old_mouse_b != mouse_b && mouse_b > 0) old_mouse_b = mouse_b;
     check_mask();
     gfx_update();
   }
   gfx_update();
 }
 if (mouse_x >= 6 && mouse_x <= columns*20+6-1 && mouse_y >= 52 && mouse_y <= rows*20+52-1 && !pause_timer ){
   game_start=TRUE;
   switch (old_mouse_b)
   {
     case 1: if (board_map[x_index][y_index] ==SHOW_BLANK && !mouse_b_lock)
                uncover_blanks(x_index,y_index);
             else if (board_map[x_index][y_index] < HIDDEN_MINE && !mouse_b_lock)
                board_mask[x_index][y_index] =board_map[x_index][y_index];
             else if(board_map[x_index][y_index] == HIDDEN_MINE && !mouse_b_lock){
                board_mask[x_index][y_index] = TRIPPED_MINE;
                lose=TRUE;
                lose_game();
               }
             else if(board_map[x_index][y_index] == SHOW_BUTTON && !mouse_b_lock)
               board_mask[x_index][y_index] = SHOW_BUTTON;
             break;
     case 2: if (board_mask[x_index][y_index] ==SHOW_BUTTON && !mouse_b_lock)
               board_mask[x_index][y_index] = FLAG_SET;
             else if (board_mask[x_index][y_index] == FLAG_SET && !mouse_b_lock)
               board_mask[x_index][y_index] = QUESTION_SET;
             else if (board_mask[x_index][y_index] == QUESTION_SET && !mouse_b_lock)
               board_mask[x_index][y_index] = SHOW_BUTTON;
             break;
     case 3:
     case 4: if ((flag_sweeper(x_index,y_index)) == (int)board_map[x_index][y_index] && board_mask[x_index][y_index] != SHOW_BUTTON)
               uncover_around_flags(x_index,y_index);
             break;
   }
 }
 check_mask();
}

void check_mask(void)
{
 int x,y,x_blit,y_blit;
 int button_size = 20;     // buttons are 20x20
 int game_width ;          // game width in pixels
 int game_hieght;
 int win_check=0;
 game_width = columns * button_size + 10;
 game_hieght = rows * button_size + 40+15;
 flag_cnt =mine_num;
 
 show_mouse(NULL);
 rectfill(gfx_buf,0,0,columns*20+10,14,27);  // Display menubar
 blit(gamedata[MENU].dat,gfx_buf,0,0,0,0,90,15);

 rect(gfx_buf,0,15,game_width,16,21);
 rect(gfx_buf,0,15,1,game_hieght,21);
 rect(gfx_buf,0,game_hieght-1,game_width,game_hieght,16);
 rect(gfx_buf,game_width-1,game_hieght,game_width,15,16);
 rectfill(gfx_buf,2,17,game_width-2,game_hieght-2,27);

 // blit Smiley button
 if (!lose)
   blit(gamedata[SMILEY].dat,gfx_buf,0,0,game_width/2-15,19,30,30);
 else
   blit(gamedata[SMILEY_SAD].dat,gfx_buf,0,0,game_width/2-15,19,30,30);

 for (y=0;y<rows;y++)
  for (x=0;x<columns;x++){
    x_blit = (x+1)*20-20+5;
    y_blit = (y+1)*20-20+51;
    if (board_mask[x][y] !=SHOW_BUTTON && board_map[x][y] != HIDDEN_MINE && board_mask[x][y] !=MOUSE_B1_DOWN&& board_mask[x][y] !=FLAG_SET && board_mask[x][y] != QUESTION_SET)
      win_check ++;
    if (win_check == rows*columns - mine_num && !lose){
      pause_timer = TRUE;
      blit(gamedata[SMILEY_WIN].dat,gfx_buf,0,0,game_width/2-15,19,30,30);
      switch (game_level){
        case 1:if (game_timer < best_score_begin)
                 add_score();
               break;
        case 2:if (game_timer < best_score_medium)
                 add_score();
               break;
        case 3:if (game_timer < best_score_expert)
                 add_score();
               break;
      }
    }
    switch(board_mask[x][y]){
      case SHOW_BLANK       :blit(gamedata[BUTTON_PRESSED].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_ONE         :blit(gamedata[ONE].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_TWO         :blit(gamedata[TWO].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_THREE       :blit(gamedata[THREE].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_FOUR        :blit(gamedata[FOUR].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_FIVE        :blit(gamedata[FIVE].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_SIX         :blit(gamedata[SIX].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_SEVEN       :blit(gamedata[SEVEN].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_EIGHT       :blit(gamedata[EIGHT].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case MOUSE_B1_DOWN    :blit(gamedata[BUTTON_PRESSED].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             blit(gamedata[SMILEY_OH].dat,gfx_buf,0,0,game_width/2-15,19,30,30);
                             board_mask[x][y] = SHOW_BUTTON;
                             break;
      case HIDDEN_MINE      :blit(gamedata[MINE].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case FLAG_SET         :blit(gamedata[FLAG].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             if (flag_cnt)flag_cnt--;
                             break;
      case QUESTION_SET     :blit(gamedata[QUESTION].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_X           :blit(gamedata[X].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case TRIPPED_MINE     :blit(gamedata[SEL_MINE].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case SHOW_BUTTON      :blit(gamedata[BUTTON].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             break;
      case STOP_WALKING_LOOP:blit(gamedata[BUTTON_PRESSED].dat,gfx_buf,0,0,x_blit,y_blit,20,20);
                             board_mask[x][y] = SHOW_BLANK;
                             board_map[x][y] = SHOW_BLANK;
                             break;
    }
  }
  show_mouse(gfx_buf);
}

void press_group(int x, int y)
{
    if(board_mask[x][y] == SHOW_BUTTON)
      board_mask[x] [y] =MOUSE_B1_DOWN;

    if (x-1 >= 0 && y - 1 >= 0)
      if (board_mask[x-1][y-1] == SHOW_BUTTON)
        board_mask[x-1] [y-1] =MOUSE_B1_DOWN;

    if (x >= 0 && y - 1 >= 0)
      if (board_mask[x][y-1] == SHOW_BUTTON)
        board_mask[x]   [y-1] =MOUSE_B1_DOWN;

    if (x+1 <= columns-1 && y-1 >= 0)
      if (board_mask[x+1][y-1] == SHOW_BUTTON)
        board_mask[x+1] [y-1] =MOUSE_B1_DOWN;

    if (x+1 <= columns-1 && y <= rows-1)
      if (board_mask[x+1][y] == SHOW_BUTTON)
        board_mask[x+1] [y]   =MOUSE_B1_DOWN;

    if (x+1 <= columns-1 && y+1 <= rows-1)
      if (board_mask[x+1][y+1] == SHOW_BUTTON)
        board_mask[x+1] [y+1] =MOUSE_B1_DOWN;

    if (x <= columns-1 && y+1 <= rows-1)
      if (board_mask[x][y+1] == SHOW_BUTTON)
        board_mask[x]   [y+1] =MOUSE_B1_DOWN;

    if (x-1 >= 0 && y+1 <= rows-1)
      if (board_mask[x-1][y+1] == SHOW_BUTTON)
        board_mask[x-1] [y+1] =MOUSE_B1_DOWN;

    if (x-1 >= 0 && y <= rows-1)
      if (board_mask[x-1][y] == SHOW_BUTTON)
        board_mask[x-1] [y]   =MOUSE_B1_DOWN;
}

void uncover_blanks(int x, int y)
{
   board_mask[x][y] = SHOW_BLANK;
   board_map[x][y] = STOP_WALKING_LOOP;
   if (x-1 >= 0 && y - 1 >= 0 )
     if (board_map[x-1][y-1] == SHOW_BLANK){
       board_mask[x-1][y-1] = SHOW_BLANK;
       uncover_blanks(x-1,y-1);
     }
     else if (board_map[x-1][y-1] != SHOW_BLANK)
       board_mask[x-1][y-1] = board_map[x-1][y-1];

   if (x >= 0 && y - 1 >= 0)
     if (board_map[x][y-1] == SHOW_BLANK){
       board_mask[x][y-1]=SHOW_BLANK;
       uncover_blanks(x, y-1);
     }
     else if(board_map[x][y-1] != SHOW_BLANK)
       board_mask[x][y-1] = board_map[x][y-1];

   if (x+1 <= columns-1 && y-1 >= 0)
     if (board_map[x+1][y-1] == SHOW_BLANK){
       board_mask[x+1][y-1] = SHOW_BLANK;
       uncover_blanks(x+1,y-1);
     }
     else if (board_map[x+1][y-1] != SHOW_BLANK)
       board_mask[x+1][y-1] = board_map[x+1][y-1];

   if (x+1 <= columns-1 && y <= rows-1)
     if (board_map[x+1][y] == SHOW_BLANK){
       board_mask[x+1][y] = SHOW_BLANK;
       uncover_blanks(x+1, y);
     }
     else if(board_map[x+1][y] != SHOW_BLANK)
       board_mask[x+1][y] = board_map[x+1][y];

   if (x+1 <= columns-1 && y+1 <= rows-1)
     if (board_map[x+1][y+1] == SHOW_BLANK){
       board_mask[x+1][y+1] = SHOW_BLANK;
       uncover_blanks(x+1,y+1);
     }
     else if (board_map[x+1][y+1] != SHOW_BLANK)
       board_mask[x+1][y+1] = board_map[x+1][y+1];

   if (x <= columns-1 && y+1 <= rows-1)
     if (board_map[x][y+1] == SHOW_BLANK){
       board_mask[x][y+1] = SHOW_BLANK;
       uncover_blanks(x, y+1);
     }
     else if(board_map[x][y+1] != SHOW_BLANK)
       board_mask[x][y+1] = board_map[x][y+1];

   if (x-1 >= 0 && y+1 <= rows-1)
     if (board_map[x-1][y+1] == SHOW_BLANK){
       board_mask[x-1][y+1] =SHOW_BLANK;
       uncover_blanks(x-1,y+1);
     }
     else if (board_map[x-1][y+1] != SHOW_BLANK)
       board_mask[x-1][y+1] = board_map[x-1][y+1];
     
   if (x-1 >= 0 && y <= rows-1)
     if (board_map[x-1][y] == SHOW_BLANK){
       board_mask[x-1][y] = SHOW_BLANK;
       uncover_blanks(x-1, y);
     }
     else if(board_map[x-1][y] != SHOW_BLANK)
       board_mask[x-1][y] = board_map[x-1][y];
}

int flag_sweeper(int x, int y)
{
  int cnt = 0;
  if (x-1 >= 0 && y - 1 >= 0)
    if (board_mask[x-1][y-1] == FLAG_SET) cnt++;
  if (x >= 0 && y - 1 >= 0)
    if (board_mask[x][y-1]   == FLAG_SET) cnt++;
  if (x+1 <= columns-1 && y-1 >= 0)
    if (board_mask[x+1][y-1] == FLAG_SET) cnt++;
  if (x+1 <= columns-1 && y <= rows-1)
    if (board_mask[x+1][y]   == FLAG_SET) cnt++;
  if (x+1 <= columns-1 && y+1 <= rows-1)
    if (board_mask[x+1][y+1] == FLAG_SET) cnt++;
  if (x <= columns-1 && y+1 <= rows-1)
    if (board_mask[x][y+1]   == FLAG_SET) cnt ++;
  if (x-1 >= 0 && y+1 <= rows-1)
    if (board_mask[x-1][y+1] == FLAG_SET) cnt ++;
  if (x-1 >= 0 && y <= rows-1)
    if (board_mask[x-1][y]   == FLAG_SET) cnt ++;
  return cnt;
}

void uncover_around_flags(int x, int y)
{
    if (x-1 >= 0 && y - 1 >= 0 ){
     if (board_mask[x-1][y-1] == FLAG_SET && board_map[x-1][y-1] != HIDDEN_MINE)
       board_mask[x-1][y-1] = SHOW_X;
     else if (board_mask[x-1][y-1] == SHOW_BUTTON && board_map[x-1][y-1] == HIDDEN_MINE){
       board_mask[x-1][y-1] = TRIPPED_MINE;
       lose = TRUE;
     }
     else if (board_map[x-1][y-1] == SHOW_BLANK)
       uncover_blanks(x-1,y-1);
     else if (board_map[x-1][y-1] != HIDDEN_MINE)
       board_mask[x-1][y-1] = board_map[x-1][y-1];
   }
   if (x >= 0 && y - 1 >= 0 ){
     if (board_mask[x][y-1] == FLAG_SET && board_map[x][y-1] != HIDDEN_MINE)
       board_mask[x][y-1] = SHOW_X;
     else if (board_mask[x][y-1] == SHOW_BUTTON && board_map[x][y-1] == HIDDEN_MINE){
       board_mask[x][y-1] = TRIPPED_MINE;
       lose = TRUE;
     }
     else if (board_map[x][y-1] == SHOW_BLANK)
       uncover_blanks(x,y-1);
     else if (board_map[x][y-1] != HIDDEN_MINE)
       board_mask[x][y-1] = board_map[x][y-1];
   }
   if (x+1 <= columns-1 && y-1 >= 0 ){
     if (board_mask[x+1][y-1] == FLAG_SET && board_map[x+1][y-1] != HIDDEN_MINE)
       board_mask[x+1][y-1] = SHOW_X;
     else if (board_mask[x+1][y-1] == SHOW_BUTTON && board_map[x+1][y-1] == HIDDEN_MINE){
       board_mask[x+1][y-1] = TRIPPED_MINE;
       lose = TRUE;
     }
     else if (board_map[x+1][y-1] == SHOW_BLANK)
       uncover_blanks(x+1,y-1);
     else if (board_map[x+1][y-1] != HIDDEN_MINE)
       board_mask[x+1][y-1] = board_map[x+1][y-1];
   }
   if (x+1 <= columns-1 && y <= rows-1 ){
     if (board_mask[x+1][y] == FLAG_SET && board_map[x+1][y] != HIDDEN_MINE)
       board_mask[x+1][y] = SHOW_X;
     else if (board_mask[x+1][y] == SHOW_BUTTON && board_map[x+1][y] == HIDDEN_MINE){
       board_mask[x+1][y] = TRIPPED_MINE;
       lose = TRUE;
     }
     else if (board_map[x+1][y] == SHOW_BLANK)
       uncover_blanks(x+1,y);
     else if (board_map[x+1][y] != HIDDEN_MINE)
       board_mask[x+1][y] = board_map[x+1][y];
   }
   if (x+1 <= columns-1 && y+1 <= rows-1){
     if (board_mask[x+1][y+1] == FLAG_SET && board_map[x+1][y+1] != HIDDEN_MINE)
       board_mask[x+1][y+1] = SHOW_X;
     else if (board_mask[x+1][y+1] == SHOW_BUTTON && board_map[x+1][y+1] == HIDDEN_MINE){
       board_mask[x+1][y+1] = TRIPPED_MINE;
       lose = TRUE;
     }
     else if (board_map[x+1][y+1] == SHOW_BLANK)
       uncover_blanks(x+1,y+1);
     else if (board_map[x+1][y+1] != HIDDEN_MINE)
       board_mask[x+1][y+1] = board_map[x+1][y+1];
   }
   if (x <= columns-1 && y+1 <= rows-1){
     if (board_mask[x][y+1] == FLAG_SET && board_map[x][y+1] != HIDDEN_MINE)
       board_mask[x][y+1] = SHOW_X;
     else if (board_mask[x][y+1] == SHOW_BUTTON && board_map[x][y+1] == HIDDEN_MINE){
       board_mask[x][y+1] = TRIPPED_MINE;
       lose = TRUE;
     }
     else if (board_map[x][y+1] == SHOW_BLANK)
       uncover_blanks(x,y+1);
     else if (board_map[x][y+1] != HIDDEN_MINE)
       board_mask[x][y+1] = board_map[x][y+1];
   }
   if (x-1 >= 0 && y+1 <= rows-1){
     if (board_mask[x-1][y+1] == FLAG_SET && board_map[x-1][y+1] != HIDDEN_MINE)
       board_mask[x-1][y+1] = SHOW_X;
     else if (board_mask[x-1][y+1] == SHOW_BUTTON && board_map[x-1][y+1] == HIDDEN_MINE){
       board_mask[x-1][y+1] = TRIPPED_MINE;
       lose = TRUE;
     }
     else if (board_map[x-1][y+1] == SHOW_BLANK)
       uncover_blanks(x-1,y+1);
     else if (board_map[x-1][y+1] != HIDDEN_MINE)
       board_mask[x-1][y+1] = board_map[x-1][y+1];
   }
   if (x-1 >= 0 && y <= rows-1 ){
     if (board_mask[x-1][y] == FLAG_SET && board_map[x-1][y] != HIDDEN_MINE)
       board_mask[x-1][y] = SHOW_X;
     else if (board_mask[x-1][y] == SHOW_BUTTON && board_map[x-1][y] == HIDDEN_MINE){
       board_mask[x-1][y] = TRIPPED_MINE;
       lose = TRUE;
     }
     else if (board_map[x-1][y] == SHOW_BLANK)
       uncover_blanks(x-1,y);
      else if (board_map[x-1][y] != HIDDEN_MINE)
       board_mask[x-1][y] = board_map[x-1][y];
   }
   if (lose) lose_game ();
}

void lose_game (void)
{
 int x,y;
 game_start = FALSE;
 for(y=0;y<rows;y++)
  for(x=0;x<columns;x++){
   if (board_map[x][y] == HIDDEN_MINE && board_mask[x][y] != FLAG_SET && board_mask[x][y] != TRIPPED_MINE)
     board_mask[x][y] = board_map[x][y];
   else if (board_map[x][y] != HIDDEN_MINE && board_mask[x][y] == FLAG_SET)
     board_mask[x][y] = SHOW_X;
  }
}

void led_print(int num,int x,int y)
{
 int x_blit, y_blit,cnt,old_num,num_buf[3];
 old_num = num;
 x_blit = x;
 y_blit = y;

 if (old_num > 99){
    num_buf[0] = num / 100;
    num = num - num_buf[0]*100;
 }
 else
    num_buf[0] = 10;

 if (old_num > 9){
   num_buf[1] = num /10;
   num = num - num_buf[1]*10;
 }
 else
   num_buf[1] = 10;

 num_buf[2] = num;
 show_mouse(NULL);
 for(cnt =0; cnt < 3; cnt++,x_blit += 15)
  {
    switch(num_buf[cnt]){
      case 0:blit(gamedata[LED_0].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 1:blit(gamedata[LED_1].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 2:blit(gamedata[LED_2].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 3:blit(gamedata[LED_3].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 4:blit(gamedata[LED_4].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 5:blit(gamedata[LED_5].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 6:blit(gamedata[LED_6].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 7:blit(gamedata[LED_7].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 8:blit(gamedata[LED_8].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 9:blit(gamedata[LED_9].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
      case 10:blit(gamedata[LED_BLANK].dat,gfx_buf,0,0,x_blit,y_blit,15,20);
             break;
     }
  }
  show_mouse(gfx_buf);
}

void update_timer(void)
{
  static int sec_monitor;
  int game_width = columns * 20 + 10;
  int x_timer,y_timer;
  x_timer = (game_width) - ((3*15)+5);
  y_timer = 24;
  if (game_start == FALSE){
    sec_monitor = time(0) % 2;
    game_timer=0;
  }
  else if (game_start == TRUE && sec_monitor != time(0) % 2){
    sec_monitor = time(0) % 2;
    if (!pause_timer)
       game_timer++;
  }
  led_print(game_timer,x_timer,y_timer);
  if (!menu_open)led_print(flag_cnt,5,24);
}

int custom_menu(void)
{
  int x_blit = x_rev /2 -(128/2);
  int y_blit = y_rev /2 -(128/2);
  int x_ok = x_blit +10;
  int x_cancel = x_blit +70;
  int y_buttons = y_blit+100;
  int x_cur = x_blit + 62;
  int y_cur = y_blit + 8;
  int cur_update = 0;
  int textbox =1;
  int str2int;
  int cols_buf,rows_buf,mines_buf;
  char key_buf[2];
  char cus_rows[3]={""},cus_cols[3]={""}, cus_mines[4]={""};
  
  char *ptr;

  show_mouse(NULL);
  blit(gamedata[CUSTOM].dat, gfx_buf,0,0,x_blit,y_blit,128,128);
  blit(gamedata[BUTTON_OK].dat, gfx_buf,0,0,x_ok,y_buttons,46,17);
  blit(gamedata[BUTTON_CANCEL].dat, gfx_buf,0,0,x_cancel,y_buttons,46,17);
  show_mouse(screen);
  while (1){
    if (mouse_x >=x_blit+60 && mouse_x <=x_blit +93 && mouse_y >= y_blit+7 && mouse_y <=y_blit+19 && mouse_b ==1){
       vline(gfx_buf, x_cur, y_cur, y_cur + 9,10);
       textbox =1;
       y_cur = y_blit + 8;
    }
    else if (mouse_x >=x_blit+60 && mouse_x <=x_blit+93 && mouse_y >= y_blit+27 && mouse_y <=y_blit+38 && mouse_b ==1){
       vline(gfx_buf, x_cur, y_cur, y_cur + 9,10);
       textbox =2;
       y_cur= y_blit + 27;
    }
    else if (mouse_x >=x_blit+60 && mouse_x <=x_blit+93 && mouse_y >= y_blit+45 && mouse_y <=y_blit+56 && mouse_b ==1){
       vline(gfx_buf, x_cur, y_cur, y_cur + 9,10);
       textbox =3;
       y_cur= y_blit + 45;
    }
	  vline(gfx_buf, x_cur, y_cur, y_cur + 9,26);

     if (keypressed()){
      key_buf[0] = readkey();
      sprintf(key_buf,"%c",key_buf[0]);
      str2int =atoi(key_buf);
      switch(textbox){
        case 1: if (str2int >=1 && strlen(cus_rows) < 2 || key_buf[0] == 48 && strlen(cus_rows) < 2 )
                  strcat(cus_rows,key_buf);
                else if(key[KEY_DEL] && strlen(cus_rows) > 0){
                  ptr = cus_rows;
                  strcpy(cus_rows,ptr+1);
                  show_mouse(NULL);
                  textprintf(gfx_buf,font,x_cur+2,y_cur+1,2,"%s","  ");
                  show_mouse(screen);
                }
                break;
        case 2: if (str2int >=1 && strlen(cus_cols) < 2 || key_buf[0] == 48 && strlen(cus_cols) < 2 )
                  strcat(cus_cols,key_buf);
                else if(key[KEY_DEL] && strlen(cus_cols) > 0){
                  ptr = cus_cols;
                  strcpy(cus_cols,ptr+1);
                  show_mouse(NULL);
                  textprintf(gfx_buf,font,x_cur+1,y_cur+1,2,"%s","  ");
                  show_mouse(screen);
                }
                break;
        case 3: if (str2int >=1 && strlen(cus_mines) < 3 || key_buf[0] == 48 && strlen(cus_mines) < 3 )
                  strcat(cus_mines,key_buf);
                else if(key[KEY_DEL] && strlen(cus_mines) > 0){
                 ptr = cus_mines;
                 strcpy(cus_mines,ptr+1);
                 show_mouse(NULL);
                 textprintf(gfx_buf,font,x_cur+1,y_cur+1,2,"%s","   ");
                 show_mouse(screen);
                }
                break;
      }
    }
    if (mouse_x >x_ok && mouse_x <= x_ok +46 && mouse_y > y_buttons && mouse_y <=y_buttons +17 && mouse_b ==1){
      show_mouse(NULL);
      blit(gamedata[BUTTON_OK_PRESSED].dat, gfx_buf,0,0,x_ok,y_buttons,46,17);
      show_mouse(screen);
      while (mouse_b==1)
         gfx_update();
      show_mouse(NULL);
      blit(gamedata[BUTTON_OK].dat, gfx_buf,0,0,x_ok,y_buttons,46,17);
      show_mouse(screen);
      // OK button pressed
      if (mouse_x >x_ok && mouse_x <= x_ok +46 && mouse_y > y_buttons && mouse_y <=y_buttons +17){
        cols_buf=atoi(cus_cols);
        rows_buf=atoi(cus_rows);
        mines_buf=atoi(cus_mines);
        switch (x_rev){
          case 640:if (cols_buf>31)
                     cols_buf=31;
                   if (rows_buf>21)
                     rows_buf=21;
                   break;
          case 800:if (cols_buf>39)
                     cols_buf=39;
                   if (rows_buf>27)
                     rows_buf=27;
                   break;
          case 1024:if (cols_buf>50)
                     cols_buf=50;
                   if (rows_buf>35)
                     rows_buf=35;
                   break;
        }
        if (cols_buf <7)
          cols_buf=7;
        if (rows_buf <7)
          rows_buf=7;
        if (mines_buf > (cols_buf*rows_buf))
          mines_buf = cols_buf*rows_buf;
        if (mines_buf <1)
          mines_buf = 1;
        columns = cols_buf;
        rows = rows_buf;
        mine_num = mines_buf;
        return 1;
      }
    }
    // cancel button pressed
    if (mouse_x >x_cancel && mouse_x <= x_cancel +46 && mouse_y > y_buttons && mouse_y <=y_buttons +17 && mouse_b ==1){
      show_mouse(NULL);
      blit(gamedata[BUTTON_CANCEL_PRESSED].dat, gfx_buf,0,0,x_cancel,y_buttons,46,17);
      show_mouse(screen);
      while (mouse_b==1)
         gfx_update();
      show_mouse(NULL);
      blit(gamedata[BUTTON_CANCEL].dat, gfx_buf,0,0,x_cancel,y_buttons,46,17);
      show_mouse(screen);
      if (mouse_x >x_cancel && mouse_x <= x_cancel +46 && mouse_y > y_buttons && mouse_y <=y_buttons +17)
        show_mouse(NULL);
        clear(gfx_buf);
        show_mouse(screen);
        check_mask();
        return 0;
    }
    textprintf(gfx_buf,font,x_blit + 64,y_blit +9,2,"%s",cus_rows);
    textprintf(gfx_buf,font,x_blit + 64,y_blit +28,2,"%s",cus_cols);
    textprintf(gfx_buf,font,x_blit + 64,y_blit +47,2,"%s",cus_mines);
    gfx_update();
  }

}

void load_settings(void)
{
  FILE *file_ptr;
//  char buf[25];
  if(exists(GAME_SETTINGS)){
    file_ptr=fopen(GAME_SETTINGS,"rt");
    
    fscanf(file_ptr,"%4d",&x_rev);
    fscanf(file_ptr,"%4d",&y_rev);

    fscanf(file_ptr,"%2d",&columns);
    fscanf(file_ptr,"%2d",&rows);
    fscanf(file_ptr,"%3d",&mine_num);
    fscanf(file_ptr,"%1d",&game_level);

    fscanf(file_ptr,"%3d",&best_score_begin);
    fscanf(file_ptr,"%s",champ_name_begin);

    fscanf(file_ptr,"%3d",&best_score_medium);
    fscanf(file_ptr,"%s",champ_name_medium);

    fscanf(file_ptr,"%3d",&best_score_expert);
    fscanf(file_ptr,"%s",champ_name_expert);

    fclose(file_ptr);
  }
}

void save_settings(void)
{
  FILE *file_ptr;
  file_ptr = fopen(GAME_SETTINGS,"wt");
  fseek(file_ptr,0,SEEK_SET);

  fprintf(file_ptr,"%-4d\n",x_rev);
  fprintf(file_ptr,"%-4d\n",y_rev);

  fprintf(file_ptr,"%-2d\n",columns);
  fprintf(file_ptr,"%-2d\n",rows);
  fprintf(file_ptr,"%-3d\n",mine_num);
  fprintf(file_ptr,"%-1d\n",game_level);

  fprintf(file_ptr,"%-3d\n",best_score_begin);
  fprintf(file_ptr,"%-10s\n",champ_name_begin);

  fprintf(file_ptr,"%-3d\n",best_score_medium);
  fprintf(file_ptr,"%-10s\n",champ_name_medium);

  fprintf(file_ptr,"%-3d\n",best_score_expert);
  fprintf(file_ptr,"%-10s\n",champ_name_expert);

  fclose(file_ptr);
}

int show_score(void)
{
  //  320x160
  int x_blit = x_rev /2 -(320/2);
  int y_blit = y_rev /2 -(160/2);
  int y_buttons = y_blit+120;
  int x_reset = x_blit +60;
  int x_ok = x_blit + 200;

  show_mouse(NULL);
  blit(gamedata[SCORES].dat, gfx_buf,0,0,x_blit,y_blit,320,160);
  blit(gamedata[BUTTON_OK].dat, gfx_buf,0,0,x_ok,y_buttons,46,17);
  blit(gamedata[BUTTON_RESET].dat, gfx_buf,0,0,x_reset,y_buttons,92,17);
  textprintf(gfx_buf,font,x_blit +110,y_blit + 38,2,"%-3d seconds   %-10s",best_score_begin,champ_name_begin);
  textprintf(gfx_buf,font,x_blit +110,y_blit + 63,2,"%-3d seconds   %-10s",best_score_medium,champ_name_medium);
  textprintf(gfx_buf,font,x_blit +110,y_blit + 86,2,"%-3d seconds   %-10s",best_score_expert,champ_name_expert);
  show_mouse(screen);
  while(1){
    if (mouse_x > x_reset && mouse_x <=x_reset + 92 && mouse_y > y_buttons && mouse_y <= y_buttons +17 && mouse_b == 1){
      show_mouse(NULL);
      blit(gamedata[BUTTON_RESET_PRESSED].dat, gfx_buf,0,0,x_reset,y_buttons,92,17);
      show_mouse(screen);
      while(mouse_b == 1)
       gfx_update();

      show_mouse(NULL);
      blit(gamedata[BUTTON_RESET].dat, gfx_buf,0,0,x_reset,y_buttons,92,17);
      show_mouse(screen);

      if (mouse_x > x_reset && mouse_x <=x_reset + 92 && mouse_y > y_buttons && mouse_y <= y_buttons +17){
        best_score_begin  = 999;
        best_score_medium = 999;
        best_score_expert = 999;
        strcpy(champ_name_begin,"Anonymous");
        strcpy(champ_name_medium,"Anonymous");
        strcpy(champ_name_expert,"Anonymous");
        show_mouse(NULL);
        textprintf(gfx_buf,font,x_blit +110,y_blit + 38,2,"%-3d seconds   %-10s",best_score_begin,champ_name_begin);
        textprintf(gfx_buf,font,x_blit +110,y_blit + 63,2,"%-3d seconds   %-10s",best_score_medium,champ_name_medium);
        textprintf(gfx_buf,font,x_blit +110,y_blit + 86,2,"%-3d seconds   %-10s",best_score_expert,champ_name_expert);
        show_mouse(screen);
        save_settings();
      }
    }
    gfx_update();
    if (mouse_x > x_ok && mouse_x <=x_ok + 46 && mouse_y > y_buttons && mouse_y <= y_buttons +17 && mouse_b == 1){
      show_mouse(NULL);
      blit(gamedata[BUTTON_OK_PRESSED].dat, gfx_buf,0,0,x_ok,y_buttons,46,17);
      show_mouse(screen);
      while(mouse_b == 1)
       gfx_update();

      show_mouse(NULL);
      blit(gamedata[BUTTON_OK].dat, gfx_buf,0,0,x_ok,y_buttons,46,17);
      show_mouse(screen);
      if (mouse_x > x_ok && mouse_x <=x_ok + 46 && mouse_y > y_buttons && mouse_y <= y_buttons +17){
        show_mouse(NULL);
        clear(gfx_buf);
        show_mouse(screen);
        check_mask();
        return 0;
      }
    }
  }
}


int add_score(void)
{
  int x_blit = x_rev /2 -(200/2);
  int y_blit = y_rev /2 -(100/2);
  int x_ok = x_blit + 5;
  int y_ok = y_blit+79;
  int exit_loop = FALSE;
  int x_cur = x_blit + 15;
  int y_cur = y_blit + 60;
  int cur_update = 0;
  char name_buf[11]={""};
  char level_name[15];
  char key_buf[2];
  char *ptr;

  switch (game_level){
    case 1: strcpy(level_name,"beginner");
            break;
    case 2: strcpy(level_name,"intermediate");
            break;
    case 3: strcpy(level_name,"expert");
            break;
  }
  show_mouse(NULL);
  blit(gamedata[SET_SCORE].dat, gfx_buf,0,0,x_blit,y_blit,250,100);
  blit(gamedata[BUTTON_OK].dat, gfx_buf,0,0,x_ok,y_ok,46,17);
  textout(gfx_buf,font,"You have the fastest time for",x_blit+7,y_blit+22,2);
  textprintf(gfx_buf,font,x_blit+7,y_blit+32,2,"the %s level.",level_name);
  textout(gfx_buf,font,"Please type your name.",x_blit+7,y_blit+42,2);
  show_mouse(screen);
  while(1){
    vline(gfx_buf, x_cur, y_cur, y_cur + 9,26);
    if (keypressed()){
      key_buf[0] = readkey();
      sprintf(key_buf,"%c",key_buf[0]);
      if (strlen(name_buf) < 10 && key_buf[0] >= '!' && key_buf[0] <= '~')
        strcat(name_buf,key_buf);
      if(key[KEY_DEL] && strlen(name_buf) > 0){
        ptr = name_buf;
        strcpy(name_buf,ptr+1);
        show_mouse(NULL);
        textprintf(gfx_buf,font,x_cur+2,y_cur+1,2,"%s","            ");
        show_mouse(screen);
      }
      if (key[KEY_ESC])
        return 0;
      if (key[KEY_ENTER] && strlen(name_buf) >0){
        switch (game_level){
          case 1:best_score_begin = game_timer ;
                 strcpy(champ_name_begin,name_buf);
                 break;
          case 2:best_score_medium = game_timer ;
                 strcpy(champ_name_medium,name_buf);
                 break;
          case 3:best_score_expert = game_timer ;
                 strcpy(champ_name_expert,name_buf);
                 break;
        }
        save_settings();
        show_mouse(NULL);
        clear(gfx_buf);
        show_mouse(screen);
        check_mask();
        return 0;
      }
    }
    if (mouse_x >x_ok && mouse_x <= x_ok+46 && mouse_y > y_ok && mouse_y <=y_ok +17 && mouse_b ==1){
      show_mouse(NULL);
      blit(gamedata[BUTTON_OK_PRESSED].dat, gfx_buf,0,0,x_ok,y_ok,46,17);
      show_mouse(screen);
      while (mouse_b==1)
        gfx_update();
      if (mouse_x >x_ok && mouse_x <= x_ok+46 && mouse_y > y_ok && mouse_y <=y_ok +17 && strlen(name_buf) >0){
        switch (game_level){
          case 1:best_score_begin = game_timer ;
                 strcpy(champ_name_begin,name_buf);
                 break;
          case 2:best_score_medium = game_timer ;
                 strcpy(champ_name_medium,name_buf);
                 break;
          case 3:best_score_expert = game_timer ;
                 strcpy(champ_name_expert,name_buf);
                 break;
        }
        save_settings();
        show_mouse(NULL);
        clear(gfx_buf);
        show_mouse(screen);
        check_mask();
        return 0;
      }
      else {
      show_mouse(NULL);
      blit(gamedata[BUTTON_OK].dat, gfx_buf,0,0,x_ok,y_ok,46,17);
      show_mouse(screen);
      }
    }
    textprintf(gfx_buf,font,x_cur+2,y_cur+1,1,"%s",name_buf);
    gfx_update();
  }
}

void show_help (void)
{
}

void update_gfx_speed_control(void)
{
  if(!gfx_speed_control)gfx_speed_control++;
}

