/**\file n_sound.c
*
*  Header to include for use sounds
*
*\author Castagnier Mickal
*
*\version 1.0
*
*\date 24/03/05
*/



#include "nilorea.h"



/*!\fn create_libsound( LIBSOUND **lib , char *name , int max )
 *
 *\brief Create an empty LIBSOUND *lib, with 'max' element allowed
 *
 *\param lib An LIBSOUND *library to create
 *\param name A name for your library
 *\param max The maximum of elements allowed
 *
 *\return TRUE or FALSE
 */

int create_libsound( LIBSOUND **lib , char *name , int max )
    {

    int it;

    if ( ( *lib ) )
        return FALSE;

    Malloc( ( *lib ) , LIBSOUND , 1 );

    if ( !( *lib ) )
        return FALSE;

    Malloc( ( *lib ) -> data , SOUND , max );

    if ( !( *lib ) -> data )
        return FALSE;

    for ( it = 0 ; it < max ; it ++ ) {
            ( *lib ) -> data[ it ] . state = NONE ;
            ( *lib ) -> data[ it ] . sample = NULL ;
            ( *lib ) -> data[ it ] . name = NULL ;
            }


    if ( name ) {
            Malloc( ( *lib ) -> name , char , ustrlen( name ) );

            if ( !( *lib ) -> name )
                return FALSE;

            ustrcpy( ( *lib ) -> name , name );

            }

    ( *lib ) -> max = max;
    ( *lib ) -> current = 0 ;

    return TRUE;

    } /* create_libsound(...) */



/*!\fn destroy_libsound( LIBSOUND **lib )
 *
 *\brief Destroy a sound library
 *
 *\param lib The lib to destroy
 *
 *\return TRUE of FALSE
 */

int destroy_libsound( LIBSOUND **lib )
    {

    int it = 0 ;

    if ( !( *lib ) )
        return FALSE;

    if ( ( *lib ) -> name )
        Free( ( *lib ) -> name )

        for ( it = 0 ; it < ( *lib ) -> current ; it ++ )
            remove_sample_from( &( *lib ) , it );

    Free( ( *lib ) -> data );

    Free( ( *lib ) );

    return TRUE;

    } /* destroy_libsound( ... ) */



/*!\fn add_sample_to_lib( LIBSOUND **lib , SAMPLE *sample , char *name )
 *
 *\brief Add a sample to a library
 *
 *\param lib The aimed library
 *\param sample A sample to add to the library
 *\param name The name of the sample
 *
 *\return The index of the sample in the library, -1 if error or full
 */

int add_sample_to_lib( LIBSOUND **lib , SAMPLE *sample , char *name )
    {

    int current = 0 ,
                  done = 0;

    if ( !( *lib ) || !( *lib ) -> data || !sample )
        return -1;

    while ( current < ( *lib ) -> max && done == 0 ) {
            if ( ( *lib ) -> data[ current ] . state != NONE )
                current++;
            else
                done = 1;
            }

    if ( current >= ( *lib ) -> max )
        return -1;


    ( *lib ) -> data[ current ] . sample = sample;

    ( *lib ) -> data[ current ] . state = IDLE ;

    if ( name ) {
            Malloc( ( *lib ) -> data[ current ] . name , char , uconvert_size( name , U_CURRENT, U_UNICODE ) );
            ( *lib ) -> data[ current ] . state = IDLE;
            ustrcpy( ( *lib ) -> data[ current ] . name , name );
            }

    if ( current >= ( *lib ) -> current ) {
            if ( ( *lib ) -> current < ( *lib ) -> max )
                ( *lib ) -> current = ( *lib ) -> current + 1 ;
            }


    return current;


    } /* add_sample_to_lib( ... ) */



/*!\fn replace_sample( LIBSOUND **lib , int id , SAMPLE *sample )
 *
 *\brief Free and replace a sample in a library
 *
 *\param lib The library to modify
 *\param id The position of the library
 *\param sample The new sample
 *
 *\return TRUE of FALSE
 */

int replace_sample( LIBSOUND **lib , int id , SAMPLE *sample )
    {

    if ( id >= ( *lib ) -> max )
        return FALSE;

    if ( ( *lib ) -> data[ id ] . state != NONE )
        destroy_sample( ( *lib ) -> data[ id ] . sample );

    ( *lib ) -> data[ id ] . sample = sample;

    ( *lib ) -> data[ id ] . state = IDLE;

    return TRUE;

    } /* replace_sample( ... ) */



/*!\fn remove_sample_from( LIBSOUND **lib , int id )
 *
 *\brief Remove a sample from the library
 *
 *\param lib The library where delete the sample
 *\param id The id of the sample to remove
 *
 *\return TRUE of FALSE
 */

int remove_sample_from( LIBSOUND **lib , int id )
    {

    if ( !( *lib ) )
        return FALSE;


    if ( ( *lib ) -> data[ id ] . sample )
        destroy_sample( ( *lib ) -> data[ id ] . sample );

    if ( ( *lib ) -> data[ id ] . name )
        Free( ( *lib ) -> data[ id ] . name );

    ( *lib ) -> data[ id ] . state = NONE;


    return TRUE;

    } /* remove_sample_from( ... ) */



/*!\fn save_sample_in_datafile( SAMPLE *spl , PACKFILE *f )
 *
 *\brief Save a loaded sample in the specified opened datafile
 *
 *\param spl The SAMPLE *sample to save
 *\param f An Opened PACKFILE *file
 *
 *\return TRUE or FALSE
 */

int save_sample_in_datafile( SAMPLE *spl , PACKFILE *f )
    {
    *allegro_errno = 0;

    pack_mputw( ( spl->stereo ) ? -spl->bits : spl->bits, f );
    pack_mputw( spl->freq, f );
    pack_mputl( spl->len, f );

    if ( spl->bits == 8 ) {
            pack_fwrite( spl->data, spl->len * ( ( spl->stereo ) ? 2 : 1 ), f );
            }

    else {
            int i;

            for ( i = 0; i < ( int ) spl->len * ( ( spl->stereo ) ? 2 : 1 ); i++ ) {
                    pack_iputw( ( ( int16_t * ) spl->data ) [ i ], f );
                    }
            }

    if ( *allegro_errno )
        return FALSE;
    else
        return TRUE;
    } /* save_sample_in_datafile( ... ) */



/*!\fn save_libsound( LIBSOUND *lib , PACKFILE *file )
 *
 *\brief Save a sound library in an opened packfile
 *
 *\param lib The library to save
 *\param file The opened packfile where to save the library
 *
 *\return TRUE of FALSE
 */

int save_libsound( LIBSOUND *lib , PACKFILE *file )
    {

    int it;

    if ( !lib || !file )
        return FALSE;

    pack_iputl( lib -> current , file );

    pack_iputl( lib -> max , file );

    pack_iputl( uconvert_size( lib -> name , U_CURRENT, U_UNICODE ) , file );

    pack_fwrite( lib -> name , uconvert_size( lib -> name , U_CURRENT, U_UNICODE ) , file );

    for ( it = 0 ; it < lib -> current ; it ++ ) {
            if ( lib -> data[ it ] . state != NONE ) {
                    pack_iputl( 1 , file );
                    pack_iputl( uconvert_size( lib -> data[ it ] . name , U_CURRENT, U_UNICODE ) , file );
                    pack_fwrite( lib -> data[ it ] . name , uconvert_size( lib -> data[ it ] . name , U_CURRENT, U_UNICODE ) , file );
                    save_sample_in_datafile( lib -> data[ it ] . sample , file );
#ifdef DEBUGMODE

                    fprintf( stderr , "%s saved at %d\n" , lib -> data[ it ] . name , it );
#endif

                    }

            else
                pack_iputl( 0 , file );

            }

    return TRUE;


    } /* save_libsound( ... ) */



/*!\fn load_sample_from_datafile( PACKFILE *f)
 *
 *\brief Load a sample from a datafile
 *
 *\param f An opened packfile where to load the sample
 *
 *\return SAMPLE or NULL
 */

SAMPLE *load_sample_from_datafile( PACKFILE *f )
    {
    char buffer[ 25 ];
    int i;
    int length, len;
    int freq = 22050;
    int bits = 8;
    int channels = 1;
    int s;
    SAMPLE *spl = NULL;
    ASSERT( f );

    memset( buffer, 0, sizeof buffer );

    pack_fread( buffer, 12, f );          /* check RIFF header */

    if ( memcmp( buffer, "RIFF", 4 ) || memcmp( buffer + 8, "WAVE", 4 ) )
        goto getout;

    while ( TRUE ) {
            if ( pack_fread( buffer, 4, f ) != 4 )
                break;

            length = pack_igetl( f );          /* read chunk length */

            if ( memcmp( buffer, "fmt ", 4 ) == 0 ) {
                    i = pack_igetw( f );            /* should be 1 for PCM data */
                    length -= 2;

                    if ( i != 1 )
                        goto getout;

                    channels = pack_igetw( f );     /* mono or stereo data */

                    length -= 2;

                    if ( ( channels != 1 ) && ( channels != 2 ) )
                        goto getout;

                    freq = pack_igetl( f );         /* sample frequency */

                    length -= 4;

                    pack_igetl( f );                /* skip six bytes */

                    pack_igetw( f );

                    length -= 6;

                    bits = pack_igetw( f );         /* 8 or 16 bit data? */

                    length -= 2;

                    if ( ( bits != 8 ) && ( bits != 16 ) )
                        goto getout;
                    }

            else if ( memcmp( buffer, "data", 4 ) == 0 ) {
                    len = length / channels;

                    if ( bits == 16 )
                        len /= 2;

                    spl = create_sample( bits, ( ( channels == 2 ) ? TRUE : FALSE ), freq, len );

                    if ( spl ) {
                            if ( bits == 8 ) {
                                    if ( pack_fread( spl->data, length, f ) < length ) {
                                            destroy_sample( spl );
                                            spl = NULL;
                                            }
                                    }

                            else {
                                    for ( i = 0; i < len*channels; i++ ) {
                                            if ( ( s = pack_igetw( f ) ) == EOF ) {
                                                    destroy_sample( spl );
                                                    spl = NULL;
                                                    break;
                                                    }

                                            ( ( signed short * ) spl->data ) [ i ] = ( signed short ) s ^ 0x8000;
                                            }
                                    }

                            length = 0;
                            }
                    }

            while ( length > 0 ) {             /* skip the remainder of the chunk */

                    if ( pack_getc( f ) == EOF )
                        break;

                    length--;
                    }
            }

getout:
    return spl;

    } /* load_sample_from_datafile( ... ) */



/*!\fn load_libsound( LIBSOUND **lib , PACKFILE *file )
 *
 *\brief Load a LIBSOUND *lib in the given opened PACKFILE *file
 *
 *\param lib An LIBSOUND *library to fill
 *\param file An opened PACKFILE *file
 *
 *\return TRUE or FALSE
 */

int load_libsound( LIBSOUND **lib , PACKFILE *file )
    {

    int it = 0, it1 = 0;

    if ( !file )
        return FALSE;

    if ( ( *lib ) )
        destroy_libsound( &( *lib ) );

    it = pack_igetl( file );

    it1 = pack_igetl( file );

    fprintf( stderr, "loading %d %d\n", it, it1 );

    if ( create_libsound( &( *lib ) , NULL , it1 ) == FALSE )
        return FALSE;

    ( *lib ) -> current = it ;

    it = pack_igetl( file );

    Malloc( ( *lib ) -> name , char , it );

    pack_fread( ( *lib ) -> name , it , file );

    for ( it = 0 ; it < ( *lib ) -> current ; it ++ ) {
            if ( pack_igetl( file ) == 1 ) {
                    it1 = pack_igetl( file );
                    Malloc( ( *lib ) -> data[ it ] . name , char , it1 );
                    pack_fread( ( *lib ) -> data[ it ] . name , it1 , file );
                    ( *lib ) -> data[ it ] . sample = load_sample_from_datafile( file );
                    ( *lib ) -> data[ it ] . state = IDLE;
#ifdef DEBUGMODE

                    fprintf( stderr , "%s loaded at %d\n" , ( *lib ) -> data[ it ] . name , it );
#endif

                    }

            else
                ( *lib ) -> data[ it ] . state = NONE;

            }

    return TRUE;


    } /* load_libsound( ... ) */



/*!\fn search_sample_in( LIBSOUND *lib , char *name )
 *
 *\brief Search for a sample in a library, return the first free place if name is NULL
 *
 *\param lib Library for searching
 *\param name The name of a sample, NULL if not
 *
 *\return The sample index if founded, -1 if no result
 */

int search_sample_in( LIBSOUND *lib , char *name )
    {

    int it = 0,
             done = -1,
                    len = 0;

    if ( name ) {
            len = uconvert_size( name , U_CURRENT, U_UNICODE );

            while ( it < lib -> current && done == -1 ) {
                    if ( lib -> data[ it ] . state != NONE && lib -> data[ it ] .name ) {
                            if ( len == uconvert_size( lib -> data[ it ] .name , U_CURRENT, U_UNICODE ) &&
                                    ustrcmp( lib -> data[ it ] .name , name ) == 0 )
                                done = it;
                            }

                    it++;
                    }

            return done;
            }

    else {
            while ( it < lib -> current && done == -1 ) {
                    if ( lib -> data[ it ] . state == NONE )
                        done = it;

                    it++;
                    }

            return done;
            }

    } /* search_sample_in( ... ) */





