/**\file network.c
*
*  network function
*
*  Network Engine
*
*\author Castagnier Mickal
*
*\version 1.0
*
*\date 10/05/2005
*
*/



#include "nilorea.h"



#ifndef NONETWORK



/*!\fn Handle_Wsa( int mode , int v1 , int v2 )
 *
 *\brief Do not directly use, it's used by Init & Close network
 *
 *\param mode 1 for opening 0 for close
 *\param v1 First digit of version requested
 *\param v2 Second digit of version requested
 *
 *\return TRUE on success FALSE on error
 */

int Handle_Wsa( int mode , int v1 , int v2 )
    {

#ifndef LINUX

    static WSADATA WSAdata; /*WSA worlf*/

    static int WSA_IS_INITIALIZED = 0; /*status checking*/

    switch ( mode ) {

                /*returning WSA status*/

                case 2 :

                return WSA_IS_INITIALIZED;

                break;

                /*loading WSA Dll*/

                case 1 :

                if ( WSA_IS_INITIALIZED == 1 )
                    return TRUE; /*already loaded*/


                if ( ( WSAStartup( MAKEWORD( v1 , v2 ), &WSAdata ) ) != 0 ) {

                        WSA_IS_INITIALIZED = 0;
                        return FALSE;
                        }

                else {
                        WSA_IS_INITIALIZED = 1 ;
                        return TRUE;
                        }

                break;

                /*unloading (closing) WSA */

                case 0 :

                if ( WSA_IS_INITIALIZED == 0 )
                    return TRUE; /*already loaded*/

                if ( WSACleanup() == 0 ) {
                        WSA_IS_INITIALIZED = 0;
                        return TRUE;
                        }

                break;

            } /*switch(...)*/


#endif /*#ifndef LINUX*/

    return TRUE;
    } /*Handle_Wsa(...)*/



/*!\fn Init_All_Network( int version1, int version2 )
 *
 *\brief This do nothing under linux, but load the right winsock32 dll under Windows
 *
 *\param version1 an int for the first digit of version requested
 *\param version2 an int for the second digit of version requested
 *
 *\return TRUE if successfull, FALSE on error
 */

int Init_All_Network( int version1 , int version2 )
    {

    return Handle_Wsa( 1 , version1 , version2 );

    } /*Init_All_Network(...)*/



/*!\fn Close_All_Network( void )
 *
 *\brief This do nothing under linux, but unload the right winsock32 dll under Windows
 *
 *\return TRUE if successfull, FALSE on error
 */

int Close_All_Network()
    {

    return Handle_Wsa( 0 , 0 , 0 );

    } /*Close_All_Network()*/



/*!\fn Connect_Network( NETWORK **netw , char *ip , int port )
 *
 *\brief Use this to connect a NETWORK to any listening one
 *
 *\param netw a NETWORK **object
 *\param ip a char*ip_to_join
 *\param port an int to put the dest PORT
 *
 *\return TRUE if successfull, FALSE if any error occurs
 */

int Connect_Network( NETWORK **netw , char *ip , int port )
    {

    int tmp = 0;

    /*checking WSA when under windows*/

    if ( Handle_Wsa( 2 , 0 , 0 ) == FALSE )
        return FALSE;

    /*do not work over an already used netw*/
    if ( ( *netw ) )
        return FALSE;

    /*creating array*/
    Malloc( ( *netw ) , NETWORK , 1 );

    if ( ( ( *netw ) -> link . sock = socket( AF_INET , SOCK_STREAM , 0 ) ) == INVALID_SOCKET ) {
            fprintf( stderr ,
                     "Error creating socket at line %d of %s\n",
                     __LINE__ , __FILE__ );
            return FALSE;
            }

    /* disable naggle algorithm */
    tmp = 1;

    if ( setsockopt ( ( *netw ) -> link . sock, IPPROTO_TCP, TCP_NODELAY, ( const char * ) & tmp, sizeof( tmp ) ) == -1 ) {
            fprintf( stderr , "Error setting TCP_NODELAY option on %d\n" , ( *netw ) -> link . sock );
            return FALSE;
            }


    /* reducing buffer size */
    tmp = 10240;

    if ( setsockopt ( ( *netw ) -> link . sock, SOL_SOCKET, SO_SNDBUF, ( const char * ) & tmp, sizeof( tmp ) ) == -1 ) {
            fprintf( stderr , "Error setting SO_SNDBUF option on %d\n" , ( *netw ) -> link . sock );
            return FALSE;
            }



    /* reducing buffer size */
    tmp = 10240;

    if ( setsockopt ( ( *netw ) -> link . sock, SOL_SOCKET, SO_RCVBUF, ( const char * ) & tmp, sizeof( tmp ) ) == -1 ) {
            fprintf( stderr , "Error setting SO_RCVBUF option on %d\n" , ( *netw ) -> link . sock );
            return FALSE;
            }



    /* lose the pesky "Address already in use" error message*/
    if ( setsockopt( ( *netw ) -> link . sock , SOL_SOCKET, SO_REUSEADDR, "1", sizeof( int ) ) == -1 ) {
            fprintf( stderr , "Error setting SO_REUSEADDR option on %d\n" , ( *netw ) -> link . sock );
            return FALSE;
            }


    /*filling host information*/
    ( *netw ) -> link . haddr . sin_addr . s_addr = inet_addr( ip );

    ( *netw ) -> link . haddr . sin_family = AF_INET;

    ( *netw ) -> link . haddr . sin_port = htons( port );

    /*filling local information*/
    ( *netw ) -> link . laddr . sin_addr . s_addr = htonl( INADDR_ANY );

    ( *netw ) -> link . laddr . sin_family = AF_INET;

    ( *netw ) -> link . laddr . sin_port = htons( port );

    ( *netw ) -> link . port = port;

    memset( &( ( *netw ) -> link . laddr . sin_zero ), 0 , 8 );  /*zero the rest of the struct */

    memset( &( ( *netw ) -> link . haddr . sin_zero ), 0 , 8 ); /* zero the rest of the struct*/

    /*connecting*/

    if ( connect( ( *netw ) -> link .sock , ( struct sockaddr * ) & ( ( *netw ) -> link . haddr ),

                  sizeof( struct sockaddr ) ) < 0 ) {

            fprintf( stderr , "Error connecting to %s on port %d socket %d\n", inet_ntoa( ( *netw ) -> link . haddr . sin_addr ) , ( *netw ) -> link . port , ( *netw ) -> link .sock );
            return FALSE;
            }

    Malloc( ( *netw ) -> link . ip , char , strlen( ip ) + 1 );

    /*storing connected to ip adress*/
    strcpy( ( *netw ) -> link . ip , ip );

    /*initialize queue with HUGELIST*/

    if ( init_list( &( *netw ) -> send_buf , HUGELIST ) == FALSE )
        return FALSE;

    if ( init_list( &( *netw ) -> recv_buf , HUGELIST ) == FALSE )
        return FALSE;

    /*initiliaze mutex*/
    if ( pthread_mutex_init( &( *netw ) -> sendbolt , NULL ) != 0 )
        return FALSE;

    if ( pthread_mutex_init( &( *netw ) -> recvbolt , NULL ) != 0 )
        return FALSE;


    return TRUE;

    } /*Connect_Network(...)*/



/*!\fn Close_Network( NETWORK *netw )
 *
 *\brief Closing a specified Network
 *
 *\param netw A NETWORK *network to close
 *
 *\return TRUE on success , FALSE on failure
 */

int Close_Network( NETWORK *netw )
    {

    /*closing connection*/
    closesocket( netw -> link . sock );

    /*list freeing*/
    empty_list ( netw -> send_buf );
    empty_list ( netw -> recv_buf );

    Free( netw -> send_buf );
    Free( netw -> recv_buf );

    Free( netw );

    return TRUE;

    } /*Close_Network(...)*/



/*!\fn Make_Listening_Network( NETWORK **netw , int PORT , int nbpending )
 *
 *\brief Make a NETWORK be a Listening network
 *
 *\param netw A NETWORK **network to make listening
 *
 *\param PORT For choosing a PORT to listen to
 *
 *\param nbpending Number of pending connection when listening
 *
 *\return TRUE on success, FALSE on error
 */

int Make_Listening_Network( NETWORK **netw , int PORT , int nbpending )
    {

    int tmp = 0;

    /*checking WSA when under windows*/

    if ( Handle_Wsa( 2 , 0 , 0 ) == FALSE )
        return FALSE;

    if ( *netw )
        return FALSE; /*already used network*/

    /*creating array*/
    Malloc( *netw , NETWORK , 1 );


    ( *netw ) -> link . sock = socket( AF_INET, SOCK_STREAM, 0 );


    /* disable naggle algorithm */
    tmp = 1;

    if ( setsockopt ( ( *netw ) -> link . sock, IPPROTO_TCP, TCP_NODELAY, ( const char * ) & tmp, sizeof( tmp ) ) == -1 ) {
            fprintf( stderr , "Error setting TCP_NODELAY option on %d\n" , ( *netw ) -> link . sock );
            return FALSE;
            }



    /* reducing buffer size */
    tmp = 2048;

    if ( setsockopt ( ( *netw ) -> link . sock, SOL_SOCKET, SO_SNDBUF, ( const char * ) & tmp, sizeof( tmp ) ) == -1 ) {
            fprintf( stderr , "Error setting SO_SNDBUF option on %d\n" , ( *netw ) -> link . sock );
            return FALSE;
            }



    /* reducing buffer size */
    tmp = 2048;

    if ( setsockopt ( ( *netw ) -> link . sock, SOL_SOCKET, SO_RCVBUF, ( const char * ) & tmp, sizeof( tmp ) ) == -1 ) {
            fprintf( stderr , "Error setting SO_RCVBUF option on %d\n" , ( *netw ) -> link . sock );
            return FALSE;
            }



    /* lose the pesky "Address already in use" error message*/
    if ( setsockopt( ( *netw ) -> link . sock , SOL_SOCKET, SO_REUSEADDR, "1", sizeof( int ) ) == -1 ) {
            fprintf( stderr , "Error setting SO_REUSEADDR option on %d\n" , ( *netw ) -> link . sock );
            return FALSE;
            }



    /*filling local information*/
    ( *netw ) -> link . laddr . sin_addr.s_addr = INADDR_ANY;

    ( *netw ) -> link . laddr . sin_family = AF_INET;

    ( *netw ) -> link . laddr . sin_port = htons( PORT );

    ( *netw ) -> nb_pending = nbpending;

    memset( &( ( *netw ) -> link . laddr . sin_zero ), 0 , 8 ); /* zero the rest of the struct*/

    memset( &( ( *netw ) -> link . haddr . sin_zero ), 0 , 8 ); /* zero the rest of the struct*/


    /*binding information to socket*/
    if ( bind( ( *netw ) -> link . sock,

               ( struct sockaddr * ) & ( ( *netw ) -> link . laddr ),

               sizeof( struct sockaddr ) ) == -1 ) {
            fprintf( stderr , "Error binding socket %d\n" , ( *netw ) -> link . sock );
            return FALSE;
            }

    /*enqueing nb_pending connections*/
    listen( ( *netw ) -> link . sock, ( *netw ) -> nb_pending );


    return TRUE;

    } /*Make_Listening_Network(...)*/



/*!\fn Accept_From_Network( NETWORK *from )
 *
 *\brief make a normal 'accept' with the given *from NETWORK
 *
 *\param from the network from where we accept
 *
 *\return NULL on failure, if not a pointer to the connected network
 */

NETWORK *Accept_From_Network( NETWORK *from )
    {

    int sin_size;

    NETWORK *netw;

    if ( !from )
        return FALSE;

    Malloc( netw , NETWORK , 1 );

    sin_size = sizeof( struct sockaddr_in );


    if ( ( netw -> link . sock =
                    accept( from -> link . sock, ( struct sockaddr * ) & ( netw -> link . haddr ) , &sin_size ) ) == INVALID_SOCKET ) {
            fprintf( stderr , "error accepting on %d\n" , netw -> link . sock );
            return NULL;
            }

    netw -> mode = CLIENT;
    netw -> link . port = from -> link . port;

    /*initialize queue with HUGELIST*/

    if ( init_list( &( netw ) -> send_buf , HUGELIST ) == FALSE )
        return FALSE;

    if ( init_list( &( netw ) -> recv_buf , HUGELIST ) == FALSE )
        return FALSE;

    /*initiliaze mutex*/
    if ( pthread_mutex_init( &netw -> sendbolt , NULL ) != 0 )
        return FALSE;

    if ( pthread_mutex_init( &netw -> recvbolt , NULL ) != 0 )
        return FALSE;

    return netw;

    } /*Accept_from_Network*/



/*!\fn Add_Msg( NETWORK *netw , char *msg , int nboct )
 *
 *\brief Add a message to send in aimed NETWORK
 *
 *\param netw NETWORK where add the message
 *\param msg  the message to add
 *\param nboct the size of msg
 *
 *return TRUE if success FALSE on error
 */

int Add_Msg( NETWORK *netw , char *msg , int nboct )
    {


    pthread_mutex_lock( &netw -> sendbolt );

    if ( add_last( netw -> send_buf , msg , nboct ) == FALSE ) {

            pthread_mutex_unlock( &netw -> sendbolt );

            return FALSE;

            }

    pthread_mutex_unlock( &netw -> sendbolt );

    return TRUE;

    } /*Add_Msg(...)*/



/*!\fn Get_Msg( NETWORK *netw , char **msg , int *nboct )
 *
 *\brief Get a message from aimed NETWORK
 *
 *\param netw NETWORK where get the msg
 *\param msg Place where storing the message
 *\param nboct size of message
 *
 *\warning msg must be free before use
 *
 *\return TRUE or FALSE
 */

int Get_Msg( NETWORK *netw , char **msg , int *nboct )
    {

    N_STR * ptr;

    pthread_mutex_lock( &netw -> recvbolt );

    ptr = get_first( netw -> recv_buf );

    if ( !ptr ) {
            pthread_mutex_unlock( &netw -> recvbolt );
            return FALSE;
            }

    if ( ( *msg ) )
        Free( ( *msg ) );

    Malloc( ( *msg ) , char , ptr -> length );

    if ( ! ( *msg ) ) {
            pthread_mutex_unlock( &netw -> recvbolt );
            return FALSE;
            }

    Strcpy( ptr -> data , ( *msg ) , ptr -> length );

    *nboct = ptr -> length;

    kill_first_item( netw -> recv_buf );

    pthread_mutex_unlock( &netw -> recvbolt );

    return TRUE;

    } /*Get_Msg(...)*/


/*!\fn Do_Connecting_Network( NETWORK * netw )
 *
 *\brief Connecting side networking
 *
 *\param netw The Network to manage
 *
 *\return TRUE if something was do FALSE if error or disconnected
 */

int Do_Connecting_Network( NETWORK *netw )
    {


    N_STR * ptr;      /*pointer to read data*/

    char *recvdmsg,   /*array for temp store of new received message*/
    nboct[ 4 ];  /*to send and receive the size of message*/

    int nboctet;    /*to store the converted to int (char nboct)*/

#ifdef NETW_DEBUG

    int it;         /*iterator,Global Use*/
#endif


    if ( !netw )
        return FALSE;


    /*******************************************************************************
    **************************** SENDING PART OF NETWORK ***************************
    *******************************************************************************/


    pthread_mutex_lock( &netw -> sendbolt );

    ptr = get_first( netw -> send_buf ) ;

    /*if no message send blank*/

    if ( !ptr ) {

            nboctet = htonl( nboctet );
            memcpy( nboct , &nboctet , sizeof( nboctet ) );

            if ( send_data( netw -> link . sock , nboct , sizeof( nboct ) ) == FALSE ) {

                    pthread_mutex_unlock( &netw -> sendbolt );

#ifdef NETW_DEBUG

                    fprintf( stderr , "Blank packet sending has failed\n" );
#endif

                    return FALSE;
                    }

#ifdef NETW_DEBUG
            fprintf( stderr , "%d has sended an empty packet\n" , netw -> link . sock );

#endif

            }

    else {

            nboctet = ptr -> length;

#ifdef NETW_DEBUG

            fprintf( stderr , "%d ready to send %d octets\n" , netw -> link . sock , nboctet );
#endif

            nboctet = htonl( nboctet );
            memcpy( nboct , &nboctet , sizeof( nboctet ) );

            if ( send_data( netw -> link . sock , nboct , sizeof( nboct ) ) == FALSE ) {
                    pthread_mutex_unlock( &netw -> sendbolt );
                    return FALSE;
                    }



            if ( send_data( netw -> link . sock , ptr -> data , ptr -> length ) != FALSE ) {

#ifdef NETW_DEBUG

                    fprintf( stderr , "%d packet dump:\n" , netw -> link . sock );

                    for ( it = 0 ; it < ptr -> length ; it ++ )
                        fprintf( stderr , "%c" , ptr -> data[ it ] );

                    fprintf( stderr , "\n" );

#endif

                    kill_first_item( netw -> send_buf );

                    pthread_mutex_unlock( &netw -> sendbolt );

#ifdef NETW_DEBUG

                    fprintf( stderr , "%d KILL: last sended packet\n" , netw -> link . sock );

#endif

                    }

            else {

                    pthread_mutex_unlock( &netw -> sendbolt );
                    return FALSE;
                    }

            }


    /*******************************************************************************
    ************************** RECEIVING PART OF NETWORK ***************************
    *******************************************************************************/


#ifdef NETW_DEBUG
    fprintf( stderr , "%d entering receiving part\n" , netw -> link . sock );

#endif

    if ( recv_data( netw -> link . sock , nboct , sizeof( nboct ) ) == FALSE )
        return FALSE;

    memcpy( &nboctet , nboct , sizeof( nboctet ) );

    nboctet = ntohl( nboctet );

#ifdef NETW_DEBUG

    fprintf( stderr , "%d is going to receive %d octets\n" , netw -> link . sock , nboctet );

#endif

    if ( nboctet != 0 ) {

            Malloc( recvdmsg , char , nboctet );

            fill_str( recvdmsg , 0 , nboctet );

            if ( recv_data( netw -> link . sock , recvdmsg , nboctet ) == FALSE )
                return FALSE;

#ifdef NETW_DEBUG

            fprintf( stderr , "%d packet dump:\n" , netw -> link . sock );

            for ( it = 0 ; it < nboctet ; it ++ )
                fprintf( stderr , "%c" , recvdmsg[ it ] );

            fprintf( stderr , "\n" );

#endif

            pthread_mutex_lock( &netw -> recvbolt );

            if ( add_last( netw -> recv_buf , recvdmsg , nboctet ) == FALSE ) {
                    Free( recvdmsg );
                    pthread_mutex_unlock( &netw -> recvbolt );
                    return FALSE;

                    }

            Free( recvdmsg );
            pthread_mutex_unlock( &netw -> recvbolt );

            }

    else {

#ifdef NETW_DEBUG
            fprintf( stderr , "%d has received zero octets\n" , netw -> link . sock );
#endif

            }

    return TRUE;

    } /* Do_Connecting_Network(...) */



/*!\fn Do_Accepting_Network( NETWORK * netw )
 *
 *\brief Accepting side networking
 *
 *\param netw The Network to manage
 *
 *\return TRUE if something was do FALSE if ERROR or disconnected
 */

int Do_Accepting_Network( NETWORK *netw )
    {


    N_STR * ptr;      /*pointer to read data*/
    char *recvdmsg,   /*array for temp store of new received message*/
    nboct[ 4 ];  /*to send and receive the size of message*/

    int nboctet;    /*to store the converted to int (char nboct)*/

#ifdef NETW_DEBUG

    int it;         /*iterator,Global Use*/
#endif


    if ( !netw )
        return FALSE;


    /*******************************************************************************
    ************************** RECEIVING PART OF NETWORK ***************************
    *******************************************************************************/

#ifdef NETW_DEBUG

    fprintf( stderr , "%d entering receiving part\n" , netw -> link . sock );

#endif

    if ( recv_data( netw -> link . sock , nboct , sizeof( nboct ) ) == FALSE )
        return FALSE;

    memcpy( &nboctet , nboct , sizeof( nboctet ) );

    nboctet = ntohl( nboctet );


    if ( nboctet != 0 ) {

#ifdef NETW_DEBUG
            fprintf( stderr , "%d is receiving %d octets\n" , netw -> link . sock , nboctet );
#endif

            Malloc( recvdmsg , char , nboctet );

            if ( !recvdmsg )
                return FALSE;

            fill_str( recvdmsg , 0 , nboctet );

            if ( recv_data( netw -> link . sock , recvdmsg , nboctet ) == FALSE )
                return FALSE;

#ifdef NETW_DEBUG

            fprintf( stderr , "%d packet dump:\n" , netw -> link . sock );

            for ( it = 0 ; it < nboctet ; it ++ )
                fprintf( stderr , "%c" , recvdmsg[ it ] );

            fprintf( stderr , "\n" );

#endif

            pthread_mutex_lock( &netw -> recvbolt );

            if ( add_last( netw -> recv_buf , recvdmsg , nboctet ) == FALSE ) {
                    Free( recvdmsg );
                    pthread_mutex_unlock( &netw -> recvbolt );
                    return FALSE;
                    }

            Free( recvdmsg );
            pthread_mutex_unlock( &netw -> recvbolt );

            }

    else {
#ifdef NETW_DEBUG
            fprintf( stderr , "%d has received an empty packet\n" , netw -> link . sock );
#endif

            }

    /*******************************************************************************
    **************************** SENDING PART OF NETWORK ***************************
    *******************************************************************************/
#ifdef NETW_DEBUG
    fprintf( stderr , "%d entering sending part\n" , netw -> link . sock );

#endif

    pthread_mutex_lock( &netw -> sendbolt );

    ptr = get_first( netw -> send_buf ) ;


    /*if no message send blank*/

    if ( !ptr ) {

            nboctet = 0;
            nboctet = htonl( nboctet );
            memcpy( nboct , &nboctet , sizeof( nboctet ) );

            if ( send_data( netw -> link . sock , nboct , sizeof( nboct ) ) == FALSE ) {
                    pthread_mutex_unlock( &netw -> sendbolt );
                    return FALSE;
                    }

#ifdef NETW_DEBUG
            fprintf( stderr, "%d has sended an empty packet\n" , netw -> link . sock );

#endif

            }

    else {

            nboctet = ptr -> length;
            nboctet = htonl( nboctet );
            memcpy( nboct , &nboctet , sizeof( nboctet ) );

            if ( send_data( netw -> link . sock , nboct , sizeof( nboct ) ) == FALSE ) {
                    pthread_mutex_unlock( &netw -> sendbolt );
                    return FALSE;
                    }

            if ( send_data( netw -> link . sock , ptr -> data , ptr -> length ) != FALSE ) {

#ifdef NETW_DEBUG
                    fprintf( stderr , "%d has sended %d octets\n" , netw -> link . sock , ntohl( nboctet ) );
#endif

                    kill_first_item( netw -> send_buf );

                    pthread_mutex_unlock( &netw -> sendbolt );

                    }

            else {

                    pthread_mutex_unlock( &netw -> sendbolt );

                    return FALSE;
                    }
            }

    return TRUE;

    } /* Do_Accepting_Network(...) */



/*!\fn send_data( SOCKET s , char *buf, int n )
 *
 *\brief send data onto the socket
 *
 *\param s connected socket
 *\param buf pointer to buffer
 *\param n number of characters we want
 *
 *\return -1 on error, n on success
 */

int send_data( SOCKET s,   /* connected socket */
               char *buf,  /* pointer to the buffer */
               int n      /* number of characters (bytes) we want, max is 1000000 characters*/
             )
    {
    int bcount; /* counts bytes read */
    int br;     /* bytes read this pass */

    bcount = 0;
    br = 0;

    while ( bcount < n ) {             /* loop until full buffer */

            if ( ( br = send( s, buf, n - bcount, NETFLAGS ) ) > 0 ) {
                    bcount += br;                /* increment byte counter */
                    buf += br;                   /* move buffer ptr for next read */
                    }

            else if ( br <= 0 ) {             /* signal an error to the caller */
                    fprintf( stderr , "Socket %d sending Error\n", s );
                    return FALSE;
                    }
            }

    return bcount;

    } /*send_data(...)*/



/*!\fn recv_data( SOCKET s , char *buf, int n )
 *
 *\brief recv data from the socket
 *
 *\param s connected socket
 *\param buf pointer to buffer
 *\param n number of characters we want
 *
 *\return -1 on error, n on success
 */

int recv_data( SOCKET s,   /* connected socket */
               char *buf,  /* pointer to the buffer */
               int n )
    {
    int bcount; /* counts bytes read */
    int br;     /* bytes read this pass */


    bcount = 0;
    br = 0;

    while ( bcount < n ) {             /* loop until full buffer */

            if ( ( br = recv( s, buf, n - bcount, NETFLAGS ) ) > 0 ) {
                    bcount += br;                /* increment byte counter */
                    buf += br;                   /* move buffer ptr for next read */
                    }

            else if ( br <= 0 ) {               /* signal an error to the caller */
                    fprintf( stderr , "Socket %d receive Error\n", s );
                    return FALSE;
                    }
            }

    return bcount;
    } /*recv_data(...)*/



/*!\fn load_server_config( char *file , int *MAX_CLIENTS , int *RESERVED , int *PORT , int *ADMPORT , int *LOGGING , int *sendlimit , int *recvlimit )
 *
 *\brief Load the server config file, create it on error
 *
 *\param file The config filename
 *\param MAX_CLIENTS Number of client allowed
 *\param RESERVED   Number of admin-reserved places
 *\param PORT Port where the server will accept connection from Clients
 *\param ADMPORT Port where the server will accept connection from Admins
 *\param LOGGING Enable or Disable Serveur Logging
 *\param sendlimit If != 0 enable the sending speed limit
 *\param recvlimit If != 0 enable the recving speed limit
 *
 *\return TRUE if loaded, FALSE if filled by default
 */

int load_server_config( char *file ,
                        int *MAX_CLIENTS , int *RESERVED , int *PORT ,
                        int *ADMPORT , int *LOGGING , int *sendlimit ,
                        int *recvlimit )
    {

    int it,
    nb_fail = 0 ;

    char tmpstr[ 256 ];

    FILE *settings;

    /* setting default value if file isn't existing */
    ( *MAX_CLIENTS ) = 100 ;
    ( *RESERVED ) = 10;
    ( *PORT ) = 15243 ;
    ( *ADMPORT ) = 34251 ;
    ( *LOGGING ) = 1 ;
    ( *sendlimit ) = 20000 ;
    ( *recvlimit ) = 20000 ;

    settings = fopen( file , "rt+" );

    if ( !settings ) {
            fprintf( stderr , "ERROR: Cannot read %s, default loaded\n" , file );
            return FALSE;
            }

    /* reading comment */
    for ( it = 0 ; it < 11 ; it ++ )
        fgets( tmpstr , 256 , settings );

    /* reading nb max client */
    fill_str( tmpstr , 0 , 256 );

    fgets( tmpstr , 256 , settings );

    ( *MAX_CLIENTS ) = atoi( tmpstr );

    if ( ( *MAX_CLIENTS ) == 0 ) {
            ( *MAX_CLIENTS ) = 100 ;
            nb_fail++;
            fprintf( stderr , "FAILED: MAX_CLIENT HAS DEFAULT VALUE 100\n" );
            };

    /* reading comment */
    fgets( tmpstr , 256 , settings );

    /* reading port number */
    fill_str( tmpstr , 0 , 256 );

    fgets( tmpstr , 256 , settings );

    ( *PORT ) = atoi( tmpstr );

    if ( ( *PORT ) == 0 ) {
            ( *PORT ) = 15243 ;
            nb_fail++;
            fprintf( stderr , "FAILED: PORT HAS DEFAULT VALUE 15243\n" );
            };

    /* reading comment */
    fgets( tmpstr , 256 , settings );

    /* reading admin port number */
    fill_str( tmpstr , 0 , 256 );

    fgets( tmpstr , 256 , settings );

    ( *ADMPORT ) = atoi( tmpstr );

    if ( ( *ADMPORT ) == 0 ) {
            ( *ADMPORT ) = 34251 ;
            nb_fail++;
            fprintf( stderr , "FAILED: ADMPORT HAS DEFAULT VALUE 34251\n" );
            };

    /* reading comment */
    fgets( tmpstr , 256 , settings );

    /* reading a logging number */
    fill_str( tmpstr , 0 , 256 );

    fgets( tmpstr , 256 , settings );

    ( *LOGGING ) = atoi( tmpstr );

    if ( ( *LOGGING ) == 0 ) {
            ( *LOGGING ) = 1 ;
            nb_fail++;
            fprintf( stderr , "FAILED: LOGGING HAS DEFAULT VALUE 1\n" );
            };

    /* reading comment */
    fgets( tmpstr , 256 , settings );

    /* reading sending limit number */
    fill_str( tmpstr , 0 , 256 );

    fgets( tmpstr , 256 , settings );

    ( *sendlimit ) = atoi( tmpstr );

    if ( ( *sendlimit ) == 0 ) {
            ( *sendlimit ) = 20000 ;
            nb_fail++;
            fprintf( stderr , "FAILED: sendlimit HAS DEFAULT VALUE 20000\n" );
            };

    /* reading comment */
    fgets( tmpstr , 256 , settings );

    /* reading receive limit number */
    fill_str( tmpstr , 0 , 256 );

    fgets( tmpstr , 256 , settings );

    ( *recvlimit ) = atoi( tmpstr );

    if ( ( *recvlimit ) == 0 ) {
            ( *recvlimit ) = 20000 ;
            nb_fail++;
            fprintf( stderr , "FAILED: recvlimit HAS DEFAULT VALUE 20000\n" );
            };

    /* reading comment */
    fgets( tmpstr , 256 , settings );

    /* reading RESERVED */
    fill_str( tmpstr , 0 , 256 );

    fgets( tmpstr , 256 , settings );

    ( *RESERVED ) = atoi( tmpstr );

    if ( ( *RESERVED ) == 0 ) {
            ( *RESERVED ) = 10 ;
            nb_fail++;
            fprintf( stderr , "FAILED: RESERVED HAS DEFAULT VALUE 10\n" );
            };

    if ( nb_fail != 0 )
        return FALSE;

    return TRUE;

    } /* load_server_config( ... ) */



/*!\fn save_server_config( char *file , int MAX_CLIENTS , int RESERVED , int PORT , int ADMPORT , int LOGGING , int sendlimit , int recvlimit )
 *
 *\brief Save the server config file, creating it if needed
 *
 *\param file The config filename
 *\param MAX_CLIENTS Number of client allowed
 *\param RESERVED   Number of admin-reserved places
 *\param PORT Port where the server will accept connection from Clients
 *\param ADMPORT Port where the server will accept connection from Admins
 *\param LOGGING Enable or Disable Serveur Logging
 *\param sendlimit If != 0 enable the sending speed limit
 *\param recvlimit If != 0 enable the recving speed limit
 *
 *\return TRUE if loaded or created, FALSE if error
 */

int save_server_config( char *file ,
                        int MAX_CLIENTS , int RESERVED , int PORT ,
                        int ADMPORT , int LOGGING ,
                        int sendlimit , int recvlimit )
    {

    FILE * settings;

    char tmpstr[ 256 ];

    settings = fopen( file , "wt+" );

    if ( !settings ) {
            fprintf( stderr , "FATAL ERROR: Cannot create file for saving server\n" );
            fclose( settings );
            return FALSE;
            }


    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# NILOREA ENGINE CLIENT CONFIG FILE\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# Do not modify without knowing exactly what you are doing\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# This file is automatically generated,loaded and saved by the server\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# Hints: -Place is important, only modify the values.\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#        -If the server is running change the values by the server command\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#        -This file will be automatically generated by the server if not found or if an error occur.\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#         The fields will be filled by default values.\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# MAX CLIENT ON THE SERVER\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%d\n" , MAX_CLIENTS );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# CONNECTION PORT\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%d\n" , PORT );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# ADMIN CONNECTION PORT\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%d\n" , ADMPORT );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# LOGGING\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%d\n" , LOGGING );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# SENDING LIMIT\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%d\n" , sendlimit );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# RECEIVING LIMIT\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%d\n" , recvlimit );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# NB RESERVED ADMIN CONNECTION\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%d\n" , RESERVED );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );

    fclose( settings );

    return TRUE;

    } /* save_server_config(...) */



/*!\fn load_client_config( char *file , char **ip , int *PORT , int *LOGGING )
 *
 *\brief Load the client config file, create it on error
 *
 *\param file The config filename
 *\param ip Ip adress of the server to connect
 *\param PORT Port where the server will accept connection from Clients
 *\param LOGGING Enable or Disable Serveur Logging
 *
 *\return TRUE if loaded, FALSE if filled by default
 */

int load_client_config( char *file ,
                        char **ip , int *PORT , int *LOGGING )
    {

    int it,
    nb_fail = 0 ;

    char tmpstr[ 256 ];

    FILE *settings;

    /* setting default value if file isn't existing */
    Malloc( *ip , char , 256 );
    fill_str( *ip , 0 , 256 );
    strcpy( *ip , "127.0.0.1" );
    ( *PORT ) = 15243 ;
    ( *LOGGING ) = 1 ;

    settings = fopen( file , "rt+" );

    if ( !settings ) {
            fprintf( stderr , "ERROR: Cannot read %s, default loaded\n" , file );
            return FALSE;
            }

    /* reading comment */
    for ( it = 0 ; it < 11 ; it ++ )
        fgets( tmpstr , 256 , settings );

    /* reading nb max client */
    fill_str( tmpstr , 0 , 256 );

    if ( fgets( tmpstr , 256 , settings ) ) {
            if ( strlen( tmpstr ) >= 8 ) {
                    fill_str( *ip , 0 , 256 );
                    Strcpy( tmpstr , *ip , strlen( tmpstr ) - 1 );
                    }

            else
                fprintf( stderr , "Invalid Ip adress, default loaded\n" );
            }

    else
        fprintf( stderr , "No Ip to read, default loaded\n" );


    /* reading comment */
    fgets( tmpstr , 256 , settings );

    /* reading port number */
    fill_str( tmpstr , 0 , 256 );

    fgets( tmpstr , 256 , settings );

    ( *PORT ) = atoi( tmpstr );

    if ( ( *PORT ) == 0 ) {
            ( *PORT ) = 15243 ;
            nb_fail++;
            fprintf( stderr , "FAILED: PORT HAS DEFAULT VALUE 15243\n" );
            };

    /* reading comment */
    fgets( tmpstr , 256 , settings );

    /* reading logging */
    fill_str( tmpstr , 0 , 256 );

    fgets( tmpstr , 256 , settings );

    ( *LOGGING ) = atoi( tmpstr );

    if ( ( *LOGGING ) == 0 ) {
            ( *LOGGING ) = 1 ;
            nb_fail++;
            fprintf( stderr , "FAILED: LOGGING HAS DEFAULT VALUE 1\n" );
            };

    if ( nb_fail != 0 )
        return FALSE;

    return TRUE;

    } /* load_client_config( ... ) */



/*!\fn save_client_config( char *file , char *ip , int PORT , int LOGGING )
 *
 *\brief Save the server config file, creating it if needed
 *
 *\param file The config filename
 *\param ip Ip adress or name of the server
 *\param PORT Port where the server will accept connection from Clients
 *\param LOGGING Enable or Disable Serveur Logging
 *
 *\return TRUE if loaded or created, FALSE if error
 */

int save_client_config( char *file ,
                        char *ip , int PORT , int LOGGING )
    {

    FILE * settings;

    char tmpstr[ 256 ];

    settings = fopen( file , "wt+" );

    if ( !settings ) {
            fprintf( stderr , "FATAL ERROR: Cannot create file for saving client config\n" );
            fclose( settings );
            return FALSE;
            }


    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# NILOREA ENGINE CLIENT CONFIG FILE\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# Do not modify without knowing exactly what you are doing\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# This file is automatically generated,loaded and saved by the server\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# Hints: -Place is important, only modify the values.\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#        -If the server is running change the values by the server command\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#        -This file will be automatically generated by the server if not found or if an error occur.\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#         The fields will be filled by default values.\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "#\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# IP TO CONNECT PORT\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%s\n" , ip );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# CONNECTION PORT\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%d\n" , PORT );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    strcpy( tmpstr , "# LOGGING\n" );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );
    fill_str( tmpstr , 0 , 256 );
    sprintf( tmpstr , "%d\n" , LOGGING );
    fwrite( tmpstr , strlen( tmpstr ) , sizeof( char ) , settings );

    fclose( settings );

    return TRUE;

    } /* save_client_config(...) */



#endif /* #ifndef NONETWORK */
