/**\file n_strlist.c
*
*  N_STR linked list functions
*
*  Everything you need to create, delete , append list is here
*
*\author Castagnier Mickal
*
*\version 1.0
*
*\date 24/03/05
*
*\warning Never forget to init_list(...) before using a N_STRLIST *LIST object
*/



#include "nilorea.h"



#ifndef NOSTR



/*!\fn int init_list( N_STRLIST **LIST , int max )
 *
 *\brief Always apply this function to any N_STRLIST *object before first use
 *
 *\param LIST an N_STRLIST object
 *\param max  an integer to assign 'max' item limitation to N_STRLIST **LIST, -1 for undeterminated
 *
 *\return TRUE on success, FALSE on failure (generaly due to memory lake, report is written to stderr)
 */

int init_list( N_STRLIST **LIST , int max )
    {

    /*point to nothing*/
    *LIST = NULL;

    /* allocating memory, returning FALSE  if impossible*/
    Malloc( *LIST , N_STRLIST , 1 );

    if ( !( *LIST ) )
        return FALSE;


    /* initilize list elements*/

    /*there is no item in the list*/
    ( *LIST ) -> nb_item = 0 ;

    /* we want 'max' element in our list*/
    ( *LIST ) -> nb_max_item = max;

    /* after allocation, we already have no item so we point nowhere*/
    ( *LIST ) -> start = ( *LIST ) -> end = NULL;

    /*returning success of allocation*/
    return TRUE;

    } /*init_list(...)*/



/*!\fn int add_first( N_STRLIST *LIST , char *item , int length )
  *
  *\brief use this to add a string item at the begin of your N_STRLIST *list
  *
  *\param LIST an N_STRLIST *object
  *\param item a char *string
  *\param length an int to specify the real length we wanna use for this string
  *
  *\return TRUE on success, FALSE on failure (generaly due to memory lake)
  */

int add_first( N_STRLIST *LIST , char *item , int length )
    {

    N_STR * newstr;

    if ( LIST != NULL )    /*it's an active list*/
            {

            /*checking if full or undeterminated*/

            if ( LIST-> nb_max_item != -1 && ( LIST -> nb_item + 1 >= LIST -> nb_max_item ) )
                return FALSE;

            if ( LIST -> nb_item == 0 )    /*it's an empty list we add the root*/
                    {

                    Malloc ( LIST->start, N_STR , 1 );                     /*memory allocation*/

                    LIST -> start -> next = LIST -> start -> prev = NULL; /*initialize LIST*/

                    Malloc ( LIST -> start -> data, char , length ); /*allocating data array*/
                    Strcpy( item , LIST -> start -> data , length ); /*filling data array*/

                    LIST -> start -> length = length ; /*saving the length of our string*/
                    LIST -> end = LIST -> start;      /*first node is OK now*/

                    }
            else
                    {
                    /*create new node*/
                    Malloc( newstr , N_STR , 1 );
                    Malloc( newstr -> data , char , length );
                    /*modify the next*/
                    newstr -> next = LIST -> start;
                    /*copying the string*/
                    Strcpy( item , newstr->data , length );
                    /*modify the new next node*/
                    LIST -> start -> prev = newstr;
                    /*moving to the new start position*/
                    LIST -> start = newstr;
                    }

            }
    else
        return FALSE;                       /*that was not an initialized list*/


    LIST -> nb_item = LIST -> nb_item + 1; /*here we have a new element in*/

    return TRUE;                            /*successful*/

    } /*add_first(...)*/



/*!\fn add_before( N_STRLIST *LIST , char *item , int length , int pos)
 *
 *\brief use this to add an item before the aimed one in your N_STRLIST *LIST
 *
 *\param LIST an N_STRLIST *object
 *\param item a char *string
 *\param length an int to specify the real length we wanna use for this string
 *\param pos an unsigned int to specify the position we wanna put our string
 *
 *\return TRUE on success, FALSE on failure (if pos isn't between 0 and LIST->nb_item, or memory lake)
 */

int add_before( N_STRLIST *LIST , char *item , int length , int pos )
    {

    N_STR * tmpstr,   /*new item*/
    *pointer; /*pointer to seek*/

    /*if LIST and pos are right*/

    if ( LIST != NULL && LIST -> nb_item > 0 && pos >= 1 && pos <= LIST -> nb_item ) {
            /*checking if full or undeterminated*/

            if ( LIST-> nb_max_item != -1 && ( LIST -> nb_item + 1 >= LIST -> nb_max_item ) )
                return FALSE;

            if ( pos == 1 )                                        /*if it's the first*/
                return add_first( LIST , item , length );   /* return the status of add_first */

            /* get the aimed item*/
            pointer = SeekAt( LIST, pos );

            if ( pointer != NULL )    /*if it is a right knot*/
                    {
                    /*create the new item*/
                    Malloc( tmpstr , N_STR , 1 );            /*memory allocating*/
                    Malloc( tmpstr->data , char , length ); /*string allocating*/
                    Strcpy( item , tmpstr->data , length ); /*string copy*/

                    /*modify the item to insert*/
                    tmpstr->prev = pointer->prev;
                    tmpstr->next = pointer;

                    /*modify the list*/
                    pointer->prev->next = tmpstr;
                    pointer->prev = tmpstr;

                    /*one more*/
                    ++LIST -> nb_item;

                    }
            else
                return FALSE;   /*if(pointer!=NULL)*/

            }

    else
        return FALSE;   /*if(LIST != NULL && pos >= 1 ...)*/

    /*Normal end*/
    return TRUE;

    } /*add_before(...)*/



/*!\fn add_after( N_STRLIST *LIST , char *item , int length , int pos)
 *
 *\brief use this to add an item after the aimed one in your N_STRLIST *LIST
 *
 *\param LIST an N_STRLIST *object
 *\param item a char *string
 *\param length an int to specify the real length we wanna use for this string
 *\param pos an unsigned int to specify the position we wanna put our string
 *
 *\return TRUE on success, FALSE on failure (if pos isn't between 0 and LIST->nb_item, or memory lake)
 */

int add_after( N_STRLIST *LIST , char *item , int length , int pos )
    {

    N_STR * tmpstr,  /*new node*/
    *pointer; /*pointer to seek*/

    /*if LIST and pos are right*/

    if ( LIST != NULL && LIST -> nb_item > 0 && pos >= 1 && pos <= LIST -> nb_item ) {

            /*checking if full or undeterminated*/

            if ( LIST-> nb_max_item != -1 && ( LIST -> nb_item + 1 >= LIST -> nb_max_item ) )
                return FALSE;

            if ( pos == LIST -> nb_max_item )    /*if it's the last*/
                return add_last( LIST , item , length ); /*return the status of add_last*/


            /*get the aimed node*/
            pointer = SeekAt( LIST , pos );

            if ( pointer != NULL )    /*if it is a right node*/
                    {

                    /*create the new item*/
                    Malloc( tmpstr , N_STR , 1 );            /*memory allocating*/
                    Malloc( tmpstr->data , char , length ); /*string allocating*/
                    Strcpy( item , tmpstr->data , length ); /*string copy*/

                    /*modify the item to insert*/
                    tmpstr -> prev = pointer;
                    tmpstr -> next = pointer -> next;

                    /*modify the list*/
                    pointer -> next -> prev = tmpstr;
                    pointer -> next = tmpstr;

                    }
            else
                return FALSE;   /* Retourne une erreur ou avertissement.     */
            }

    else
        return FALSE;   /* Retourne une erreur ou avertissement.             */

    return TRUE;

    } /*add_after(...)*/



/*!\fn int add_last( N_STRLIST *LIST , char *item , int length )
  *
  *\brief use this to add a string item at the end of your N_STRLIST *list
  *
  *\param LIST an N_STRLIST *object
  *\param item a char *string
  *\param length an int to specify the real length we wanna use for this string
  *
  *\return TRUE on success, FALSE on failure (generaly due to memory lake)
  */

int add_last( N_STRLIST *LIST , char *item , int length )
    {


    /*not initialized list*/

    if ( !LIST )
        return FALSE;

    /*checking if full or undeterminated*/
    if ( LIST-> nb_max_item != -1 && ( LIST -> nb_item + 1 >= LIST -> nb_max_item ) )
        return FALSE;


    /*if it's a 'new' (never used) list*/
    if ( LIST -> nb_item == 0 ) {

            Malloc ( LIST->start, N_STR , 1 );                     /*memory allocation*/

            LIST -> start -> next = LIST -> start -> prev = NULL; /*initialize LIST*/

            Malloc ( LIST -> start -> data, char , length ); /*allocating data array*/
            Strcpy( item , LIST -> start -> data , length ); /*filling data array*/
            LIST -> start -> length = length ; /*saving the length of our string*/

            LIST -> end = LIST -> start;      /*first node is OK now*/


            }

    else {  /*we already have a first node*/

            Malloc ( LIST -> end -> next , N_STR , 1 ); /*allocating next item array*/

            /*moving around nodes to make the new item be the last of the list*/
            LIST -> end -> next -> prev = LIST -> end;
            LIST -> end = LIST -> end -> next;
            LIST -> end -> next = NULL;

            Malloc( LIST -> end -> data, char , length ); /*allocating data array*/
            Strcpy( item , LIST -> end -> data , length ); /*filling data array*/
            LIST -> end -> length = length ; /*saving the length of our string*/
            }

    /*if we are here, we have successfully add one item, so*/
    LIST->nb_item++; /*we increase nb_item*/

    return TRUE; /*that's ok for us , let's say it to our caller ;) */

    } /*add_last(...)*/



/*!\fn N_STR *get_first( N_STRLIST *LIST )
 *
 *\brief use this to get a pointer on the first list element
 *
 *\param LIST an N_STRLIST *object
 *
 *\return A pointer to the first element, else NULL
 */

N_STR *get_first( N_STRLIST *LIST )
    {

    /*check if the list is exisiting and filled by one element at least*/

    if ( LIST && LIST -> nb_item != 0 )
        return LIST->start;

    /*else return NULL*/
    return NULL;

    } /*get_first(...)*/



/*!\fn get_prev( N_STR **STR)
 *
 *\brief change the aim of the STR pointer to the previous item if existing
 *
 *\param STR an N_STR *object , who aim an item of the list
 *
 *\return TRUE if we got an item, FALSE if not
 */

int get_prev( N_STR **STR )
    {

    /*we have something to return*/

    if ( ( *STR ) && ( *STR ) -> prev != NULL ) {
            ( *STR ) = ( *STR ) -> prev;
            return TRUE;
            }

    /* it's finished*/
    else
        return FALSE;

    } /*get_prev(...)*/



/*!\fn get_next( N_STR **STR)
 *
 *\brief change the aim of the STR pointer to the next item if existing
 *
 *\param STR an N_STR *object , who aim an item of the list
 *
 *\return TRUE if we got an item, FALSE if not
 */

int get_next( N_STR **STR )
    {

    /*we have something to return*/

    if ( ( *STR ) && ( *STR ) -> next != NULL ) {
            ( *STR ) = ( *STR ) -> next;
            return TRUE;
            }

    /* it's finished*/
    else
        return FALSE;

    } /*get_next(...)*/



/*!\fn SeekAt( N_STRLIST *LIST , int pos)
 *
 *\brief return a pointer to the aimed item
 *
 *\param LIST an N_STRLIST *object
 *\param pos an int meaning the position you wanna point
 *
 *\return a pointer to 'pos' , NULL if nothing to aim
 */

N_STR *SeekAt( N_STRLIST *LIST , int pos )
    {

    /*tmp pointer*/
    N_STR * tmp;

    /*iterator*/

    int it = 1 ;

    /*check if LIST is ok, 'pos' is in range*/

    if ( !LIST || pos < 1 || pos > LIST -> nb_item )
        return NULL;

    tmp = LIST -> start;

    while ( it < pos && tmp != NULL ) {
            it++;
            tmp = tmp -> next;
            }

    return tmp;

    } /*SeekAt*/



/*!\fn int kill_first_item( N_STRLIST *LIST )
  *
  *\brief use this to kill the first item in the N_STRLIST *list
  *
  *\param LIST an N_STRLIST *object
  *
  *\return EMPTY if there is nothing to kill, TRUE if an item was killed, FALSE if the list was not created
  */

int kill_first_item( N_STRLIST *LIST )
    {

    N_STR * search = NULL; /*pointer to NULL*/

    if ( LIST ) { /*if it's an existing list*/

            if ( LIST -> nb_item == 0 )
                return EMPTY; /*if there was no item in*/

            if ( LIST -> nb_item > 1 ) { /*if we have more than one element*/

                    search = LIST -> start;     /*we wanna kill the first*/

                    LIST -> start = LIST -> start -> next; /*make the start point to the next element*/
                    LIST -> start -> prev = NULL; /*there is nothing before the first*/

                    /*freeing memory allocation*/
                    Free ( search -> data );
                    Free ( search );

                    LIST -> nb_item--; /*decrease the number of item*/

                    return TRUE; /*success*/

                    }

            else { /*else we have only one element, root of the list*/

                    /*freeing memory*/
                    Free( LIST -> start -> data );
                    Free( LIST -> start );

                    /*list is empty, so initialize nb_item and point to nothing*/
                    LIST -> nb_item = 0 ;
                    LIST -> start = LIST -> end = NULL ;

                    return TRUE;

                    } /*if( LIST -> nb_item > 1 )*/

            } /*if( LIST )*/

    return FALSE; /*the list was not an initialized list*/

    } /*kill_first_item(...)*/



/*!\fn kill_item_at( N_STRLIST *LIST , int pos )
*
*\brief kill the item at 'pos'
*
*\param LIST an N_STRLIST *object
*\param pos an int to tell the place you wanna kill
*
*\return TRUE on success, FALSE if not reachable, EMPTY if there was nothing in the list
*/

int kill_item_at( N_STRLIST *LIST , int pos )
    {

    N_STR * tmp; /*tmp pointer*/

    /*if list is initialized, if there are elements and if 'pos' is in data array*/

    if ( !LIST )
        return FALSE;

    if ( LIST -> nb_item <= 0 )
        return EMPTY;

    if ( pos > 0 && pos <= LIST -> nb_item ) {
            if ( pos == 1 ) {
                    /* ... clear root... */

                    if ( kill_first_item( LIST ) != TRUE )
                        return FALSE;   /* error when killing item */

                    return TRUE; /*success*/
                    }

            if ( pos == LIST -> nb_item ) {
                    /* ... kill the end... */

                    if ( kill_last_item( LIST ) != TRUE )
                        return -1;
                    }

            else {
                    /* ... get the aimed knot... */
                    tmp = SeekAt( LIST , pos );

                    if ( tmp != NULL )    /* ...if it is a right knot... */
                            {
                            /* ...unlink from previous and next... */
                            tmp -> prev -> next = tmp -> next;
                            tmp -> next -> prev = tmp -> prev;
                            /* ...free the node... */
                            free( tmp -> data );
                            free( tmp );
                            /* ... - 1 nb_item ... */
                            --LIST -> nb_item;
                            }
                    else
                        return FALSE;   /* not a right knot*/
                    }
            }

    else
        return FALSE;   /*out of list range*/

    /* if we're here, it's cool because we have successfull use the function*/
    return TRUE;

    } /*kill_item_at(...)*/



/*!\fn kill_last_item( N_STRLIST *LIST )
 *
 *\brief kill the item at the end of list
 *
 *\param LIST an N_STRLIST *object
 *
 *\return TRUE on success, FALSE if not reachable, EMPTY if there was nothing in the list
 */

int kill_last_item( N_STRLIST *LIST )
    {

    /*tmp pointer*/
    N_STR * tmp;

    /* not an initialized list*/

    if ( !LIST )
        return FALSE;

    if ( LIST -> nb_item > 0 ) {
            /* ... get the one before the last...               */
            tmp = LIST -> end -> prev;
            /* ... get it point to null on next item...         */
            tmp -> next = NULL;
            /* ...    free the last    ...                      */
            free( LIST -> end );
            /* ... and make the new be the last  ...            */
            LIST -> end = tmp;
            /* ... decrease nb item ...                         */
            --LIST -> nb_item;
            }

    else
        return EMPTY;   /* no element to kill*/

    /*it's all right*/
    return TRUE;

    } /*kill_last_item(...)*/



/*!\fn empty_list( N_STRLIST *LIST )
  *
  *\brief use this to empty an N_STRLIST *list
  *
  *\param LIST an N_STRLIST *object
  *
  *\return EMPTY if empty list, FALSE if it was already empty or never filled
  */

int empty_list( N_STRLIST *LIST )
    {

    int it; /*iterator*/

    it = kill_first_item ( LIST ); /*getting first shoot result*/

    while ( it == TRUE )
        it = kill_first_item ( LIST ); /*killing all killable*/

    return it;

    } /*empty_list(...)*/



#endif /* #ifndef NOSTR */








