/**\file iso_anim.c
*
*  2D animations functions and library for IsoEngine
*
*\author Castagnier Mickal
*
*\version 1.0
*
*\date 19/07/05
*
*/



#include "nilorea.h"



#ifndef NOISOENGINE



/*******************************************************************************
                              SPRITE ROUTINES
*******************************************************************************/



/*!\fn init_sprite( SPRITE **spr , char *name , int w , int h , int x , int y , int id , int type , int animated , int delay , int max )
 *
 *\brief  init a SPRITE object with given parameters (pass -1 for id and delay if unavailable)
 *
 *\param spr SPRITE object to init
 *\param name the name of the sprite
 *\param w width of the sprite(s) in the SPRITE object
 *\param h height of the sprite(s) in the SPRITE object
 *\param x X position of the center of the sprite
 *\param y Y position of the center of the sprite
 *\param id identifiant of sprite in  a lib , -1 if unavailable
 *\param type  0 -> normal ; blit 1 -> MASKED ; 2 -> translucent
 *\param animated TRUE or FALSE
 *\param delay -1 or number of milisec between frames
 *\param max number of maximum images in SPRITE object
 *
 *\return TRUE or FALSE
 */

int init_sprite( SPRITE **spr , char *name ,
                 int w , int h ,
                 int x , int y ,
                 int id,
                 int type ,
                 int animated ,
                 int delay ,
                 int max )
    {

    /* allocating */

    Malloc( *spr , SPRITE , 1 );

    if ( !spr )
        return FALSE;

    if ( name ) {

            ( *spr ) -> namelenght = strlen( name );

            Malloc( ( *spr ) -> name , char , ( *spr ) -> namelenght + 1 );
            fill_str( ( *spr ) -> name , 0 , ( *spr ) -> namelenght + 1 );
            Strcpy( name , ( *spr ) -> name , ( *spr ) -> namelenght );
            }


    ( *spr ) -> w = w ;
    ( *spr ) -> h = h ;
    ( *spr ) -> x = x ;
    ( *spr ) -> y = y ;
    ( *spr ) -> id = id ;
    ( *spr ) -> type = type ;
    ( *spr ) -> animated = animated ;
    ( *spr ) -> delay = delay ;
    ( *spr ) -> current = 0 ;
    ( *spr ) -> alive = FALSE ;

    if ( max != -1 ) {

            ( *spr ) -> max = max;

            /* allocating image data array */

            Malloc( ( *spr ) -> spr , BITMAP * , max );

            if ( !( *spr ) -> spr )
                return FALSE;

            Malloc( ( *spr ) -> handles , GLuint , max );

            if ( !( ( *spr ) -> handles ) )
                return FALSE;

            }


    return TRUE;

    } /* init_sprite(...) */



/*!\fn add_bmp_to_spr( BITMAP *bmp , SPRITE *spr )
 *
 *\brief Add if possible a bimtap to a sprite
 *
 *\param bmp a bitmap to add to the sprite
 *\param spr a SPRITE where to add the bmp
 *
 *\return FALSE if spr is full or if error, TRUE else
 */

int add_bmp_to_spr( BITMAP *bmp , SPRITE *spr )
    {

    /* checking space */

    if ( spr -> current >= spr -> max )
        return FALSE;

    spr -> spr[ spr -> current ] = bmp ;

    spr -> alive = TRUE ;

    if ( gfx_status( GET , 0 ) == MODE_OGL )
        spr -> handles[ spr -> current ] = allegro_gl_make_texture( spr -> spr[ spr -> current ] );

    spr -> current += 1;


    return TRUE;

    } /* add_bmp_to_sprite(...) */



/*!\fn read_sprite_in_file( PACKFILE *file , SPRITE **spr )
 *
 *\brief read a sprite in an opened file
 *
 *\param file an opened PACKFILE *file to read in
 *\param spr a SPRITE pointer to point the sprite in memory
 *
 *\return FALSE if not valid file or writing error or memory error TRUE else
 */

int read_sprite_in_file( PACKFILE *file , SPRITE **spr )
    {

    int it;

    /* error checking*/

    if ( !file )
        return FALSE;

    if ( init_sprite( &( *spr ) , NULL , -1, -1, -1, -1, -1, -1, -1, -1, -1 ) == FALSE )
        return FALSE;


    ( *spr ) -> w = pack_igetl( file );

    ( *spr ) -> h = pack_igetl( file );

    ( *spr ) -> x = pack_igetl( file );

    ( *spr ) -> y = pack_igetl( file );

    ( *spr ) -> id = pack_igetl( file );

    ( *spr ) -> type = pack_igetl( file );

    ( *spr ) -> animated = pack_igetl( file );

    ( *spr ) -> delay = pack_igetl( file );

    ( *spr ) -> current = pack_igetl( file );

    ( *spr ) -> max = pack_igetl( file );

    ( *spr ) -> alive = pack_igetl( file );

    if ( ( *spr ) -> alive == TRUE ) {

            ( *spr ) -> namelenght = pack_igetl( file );

            Malloc( ( *spr ) -> name , char , ( *spr ) -> namelenght + 1 );

            fill_str( ( *spr ) -> name , 0 , ( *spr ) -> namelenght + 1 );

            pack_fread( ( *spr ) -> name , ( *spr ) -> namelenght , file );

            Malloc( ( *spr ) -> spr , BITMAP * , ( *spr ) -> max );

            Malloc( ( *spr ) -> handles , GLuint , ( *spr ) -> max );

            for ( it = 0 ; it < ( *spr ) -> current ; it ++ ) {

                    ( *spr ) -> spr[ it ] = load_bmp_pf( file , NULL );

                    if ( gfx_status( GET , 0 ) == MODE_OGL )
                        ( *spr ) -> handles[ it ] = allegro_gl_make_texture( ( *spr ) -> spr[ it ] );

                    } /* for(it...) */
            }


    return TRUE;


    } /* read_sprite_in_file(...) */



/*!\fn write_sprite_in_file( PACKFILE *file , SPRITE *spr )
 *
 *\brief write a sprite in an opened file
 *
 *\param file an opened PACKFILE *file to write in
 *\param spr a SPRITE pointer to know the data
 *
 *\return FALSE if not valid file or writing error or memory error TRUE else
 */

int write_sprite_in_file( PACKFILE *file , SPRITE *spr )
    {

    int it;

    /* error checking*/

    if ( !file )
        return FALSE;

    if ( !spr )
        return FALSE;

    pack_iputl( spr -> w , file );

    pack_iputl( spr -> h , file );

    pack_iputl( spr -> x , file );

    pack_iputl( spr -> y , file );

    pack_iputl( spr -> id , file );

    pack_iputl( spr -> type , file );

    pack_iputl( spr -> animated , file );

    pack_iputl( spr -> delay , file );

    pack_iputl( spr -> current , file );

    pack_iputl( spr -> max , file );

    pack_iputl( spr -> alive , file );

    if ( spr -> alive == TRUE ) {

            pack_iputl( spr -> namelenght , file );
            pack_fwrite( spr -> name , spr -> namelenght , file );

            for ( it = 0 ; it < spr -> current ; it ++ ) {
                    save_bmp_pf( file , spr -> spr[ it ] , NULL );

                    }
            }

    return TRUE;


    } /* write_sprite_in_file( ... ) */



/*!\fn blit_sprite( SPRITE *sprite , int x , int y , int frame , BITMAP *bmp )
 *
 *\brief draw a sprite at the right position
 *
 *\param sprite a pointer where the data are
 *\param x where to put sprite on BITMAP , X
 *\param y where to put sprite on BITMAP , Y
 *\param frame frame of sprite to blit
 *\param bmp BITMAP object where blit sprite
 *
 *\return TRUE or FALSE (if not valid frame ie)
 */

int blit_sprite( SPRITE *sprite , int x , int y , int frame , BITMAP *bmp )
    {


    if ( !sprite || sprite -> alive == FALSE || frame < 0 || frame >= sprite -> current )
        return FALSE;

    switch ( sprite -> type ) {

                /* normal blit */

                case NORMAL_SPRITE :

                if ( bmp )
                    blit( sprite -> spr[ frame ] , bmp ,
                          0 , 0 ,
                          x - sprite -> x ,
                          y - sprite -> y ,
                          sprite -> w,
                          sprite -> h
                        );
                else
                    ogl_blit( sprite -> handles[ frame ] , x - sprite -> x , y - sprite -> y , 1 , 1, 1 , 0, sprite -> w , sprite -> h , sprite -> type );



                return TRUE;

                break;


                /* masked blit */

                case MASKED_SPRITE:

                if ( bmp ) {
                        draw_sprite( bmp , sprite -> spr[ frame ] , x - sprite -> x , y - sprite -> y );
                        }

                else


                    ogl_blit( sprite -> handles[ frame ] , x - sprite -> x , y - sprite -> y , 1 , 1, 1 , 0, sprite -> w , sprite -> h , sprite -> type );


                return TRUE;

                break;


                /* translucent blit */

                case TRANS_SPRITE:

                if ( bmp ) {
                        set_alpha_blender();
                        draw_trans_sprite( bmp , sprite -> spr[ frame ] , x - sprite -> x , y - sprite -> y );
                        }

                else

                    ogl_blit( sprite -> handles[ frame ] , x - sprite -> x , y - sprite -> y , 1 , 1 , 1 , 0, sprite -> w , sprite -> h , sprite -> type );


                return TRUE;

                break;


                /* Sprite has no type ! */

                default:

                return FALSE;

                break;

            } /*switch(...)*/

    return TRUE;

    } /* blit_sprite(...) */



/*!\fn free_sprite( SPRITE **spr )
 *
 *\brief Free a sprite from memory
 *
 *\param spr A SPRITE *spr to kill
 *
 *\return TRUE
 */

int free_sprite( SPRITE **spr )
    {

    int it;

    for ( it = 0 ; it < ( *spr ) -> max ; it ++ )
        Free( ( *spr ) -> spr[ it ] );

    Free( ( *spr ) -> spr );

    Free( ( *spr ) -> name );

    Free( ( *spr ) );

    return TRUE;

    } /* free_sprite(..) */



/*******************************************************************************
                             ANIMATION ROUTINES
*******************************************************************************/



/*!fn create_animlib( ANIM **lib , char *name , int nbspr )
 *
 *\brief create a big library of animations
 *
 *\param lib an animations library
 *\param name name of library
 *\param nbspr number of animations in library
 *
 *\return TRUE or FALSE
 */

int create_animlib( ANIM **lib , char *name , int nbspr )
    {

    int it = 0;

    /* allocating and testing anim struct */
    Malloc( lib , ANIM *, 1 );

    if ( !( *lib ) )
        return FALSE;


    /* allocating and testing sprite array of anim */
    Malloc( ( *lib ) -> data , SPRITE * , nbspr );

    for ( it = 0 ; it < nbspr ; it ++ ) {
            ( *lib ) -> data[ it ] -> alive = FALSE;
            }


    if ( !( *lib ) ->data )
        return FALSE;


    /* saving nb max spr */
    ( *lib ) -> max = nbspr;


    /* allocating, testing and saving name */
    if ( name ) {

            Malloc( ( *lib ) -> name , char , strlen( name ) + 1 );

            if ( !( *lib ) -> name )
                return FALSE;

            strcpy( name , ( *lib ) -> name );

            }

    /* initialising counter */
    ( *lib ) -> hightest = 0;

    return TRUE;

    } /* create_animlib(...) */



/*!\fn load_animlib( PACKFILE *file , ANIM **lib )
 *
 *\brief  load an anim file from disk , return FALSE if : not existing and error creating : existing and error reallocating if not enought big)
 *
 *\param file an opened PACKFILE *file where read an anim lib
 *\param lib a pointer to the futur library
 *
 *\return TRUE on success FALSE on failure
 */

int load_animlib( PACKFILE *file , ANIM **lib )
    {


    int it = 0;

    /* error checking */
    Malloc( ( *lib ) , ANIM , 1 );

    if ( !( *lib ) )
        return FALSE;

    if ( !file )
        return FALSE;

    /*reading data*/
    ( *lib ) -> hightest = pack_igetl( file );

    ( *lib ) -> max = pack_igetl( file );

    ( *lib ) -> namelenght = pack_igetl( file );

    Malloc( ( *lib ) -> name , char , ( *lib ) -> namelenght + 1 );

    fill_str( ( *lib ) -> name , 0 , ( *lib ) -> namelenght + 1 );

    pack_fread( &( *lib ) -> name[ it ], ( *lib ) -> namelenght , file );

    Malloc( ( *lib ) -> data , SPRITE * , ( *lib ) -> hightest );

    for ( it = 0 ; it < ( *lib ) -> hightest ; it ++ )

        read_sprite_in_file( file, &( *lib ) -> data[ it ] );

    return TRUE;

    } /* load_animlib(...) */



/*!\fn save_animlib( PACKFILE *file , ANIM *lib )
 *
 *\brief save an anim library to disk ( cropping the various arrays )
 *
 *\param file an opened PACKFILE *file where save an anim lib
 *\param lib the library to save
 *
 *\return FALSE if error TRUE else
 */

int save_animlib( PACKFILE *file , ANIM *lib )
    {

    int it;

    /* error checking */

    if ( !lib )
        return FALSE;


    if ( !file )
        return FALSE;

    /*writting data*/

    pack_iputl( lib -> hightest , file );

    pack_iputl( lib -> max , file );

    pack_iputl( lib -> namelenght , file );

    pack_fwrite( lib -> name , lib -> namelenght , file );

    for ( it = 0 ; it < lib -> max ; it ++ )
        write_sprite_in_file( file , lib -> data[ it ] );

    return TRUE;

    } /* save_animlib(...) */



/*!\fn add_sprite_to_lib( SPRITE *spr , ANIM *lib )
 *
 *\brief add sprite to lib
 *
 *\param spr SPRITE to add
 *\param lib ANIM lib where add the sprite
 *
 *\return TRUE on success, FALSE on error
 */

int add_sprite_to_lib( SPRITE *spr , ANIM *lib )
    {

    int current = 0 ;

    /* error checking */

    if ( lib -> hightest + 1 > lib -> max )
        return FALSE;

    /* searching first free place */
    while ( lib -> data[ current ] -> spr )
        current ++;

    /* pointing to anim */
    lib -> data[ current ] = spr;

    /* saving sprite id */
    lib -> data[ current ] -> id = current;

    /* updating new position */
    lib -> hightest += 1;

    return TRUE;

    } /* add_sprite_to_lib(...) */



/*!\fn remove_sprite_from_lib( ANIM *lib , int id )
 *
 *\brief remove sprite from library
 *
 *\param lib the lib where remove sprite
 *\param id the id of the sprite to remove
 *
 *\return TRUE on success, FALSE on error
 */

int remove_sprite_from_lib( ANIM *lib , int id )
    {


    if ( !lib -> data[ id ] )
        return FALSE;


    free_sprite( &lib -> data[ id ] );

    if ( id == lib -> data[ id ] -> current ) {
            lib -> data[ id ] -> current -= 1;
            }


    return TRUE;

    } /* remove_sprite(...)*/



/*!\fn  free_anim( ANIM *lib )
 *
 *\brief free an anim library
 *
 *\param lib a lib to free
 *
 *\return TRUE or FALSE
 */

int free_anim( ANIM *lib )
    {

    int it = 0;

    if ( !lib )
        return FALSE;

    for ( it = 0 ; it < lib -> max ; it ++ )
        if ( lib -> data[ it ] )
            free_sprite( &lib -> data[ it ] );

    Free( lib -> name );

    Free( lib -> data );

    Free( lib );

    return TRUE;

    } /* free_anim(...) */



/*!\fn resize_anim_lib( ANIM **lib , int nb )
 *
 *\brief Resize your lib for reach (if possible) nb elements. It will reduce to a maximum of lib->hightest elements, or grow to memory allowed maximum elements. If an error occur, a roll back will be performed.
 *
 *\param lib A pointer to a ANIM *library
 *\param nb An int value to reach
 *
 *\return 1 if not possible but not error , 2 if not possible but roll back is successfull , 3 if success
 */

int resize_anim_lib( ANIM **lib , int nb )
    {

    ANIM * tmp;

    int it = 0 , it1 = 0;

    if ( nb < ( *lib ) -> hightest )
        return 2;

    if ( nb == ( *lib ) -> hightest || nb == ( *lib ) -> max )
        return 3;

    if ( create_animlib( &tmp , ( *lib ) -> name , nb ) == FALSE ) {
            free_anim( tmp );
            return 2;
            }

    tmp -> nbsprite = ( *lib ) -> nbsprite;
    tmp -> hightest = ( *lib ) -> hightest;
    tmp -> max = ( *lib ) -> max;

    for ( it = 0 ; it < tmp -> hightest ; it ++ ) {
            if ( ( *lib ) -> data[ it ] ) {

                    tmp -> data[ it ] -> w = ( *lib ) -> data[ it ] -> w;
                    tmp -> data[ it ] -> h = ( *lib ) -> data[ it ] -> h;
                    tmp -> data[ it ] -> x = ( *lib ) -> data[ it ] -> x;
                    tmp -> data[ it ] -> y = ( *lib ) -> data[ it ] -> y;
                    tmp -> data[ it ] -> id = ( *lib ) -> data[ it ] -> id;
                    tmp -> data[ it ] -> type = ( *lib ) -> data[ it ] -> type;
                    tmp -> data[ it ] -> animated = ( *lib ) -> data[ it ] -> animated ;
                    tmp -> data[ it ] -> delay = ( *lib ) -> data[ it ] -> delay;
                    tmp -> data[ it ] -> current = ( *lib ) -> data[ it ] -> current;
                    tmp -> data[ it ] -> max = ( *lib ) -> data[ it ] -> max;

                    for ( it1 = 0 ; it1 < tmp -> max ; it1++ ) {


                            if ( ( *lib ) -> data[ it ] -> handles ) {
                                    tmp -> data[ it1 ] -> handles = ( *lib ) -> data[ it1 ] -> handles ;
                                    }

                            if ( ( *lib ) -> data[ it ] -> spr [ it1 ] ) {
                                    tmp -> data[ it ] -> spr[ it1 ] = create_bitmap( tmp -> data[ it ] -> w , tmp -> data[ it ] -> h );

                                    if ( !tmp -> data[ it ] -> spr[ it1 ] ) {
                                            free_anim( tmp );
                                            return 2;
                                            }

                                    /*            blit( (*lib) -> data[ it ] -> spr[ it1 ] , tmp -> data[ it ] -> spr[ it1 ] , 0 , 0 , 0 , 0 , tmp -> data[ it ] ->w , tmp -> data[ it ] -> h;
                                      */
                                    memcpy( tmp -> data[ it ] -> spr[ it1 ] , ( *lib ) -> data[ it ] -> spr[ it1 ] , sizeof( ( *lib ) -> data[ it ] -> spr[ it1 ] ) );
                                    }

                            }

                    }

            }

    free_anim( ( *lib ) );

    ( *lib ) = tmp;

    return 3;

    }



/*!\fn blit_anim( ANIM *lib, int id , int x , int y , int frame , BITMAP *bmp )
 *
 *\brief draw a sprite at the right position
 *
 *\param lib an ANIM library where found the sprite
 *\param id ident of the sprite to blit
 *\param x where to put sprite on BITMAP , X
 *\param y where to put sprite on BITMAP , Y
 *\param frame frame of sprite to blit
 *\param bmp BITMAP object where blit sprite
 *
 *\return TRUE or FALSE (if not valid frame ie)
 */

int blit_anim( ANIM *lib , int id , int x , int y , int frame , BITMAP *bmp )
    {

    if ( !lib -> data[ id ] )
        return FALSE;

    return blit_sprite( lib -> data[ id ] , x , y, frame , bmp );

    } /* blit_anim(...) */



/*!\fn exist_sprite( ANIM *lib , char *name )
 *
 *\brief check if a sprite already exist in a lib
 *
 *\return If exist, it return the aimed sprite index, it return -2
 */

int exist_sprite( ANIM *lib , char *name )
    {

    int it = 0;
    int found = -2;

    if ( !lib )
        return -2;

    if ( lib -> hightest == 0 )
        return -2;


    while ( found == -2 && it < lib -> hightest ) {

            if ( lib -> data[ it ] && lib -> data[ it ] -> name ) {
                    if ( strcmp( lib -> data[ it ] -> name , name ) != 0 )
                        it ++;
                    else
                        found = it;
                    } /* if( ... ) */

            } /* while( ... ) */

    return found;


    } /* exist_sprite( ... ) */



#endif /* #ifndef NOISOENGINE */








