/**\file common.c
*
*  common function
*
*\author Castagnier Mickal
*
*\version 1.0
*
*\date 24/03/05
*
*/



#include "nilorea.h"



/*!\fn PAUSE(void)
 *
 *\brief make a pause in a terminal
 */

void PAUSE()
    {

    char k[ 2 ];

    printf( "Press enter to continue..." );

    fflush( stdin );

    scanf( "%c", k );

    printf( "\n" );

    } /* PAUSE(...) */



/*!\fn start_HiTimer( N_TIME *timer )
 *
 *\brief Initialize or restart from zero any N_TIME HiTimer
 *
 *\param timer Any N_TIMER *timer you wanna start or reset
 *
 */

void start_HiTimer( N_TIME *timer )
    {

    timer -> delta = 0;

#ifndef LINUX
    QueryPerformanceFrequency( ( LARGE_INTEGER * ) & timer -> freq );
    QueryPerformanceCounter( &timer -> startTime );
#else
    gettimeofday( &timer -> startTime, 0 );
#endif

    } /* init_HiTimer(...) */



/*!\fn get_usec( N_TIME  *timer )
 *
 *\brief Poll any N_TIME HiTimer, returning usec, and moving currentTime to startTime
 *
 *\param timer Any N_TIMER *timer you wanna poll
 *
 *\return The elapsed number of usec for the given N_TIME *timer
 */

time_t get_usec( N_TIME *timer )
    {

#ifdef WIN32

    QueryPerformanceCounter( ( LARGE_INTEGER * ) & timer -> currentTime );

    timer -> delta = 1000000 * ( timer -> currentTime . QuadPart - timer -> startTime . QuadPart ) / timer -> freq . QuadPart ;

    timer -> startTime = timer -> currentTime ;

#else

    gettimeofday( &timer -> currentTime, 0 );

    timer -> delta = ( timer -> currentTime . tv_sec   - timer -> startTime . tv_sec  ) * 1000000 +
            ( timer -> currentTime . tv_usec  - timer -> startTime . tv_usec );

    timer -> startTime . tv_sec  = timer -> currentTime . tv_sec ;
    timer -> startTime . tv_usec = timer -> currentTime . tv_usec ;

#endif

    return timer -> delta;

    } /* get_usec( ... ) */




/*!\fn get_msec( N_TIME  *timer )
 *
 *\brief Poll any N_TIME HiTimer, returning msec, and moving currentTime to startTime
 *
 *\param timer Any N_TIMER *timer you wanna poll
 *
 *\return The elapsed number of msec for the given N_TIME *timer
 */

time_t get_msec( N_TIME *timer ){

#ifdef WIN32

    QueryPerformanceCounter( ( LARGE_INTEGER * ) & timer -> currentTime );

    timer -> delta = 1000 * ( timer -> currentTime . QuadPart - timer -> startTime . QuadPart ) / timer -> freq . QuadPart ;

    timer -> startTime = timer -> currentTime ;

#else

    gettimeofday( &timer -> currentTime, 0 );

    timer -> delta = ( timer -> currentTime . tv_sec   - timer -> startTime . tv_sec  ) * 1000 +
            ( timer -> currentTime . tv_usec  - timer -> startTime . tv_usec ) / 1000 ;

    timer -> startTime . tv_sec  = timer -> currentTime . tv_sec ;
    timer -> startTime . tv_usec = timer -> currentTime . tv_usec ;

#endif

    return timer -> delta;
} /* get_msec(...) */



/*!\fn get_sec( N_TIME  *timer )
 *
 *\brief Poll any N_TIME HiTimer, returning sec, and moving currentTime to startTime
 *
 *\param timer Any N_TIMER *timer you wanna poll
 *
 *\return The elapsed number of sec for the given N_TIME *timer
 */

time_t get_sec( N_TIME *timer ){

#ifdef WIN32

    QueryPerformanceCounter( ( LARGE_INTEGER * ) & timer -> currentTime );

    timer -> delta = ( timer -> currentTime . QuadPart - timer -> startTime . QuadPart ) / timer -> freq . QuadPart ;

    timer -> startTime = timer -> currentTime ;

#else

    gettimeofday( &timer -> currentTime, 0 );

    timer -> delta = ( timer -> currentTime . tv_sec   - timer -> startTime . tv_sec  )  +
            ( timer -> currentTime . tv_usec  - timer -> startTime . tv_usec ) / 1000000 ;

    timer -> startTime . tv_sec  = timer -> currentTime . tv_sec ;
    timer -> startTime . tv_usec = timer -> currentTime . tv_usec ;

#endif

    return timer -> delta;

}/* get_sec(...) */



#ifndef NOPTHREAD



/*!\fn GetAppStatus( PROCESS *proc)
 *
 *\brief Get the status of the given process
 *
 *\param proc Any Initialized PROCESS *variable
 *
 *\return Return the current status of an application, or FALSE on error.
 *
 */

int GetAppStatus( PROCESS *proc )
    {

    int tmp = DEFAULT;

    if( !proc )
        return FALSE;

    pthread_mutex_lock( &proc -> APP_STATUS_MUTEX );

    tmp = proc -> APP_STATUS ;

    pthread_mutex_unlock( &proc -> APP_STATUS_MUTEX );

    return tmp;

    } /* GetAppStatus(...) */



/*!\fn SetAppStatus( PROCESS *proc , int status )
 *
 *\brief Set the status of an application
 *
 *\param proc A PROCESS *proc to change
 *
 *\param status Value to assign: DEFAULT,RUNNING,STOPWANTED,STOPPED,PAUSED
 *
 *\warning If proc isn't right, we will try to Malloc it
 */

int SetAppStatus( PROCESS *proc , int status )
    {

    if ( !proc ){

        Malloc( proc , PROCESS , 1 );

    if ( !proc )
        return FALSE;

    pthread_mutex_init( &proc -> APP_STATUS_MUTEX , NULL );

    }

    pthread_mutex_lock( &proc -> APP_STATUS_MUTEX );

    switch ( status ) {

                case RUNNING:
                proc -> APP_STATUS = RUNNING;
                break;

                case STOPWANTED:
                proc -> APP_STATUS = STOPWANTED;
                break;

                case STOPPED:
                proc -> APP_STATUS = STOPPED;
                break;

                case PAUSED:
                proc -> APP_STATUS = PAUSED;
                break;

                case DEFAULT:
                proc -> APP_STATUS = DEFAULT;
                break;

                default:
                proc -> APP_STATUS = DEFAULT;
                break;
            }

    pthread_mutex_unlock( &proc -> APP_STATUS_MUTEX );

    return TRUE;

    } /* SetAppStatus(...) */



/*!\fn open_safe_logging( TS_LOG *log , char *pathname , char *opt )
 *
 *\brief Open a thread-safe logging file
 *
 *\param log A TS_LOG handler
 *\param pathname The file path (if any) and name
 *\param opt Options for opening (please never forget to use "w")
 *
 *\return TRUE on success , FALSE on error , -1000 if already open
 */

int open_safe_logging( TS_LOG *log , char *pathname , char *opt ) {

    if( log ){

        if( log -> file )
            return -1000; /* already open */

        return FALSE;
    }

    if( !pathname ) return FALSE; /* no path/filename */

    Malloc( log , TS_LOG , 1 );

    if( !log )
        return FALSE;

    pthread_mutex_init( &log -> LOG_MUTEX , NULL );

    log -> file = fopen( pathname , opt );

    if( !log -> file )
        return FALSE;

    return TRUE;

}/* open_safe_logging(...) */



/*!\fn write_safe_log( TS_LOG *log , char *pat , ... )
 *
 *\brief write to a thread-safe logging file
 *
 *\param log A TS_LOG handler
 *\param pat Pattern for writting (i.e "%d %d %s")
 *
 *\return TRUE on success, FALSE on error
 */

int write_safe_log( TS_LOG *log , char *pat , ... ) {

    /* argument list */
    va_list arg;
    char str[2048];

    if( !log )
        return FALSE;

    va_start( arg , pat );

    uvszprintf( str , sizeof(str), pat, arg );

    va_end( arg );

    pthread_mutex_lock( &log -> LOG_MUTEX );

    fprintf( log -> file , "%s" , str );

    pthread_mutex_unlock( &log -> LOG_MUTEX );

    return TRUE;

} /* write_safe_log( ... ) */



/*!\fn close_safe_logging( TS_LOG *log )
 *
 *\brief close a thread-safe logging file
 *
 *\param log A TS_LOG handler
 *
 *\return TRUE on success, FALSE on error, -1000 if already closed
 */

int close_safe_logging( TS_LOG *log ){

    if( !log )
        return -1000;

    pthread_mutex_destroy( &log -> LOG_MUTEX );

    fclose( log -> file );

    return TRUE;

}/* close_safe_logging( ...) */



#endif /* #ifndef NOPTHREAD */



#ifndef NOALLEGRO

/*!\fn get_keyboard( char *keybuf , int *cur , int max ){
 *
 *\brief Function for getting a keyboard input, unicode
 *
 *\param keybuf a valid char buffer
 *\param cur a pointer to the current_cursor_position
 *\param max the string size limit
 */

void get_keyboard( char *keybuf , int *cur , int max )
    {

    int k = 0;

    if ( keypressed() ) {

            k = ureadkey( NULL );

            if ( k != 0x1b &&
                    k != 0x09 &&
                    k != 0x00 &&
                    k != 0x0d &&
                    k != 0x08 &&
                    ( *cur ) < max ) {

                    usprintf( keybuf + strlen( keybuf ) , "%c" , k );

                    ( *cur ) ++;

                    } /* if( k != ... && ) */

            if ( k == 0x08 && (*cur) > 0 )  /* backspace */
                    {

                    ( *cur ) --;

                    uremove( keybuf , ( *cur ) );
                    }

            } /* if( keypressed( ... ) ) */

    } /* get_keyboard(...) */


#endif /* #ifndef NOALLEGRO */
