// Wormlings main function and init/shutdown routines.
// -----------------------------------------------------------------------------
// $Revision: 150 $ $Date: 2005-03-03 13:42:08 +0100 (to, 03 mar 2005) $
// -----------------------------------------------------------------------------
// Copyright 2000-2005 Daniel Schlyder.
// Distributed under the GNU General Public License; as published by the Free
// Software Foundation; either version 2 of the License, or (at your option) any
// later version. (See accompanying file LICENSE.txt or copy at
// http://www.gnu.org/licenses/gpl.html)



#include "graphics.hpp"
#include "audio.hpp"
#include "menu.hpp"
#include "game.hpp"
#include "utility.hpp"

#include <allegro.h>
#include <jpgalleg.h>

#include <string>
#include <vector>
#include <algorithm>
#include <cstdlib> // srand()
#include <ctime> // time()



namespace {

bool init(bool runWindowedIsSet);
void shutdown();
void loadGameSettings();
int loadGameSetting(
    std::string const &name,
    std::vector<int> const &valid_values, int default_value
);

} // namespace



int main(int argc, char *argv[])
{
#ifdef NDEBUG
    three_finger_flag = false;
#endif

    // these have been put here instead of in init() so allegro_message()
    // can be used to give feedback on command-line input
    allegro_init();
    set_window_title("Wormlings");
    
    // process command-line options and initialize program
    bool runWindowedIsSet(false);

    for (int i = 1; i < argc; ++i)
    {
        std::string argument(argv[i]);
        if (argument == "-w" || argument == "--windowed")
        {
            runWindowed = true;
            runWindowedIsSet = true;
        }
        else if (argument == "-f" || argument == "--fullscreen")
        {
            runWindowed = false;
            runWindowedIsSet = true;
        }
        else if (argument == "--fps")
        {
            display_fps_counter = true;
        }
        else
        {
            allegro_message(
                "Usage: wormlings [OPTIONS]\n\n"
                "Options:\n\n"
                "-w, --windowed\n\tStart game in windowed mode.\n"
                "-f, --fullscreen\n\tStart game in fullscreen mode.\n\n"
            );
            return 1;
        }
    }
    
    // initialise everything
    if (!init(runWindowedIsSet))
    {
        return 1;
    }
    
    menu();

    // clean up and exit
    shutdown();
    return 0;
}
END_OF_MAIN()



namespace {

// initialise Allegro components and other stuff
bool init(bool runWindowedIsSet)
{
    // install Allegro subsystems
    install_timer();
    install_keyboard();
   
    jpgalleg_init();
   
    override_config_file(fix_path("wormlings.cfg").c_str());
    
    if (!audio::instance().init())
    {
        return false;
    }
    
    int volume(get_config_int("audio", "volume", 200));
    if (!(volume % 25))
    {
        audio::instance().set_volume(volume);
    }
    
    if (!runWindowedIsSet)
    {
        runWindowed = get_config_int("Graphics", "runWindowed", 0);
    }
    
    g_colour_depth = std::max(15, desktop_color_depth());
    
    if (!initGraphics())
    {
        return false;
    }

    std::srand(std::time(0));
    
    loadGameSettings();

    // load control keys from config. file..
    // ..player 1
    players[0].keyLeft = get_config_int("Player1", "keyLeft", KEY_LEFT);
    players[0].keyRight = get_config_int("Player1", "keyRight", KEY_RIGHT);
    players[0].keyTurbo = get_config_int("Player1", "keyTurbo", KEY_UP);
    players[0].keyStop = get_config_int("Player1", "keyStop", KEY_DOWN);
    players[0].name = get_config_string("Player1", "name", "Red player");
    
    // ..player 2
    players[1].keyLeft = get_config_int("Player2", "keyLeft", KEY_A);
    players[1].keyRight = get_config_int("Player2", "keyRight", KEY_D);
    players[1].keyTurbo = get_config_int("Player2", "keyTurbo", KEY_W);
    players[1].keyStop = get_config_int("Player2", "keyStop", KEY_S);
    players[1].name = get_config_string("Player2", "name", "Green player");
    
    // ..player 3
    players[2].keyLeft = get_config_int("Player3", "keyLeft", KEY_4_PAD);
    players[2].keyRight = get_config_int("Player3", "keyRight", KEY_6_PAD);
    players[2].keyTurbo = get_config_int("Player3", "keyTurbo", KEY_8_PAD);
    players[2].keyStop = get_config_int("Player3", "keyStop", KEY_5_PAD);
    players[2].name =
        get_config_string("Player3", "name", "Yellow player");
    
    // ..player 4
    players[3].keyLeft = get_config_int("Player4", "keyLeft", KEY_J);
    players[3].keyRight = get_config_int("Player4", "keyRight", KEY_L);
    players[3].keyTurbo = get_config_int("Player4", "keyTurbo", KEY_I);
    players[3].keyStop = get_config_int("Player4", "keyStop", KEY_K);
    players[3].name = get_config_string("Player4", "name", "Cyan player");

    setPlayersColour();

    return true;
}



void shutdown()
{
    set_config_file(fix_path("wormlings.cfg").c_str());
    
    set_config_int("audio", "volume", audio::instance().volume());

    set_config_int("Graphics", "runWindowed", runWindowed);

    set_config_int("Game", "playersNum", playersNum);
    set_config_int("Game", "playersSize", playersSize);
    set_config_int("Game", "playersMaxLength", playersMaxLength);
    set_config_int("Game", "playersSlowGrowth", playersSlowGrowth);
    set_config_int("Game", "playersTurnSpeed", playersTurnSpeed);
    set_config_int("Game", "powerMax", powerMax);
    set_config_int("Game", "powerRegainSpeed", powerRegainSpeed);
    set_config_int("Game", "eraseTheDead", eraseTheDead);
    set_config_int("Game", "ballsAmount", ballsAmount);
    set_config_int("Game", "scoreLimit", scoreLimit);

    for (int i = 0; i < 4; ++i)
    {
        std::string section("Player" + toString(i + 1));
        set_config_string(section.c_str(), "name", players[i].name.c_str());
        set_config_int(section.c_str(), "keyLeft", players[i].keyLeft);
        set_config_int(section.c_str(), "keyRight", players[i].keyRight);
        set_config_int(section.c_str(), "keyTurbo", players[i].keyTurbo);
        set_config_int(section.c_str(), "keyStop", players[i].keyStop);
    }

    shutdownGraphics();
}



void loadGameSettings()
{
    std::vector<int> values;
    
    values.push_back(2);
    values.push_back(3);
    values.push_back(4);
    
    playersNum = loadGameSetting("playersNum", values, 2);
    
    values.clear();
    values.push_back(1);
    values.push_back(2);
    values.push_back(3);

    playersSize = loadGameSetting("playersSize", values, 2);

    values.clear();
    values.push_back(250);
    values.push_back(500);
    values.push_back(750);
    values.push_back(1000);
    values.push_back(1250);
    values.push_back(0);

    playersMaxLength = loadGameSetting("playersMaxLength", values, 750);

    values.clear();
    values.push_back(0);
    values.push_back(1);

    playersSlowGrowth = loadGameSetting("playersSlowGrowth", values, 1);

    values.clear();
    values.push_back(1);
    values.push_back(2);
    values.push_back(3);
    values.push_back(4);
    values.push_back(5);

    playersTurnSpeed = loadGameSetting("playersTurnSpeed", values, 3);

    values.clear();
    values.push_back(0);
    values.push_back(125);
    values.push_back(250);
    values.push_back(375);
    values.push_back(500);

    powerMax = loadGameSetting("powerMax", values, 250);

    values.clear();
    values.push_back(3);
    values.push_back(5);
    values.push_back(7);

    powerRegainSpeed = loadGameSetting("powerRegainSpeed", values, 5);

    values.clear();
    values.push_back(0);
    values.push_back(1);

    eraseTheDead = loadGameSetting("eraseTheDead", values, 1);

    values.clear();
    values.push_back(0);
    values.push_back(10);
    values.push_back(20);
    values.push_back(30);
    values.push_back(40);
    values.push_back(50);
    values.push_back(60);
    values.push_back(70);
    values.push_back(80);
    values.push_back(90);
    values.push_back(100);

    ballsAmount = loadGameSetting("ballsAmount", values, 50);

    values.clear();
    values.push_back(1);
    values.push_back(2);
    values.push_back(3);
    values.push_back(4);
    values.push_back(5);
    values.push_back(6);
    values.push_back(7);
    values.push_back(8);
    values.push_back(9);
    values.push_back(10);

    scoreLimit = loadGameSetting("scoreLimit", values, 5);
}



int loadGameSetting(
    std::string const &name,
    std::vector<int> const &valid_values, int default_value
)
{
    int value(get_config_int("Game", name.c_str(), default_value));
    
    if (value == default_value)
    {
        return value;
    }
    
    for (
        std::vector<int>::const_iterator it = valid_values.begin();
        it != valid_values.end();
        ++it
    )
    {
        if (*it == value)
        {
            return value;
        }
    }
    
    return default_value;
}

} // namespace
