// -----------------------------------------------------------------------------
// $Revision: 157 $ $Date: 2005-03-04 11:07:14 +0100 (fr, 04 mar 2005) $
// -----------------------------------------------------------------------------
// Copyright 2000-2005 Daniel Schlyder.
// Distributed under the GNU General Public License; as published by the Free
// Software Foundation; either version 2 of the License, or (at your option) any
// later version. (See accompanying file LICENSE.txt or copy at
// http://www.gnu.org/licenses/gpl.html)



#include "graphics.hpp"
#include "utility.hpp"
#include "alertstream.hpp"

#include <sstream>
#include <cassert>



BITMAP
    *buffer,
    *background,
    *logo,
    *g_bmp_char,
    *g_bmp_countdown;

bool runWindowed;

int g_colour_depth;



namespace {

bool setGraphicsMode(int w, int h, int depth, bool windowed);

} // namespace



bool initGraphics()
{
    if (!setGraphicsMode(640, 480, g_colour_depth, runWindowed))
    {
        switch (g_colour_depth)
        {
            case 15: g_colour_depth = 16; break;
            case 16: g_colour_depth = 15; break;
            case 24: g_colour_depth = 32; break;
            case 32: g_colour_depth = 24; break;
        }
        
        if (!setGraphicsMode(640, 480, g_colour_depth, runWindowed))
        {
            alertstream() << "Allegro error: " << allegro_error;
            return false;
        }
    }

    buffer = create_bitmap(SCREEN_W, SCREEN_H);
    clear_bitmap(buffer);

    if (set_display_switch_mode(SWITCH_PAUSE) == -1)
    {
        set_display_switch_mode(SWITCH_AMNESIA);
    }
    set_display_switch_callback(SWITCH_IN, displayBuf);

    // Load background texture and fill background bitmap.
    
    BITMAP *texture(load_bitmap(fix_path("data/bgtile.jpg").c_str(), 0));

    if (!texture)
    {
        alertstream() << "Error: Failed to load 'data/bgtile.jpg'";
        return false;
    }

    background = create_bitmap(SCREEN_W, SCREEN_H);
    for (int y = 0; y < SCREEN_H; y += texture->h)
    {
        for (int x = 0; x < SCREEN_W; x += texture->w)
        {
            blit(texture, background, 0, 0, x, y, texture->w, texture->h);
        }
    }
    destroy_bitmap(texture);
   
    logo = load_bitmap(fix_path("data/logo.jpg").c_str(), 0);
    if (!logo)
    {
        alertstream() << "Error: Failed to load 'data/logo.jpg'";
        return false;
    }
    
    g_bmp_char = create_bitmap(8, 8);
    if (!g_bmp_char)
    {
        alertstream() << "Error: Failed to create bitmap 'bmp_char'";
        return false;
    }
    
    g_bmp_countdown = create_bitmap(96, 96);
    if (!g_bmp_countdown)
    {
        alertstream() << "Error: Failed to create bitmap 'bmp_countdown'";
        return false;
    }
    
#ifdef USE_OLD_TEXT_API
    text_mode(-1);
#endif
    
    return true;
}



void shutdownGraphics()
{
    destroy_bitmap(g_bmp_countdown);
    destroy_bitmap(g_bmp_char);
    destroy_bitmap(logo);
    destroy_bitmap(background);
    destroy_bitmap(buffer);
}



void displayBuf()
{
    acquire_screen();
    blit(buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
    release_screen();
}



void screenshot()
{
    static int num(0);
    std::ostringstream os;

    while (true)
    {
        os << "wormlings-" << WORMLINGS_VERSION << "-shot";
        os.width(4);
        os.fill('0');
        os << ++num << ".tga";
        
        if (!exists(os.str().c_str()))
        {
            break;
        }
        else
        {
            os.seekp(0);
        }
    }

    BITMAP *bmp(create_sub_bitmap(screen, 0, 0, SCREEN_W, SCREEN_H));

    if (bmp)
    {
        save_tga(os.str().c_str(), bmp, 0);
        destroy_bitmap(bmp);
    }
}



void print(
    BITMAP *bmp, FONT *fnt, int x, int y, int col,
    std::string const &text, int align, int shadowOffset
)
{
    assert(bmp);
    assert(fnt);
    
    switch (align)
    {
        case -1: // left
#ifdef USE_OLD_TEXT_API
        textout(bmp, fnt, text.c_str(), x + shadowOffset, y + shadowOffset, 0);
        textout(bmp, fnt, text.c_str(), x, y, col);
#else
        textout_ex(
            bmp, fnt, text.c_str(), x + shadowOffset, y + shadowOffset,
            0, -1
        );
        textout_ex(bmp, fnt, text.c_str(), x, y, col, -1);
#endif
        break;
        
        case 0: // centre
#ifdef USE_OLD_TEXT_API
        textout_centre(
            bmp, fnt, text.c_str(), x + shadowOffset,
            y + shadowOffset, 0
        );
        textout_centre(bmp, fnt, text.c_str(), x, y, col);
#else
        textout_centre_ex(
            bmp, fnt, text.c_str(), x + shadowOffset,
            y + shadowOffset, 0, -1
        );
        textout_centre_ex(bmp, fnt, text.c_str(), x, y, col, -1);
#endif
        break;
        
        case 1: // right
#ifdef USE_OLD_TEXT_API
        textout_right(bmp, fnt, text.c_str(), x, y + shadowOffset, 0);
        textout_right(bmp, fnt, text.c_str(), x - shadowOffset, y, col);
#else
        textout_right_ex(bmp, fnt, text.c_str(), x, y + shadowOffset, 0, -1);
        textout_right_ex(bmp, fnt, text.c_str(), x - shadowOffset, y, col, -1);
#endif
        break;
    }
}



void printMultiColCentred(
    BITMAP *bmp, FONT *fnt, int x, int y,
    int colMiddle, int colSurround,
    std::string const &start, std::string const &middle, std::string const &end
)
{
    assert(bmp);
    assert(fnt);
    
    print(
        bmp, fnt, x - text_length(fnt, middle.c_str()) / 2, y,
        colSurround, start, 1
    );
    print(bmp, fnt, x, y, colMiddle, middle);
    print(
        bmp, fnt, x + text_length(fnt, middle.c_str()) / 2, y,
        colSurround, end, -1
    );
}



void doCircleFill(
    BITMAP *bmp, int x, int y, int r, int d,
    void (*proc)(BITMAP *, int, int, int)
)
{
    assert(bmp);
    
    int
        cx(0),
        cy(r),
        df(1 - r),
        d_e(3),
        d_se(-2 * r + 5);

    do
    {
        do_line(bmp, x - cy, y - cx, x + cy, y - cx, d, proc);

        if (cx)
        {
            do_line(bmp, x - cy, y + cx, x + cy, y + cx, d, proc);
        }

        if (df < 0)
        {
            df += d_e;
            d_se += 2;
        }
        else
        {
            if (cx != cy)
            {
                do_line(bmp, x - cx, y - cy, x + cx, y - cy, d, proc);

                if (cy)
                {
                    do_line(bmp, x - cx, y + cy, x + cx, y + cy, d, proc);
                }
            }

            df += d_se;
            d_se += 4;
            --cy;
        }

        d_e += 2;
        ++cx;
    }
    while (cx <= cy);
}



int getDarkerCol(int col)
{
    return makecol(
        MAX(0, getr(col) - 64), MAX(0, getg(col) - 64),
        MAX(0, getb(col) - 64)
    );
}



namespace {

bool setGraphicsMode(int w, int h, int depth, bool windowed)
{
    set_color_depth(depth);
    if (set_gfx_mode(
            windowed ? GFX_AUTODETECT_WINDOWED : GFX_AUTODETECT_FULLSCREEN,
            w, h, 0, 0
        )
    )
    {
        return false;
    }

    return true;
}

} // namespace
