// -----------------------------------------------------------------------------
// $Revision: 161 $ $Date: 2005-03-04 11:43:07 +0100 (fr, 04 mar 2005) $
// -----------------------------------------------------------------------------
// Copyright 2005 Daniel Schlyder.
// Distributed under the GNU General Public License; as published by the Free
// Software Foundation; either version 2 of the License, or (at your option) any
// later version. (See accompanying file LICENSE.txt or copy at
// http://www.gnu.org/licenses/gpl.html)



#include "audio.hpp"
#include "utility.hpp"
#include "alertstream.hpp"

#include <algorithm> // fill()
#include <stdexcept>
#include <cassert>



audio::audio()
:
    m_initialised(false),
    m_volume(200)
{
    std::fill(m_samples, m_samples + sid_end, static_cast<SAMPLE *>(0));
}



audio::~audio()
{
    exit();
}



bool audio::init()
{
    if (install_sound(DIGI_AUTODETECT, MIDI_NONE, 0))
    {
        alertstream()
            << "Error: Failed to initialise audio.\n\nAllegro error:\n"
            << allegro_error;
        return false;
    }
    
    try
    {
        load_sample(sid_ball_bounce, "ball_bounce");
        load_sample(sid_countdown, "countdown");
        load_sample(sid_countdown_done, "countdown_done");
        load_sample(sid_menu_loop, "menu_loop");
        load_sample(sid_wormling_death, "wormling_death");
        load_sample(sid_wormling_stop, "wormling_stop");
        load_sample(sid_wormling_turbo, "wormling_turbo");
    }
    catch (std::runtime_error const &except)
    {
        alertstream() << "Error: Failed to load '" << except.what() << "'";
        return false;
    }
    
    m_initialised = true;
    return true;
}



void audio::exit()
{
    for (unsigned idx(0); idx != sid_end; ++idx)
    {
        destroy_sample(m_samples[idx]);
    }
    
    remove_sound();
    m_initialised = false;
}



void audio::play(
    sample_id sid, float volume, unsigned char pan, int freq, bool loop
)
{
    if (m_initialised)
    {
        volume = MID(0.0, volume, 1.0);
        int voice(
            play_sample(
                m_samples[sid],
                static_cast<int>(m_volume * volume), pan, freq, loop
            )
        );
        if (voice >= 0 && loop)
        {
            m_looping_samples.insert(sid_voice(sid, voice));
        }
    }
}



void audio::play_at_x(
    sample_id sid, double x, float volume, int freq, bool loop
)
{
    play(sid, volume, static_cast<int>((256.0 / SCREEN_W) * x), freq, loop);
}



void audio::stop(sample_id sid)
{
    if (m_initialised)
    {
        stop_sample(m_samples[sid]);
        m_looping_samples.erase(sid);
    }
}



int audio::volume() const
{
    return m_volume;
}



bool audio::set_volume(int volume)
{
    if (volume > 255 || volume < 0)
    {
        return false;
    }
    else
    {
        m_volume = volume;
        
        if (m_initialised)
        {
            sid_voice_multimap::const_iterator it_end(m_looping_samples.end());
            for (
                sid_voice_multimap::const_iterator it(
                    m_looping_samples.begin()
                );
                it != it_end;
                ++it
            )
            {
                voice_set_volume(it->second, m_volume);
            }
        }
        
        return true;
    }
}



bool audio::adjust_volume(int amount)
{
    return set_volume(m_volume + amount);
}



void audio::load_sample(sample_id sid, std::string const &basename)
{
    assert(!m_samples[sid]);
    
    std::string path(fix_path("data/sfx/" + basename + ".wav"));
    m_samples[sid] = ::load_sample(path.c_str());
    if (!m_samples[sid])
    {
        throw std::runtime_error(path);
    }
}
