// -----------------------------------------------------------------------------
// $Revision: 158 $ $Date: 2005-03-04 11:11:22 +0100 (fr, 04 mar 2005) $
// -----------------------------------------------------------------------------
// Copyright 2000-2005 Daniel Schlyder.
// Distributed under the GNU General Public License; as published by the Free
// Software Foundation; either version 2 of the License, or (at your option) any
// later version. (See accompanying file LICENSE.txt or copy at
// http://www.gnu.org/licenses/gpl.html)



#include "graphics.hpp"
#include "audio.hpp"
#include "game.hpp"
#include "latencyTimer.hpp"
#include "utility.hpp"
#include "alertstream.hpp"

#include <allegro.h>

#include <string>
#include <sstream>
#include <algorithm>
#include <vector>
#include <utility>
#include <cctype> // isprint()



namespace {

void menuDraw();
void printCopyright();
void configureGame();
void configureNames();
void configureControls();
void setConfigValue(
    std::string const &name, int &value,
    std::vector<std::pair<int, std::string> > const &values, int pos = 0
);
std::string getInputString(
    std::string value, unsigned int maxWidth,
    int x, int y, int col, int align = 0
);

void wait_for_key_up(int key_id);

} // namespace



void menu()
{
    menuDraw();
    audio::instance().play(sid_menu_loop, 1.0, 128, 1000, true);
    clear_keybuf();

    bool exitGame(false);
    while (!exitGame)
    {
        if (keypressed())
        {
            switch (readkey() >> 8)
            {
                case KEY_F1:
                audio::instance().stop(sid_menu_loop);
                game();
                menuDraw();
                break;
                
                case KEY_F2:
                configureGame();
                menuDraw();
                break;
                
                case KEY_F3:
                configureNames();
                menuDraw();
                break;
                
                case KEY_F4:
                configureControls();
                menuDraw();
                break;
                
                case KEY_PLUS_PAD:
                if (audio::instance().adjust_volume(25))
                {
                    menuDraw();
                }
                break;
                
                case KEY_MINUS_PAD:
                if (audio::instance().adjust_volume(-25))
                {
                    menuDraw();
                }
                break;
                
                case KEY_F9:
                shutdownGraphics();
                runWindowed = !runWindowed;
                if (!initGraphics())
                {
                    alertstream()
                        << "Error: switch to "
                        << (runWindowed ? "windowed" : "fullscreen")
                        << " mode failed";
                    exit(1);
                }
                setPlayersColour();
                menuDraw();
                break;
                
                case KEY_F12:
                screenshot();
                break;
                
                case KEY_ESC:
                exitGame = true;
                break;
            }
            
            clear_keybuf();
        }
        else
        {
            rest(1);
        }
    }
    
    audio::instance().stop(sid_menu_loop);
}



namespace {

void menuDraw()
{
    blit(background, buffer, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
    printCopyright();
    
    blit(logo, buffer, 0, 0, 30, 30, logo->w, logo->h);
    
    print(
        buffer, font, SCREEN_W / 2 - 44, SCREEN_H / 2 - 68,
        makecol(255, 255, 255), "F1... Play", -1
    );
    print(
        buffer, font, SCREEN_W / 2 - 44, SCREEN_H / 2 - 36,
        makecol(255, 255, 255), "F2... Configure game", -1
    );
    print(
        buffer, font, SCREEN_W / 2 - 44, SCREEN_H / 2 - 4,
        makecol(255, 255, 255), "F3... Configure player names", -1
    );
    print(
        buffer, font, SCREEN_W / 2 - 44, SCREEN_H / 2 + 28,
        makecol(255, 255, 255), "F4... Configure player controls", -1
    );
    
    std::stringstream oss;
    oss << "+/-.. Adjust audio volume (" << audio::instance().volume() << ")";
    print(
        buffer, font, SCREEN_W / 2 - 44, SCREEN_H / 2 + 60,
        makecol(255, 255, 255), oss.str().c_str(), -1
    );
    
    std::string label;
    if (runWindowed)
    {
        label = "F9... Switch to fullscreen mode";
    }
    else
    {
        label = "F9... Switch to windowed mode";
    }
    print(
        buffer, font, SCREEN_W / 2 - 44, SCREEN_H / 2 + 92,
        makecol(255, 255, 255), label.c_str(), -1
    );
    
    print(
        buffer, font, SCREEN_W / 2 - 44, SCREEN_H / 2 + 124,
        makecol(255, 255, 255), "ESC.. Exit", -1
    );
    
    displayBuf();
}



// print copyright notice and version number
void printCopyright()
{
    print(
        buffer, font, SCREEN_W / 2, SCREEN_H - 12,
        makecol(255, 255, 255),
        "Version " + std::string(WORMLINGS_VERSION)
            + " -- Copyright 2000-2005 Daniel Schlyder"
            " -- bitblaze.com"
    );
}



// Configure game settings.
void configureGame()
{
    blit(background, buffer, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
    printCopyright();
    print(
        buffer, font, SCREEN_W / 2, 40, makecol(255, 255, 255),
        "Configure game..."
    );
    print(
        buffer, font, SCREEN_W / 2, 64, makecol(192, 192, 192),
        "(Change values using left and right arrow keys;"
            " press Enter to accept.)"
    );
    displayBuf();

    std::vector<std::pair<int, std::string> > values;
    
    typedef std::pair<int, std::string> pis;

    values.push_back(pis(2, ""));
    values.push_back(pis(3, ""));
    values.push_back(pis(4, ""));
    
    setConfigValue("Players", playersNum, values, 104);
    
    values.clear();
    values.push_back(pis(1, "thin"));
    values.push_back(pis(2, "medium"));
    values.push_back(pis(3, "thick"));

    setConfigValue("Wormlings size", playersSize, values);

    values.clear();
    values.push_back(pis(250, "very short"));
    values.push_back(pis(500, "short"));
    values.push_back(pis(750, "medium"));
    values.push_back(pis(1000, "long"));
    values.push_back(pis(1250, "very long"));
    values.push_back(pis(0, "infinite"));

    setConfigValue("Wormlings max length", playersMaxLength, values);

    values.clear();
    values.push_back(pis(0, "no"));
    values.push_back(pis(1, "yes"));

    setConfigValue("Wormlings grow slowly", playersSlowGrowth, values);

    values.clear();
    values.push_back(pis(1, "very slow"));
    values.push_back(pis(2, "slow"));
    values.push_back(pis(3, "medium"));
    values.push_back(pis(4, "fast"));
    values.push_back(pis(5, "very fast"));

    setConfigValue("Wormlings turn speed", playersTurnSpeed, values);

    values.clear();
    values.push_back(pis(0, "off"));
    values.push_back(pis(125, "low"));
    values.push_back(pis(250, "medium"));
    values.push_back(pis(375, "high"));
    values.push_back(pis(500, "very high"));

    setConfigValue("Energy", powerMax, values);

    if (powerMax)
    {
        values.clear();
        values.push_back(pis(3, "slow"));
        values.push_back(pis(5, "medium"));
        values.push_back(pis(7, "fast"));
    
        setConfigValue("Energy regained speed", powerRegainSpeed, values);
    }

    values.clear();
    values.push_back(pis(0, "no"));
    values.push_back(pis(1, "yes"));

    setConfigValue("Remove dead wormlings", eraseTheDead, values);

    values.clear();
    values.push_back(pis(0, ""));
    values.push_back(pis(10, ""));
    values.push_back(pis(20, ""));
    values.push_back(pis(30, ""));
    values.push_back(pis(40, ""));
    values.push_back(pis(50, ""));
    values.push_back(pis(60, ""));
    values.push_back(pis(70, ""));
    values.push_back(pis(80, ""));
    values.push_back(pis(90, ""));
    values.push_back(pis(100, ""));

    setConfigValue("Amount of Pink Balls of Death (%)", ballsAmount, values);
    
    values.clear();
    values.push_back(pis(1, toString(1 * playersNum)));
    values.push_back(pis(2, toString(2 * playersNum)));
    values.push_back(pis(3, toString(3 * playersNum)));
    values.push_back(pis(4, toString(4 * playersNum)));
    values.push_back(pis(5, toString(5 * playersNum)));
    values.push_back(pis(6, toString(6 * playersNum)));
    values.push_back(pis(7, toString(7 * playersNum)));
    values.push_back(pis(8, toString(8 * playersNum)));
    values.push_back(pis(9, toString(9 * playersNum)));
    values.push_back(pis(10, toString(10 * playersNum)));

    setConfigValue("Score limit", scoreLimit, values);
}



// configure player names
void configureNames()
{
    blit(background, buffer, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
    printCopyright();
    print(
        buffer, font, SCREEN_W / 2, 40, makecol(255, 255, 255),
        "Configure player names..."
    );
    print(
        buffer, font, SCREEN_W / 2, 64, makecol(192, 192, 192),
        "(Type in name, then press Enter to accept.)"
    );

    for (int i = 0; i < 4; ++i)
    {
        print(
            buffer, font, SCREEN_W / 2, 104 + 64 * i, players[i].col,
            "Player " + toString(i + 1) + "'s name:"
        );
        players[i].name = getInputString(
            players[i].name, SCREEN_W / 2,
            SCREEN_W / 2, 128 + 64 * i, makecol(255, 255, 255)
        );
    }
}



// configure player controls
void configureControls()
{
    clear_keybuf();
    
    blit(background, buffer, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
    printCopyright();
    print(
        buffer, font, SCREEN_W / 2, 40, makecol(255, 255, 255),
        "Configure player controls..."
    );
    
    for (int i = 0; i < 4; ++i)
    {
        print(
            buffer, font, SCREEN_W / 2, 80 + 92 * i, players[i].col,
            players[i].name + "'s keys:"
        );
       
        // left
        printMultiColCentred(
            buffer, font, SCREEN_W / 2, 104 + 92 * i,
            makecol(255, 255, 255), makecol(192, 192, 192),
            "press ", "Turn Left", " key"
        );
        displayBuf();
        players[i].keyLeft = readkey() >> 8;
        wait_for_key_up(players[i].keyLeft);
        
        // right
        printMultiColCentred(
            buffer, font, SCREEN_W / 2, 104 + 92 * i + 12,
            makecol(255, 255, 255), makecol(192, 192, 192),
            "press ", "Turn Right", " key"
        );
        displayBuf();
        players[i].keyRight = readkey() >> 8;
        wait_for_key_up(players[i].keyRight);
        
        // turbo
        printMultiColCentred(
            buffer, font, SCREEN_W / 2, 104 + 92 * i + 24,
            makecol(255, 255, 255), makecol(192, 192, 192),
            "press ", "Turbo", " key"
        );
        displayBuf();
        players[i].keyTurbo = readkey() >> 8;
        wait_for_key_up(players[i].keyTurbo);
        
        // stop
        printMultiColCentred(
            buffer, font, SCREEN_W / 2, 104 + 92 * i + 36,
            makecol(255, 255, 255), makecol(192, 192, 192),
            "press ", "Stop", " key"
        );
        displayBuf();
        players[i].keyStop = readkey() >> 8;
        wait_for_key_up(players[i].keyStop);
    }
}



// Set value of configuration setting.
void setConfigValue(
    std::string const &name, int &value,
    std::vector<std::pair<int, std::string> > const &values, int pos
)
{
    static int y(0);
    if (pos > 0)
    {
        y = pos;
    }
    else
    {
        y += 24;
    }
    
    print(
        buffer, font, SCREEN_W / 2 - (name.size() + 1) * 8, y,
        makecol(255, 255, 255), name + ':', -1
    );
    displayBuf();
    clear_keybuf();
    
    // Find iterator for current value.
    std::vector<std::pair<int, std::string> >::const_iterator value_it(
        values.begin()
    );
    for ( ; value_it != values.end(); ++value_it)
    {
        if (value_it->first == value)
        {
            break;
        }
    }
    
    bool fini(false);
    do
    {
        std::string str;

        if (!value_it->second.empty())
        {
            str = value_it->second;
        }
        else
        {
            str = toString(value);
        }

        blit(
            background, buffer, SCREEN_W / 2 + 7, y,
            SCREEN_W / 2 + 7, y, SCREEN_W / 2 - 7, 9
        );
        print(buffer, font, SCREEN_W / 2 + 8, y, makecol(255, 255, 0), str, -1);
        displayBuf();

        if (keypressed())
        {
            switch (readkey() >> 8)
            {
                case KEY_LEFT:
                if (value_it != values.begin())
                {
                    --value_it;
                    value = value_it->first;
                }
                break;
                
                case KEY_RIGHT:
                if (value_it != values.end() - 1)
                {
                    ++value_it;
                    value = value_it->first;
                }
                break;
                
                case KEY_ENTER:
                fini = true;
                wait_for_key_up(KEY_ENTER);
                break;
            }
        }
        else
        {
            rest(1);
        }
    }
    while (!fini);
}



// get string input from user
std::string getInputString(
    std::string value, unsigned int maxWidth,
    int x, int y, int col, int align
)
{
    // bitmap with untouched buffer used to restore background before
    // reprinting current value
    BITMAP *bmp(create_bitmap(SCREEN_W, 9));
    blit(buffer, bmp, 0, y, 0, 0, SCREEN_W, 9);
    
    clear_keybuf();
    startLatencyTimer(50);
    latency = 0;
    bool inputDone(false);
    do
    {
        while (latency && !inputDone)
        {
            if (keypressed())
            {
                int newKey(readkey());
                switch (newKey >> 8)
                {
                    case KEY_ENTER:
                    inputDone = true;
                    wait_for_key_up(KEY_ENTER);
                    break;
                    
                    case KEY_BACKSPACE:
                    if (value.length())
                    {
                        value = value.substr(0, value.length() - 1);
                    }
                    break;
                    
                    default:
                    if (std::isprint(newKey & 0xff))
                    {
                        // add new key's ASCII value to string
                        value += newKey & 0xff;

                        if ((8 * value.length() + 1) > maxWidth)
                        {
                            // remove new char if string exceeds maximum
                            // width
                            value = value.substr(0, value.length() - 1);
                        }
                    }
                }
            }

            --latency;
            if (!inputDone && !latency)
            {
                // restore background
                blit(bmp, buffer, 0, 0, 0, y, SCREEN_W, 9);

                print(buffer, font, x, y, col, value, align);
                displayBuf();
            }
        }
        
        rest(1);
    }
    while (!inputDone);
   
    stopLatencyTimer();
    destroy_bitmap(bmp);

    return value;
}



void wait_for_key_up(int key_id)
{
    while (key[key_id])
    {
        rest(1);
    }
    clear_keybuf();
}

} // namespace
