// -----------------------------------------------------------------------------
// $Revision: 162 $ $Date: 2005-03-04 12:00:38 +0100 (fr, 04 mar 2005) $
// -----------------------------------------------------------------------------
// Copyright 2000-2005 Daniel Schlyder.
// Distributed under the GNU General Public License; as published by the Free
// Software Foundation; either version 2 of the License, or (at your option) any
// later version. (See accompanying file LICENSE.txt or copy at
// http://www.gnu.org/licenses/gpl.html)



#include "game.hpp"
#include "graphics.hpp"
#include "latencyTimer.hpp"
#include "utility.hpp"

#include <allegro.h>

#include <vector>
#include <algorithm>
#include <cstdlib> // rand()
#include <cmath> // cos(), sin()



int
    playersNum,
    playersSize,
    playersMaxLength,
    playersSlowGrowth,
    playersTurnSpeed,
    powerMax,
    powerRegainSpeed,
    eraseTheDead,
    ballsAmount,
    scoreLimit;

bool display_fps_counter;

player players[4];

namespace {

struct
{
    int current, displayed, background;
} pixels[480][640];

bool dirty_rows[480];

BITMAP *bufBars[4]; // buffer before power bar is drawn

// Info for the pink balls of death.
struct ball
{
    int direction, turn, size;
    double x, y;
    // Used to counter collision detection ugliness that sometimes causes
    // multiple turns only a few frames apart.
    unsigned frames_till_sfx;
};

std::vector<ball> balls;

// Stores number of half-seconds since last time wormlings regained power.
volatile int powerRegainTimer;

// Number of players not dead in current round.
int playersAlive;

bool winner; // game won

int prev_fps, fps;

// TODO: This could be made a configuration option.
unsigned const g_countdown_max_secs(3);
unsigned const g_countdown_max(g_countdown_max_secs * 100);
unsigned g_countdown;



void initGame();
void shutdownGame();
void startTimers();
void stopTimers();
void incPowerRegainTimer();
bool getInput();
void pauseGame();
void updateWormlings();
void newHead(int player);
void deleteTail(int player);
void initBalls();
void updateBalls();
void setPixel(int x, int y, int colour);
void plot(BITMAP *, int x, int y, int col);
void plotIfClear(BITMAP *, int x, int y, int col);
void clear(BITMAP *, int x, int y, int);
void clearIfNotCol(BITMAP *, int x, int y, int col);
void plotNoWrap(BITMAP *, int x, int y, int col);
void plotIfClearNoWrap(BITMAP *, int x, int y, int col);
void clearNoWrap(BITMAP *, int x, int y, int);

// wrap around coordinates if they're outside size of screen
template <typename T>
void wrapCoords(T &x, T &y)
{
    if (x < 0)
    {
        x += SCREEN_W;
    }
    else if (x >= SCREEN_W)
    {
        x -= SCREEN_W;
    }
    if (y < 0)
    {
        y += SCREEN_H;
    }
    else if (y >= SCREEN_H)
    {
        y -= SCREEN_H;
    }
}

void cdBall(BITMAP *, int x, int y, int ball);
void cdWormling(BITMAP *, int x, int y, int player);
void cdWormlingInclDarkSelf(BITMAP *, int x, int y, int player);
void updateScores();
void drawScores();
void drawFrame();
void draw_countdown();
void drawPixels();
void drawPowerBars();
void storeBufBars();
void restoreBufBars();
void newRound();
void resetFPSCounter();

} // namespace



void setPlayersColour()
{
    players[0].col = makecol(255, 0, 0); // red
    players[1].col = makecol(0, 255, 0); // green
    players[2].col = makecol(255, 255, 0); // yellow
    players[3].col = makecol(0, 255, 255); // cyan
}



void game()
{
    initGame();

    bool abortGame(false);
    startTimers();
    while (!winner && !abortGame)
    {
        while (latency && !winner)
        {
            if (!getInput())
            {
                abortGame = true;
                audio::instance().play(sid_menu_loop, 1.0, 128, 1000, true);
                break;
            }
            
            if (g_countdown)
            {
                if (!(g_countdown % 100))
                {
                    audio::instance().play(sid_countdown);
                }
                --g_countdown;
                if (!g_countdown)
                {
                    audio::instance().play(sid_countdown_done);
                }
            }
            else
            {
                updateBalls();
                updateWormlings();
                updateScores();
            }
            
            if (!(--latency))
            {
                drawFrame();
                displayBuf();
            }

            if (playersAlive <= 1)
            {
                // make sure display is updated even if severely lagged
                drawFrame();

                drawScores();
                displayBuf();
                
                audio::instance().play(sid_menu_loop, 1.0, 128, 1000, true);
                
                if (winner)
                {
                    // wait for user to press Escape key
                    while (!key[KEY_ESC])
                    { }
                    
                    // wait for user to release Escape key again, to not make
                    // menu() exit program
                    while (key[KEY_ESC])
                    { }
                }
                else
                {
                    bool key_pressed;
                    do
                    {
                        rest(1);
                        key_pressed = false;
                        for (unsigned idx(0); idx != KEY_MAX; ++idx)
                        {
                            if (key[idx])
                            {
                                key_pressed = true;
                                break;
                            }
                        }
                    }
                    while (key_pressed);
                    
                    clear_keybuf();
                    readkey();
                    
                    audio::instance().stop(sid_menu_loop);
                    newRound();
                }
            }
        }
        
        rest(1);
    }

    stopTimers();
    shutdownGame();
}



namespace {

void initGame()
{
    // set translucency level for scoreboard background and power bars
    set_trans_blender(0, 0, 0, 128);
    
    for (int i = 0; i < playersNum; ++i)
    {
        players[i].score = 0;

        if (powerMax)
        {
            bufBars[i] = create_bitmap(125, 10);
        }
    }

    winner = false;
   
    newRound();
}



// clean up after game
void shutdownGame()
{
    if (!powerMax)
    {
        return;
    }

    for (int i = 0; i < playersNum; ++i)
    {
        destroy_bitmap(bufBars[i]);
    }
}



// start in-game timers
void startTimers()
{
    startLatencyTimer(100);
    install_int_ex(incPowerRegainTimer, BPS_TO_TIMER(2));
    if (display_fps_counter)
    {
        install_int_ex(resetFPSCounter, SECS_TO_TIMER(1));
    }
}



// stop in-game timers
void stopTimers()
{
    stopLatencyTimer();
    remove_int(incPowerRegainTimer);
    if (display_fps_counter)
    {
        remove_int(resetFPSCounter);
    }
}



// increment time since power regained
void incPowerRegainTimer()
{
    ++powerRegainTimer;
}



bool getInput()
{
    if (!g_countdown)
    {
        // player control keys
        for (int i = 0; i < playersNum; ++i)
        {
            if (!players[i].dead)
            {
                players[i].speed = 1;
                if (powerMax && players[i].power)
                {
                    if (key[players[i].keyStop])
                    {
                        players[i].speed = 0;
                        --players[i].power;
                        if (players[i].last_sid != sid_wormling_stop)
                        {
                            audio::instance().play_at_x(
                                sid_wormling_stop, players[i].x
                            );
                            players[i].last_sid = sid_wormling_stop;
                        }
                    }
                    else if (key[players[i].keyTurbo])
                    {
                        players[i].speed = 2;
                        --players[i].power;
                        if (players[i].last_sid != sid_wormling_turbo)
                        {
                            audio::instance().play_at_x(
                                sid_wormling_turbo, players[i].x
                            );
                            players[i].last_sid = sid_wormling_turbo;
                        }
                    }
                    else
                    {
                        players[i].last_sid = sid_end;
                    }
                }
                
                if (players[i].speed)
                {
                    if (key[players[i].keyLeft])
                    {
                        players[i].direction += playersTurnSpeed;
                        if (players[i].direction > 359)
                        {
                            players[i].direction = 0;
                        }
                    }
                    else if (key[players[i].keyRight])
                    {
                        players[i].direction -= playersTurnSpeed;
                        if (players[i].direction < 0)
                        {
                            players[i].direction = 359;
                        }
                    }
                }
            }
        }
    
        if (key[KEY_P])
        {
            pauseGame();
        }
    }

    // abort game
    if (key[KEY_ESC])
    {
        while (key[KEY_ESC]) { rest(1); } // release the key dammit!
        return false;
    }

    // take screen-shot
    if (key[KEY_F12])
    {
        stopTimers();
        screenshot();
        while (key[KEY_F12]) { rest(1); }
        startTimers();
    }
   
    return true;
}



void pauseGame()
{
    stopTimers();
    while (key[KEY_P]) { rest(1); } // wait until P not pressed anymore

    print(
        screen, font, SCREEN_W / 2, SCREEN_H / 2 - 20,
        makecol(255, 255, 255), "Game paused."
    );
    print(
        screen, font, SCREEN_W / 2, SCREEN_H / 2 + 12,
        makecol(255, 255, 255), "Press P to resume..."
    );

    bool resumeGame(false);
    while (!resumeGame)
    {
        if (key[KEY_P])
        {
            resumeGame = true;
        }
        else
        {
            rest(1);
        }
    }

    while (key[KEY_P]) { rest(1); }
    startTimers();
}



// update wormlings (ie. players :))
void updateWormlings()
{
    // calculate power regained and reset timer
    int powerRegained(powerRegainTimer * powerRegainSpeed);
    powerRegainTimer = 0;

    for (int i = 0; i < playersNum; ++i)
    {
        if (!players[i].dead)
        {
            players[i].power = std::min(
                players[i].power + powerRegained, powerMax
            );

            if (players[i].speed)
            {
                players[i].x += std::cos(players[i].direction * AL_PI / 180);
                players[i].y -= std::sin(players[i].direction * AL_PI / 180);
   
                wrapCoords(players[i].x, players[i].y);
   
                if (static_cast<int>(players[i].x) != players[i].head->x
                    || static_cast<int>(players[i].y) != players[i].head->y
                )
                {
                    if (players[i].eraseTail
                        || players[i].length == playersMaxLength
                    )
                    {
                        deleteTail(i);
                    }
                    if (playersSlowGrowth)
                    {
                        players[i].eraseTail = !players[i].eraseTail;
                    }
                }
            }
        }
        else
        {
            if (eraseTheDead)
            {
                deleteTail(i);
            }
        }
    }

    for (int s = 0; s < 2; ++s)
    {
        for (int i = 0; i < playersNum; ++i)
        {
            if (!players[i].dead)
            {
                if (s && players[i].speed == 2)
                {
                    players[i].x += std::cos(
                        players[i].direction * AL_PI / 180
                    );
                    players[i].y -= std::sin(
                        players[i].direction * AL_PI / 180
                    );
   
                    wrapCoords(players[i].x, players[i].y);
                }
   
                if (static_cast<int>(players[i].x) != players[i].head->x
                    || static_cast<int>(players[i].y) != players[i].head->y
                )
                {
                    if (players[i].length == playersMaxLength)
                    {
                        deleteTail(i);
                    }

                    // erase a few previous heads' borders to avoid collision
                    // detected vs these
                    pos *tmp = players[i].head;
                    for (int t = 0; t <= playersSize; ++t)
                    {
                        if (tmp)
                        {
                            doCircleFill(
                                buffer, tmp->x, tmp->y, playersSize,
                                players[i].col, clearIfNotCol
                            );
                            tmp = tmp->next;
                        }
                        else
                        {
                            break;
                        }
                    }
               
                    // collision check vs old situation
                    do_circle(
                        buffer, static_cast<int>(players[i].x),
                        static_cast<int>(players[i].y), playersSize, i,
                        cdWormlingInclDarkSelf
                    );
               
                    // redraw old heads' borders
                    tmp = players[i].head;
                    for (int t = 0; t <= playersSize; ++t)
                    {
                        if (tmp)
                        {
                            doCircleFill(
                                buffer, tmp->x, tmp->y, playersSize,
                                getDarkerCol(players[i].col), plotIfClear
                            );
                            tmp = tmp->next;
                        }
                        else
                        {
                            break;
                        }
                    }
   
                    // draw new head
                    doCircleFill(
                        buffer, static_cast<int>(players[i].x),
                        static_cast<int>(players[i].y), playersSize - 1,
                        players[i].col, plot
                    );
                    doCircleFill(
                        buffer, static_cast<int>(players[i].x),
                        static_cast<int>(players[i].y), playersSize,
                        getDarkerCol(players[i].col), plotIfClear
                    );
   
                    newHead(i);
                }
                
                if (players[i].dead)
                {
                    audio::instance().play_at_x(
                        sid_wormling_death, players[i].x
                    );
                }
            }
        }
   
        // detect collisions vs new situation
        for (int i = 0; i < playersNum; ++i)
        {
            if (!players[i].dead)
            {
                if (static_cast<int>(players[i].x) != players[i].head->x
                    || static_cast<int>(players[i].y) != players[i].head->y
                )
                {
                    do_circle(
                        buffer, static_cast<int>(players[i].x),
                        static_cast<int>(players[i].y), playersSize, i,
                        cdWormling
                    );
                }
                
                if (players[i].dead)
                {
                    audio::instance().play_at_x(
                        sid_wormling_death, players[i].x
                    );
                }
            }
        }
    }
}



// add new head piece to wormling
void newHead(int i)
{
    pos *tmp(new pos);

    if (players[i].head)
    {
        players[i].head->prev = tmp;
    }
    tmp->next = players[i].head;
    tmp->prev = 0;
    tmp->x = static_cast<int>(players[i].x);
    tmp->y = static_cast<int>(players[i].y);
    players[i].head = tmp;
    
    ++players[i].length;
}



// remove tail piece from wormling
void deleteTail(int i)
{
    pos *tmp(players[i].tail);
   
    if (tmp)
    {
        players[i].tail = tmp->prev;

        if (players[i].tail)
        {
            players[i].tail->next = 0;
         
            // erase old tail
            doCircleFill(
                buffer, tmp->x, tmp->y, playersSize, players[i].col,
                clearIfNotCol
            );
            doCircleFill(buffer, tmp->x, tmp->y, playersSize - 1, 0, clear);

            // draw new tail
            doCircleFill(
                buffer, players[i].tail->x, players[i].tail->y,
                playersSize - 1, players[i].col, plot
            );
            doCircleFill(
                buffer, players[i].tail->x, players[i].tail->y,
                playersSize, getDarkerCol(players[i].col), plotIfClear
            );
        }
        else
        {
            players[i].head = 0;
            doCircleFill(buffer, tmp->x, tmp->y, playersSize, 0, clear);
        }
      
        delete tmp;
    }
    
    --players[i].length;
}



void initBalls()
{
    balls.clear();
    
    if (ballsAmount)
    {
        int remaining_size(((240 - 40 * playersNum) * ballsAmount) / 100);
        
        while (remaining_size >= 5)
        {
            ball b;
            b.direction = std::rand() % 360;
            b.frames_till_sfx = 0;
            
            if (remaining_size > 5)
            {
                b.size = 5 + std::rand() % std::min(remaining_size - 4, 25);
            }
            else
            {
                b.size = 5;
            }
            
            remaining_size -= b.size;
            
            balls.push_back(b);
            unsigned int i(balls.size() - 1);

            do
            {
                balls[i].x = std::rand() % (SCREEN_W - 160) + 80;
                balls[i].y = std::rand() % (SCREEN_H - 120) + 60;
                balls[i].turn = false;
                
                do_circle(
                    buffer, static_cast<int>(balls[i].x),
                    static_cast<int>(balls[i].y), balls[i].size, i, cdBall
                );
            }
            while (balls[i].turn);
        }
    }
}



void updateBalls()
{
    for (unsigned int i = 0; i < balls.size(); ++i)
    {
        if (balls[i].frames_till_sfx)
        {
            --balls[i].frames_till_sfx;
        }
        
        double x(balls[i].x), y(balls[i].y);
            
        balls[i].x += std::cos(balls[i].direction * AL_PI / 180);
        balls[i].y -= std::sin(balls[i].direction * AL_PI / 180);

        if (static_cast<int>(balls[i].x) != static_cast<int>(x)
            || static_cast<int>(balls[i].y) != static_cast<int>(y)
        )
        {
            // Erase old ball position.
            doCircleFill(
                buffer, static_cast<int>(x), static_cast<int>(y),
                balls[i].size, 0, clearNoWrap
            );
         
            balls[i].turn = false;
         
            // Collision check.
            doCircleFill(
                buffer, static_cast<int>(balls[i].x),
                static_cast<int>(balls[i].y), balls[i].size, i, cdBall
            );
         
            if (balls[i].turn)
            {
                balls[i].direction = std::rand() % 360;
                balls[i].x = x;
                balls[i].y = y;
                if (!balls[i].frames_till_sfx)
                {
                    audio::instance().play_at_x(
                        sid_ball_bounce,
                        x,
                        balls[i].size / 29.0,
                        14500 / balls[i].size
                    );
                    balls[i].frames_till_sfx = 10;
                }
            }
         
            // Draw new ball position.
            do_circle(
                buffer, static_cast<int>(balls[i].x),
                static_cast<int>(balls[i].y), balls[i].size,
                makecol(128, 0, 128), plotNoWrap
            );
            doCircleFill(
                buffer, static_cast<int>(balls[i].x),
                static_cast<int>(balls[i].y), balls[i].size,
                makecol(192, 0, 192), plotIfClearNoWrap
            );
        }
    }
}



// Helper for do_circle/doCircleFill callback functions.
void setPixel(int x, int y, int colour)
{
    pixels[y][x].current = colour;
    dirty_rows[y] = true;
}



// change colour of a pixel
void plot(BITMAP *, int x, int y, int col)
{
    wrapCoords(x, y);
    setPixel(x, y, col);
}



// set colour of a pixel if it's currently clear
void plotIfClear(BITMAP *, int x, int y, int col)
{
    wrapCoords(x, y);

    if (pixels[y][x].current == pixels[y][x].background)
    {
        setPixel(x, y, col);
    }
}



// Set pixel to background colour.
void clear(BITMAP *, int x, int y, int)
{
    wrapCoords(x, y);
    setPixel(x, y, pixels[y][x].background);
}



// clear a pixel if it's currently not set to <col> colour
void clearIfNotCol(BITMAP *, int x, int y, int col)
{
    wrapCoords(x, y);

    if (pixels[y][x].current != col)
    {
        setPixel(x, y, pixels[y][x].background);
    }
}



// do_circle/doCircleFill callback function. Sets colour of pixel to value of
// d.
void plotNoWrap(BITMAP *, int x, int y, int col)
{
    setPixel(x, y, col);
}



// do_circle/doCircleFill callback function. Sets colour of pixel to value of
// d if current colour equals background colour.
void plotIfClearNoWrap(BITMAP *, int x, int y, int col)
{
    if (pixels[y][x].current == pixels[y][x].background)
    {
        setPixel(x, y, col);
    }
}



// do_circle/doCircleFill callback function. Sets colour of pixel to background
// colour.
void clearNoWrap(BITMAP *, int x, int y, int)
{
    setPixel(x, y, pixels[y][x].background);
}



// Check if ball has collided with something, and make it change direction
// in next update if it has.
void cdBall(BITMAP *, int x, int y, int i)
{
    if (x < 0 || x >= SCREEN_W || y < 0 || y >= SCREEN_H)
    {
        balls[i].turn = true;
    }
    else if (pixels[y][x].current != pixels[y][x].background)
    {
        balls[i].turn = true;
    }
}



// check if wormling has collided with something apart from itself, and
// kill it if it has
void cdWormling(BITMAP *, int x, int y, int i)
{
    wrapCoords(x, y);

    if (pixels[y][x].current != pixels[y][x].background
        && pixels[y][x].current != players[i].col
        && pixels[y][x].current != getDarkerCol(players[i].col)
    )
    {
        players[i].dead = true;
    }
}



// check if wormling has collided with something including the dark edges
// of its own skin, and kill it if it has
void cdWormlingInclDarkSelf(BITMAP *, int x, int y, int i)
{
    wrapCoords(x, y);

    if (pixels[y][x].current != pixels[y][x].background
        && pixels[y][x].current != players[i].col
    )
    {
        players[i].dead = true;
    }
}



void updateScores()
{
    // find players alive
    int aliveCount(0);
    for (int i = 0; i < playersNum; ++i)
    {
        if (!players[i].dead)
        {
            ++aliveCount;
        }
    }
   
    if (aliveCount < playersAlive)
    {
        // award difference to players still alive
        for (int i = 0; i < playersNum; ++i)
        {
            if (!players[i].dead)
            {
                players[i].score += playersAlive - aliveCount;
            }
        }
    }
   
    if (aliveCount <= 1)
    {
        // check if a player has reached the score limit
        for (int i = 0; i < playersNum; ++i)
        {
            if (players[i].score >= scoreLimit * playersNum)
            {
                winner = true;
                break;
            }
        }
    }
   
    playersAlive = aliveCount;
}



void drawScores()
{
    // find width of scoreboard background
    unsigned int width(20); // for "(Score limit: 000)"
    for (int i = 0; i < playersNum; ++i)
    {
        // " > " + name + " 000 " == 8 extra
        width = MAX(players[i].name.length() + 8, width);
    }
    width *= 8; // chars to pixels
   
    // height of scoreboard background
    unsigned int height(24 * playersNum + 64 + 16);

    // draw scoreboard background
    drawing_mode(DRAW_MODE_TRANS, 0, 0, 0);
    rectfill(
        buffer, SCREEN_W / 2 - width / 2, SCREEN_H / 2 - height / 2,
        SCREEN_W / 2 + width / 2, SCREEN_H / 2 + height / 2,
        makecol(128, 128, 128)
    );
    rect(
        buffer, SCREEN_W / 2 - width / 2 - 1,
        SCREEN_H / 2 - height / 2 - 1, SCREEN_W / 2 + width / 2 + 1,
        SCREEN_H / 2 + height / 2 + 1, makecol(255, 255, 255)
    );
    drawing_mode(DRAW_MODE_SOLID, 0, 0, 0);

    // find current highest score to determine game leaders
    int maxScore(0);
    for (int i = 0; i < playersNum; ++i)
    {
        maxScore = MAX(players[i].score, maxScore);
    }

    if (winner)
    {
        print(
            buffer, font, SCREEN_W / 2, SCREEN_H / 2 - height / 2 + 16,
            makecol(255, 255, 255), "Game Over!"
        );
    }
    else
    {
        print(
            buffer, font, SCREEN_W / 2, SCREEN_H / 2 - height / 2 + 16,
            makecol(255, 255, 255), "Scoreboard"
        );
    }
    print(
        buffer, font, SCREEN_W / 2, SCREEN_H / 2 - height / 2 + 24,
        makecol(255, 255, 255), "----------"
    );

    for (int i = 0; i < playersNum; ++i)
    {
        if (players[i].score == maxScore)
        {
            print(
                buffer, font, SCREEN_W / 2 - width / 2 + 8,
                SCREEN_H / 2 - height / 2 + 48 + 24 * i,
                makecol(255, 255, 255), ">", -1
            );
        }

        print(
            buffer, font, SCREEN_W / 2 - width / 2 + 24,
            SCREEN_H / 2 - height / 2 + 48 + 24 * i, players[i].col,
            players[i].name, -1
        );
        print(
            buffer, font, SCREEN_W / 2 + width / 2 - 8,
            SCREEN_H / 2 - height / 2 + 48 + 24 * i, makecol(255, 255, 255),
            toString(players[i].score), 1
        );
    }

    if (winner)
    {
        print(
            buffer, font, SCREEN_W / 2,
            SCREEN_H / 2 - height / 2 + 56 + 24 * playersNum,
            makecol(192, 192, 192), "Press Esc to exit!"
        );
    }
    else
    {
        print(
            buffer, font, SCREEN_W / 2,
            SCREEN_H / 2 - height / 2 + 56 + 24 * playersNum,
            makecol(192, 192, 192),
            "(Score limit: " + toString(scoreLimit * playersNum) + ")"
        );
    }
}



void drawFrame()
{
    if (powerMax)
    {
        restoreBufBars();
    }

    drawPixels();

    if (powerMax)
    {
        storeBufBars();
        drawPowerBars();
    }

    if (display_fps_counter)
    {
        ++fps;
#ifdef USE_OLD_TEXT_API
        text_mode(0);
        textprintf(
            buffer, font, SCREEN_W / 2 - 28, SCREEN_H - 8,
            makecol(255, 255, 255), "fps: %3d", prev_fps
        );
        text_mode(-1);
#else
        textprintf_ex(
            buffer, font, SCREEN_W / 2 - 28, SCREEN_H - 8,
            makecol(255, 255, 255), 0, "fps: %3d", prev_fps
        );
#endif
    }
    
    draw_countdown();
}



void draw_countdown()
{
    static bool buffer_is_dirty(false);
    
    if (g_countdown)
    {
        clear_to_color(g_bmp_char, makecol(255, 0, 255));
        
#ifdef USE_OLD_TEXT_API
        textprintf(
            g_bmp_char, font, 0, 0, makecol(255, 255, 255),
            "%d", g_countdown / 100 + 1
        );
#else
        textprintf_ex(
            g_bmp_char, font, 0, 0, makecol(255, 255, 255), -1,
            "%d", g_countdown / 100 + 1
        );
#endif
        
        blit(
            background, g_bmp_countdown,
            SCREEN_W / 2 - 40, SCREEN_H / 2 - 48, 0, 0, 96, 96
        );
        stretch_sprite(
            g_bmp_countdown, g_bmp_char, 0, 0, 96, 96
        );
        
        blit(
            background, buffer,
            SCREEN_W / 2 - 40, SCREEN_H / 2 - 48,
            SCREEN_W / 2 - 40, SCREEN_H / 2 - 48,
            96, 96
        );
        set_trans_blender(0, 0, 0, (g_countdown * 256) / g_countdown_max);
        draw_trans_sprite(
            buffer, g_bmp_countdown,
            SCREEN_W / 2 - 40, SCREEN_H / 2 - 48
        );
        set_trans_blender(0, 0, 0, 128);
        
        buffer_is_dirty = true;
    }
    else if (buffer_is_dirty)
    {
        blit(
            background, buffer,
            SCREEN_W / 2 - 40, SCREEN_H / 2 - 48,
            SCREEN_W / 2 - 40, SCREEN_H / 2 - 48,
            96, 96
        );
        buffer_is_dirty = false;
    }
}



void drawPixels()
{
    for (int y = 0; y < SCREEN_H; ++y)
    {
        if (dirty_rows[y])
        {
            for (int x = 0; x < SCREEN_W; ++x)
            {
                if (pixels[y][x].displayed != pixels[y][x].current)
                {
                    pixels[y][x].displayed = pixels[y][x].current;

                    if (g_colour_depth < 24)
                    {
                        reinterpret_cast<unsigned short *>(buffer->line[y])[x]
                            = pixels[y][x].current;
                    }
                    else if (g_colour_depth == 24)
                    {
                        WRITE3BYTES(
                            reinterpret_cast<unsigned char *>(buffer->line[y])
                                + x * 3,
                            pixels[y][x].current
                        );
                    }
                    else
                    {
                        reinterpret_cast<unsigned long *>(buffer->line[y])[x]
                            = pixels[y][x].current;
                    }
                }
            }
            
            dirty_rows[y] = false;
        }
    }
}



void drawPowerBars()
{
    drawing_mode(DRAW_MODE_TRANS, 0, 0, 0);

    int
        y,      // vertical start position of filled rectangle
        fill;   // width to fill

    for (int i = 0; i < playersNum; ++i)
    {
        if (players[i].power > 0)
        {
            fill = (players[i].power * bufBars[i]->w) / powerMax;

            if (fill > 0)
            {
                y = (i == 1 || i == 3) ? 0 : (SCREEN_H - bufBars[i]->h);

                if (i == 0 || i == 3)
                {
                    rectfill(buffer, SCREEN_W - fill, y, SCREEN_W - 1,
                        y + bufBars[i]->h - 1, players[i].col);
                }
                else
                {
                    rectfill(buffer, 0, y, fill - 1, y + bufBars[i]->h - 1,
                        players[i].col);
                }
            }
        }
    }

    drawing_mode(DRAW_MODE_SOLID, 0, 0, 0);
}



// backup buffer behind power bars
void storeBufBars()
{
    for (int i = 0; i < playersNum; ++i)
    {
        blit(
            buffer, bufBars[i],
            (i == 1 || i == 2) ? 0 : (SCREEN_W - bufBars[i]->w),
            (i == 1 || i == 3) ? 0 : (SCREEN_H - bufBars[i]->h),
            0, 0, bufBars[i]->w, bufBars[i]->h
        );
    }
}



// restore buffer to state before power bars were drawn
void restoreBufBars()
{
    for (int i = 0; i < playersNum; ++i)
    {
        blit(
            bufBars[i], buffer, 0, 0,
            (i == 1 || i == 2) ? 0 : (SCREEN_W - bufBars[i]->w),
            (i == 1 || i == 3) ? 0 : (SCREEN_H - bufBars[i]->h),
            bufBars[i]->w, bufBars[i]->h
        );
   }
}



void newRound()
{
    for (int y = 0; y < SCREEN_H; ++y)
    {
        for (int x = 0; x < SCREEN_W; ++x)
        {
            pixels[y][x].current = pixels[y][x].displayed =
                pixels[y][x].background = getpixel(background, x, y);
        }
    }
   
    blit(background, buffer, 0, 0, 0, 0, SCREEN_W, SCREEN_H);

    if (powerMax)
    {
        // store initial state of buffer behind power bars
        storeBufBars();
    }

    // set wormlings start positions
    players[0].direction = 135;
    players[0].x = SCREEN_W - 9;
    players[0].y = SCREEN_H - 9;

    players[1].direction = 315;
    players[1].x = 8;
    players[1].y = 8;

    players[2].direction = 45;
    players[2].x = 8;
    players[2].y = SCREEN_H - 9;

    players[3].direction = 225;
    players[3].x = SCREEN_W - 9;
    players[3].y = 8;

    // init other player stuff
    for (int i = 0; i < playersNum; ++i)
    {
        // remove any old pieces
        while (players[i].head)
        {
            players[i].tail = players[i].head->next;
            delete players[i].head;
            players[i].head = players[i].tail;
        }

        // create new head and tail
        newHead(i);
        players[i].tail = players[i].head;

        // awake from the dead and recharge power
        players[i].dead = false;
        players[i].length = 1;
        players[i].eraseTail = false;
        players[i].power = powerMax;
        players[i].last_sid = sid_end;
    }

    playersAlive = playersNum;
    
    initBalls();
    
    // reset timer variables
    latency = 0;
    powerRegainTimer = 0;
    
    g_countdown = g_countdown_max;
}



void resetFPSCounter()
{
    prev_fps = fps;
    fps = 0;
}

} // namespace
