#include "maingame.h"
#include "star.h"
#include "soul.h"
#include "aiinput.h"
#include "data.h"
#include "highscore.h"
#include "../base/params.h"

void MainGame::Draw(BITMAP *canvas) {
	blit(Data::space, canvas, (int)(camera->pos.x - camera->min.x), (int)(camera->pos.y - camera->min.y), 0, 0, SCREEN_W, SCREEN_H);

	for (vector<Star *>::iterator i = star.begin(); i != star.end(); ++i) {
		(*i)->Draw(canvas, camera);
	}

	for (vector<Powerup *>::iterator k = powerup.begin(); k != powerup.end(); ++k) {
		(*k)->Draw(canvas, camera);
	}

	for (vector<Soul *>::iterator j = soul.begin(); j != soul.end(); ++j) {
		(*j)->Draw(canvas, camera);
	}
	
	player->Draw(canvas, camera);
	DrawMinimap(canvas);
	DrawHP(canvas);
	DrawScore(canvas);
	
	if (player->state == Player::DEAD) {
		textout_centre(canvas, font, "GAME OVER!", SCREEN_W/2+1, SCREEN_H/2 - 3*text_height(font)/2+1, makecol(0,0,0));
		textout_centre(canvas, font, "GAME OVER!", SCREEN_W/2, SCREEN_H/2 - 3*text_height(font)/2, makecol(255,255,255));
		textout_centre(canvas, font, "Press ENTER to continue...", SCREEN_W/2+1, SCREEN_H/2 + text_height(font)/2+1, makecol(0,0,0));
		textout_centre(canvas, font, "Press ENTER to continue...", SCREEN_W/2, SCREEN_H/2 + text_height(font)/2, makecol(192,192,192));
	}
}


void MainGame::DrawMinimap(BITMAP *canvas) {
	float zoom = 0.0625f;
	int tx = 252;
	int ty = 188;
	
	if (!minimap) {
		minimap = create_bitmap(64,48);
		clear_to_color(minimap, makecol(64,160,128));
	}
	
	set_trans_blender(0,0,0,64);
	draw_trans_sprite(canvas, minimap, tx, ty);
	
	int c1 = makecol(255,255,255);
	int c2 = makecol(255,0,0);
	int c3 = makecol(0,255,255);
	int c4 = makecol(255,255,0);

	// draw corners
	hline(canvas, tx, ty, (int)(64.0f*zoom) + tx, c1);
	vline(canvas, tx, ty, (int)(64.0f*zoom) + ty, c1);
	
	hline(canvas, (int)(1024.0f*zoom) + tx, ty, (int)(960.0f*zoom) + tx, c1);
	vline(canvas, (int)(1024.0f*zoom)  + tx, ty, (int)(64.0f*zoom) + ty, c1);

	hline(canvas, tx, (int)(768.0f*zoom) + ty, (int)(64.0f*zoom) + tx, c1);
	vline(canvas, tx, (int)(768.0f*zoom) + ty, (int)(704.0f*zoom) + ty, c1);
	
	hline(canvas, (int)(1024.0f*zoom) + tx, (int)(768.0f*zoom) + ty, (int)(960.0f*zoom) + tx, c1);
	vline(canvas, (int)(1024.0f*zoom)  + tx, (int)(768.0f*zoom) + ty, (int)(704.0f*zoom) + ty, c1);

	// draw player, souls and powerups
	for (vector<Soul *>::iterator i = soul.begin(); i != soul.end(); ++i) {
		if ((*i)->state != Player::DYING) {
			putpixel(canvas, (int)((*i)->pos.x*zoom) + tx, (int)((*i)->pos.y*zoom) + ty, c2);
		}
	}
	for (vector<Powerup *>::iterator j = powerup.begin(); j != powerup.end(); ++j) {
		if ((*j)->hp > 0) {
			putpixel(canvas, (int)((*j)->pos.x*zoom) + tx, (int)((*j)->pos.y*zoom) + ty, c4);
		}
	}

	putpixel(canvas, (int)(player->pos.x*zoom) + tx, (int)(player->pos.y*zoom) + ty, c3);
}


void MainGame::DrawHP(BITMAP *canvas) {
	int y = 224;
	int x = 4;
	
	int r, g, b;
	int c1 = makecol(64,64,64);
	int c2 = makecol(40,40,40);
	int c3 = makecol(0,0,0);
	
	int maxHP = Params::hp1;
	switch (Params::difficulty) {
		case 0:		maxHP = 2*maxHP;	break;
		case 2:		maxHP = 2*maxHP/3;	break;
	}
	
	
	for (int i=0; i<30; i++) {
		if (i <= player->hp*30/maxHP) {
			hsv_to_rgb(i*4.0f, 1.0f, 1.0f, &r, &g, &b);
			rectfill(canvas, x, y, x+6, y+12, makecol(r,g,b));
			hsv_to_rgb(i*4.0f, 1.0f, 0.7f, &r, &g, &b);
			rect(canvas, x, y, x+6, y+12, makecol(r,g,b));
			putpixel(canvas, x, y, c3);
			putpixel(canvas, x+6, y, c3);
			putpixel(canvas, x, y+12, c3);
			putpixel(canvas, x+6, y+12, c3);
		}
		else {
			rectfill(canvas, x, y, x+6, y+12, c1);
			rect(canvas, x, y, x+6, y+12, c2);
			putpixel(canvas, x, y, c3);
			putpixel(canvas, x+6, y, c3);
			putpixel(canvas, x, y+12, c3);
			putpixel(canvas, x+6, y+12, c3);
		}
		x += 8;
	}
}


void MainGame::DrawScore(BITMAP *canvas) {
	textprintf(canvas, font, 209, 5, makecol(64,64,160), "pts: %08d", score.CurrentValue());
	textprintf(canvas, font, 208, 4, makecol(140,200,240), "pts: %08d", score.CurrentValue());

	for (vector<ScoreBubble *>::iterator i = bubble.begin(); i != bubble.end(); ++i) {
		(*i)->Draw(canvas, camera);
	}
}


bool MainGame::Update() {
	for (vector<Soul *>::iterator j = soul.begin(); j != soul.end();) {
		(*j)->Update();
		if ((*j)->state == Player::DEAD) {
			delete *j;
			j = soul.erase(j);
		}
		else {
			++j;
		}
	}

	for (vector<Powerup *>::iterator k = powerup.begin(); k != powerup.end();) {
		(*k)->Update();
		if ((*k)->state == Player::DEAD) {
			delete *k;
			k = powerup.erase(k);
		}
		else {
			++k;
		}
	}

	for (vector<ScoreBubble *>::iterator l = bubble.begin(); l != bubble.end();) {
		(*l)->Update();
		if ((*l)->hp <= 0) {
			delete *l;
			l = bubble.erase(l);
		}
		else {
			++l;
		}
	}
	
	player->Update();

	DoCollisionDetection();
	camera->Update();
	
	--spawnCounter;
	if (spawnCounter <= 0) {
		switch (Params::difficulty) {
			case 0:		spawnCounter = 3*Params::spawnTime/2;	break;
			case 1:		spawnCounter = Params::spawnTime;		break;
			case 2:		spawnCounter = 2*Params::spawnTime/3;	break;
		};
		SpawnSoul();
	}

	--powerupCounter;
	if (powerupCounter <= 0) {
		powerupCounter = 120;
		SpawnPowerup();
	}
	
	--sampleCounter;
	if (sampleCounter <= 0) {
		PlayScream();
		sampleCounter = rand()%10+20;
	}
	
	if (!playing) {
		play_sample(Data::amb[0], 3*Params::volSound/4, 0, 600, 1);
		play_sample(Data::amb[1], 3*Params::volSound/4, 0, 500, 1);
		playing = true;
	}

	if (delay > 0) {
		--delay;
	}
	else if (delay == 0) {
		--delay;
		play_sample(Data::amb[2], 3*Params::volSound/4, 255, 600, 1);
		play_sample(Data::amb[3], 3*Params::volSound/4, 255, 500, 1);
	}
	
	score.Update();
	
	if (GameScreen::Update() || (player->state == Player::DEAD && ::key[KEY_ENTER]) || (player->state == Player::DYING && player->stateCounter >= 60)) {
		playing = false;
		stop_sample(Data::amb[0]);
		stop_sample(Data::amb[1]);
		stop_sample(Data::amb[2]);
		stop_sample(Data::amb[3]);
		return true;
	}
	
	return false;
}


MainGame::MainGame() : GameScreen(), minimap(NULL), player(NULL), camera(NULL) {
}


MainGame::~MainGame() {
	TRACE("MainGame::~MainGame()\n");

	while (!star.empty()) {
		delete star.back();
		star.pop_back();
	}

	while (!soul.empty()) {
		delete soul.back();
		soul.pop_back();
	}
	
	while (!powerup.empty()) {
		delete powerup.back();
		powerup.pop_back();
	}

	while (!bubble.empty()) {
		delete bubble.back();
		bubble.pop_back();
	}
	
	delete camera;
	delete player;
	
	if (minimap) {
		destroy_bitmap(minimap);
		minimap = NULL;
	}
}


Error MainGame::Init() {
	// delete everything
	while (!star.empty()) {
		delete star.back();
		star.pop_back();
	}

	while (!soul.empty()) {
		delete soul.back();
		soul.pop_back();
	}

	while (!powerup.empty()) {
		delete powerup.back();
		powerup.pop_back();
	}

	while (!bubble.empty()) {
		delete bubble.back();
		bubble.pop_back();
	}

	if (camera) {
		delete camera;
	}
	
	if (player) {
		delete player;
	}
	
	// recreate everything
	player = new Player;
	camera = new Camera;
	camera->SetPlayer(player);

	for (int i=0; i<Params::nStars; i++) {
		star.push_back(new Star);
	}

	// initialize
	for (vector<Star *>::iterator i = star.begin(); i != star.end(); ++i) {
		(*i)->Init();
	}
	
	player->Init();
	camera->Init();

	for (int c=0; c<5; c++) {
		SpawnSoul();
	}
	
	spawnCounter = Params::spawnTime;
	powerupCounter = 300;
	sampleCounter = 0;
	delay = 5;
	playing = false;
	score.Reset();	

	return Error(Error::NONE);
}


GameState MainGame::NextState() {
	if ((player->state == Player::DYING) && (player->stateCounter >= 60))  {
		return PLAYING_GAME_OVER;
	}
	else if (player->state == Player::DEAD) {
		Highscore hsc("unborn.hsc");
		if (hsc.CanAdd(score.Get())) {
			return DOING_HISCORE_MENU;
		}
		else {
			return DOING_MAIN_MENU;
		}
	}
	else {
		return DOING_YESNO_MENU;
	}
}


void MainGame::SpawnSoul() {
	Soul *s = new Soul;

	while (true) {
		s->pos.x = rand()%((int)(s->max.x - s->min.x)) + (int)s->min.x;
		s->pos.y = rand()%((int)(s->max.y - s->min.y)) + (int)s->min.y;
		if (ValidatePosition(s)) {
			break;
		}
	}
	
	((AIInput *)(s->input))->player = player;
	soul.push_back(s);
}


void MainGame::SpawnPowerup() {
	powerup.push_back(new Powerup(rand()%7));
}



bool MainGame::ValidatePosition(Soul *s) {
	float mind = (float)(Params::rad*Params::rad*4);
	float dx, dy;

	for (vector<Soul *>::iterator i = soul.begin(); i != soul.end(); ++i) {
		Soul *p = *i;
		dx = p->pos.x - s->pos.x;
		dy = p->pos.y - s->pos.y;
		if (dx*dx + dy*dy <= mind) {
			return false;
		}
	}

	dx = player->pos.x - s->pos.x;
	dy = player->pos.y - s->pos.y;
	if (dx*dx + dy*dy <= 4*mind) {
		return false;
	}
	
	return true;
}


void MainGame::Collide(Player *p1, Player *p2) {
	if (p1 == p2) return;
	float mind = (float)(2*Params::rad);
	float dx = p1->pos.x - p2->pos.x;
	float dy = p1->pos.y - p2->pos.y;
	float distance = sqrt(dx*dx + dy*dy);
	if (distance <= mind) {
		// collision!
		float dvx = p1->v.x - p2->v.x;
		float dvy = p1->v.y - p2->v.y;
		float impact = -1.0f/distance*(dx*dvx + dy*dvy);
		if (distance == 0) return;
		dx /= distance;
		dy /= distance;
		if (impact < 0) return;
		// update hp
		float im2 = impact/4.0f;
		if (p1 == player) {
			play_sample(Data::thump, Params::volSound, 128, 950 + rand()%100, 0);
		}
		if (p1->v.x*p1->v.x + p1->v.y*p1->v.y > p2->v.x*p2->v.x + p2->v.y*p2->v.y) {
			p2->hp -= impact;
			if (p2->hp <= 0.0f) {
				if (p1 == player) {
					score += 2500;
					bubble.push_back(new ScoreBubble(p2->pos.x, p2->pos.y, "$2500"));
				}
				KillPlayer(p2);
			}
			if (p1 == player) {
				switch (Params::difficulty) {
					case 0:		p1->hp = MIN(p1->hp + im2, 2*Params::hp1);		break;
					case 1:		p1->hp = MIN(p1->hp + im2, Params::hp1);		break;
					case 2:		p1->hp = MIN(p1->hp + im2, 2*Params::hp1/3);	break;
				};
				int ds = (int)(impact*impact)*50;
				score += ds;
				if (p2->state != Player::DYING && ds > 0) {
					char buf[32];
					usprintf(buf, "$%d", ds);
					bubble.push_back(new ScoreBubble(p2->pos.x, p2->pos.y, buf));
				}
			}
			else {
				switch (Params::difficulty) {
					case 0:		p1->hp = MIN(p1->hp + im2, Params::hp2/2);		break;
					case 1:		p1->hp = MIN(p1->hp + im2, Params::hp2);		break;
					case 2:		p1->hp = MIN(p1->hp + im2, 3*Params::hp2/2);	break;
				};
			}
		}
		else {
			p1->hp -= impact;
			if (p1->hp <= 0.0f) {
				KillPlayer(p1);
			}
			switch (Params::difficulty) {
				case 0:		p2->hp = MIN(p2->hp + im2, Params::hp2/2);		break;
				case 1:		p2->hp = MIN(p2->hp + im2, Params::hp2);		break;
				case 2:		p2->hp = MIN(p2->hp + im2, 3*Params::hp2/2);	break;
			};
		}
		
		// update speeds
		p1->v.x += impact*dx;
		p1->v.y += impact*dy;
		p2->v.x -= impact*dx;
		p2->v.y -= impact*dy;
	}
}

void MainGame::DoCollisionDetection() {
	vector<Soul *>::iterator i;
	vector<Soul *>::iterator j;
	vector<Powerup *>::iterator k;

	if (player->state != Player::DEAD && player->state != Player::DYING) {
		for (i = soul.begin(); i != soul.end(); ++i) {
			if ((*i)->state != Player::DYING) {
				Collide(player, *i);
			}
		}
		for (k = powerup.begin(); k != powerup.end(); ++k) {
			if ((*k)->state != Player::ALIVE) continue;
			float dx = player->pos.x - (*k)->pos.x;
			float dy = player->pos.y - (*k)->pos.y;
			if (dx*dx + dy*dy < 500.0f) {
				play_sample(Data::collect, Params::volSound, 128, 1000, 0);
				switch ((*k)->type) {
					case 0:
						player->hp -= 15.0f;
						bubble.push_back(new ScoreBubble((*k)->pos.x, (*k)->pos.y, "-15HP"));
						break;
					
					case 1:
						score += 5000;
						bubble.push_back(new ScoreBubble((*k)->pos.x, (*k)->pos.y, "$5000"));
						break;
					
					case 2:
						score += 15000;
						bubble.push_back(new ScoreBubble((*k)->pos.x, (*k)->pos.y, "$15000"));
						break;
					
					case 3:
						score += 1000;
						bubble.push_back(new ScoreBubble((*k)->pos.x, (*k)->pos.y, "$1000"));
						break;
					
					case 4:
						player->hp = MIN(player->hp + 3.0f, Params::hp1);
						bubble.push_back(new ScoreBubble((*k)->pos.x, (*k)->pos.y, "+3HP"));
						break;
					
					case 5:
						player->hp -= 3.0f;
						bubble.push_back(new ScoreBubble((*k)->pos.x, (*k)->pos.y, "-3HP"));
						break;
					
					case 6:
						player->hp = MIN(player->hp + 15.0f, Params::hp1);
						bubble.push_back(new ScoreBubble((*k)->pos.x, (*k)->pos.y, "+15HP"));
						break;
				};
				if (player->hp <= 0.0f) KillPlayer(player);
				KillPowerup(*k);
			}
		}
	}
	
	for (i = soul.begin(); i != soul.end(); ++i) {
		for (j = soul.begin(); j != soul.end(); ++j) {
			Collide(*i, *j);
		}
	}
}


void MainGame::KillPlayer(Player *p) {
	if (p == player) {
		p->state = Player::DYING;
		play_sample(Data::die, Params::volSound, 128, 1000, 0);
	}
	else {
		for (vector<Soul *>::iterator i = soul.begin(); i != soul.end(); ++i) {
			if (*i == p) {
				p->state = Player::DYING;
				break;
			}
		}
	}
}	


void MainGame::KillPowerup(Powerup *p) {
	for (vector<Powerup *>::iterator i = powerup.begin(); i != powerup.end(); ++i) {
		if (*i == p) {
			p->Kill();
			break;
		}
	}
}	


void MainGame::PlayScream() {
	play_sample(Data::scream[rand()%10], Params::volSound/2, 128, 1000 - rand()%100, 0);
}
