#include <allegro.h>
#include "game.h"
#include "params.h"
#include "doublebuffer.h"

#ifndef ALLEGRO_DOS
static bool closed = false;
static void closehook(void) { closed = true; }
#else
#define closed 0
#endif


#ifdef ALLEGRO_WINDOWS
#include <winalleg.h>
#define YIELD() Sleep(1)
#else
#ifdef ALLEGRO_UNIX
#include <sys/time.h>
static void YIELD(void)
{
	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = 1;
	select(0, NULL, NULL, NULL, &tv);
}
#else
#define YIELD() yield_timeslice()
#endif
#endif


static volatile int timer = 0;
static void timer_f() {
	++timer;
} END_OF_STATIC_FUNCTION(timer_f);


Game::Game() : driver(NULL) {
	usprintf(gameTitle, "mygame");
}


Game::~Game() {
	if (driver) {
		delete driver;
		driver = NULL;
	}
	
	remove_int(timer_f);
	char buf[40];
	usprintf(buf, "%s.ini", gameTitle);
	Params::Save(buf);
}


Error Game::SetUpdateDriver(ScreenUpdate *newDriver) {
	if (driver) {
		driver->Destroy();
	}
	Error e = newDriver->Create();
	if (e) {
		delete newDriver;
		if (driver) {
			driver->Create();
		}
		return e;
	}
	else {
		if (driver) {
			delete driver;
		}
		driver = newDriver;
	}
	
	return Error(Error::NONE);
}


Error Game::Init() {
	if (allegro_init() != 0) return Error(Error::ALLEGRO);
	if (install_keyboard() != 0) return Error(Error::KEYBOARD);
	if (install_timer() != 0) return Error(Error::TIMER);
	if (install_mouse() < 0) return Error(Error::MOUSE);
	//if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL) != 0) return Error(Error::SOUND);
	reserve_voices(32,0);
	install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL);
	//set_volume(-1, -1);
	char buf[40];
	usprintf(buf, "%s.ini", gameTitle);
	Params::Load(buf);
#ifndef ALLEGRO_DOS
	set_window_title(gameTitle);
	set_window_close_hook(&closehook);
#endif
	set_display_switch_mode(SWITCH_BACKGROUND);
	set_color_depth(Params::bpp);
	if (set_gfx_mode(Params::fullscreen ? GFX_AUTODETECT : GFX_AUTODETECT_WINDOWED, Params::w, Params::h, 0, 0) != 0) {
		int alternative;
		switch (Params::bpp) {
			case 8:		alternative = 0;	break;
			case 15:	alternative = 16;	break;
			case 16:	alternative = 15;	break;
			case 24:	alternative = 32;	break;
			case 32:	alternative = 24;	break;
			default:	alternative = 16;	break;
		}
		set_color_depth(alternative);
		if (set_gfx_mode(Params::fullscreen ? GFX_AUTODETECT : GFX_AUTODETECT_WINDOWED, Params::w, Params::h, 0, 0) != 0) {
			return Error(Error::GFX);
		}
		Params::bpp = alternative;
	}
	
	LOCK_VARIABLE(timer);
	LOCK_FUNCTION(timer_f);
	if (install_int_ex(timer_f, BPS_TO_TIMER(Params::fps)) < 0) return Error(Error::TIMER);

	text_mode(-1);
	srand((unsigned)time(NULL));
	
	fps.Init(Params::fps);
	
	Error e;
	switch (Params::driver) {
		case 2:
			//Params::driver = 2;
			e = SetUpdateDriver(new TripleBuffer());
			if (!e) break;
			// fall through
		case 1:
			//Params::driver = 1;
			e = SetUpdateDriver(new PageFlipping());
			if (!e) break;
			// fall through
		case 0:
			//Params::driver = 0;
			e = SetUpdateDriver(new DoubleBuffer());
			if (!e) break;
			// fall through
		default:
			//Params::driver = -1;
			e = SetUpdateDriver(new ScreenUpdate());
			break;
	};
	
	return e;
}


void Game::Run() {
	bool done = false;
	bool needToRedraw = true;
	
	timer = 0;
	while (!done) {
		while (timer) {
			--timer;
			done = Update();
			needToRedraw = true;
			fps.Tick();
		}
		
		if (needToRedraw || Params::unlimitedFPS) {
			driver->AcquireCanvas();
			Draw(driver->GetCanvas());
			driver->ReleaseCanvas();
			driver->Draw();
			needToRedraw = false;
			fps.Frame();
		}
		
		done |= closed;
		if (Params::yield) {
			while (!timer) {
				YIELD();
			}
		}
	}
}


bool Game::Update() {
	return key[KEY_ESC];
}


void Game::Draw(BITMAP *canvas) {
	if (Params::showFPS) {
		fps.Draw(canvas);
	}
}
