/* modified by Yurand for TW-Light */
/*
   Copyright (C) 2003 by David White <davidnwhite@optusnet.com.au>
   Part of the Battle for Wesnoth Project http://wesnoth.whitevine.net

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.

   See the COPYING file for more details. 
*/

//disable the very annoying VC++ warning 4786
#ifdef WIN32
#pragma warning(disable:4786)
#endif

#include "sound.h"
#include "SDL_mixer.h"

#include <iostream>
#include <map>

#include <allegro.h>
#include "other/twconfig.h"
#include "errors.h"

SoundSystem* tw_sound = NULL;

SoundSystem::SoundSystem(bool sound_on)
{
  STACKTRACE;
  mix_ok = false;
  current_music = "";
  
  music_off = false;
  sound_off = false;

  if(!sound_on) 
    {
      return;
    }
  
  tw_set_config_file("client.ini");
  buf_size_ = get_config_int("Sound", "SoundBuffer", 1024);
  if(buf_size_<1024)
    {
      buf_size_=1024;
    }
  else if(buf_size_>4096)
    {
      buf_size_ = 4096;
    }

  
  int res = Mix_OpenAudio(MIX_DEFAULT_FREQUENCY,MIX_DEFAULT_FORMAT,2,buf_size_);
  if(res >= 0) 
    {
      mix_ok = true;
      Mix_AllocateChannels(32);
    } 
  else 
    {
      mix_ok = false;
      std::cerr << "Could not initialize audio: " << SDL_GetError() << "\n";
    }

  res = Mix_QuerySpec(&frequency, &format, &channels);
  if(!res)
    {
      mix_ok = false;
      std::cerr << "Could not initialize audio: " << SDL_GetError() << "\n";
    }
};

SoundSystem::~SoundSystem()
{
  std::cerr << "closing audio...\n";
  if(!mix_ok)
    return;
  
  Mix_HaltMusic();
  Mix_HaltChannel(-1);
  
  for(std::map<std::string,Mix_Chunk*>::iterator i = sound_cache.begin();
      i != sound_cache.end(); ++i) 
    {
      Mix_FreeChunk(i->second);
    }
  
  for(std::map<std::string,Mix_Music*>::iterator j = music_cache.begin();
      j != music_cache.end(); ++j) 
    {
      Mix_FreeMusic(j->second);
    }
  
  std::cerr << "final closing audio...\n";
  Mix_CloseAudio();
  std::cerr << "done closing audio...\n";
}
  
void SoundSystem::play_music(const std::string file, int loops)
{
  STACKTRACE;
  if(!mix_ok || current_music == file)
    return;
  
  if(music_off) 
    {
      current_music = file;
      return;
    }
  
  std::map<std::string,Mix_Music*>::const_iterator itor = music_cache.find(file);
  if(itor == music_cache.end()) 
    {
      const std::string& filename = file;
      
      if(filename.empty()) 
	{
	  return;
	}
      
      Mix_Music* const music = Mix_LoadMUS(filename.c_str());
      if(music == NULL) {
	std::cerr << "Could not load music file '" << filename << "': "
		  << SDL_GetError() << "\n";
	return;
      }
      
      itor = music_cache.insert(std::pair<std::string,Mix_Music*>(file,music)).first;
    }
  
  if(Mix_PlayingMusic()) 
    {
      Mix_FadeOutMusic(0);
    }
  
  const int res = Mix_FadeInMusic(itor->second,loops,0);
  if(res < 0) 
    {
      std::cerr << "Could not play music: " << SDL_GetError() << "\n";
    }
  
  current_music = file;
}

void SoundSystem::play_sound(const std::string file, double vol)
{
  STACKTRACE;
  if(!mix_ok || sound_off)
    return;
  
  std::map<std::string,Mix_Chunk*>::const_iterator itor = sound_cache.find(file);
  if(itor == sound_cache.end()) 
    {
      const std::string& filename = file;
      
      if(filename.empty()) 
	{
	  return;
	}
      
      Mix_Chunk* const sfx = Mix_LoadWAV(filename.c_str());
      if(sfx == NULL) 
	{
	  std::cerr << "Could not load sound file '" << filename << "': "
		    << SDL_GetError() << "\n";
	  return;
	}
      
      itor = sound_cache.insert(std::pair<std::string,Mix_Chunk*>(file,sfx)).first;
    }
  
  //play on the first available channel
  const int res = Mix_PlayChannel(-1,itor->second,0);
  if(res < 0) 
    {
      std::cerr << "error playing sound effect: " << SDL_GetError() << "\n";
    }
  if(vol!=-1)
    {
      
      Mix_VolumeChunk(itor->second, int(vol*double(MIX_MAX_VOLUME)));
    }
}

void SoundSystem::stop_sound(const std::string file)
{
  STACKTRACE;
  if(!mix_ok || sound_off)
    return;
  
  std::map<std::string,Mix_Chunk*>::const_iterator itor = sound_cache.find(file);
  if(itor == sound_cache.end()) 
    {
      return;
    }
  
  for (int i=0; i<32; i++)
    {
      Mix_Chunk * chunk = Mix_GetChunk(i);
      if(chunk == itor->second)
	{
	  Mix_HaltChannel(i);
	}
    }
}

void SoundSystem::set_music_volume(double vol)
{
  STACKTRACE;
  if(!mix_ok)
    return;
  
  tw_set_config_file("client.ini");
  set_config_float("Sound", "MusicVolume", vol);

  if(vol < 0.05) 
    {
      Mix_HaltMusic();
      music_off = true;
      return;
    }
  
  Mix_VolumeMusic(int(vol*double(MIX_MAX_VOLUME)));
  
  //if the music was off completely, start playing it again now
  if(music_off) 
    {
      music_off = false;
      const std::string music = current_music;
      current_music = "";
      if(!music.empty()) 
	{
	  play_music(music);
	}
    }
}

void SoundSystem::set_sound_volume(double vol)
{
  STACKTRACE;
  if(!mix_ok)
    return;
  
  if(vol < 0.05) 
    {
      sound_off = true;
    } 
  else 
    {
      sound_off = false;
      Mix_Volume(-1,int(vol*double(MIX_MAX_VOLUME)));
    }
  tw_set_config_file("client.ini");
  set_config_float("Sound", "SoundVolume", vol);
}

double SoundSystem::load_music_volume()
{
  STACKTRACE;
  tw_set_config_file("client.ini");
  return get_config_float("Sound", "MusicVolume", 0.5);
}

double SoundSystem::load_sound_volume()
{
  STACKTRACE;
  tw_set_config_file("client.ini");
  return get_config_float("Sound", "SoundVolume", 0.5);
}

bool SoundSystem::playing_music()
{
  STACKTRACE;
  if(!mix_ok)
    return false;
  
  return Mix_PlayingMusic();
}
