/*
This file is part of "TW-Light" 
                    http://tw-light.berlios.de/
Copyright (C) 2001-2004  TimeWarp development team

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
*/

#include <allegro.h>
#ifdef WIN32
#include <winalleg.h>
#endif

#include <algorithm>

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "other/dialogs.h"

#include "melee/mmain.h"
#include "scp.h"
#include "melee.h"
#include "melee/mcontrol.h"
#include "input.h"
#include "mhelpers.h"

/*

Various helpers included in mhelpers.cpp :

Type verication
Byte Ordering (endianness)
	offers invert_ordering, intel_ordering, motorola_ordering, 
	normal versions are 32 bit; and short versions of each are 16 bit



*/

#define PLATFORM_IS_ALLEGRO

#if defined PLATFORM_IS_ALLEGRO
#include <allegro.h>
#if defined ALLEGRO_MSVC
#include <winalleg.h>
#include <windows.h>
#endif
#else
#error unknown platform (allegro?)
#endif


#include "melee.h"

#include "scp.h"

#include "mframe.h"
#include "mgame.h"


volatile int debug_value = 0;


/*------------------------------
		Video mode
------------------------------*/
int  GFX_TIMEWARP_WINDOW =  GFX_AUTODETECT_WINDOWED;
int GFX_TIMEWARP_FULLSCREEN = GFX_AUTODETECT_FULLSCREEN;


VideoSystem* videosystem = NULL;

static void tw_display_switch_out() 
{
}

static void tw_display_switch_in() 
{
  if (get_time() > videosystem->last_poll + 1000) 
    {
      videosystem->redraw();
    }
  else 
    videosystem->screen_corrupted = true;
}

static int _gamma = -1;
static unsigned char _gamma_map[256];
int get_gamma() 
{
  return _gamma;
}

void set_gamma(int gamma) 
{
  if (gamma < 0) 
    gamma = 0;
  if (gamma > 255) 
    gamma = 255;
  _gamma = gamma;
  int i;
  for (i = 0; i < 256; i += 1) 
    {
      _gamma_map[i] = iround_down(256 * pow(i / 255.5, 1-gamma/258.));
    }
  return;
}

void gamma_color_effects (RGB *c) 
{
  if (!c->filler) 
    {
      c->r = _gamma_map[c->r];
      c->g = _gamma_map[c->g];
      c->b = _gamma_map[c->b];
    }
  else 
    {
      int alpha = (c->filler ^ 255) + 1;
      int r, g, b;
      r = (c->r << 8) / alpha;
      g = (c->g << 8) / alpha;
      b = (c->b << 8) / alpha;
      if ((r | g | b) > 255) 
	{
	  tw_error("gamma_color_effects : premultiplied alpha color invalid");
	}
      r = _gamma_map[r];
      g = _gamma_map[g];
      b = _gamma_map[b];
      r = (r * alpha) >> 8;
      g = (g * alpha) >> 8;
      b = (b * alpha) >> 8;
      c->r = r;
      c->g = g;
      c->b = b;
    }
  return;
}

int tw_color (RGB c) 
{
  videosystem->color_effects(&c);
  return makecol(c.r, c.g, c.b);
}

int tw_color (int r, int g, int b) 
{
  RGB c = {r,g,b};
  return makecol(c.r, c.g, c.b);
}

int VideoSystem::poll_redraw() 
{
  STACKTRACE;
  last_poll = get_time();
  if (screen_corrupted) 
    {
      screen_corrupted = false;
      videosystem->redraw();
      return 1;
    }
  return 0;
}

void VideoSystem::preinit() 
{
  STACKTRACE;
  int i;
  surface = NULL;
  width = -1;
  height = -1;
  bpp = -1;
  fullscreen = false;
  font_data = NULL;
  basic_font = NULL;
  palette = (RGB*)malloc(sizeof(RGB) * 256);

  FILE *f = fopen(data_full_path("palette").c_str(), "rb");
  if (f) 
    {
      for (i = 0; i < 256; i += 1) 
	{
	  palette[i].r = fgetc(f);
	  palette[i].g = fgetc(f);
	  palette[i].b = fgetc(f);
	  palette[i].filler = 0;
	}
      fclose(f);
    } 
  
  color_effects = gamma_color_effects;
  
  screen_corrupted = false;
  last_poll = -1;
  window.preinit();
  window.init( &window );
  window.locate(0,0,0,0,  0,1,0,1);
}

FONT *VideoSystem::get_font(int s) 
{
  STACKTRACE;
  if (!font_data) 
    {
      if (basic_font) 
	return basic_font;
      if (!font) 
	{
	  tw_error_exit("VideoSystem::get_font - something horribly wrong!");
	  return font;
	}
    }
  if (s < 0) s = 0;
  if (s > 7) s = 7;
  return (FONT*) font_data[s].dat;
}

void VideoSystem::set_palette(RGB *new_palette) 
{
  STACKTRACE;
  memcpy(palette, new_palette, sizeof(RGB) * 256);
  update_colors();
  return;
}

void VideoSystem::update_colors() 
{
  STACKTRACE;
  RGB tmp[256];
  if (!palette) return;
  memcpy(tmp, palette, sizeof(RGB) * 256);
  int i;	
  for (i = 1; i < 256; i += 1) 
    {
      color_effects(&tmp[i]);
      tmp[i].r = ((unsigned int)(tmp[i].r) * 63) / 255;
      tmp[i].g = ((unsigned int)(tmp[i].g) * 63) / 255;
      tmp[i].b = ((unsigned int)(tmp[i].b) * 63) / 255;
    }
  if (rgb_map) 
    create_rgb_table ( rgb_map, tmp, NULL);
  ::set_palette(tmp);
  return;
}

void VideoSystem::redraw() 
{
  STACKTRACE;
  VideoEvent ve;
  ve.type = Event::VIDEO;
  ve.subtype = VideoEvent::REDRAW;
  ve.window = &window;
  window._event(&ve);
}

int VideoSystem::set_resolution (int width, int height, int bpp, int fullscreen) 
{
  VideoEvent ve;
  ve.type = Event::VIDEO;
  ve.window = &window;
  if (width == 0) 
    width = this->width;
  if (height == 0) 
    height = this->height;
  if (bpp == 0) 
    bpp = this->bpp;
  if (!basic_font) 
    basic_font = font;
  if (!font_data) 
    font_data = load_datafile("fonts.dat");
  if ((bpp == this->bpp) && 
      (width == this->width) && 
      (height == this->height) && 
      (fullscreen == this->fullscreen)) 
    return true;

  if ((width < 300) || (height < 200)) 
    {
      char buffy[512];
      sprintf(buffy, "Error switching to graphics mode\n(%dx%d @ %d bit)\nresolution too low", width, height, bpp);
      if (this->bpp == -1) 
	{
	  tw_error_exit(buffy);
	}
      tw_alert (buffy, "Continue");
      return false;
    }
  ve.subtype = VideoEvent::INVALID;
  window._event(&ve);
  surface = NULL;
  set_color_depth(bpp);
  if ( set_gfx_mode((fullscreen ? GFX_TIMEWARP_FULLSCREEN : GFX_TIMEWARP_WINDOW), width, height, 0, 0)) 
    {
      const char *part1 = "Error switching to graphics mode";
      char part2[256];
      sprintf (part2, "(%dx%d @ %d bit)", width, height, bpp);
      const char *part3 = allegro_error;
      if (this->bpp == -1) 
	{
	  char buffy[1024];
	  sprintf(buffy, "%s\n%s\n%s", part1, part2, part3);
	  tw_error_exit(buffy);
	}
      set_color_depth(this->bpp);
      set_gfx_mode((this->fullscreen ? GFX_TIMEWARP_FULLSCREEN : GFX_TIMEWARP_WINDOW), 
		   this->width, this->height, 0, 0);
      alert (part1, part2, part3, "Continue", NULL, ' ', '\n');
      surface = screen;
      ve.subtype = VideoEvent::VALID;
      window._event(&ve);
      surface = NULL;
      redraw();
      return false;
    }
  surface = screen;
  if (set_display_switch_mode(SWITCH_BACKAMNESIA) == -1)
    set_display_switch_mode(SWITCH_BACKGROUND);
  set_display_switch_callback(SWITCH_IN, tw_display_switch_in);
  set_display_switch_callback(SWITCH_OUT, tw_display_switch_out);
  
  int owidth, oheight, obpp, ogamma, ofullscreen;
  owidth = this->width; oheight = this->height; obpp = this->bpp; 
  ogamma = this->gamma; ofullscreen = this->fullscreen;

  this->width = width;
  this->height = height;
  this->bpp = bpp;
  this->fullscreen = fullscreen;
  update_colors();
  if (font_data) 
    font = (FONT *)(font_data[15].dat);
  else 
    font = basic_font;
  if (bpp == 8) 
    {
      if (!rgb_map) rgb_map = (RGB_MAP*)malloc(1<<15);
      RGB tmp[256];
      memcpy(tmp, palette, sizeof(RGB) * 256);
      int i;	
      for (i = 0; i < 256; i += 1) 
	{
	  tmp[i].r = ((unsigned int)(tmp[i].r) * 63) / 255;
	  tmp[i].g = ((unsigned int)(tmp[i].g) * 63) / 255;
	  tmp[i].b = ((unsigned int)(tmp[i].b) * 63) / 255;
	}
      create_rgb_table ( rgb_map, tmp, NULL);
  }
  if (obpp != bpp) 
    {
      ve.subtype = VideoEvent::CHANGE_BPP;
      window._event(&ve);
    }
  if ((owidth != width) || (oheight != height)) 
    {
      ve.subtype = VideoEvent::RESIZE;
      window._event(&ve);
    }
  
  ve.subtype = VideoEvent::VALID;
  window._event(&ve);
  
  redraw();
  return true;
}

struct VW_lock_data 
{
  short int x, y, w, h;
};

void VideoWindow::lock ( ) 
{
  lock_level += 1;
  if (lock_level == 1) 
    {
      set_clip(surface, x, y, x+w-1, y+h-1);
      acquire_bitmap(surface);
    }
}

void VideoWindow::unlock ( ) 
{
  if (lock_level == 0) {
    tw_error("VideoWindow unlocked too many times");
    return;
  }
  lock_level -= 1;
  if (lock_level == 0) 
    {
      release_bitmap(surface);
      set_clip(surface, 0, 0, surface->w-1, surface->h-1);
    }
}

#define VideoWindow_callbacklist_units 4
void VideoWindow::match ( VideoWindow *old ) 
{
  if (lock_level) 
    {
      tw_error("VideoWindow - illegal while locked");
    }
  if (!parent && old->parent) 
    init( old->parent );
  locate 
    ( 
     old->const_x, old->propr_x, 
     old->const_y, old->propr_y, 
     old->const_w, old->propr_w,
     old->const_h, old->propr_h
     );
  return;
}

void VideoWindow::hide() 
{
  STACKTRACE;
  locate(0,0,0,0,0,0,0,0);
  return;
}

void VideoWindow::add_callback( BaseClass *callee ) 
{
  STACKTRACE;
  std::list<BaseClass*>::iterator cb = std::find(callback_list.begin(),callback_list.end(), callee);
  if(cb != callback_list.end())
    {
      tw_error("adding VideoWindow callback twice");
    }
  callback_list.push_back(callee);
  return;
}

void VideoWindow::remove_callback( BaseClass *callee ) 
{
  STACKTRACE;
  callback_list.remove(callee);
  return;
}

void VideoWindow::event(int subtype) 
{
  STACKTRACE;
  if (lock_level) 
    {tw_error("VideoWindow - illegal while locked");}
  VideoEvent ve;
  ve.type = Event::VIDEO;
  ve.subtype = subtype;
  ve.window = this;
  issue_event( callback_list, &ve);
  return;
}

void VideoWindow::update_pos() 
{
  STACKTRACE;
  if (lock_level) 
    {tw_error("VideoWindow - illegal while locked");}
  int nx = 0, ny = 0, nw = 0, nh = 0;
  if (parent == this) 
    {
      surface = videosystem->surface;
      if (surface) 
	{
	  nx = 0;
	  ny = 0;
	  nw = videosystem->width;
	  nh = videosystem->height;
	}
    }
  else 
    {
      if (parent) surface = parent->surface;
      else surface = NULL;
      if (surface) {
	nx = parent->x;
	ny = parent->y;
	nw = parent->w;
	nh = parent->h;
      }
    }
  
  x = nx + iround_up(const_x + propr_x * nw - 0.05);
  y = ny + iround_up(const_y + propr_y * nh - 0.05);
  w = iround_down(const_w + propr_w * nw + 0.05);
  h = iround_down(const_h + propr_h * nh + 0.05);
  
  if ((w <= 0) || (h <= 0)) surface = NULL;
  return;
}

void VideoWindow::_event( Event *e ) 
{
  STACKTRACE;
  if (e->type == Event::VIDEO) 
    {
      const VideoEvent *ve = (const VideoEvent *) e;
      if (ve->window != parent) {tw_error ("VideoWindow event not from parent?");}
      VideoEvent nve;
      nve.type = Event::VIDEO;
      nve.window = this;
      nve.subtype = ve->subtype;
      switch (ve->subtype) 
	{
	case VideoEvent::RESIZE : 
	  {
	    update_pos();
	  } 
	break;
	case VideoEvent::INVALID: 
	  {
	    x = y = w = h = 0;
	    surface = NULL;
	  } 
	  break;
	case VideoEvent::VALID: 
	  {
	    update_pos();
	  } 
	  break;
	case VideoEvent::REDRAW: 
	  {	    
	  } break;
	}
      issue_event( callback_list, &nve);
    }
  else {tw_error ( "VideoWindow got non-video event" );}
  return;
}

void VideoWindow::preinit () 
{
  parent = NULL;
  surface = NULL;
  callback_list.clear();
  
  lock_level = 0;
  
  const_x = const_y = const_w = const_h = 0;
  propr_x = propr_y = 0;
  propr_w = propr_h = 1;
  x = y = w = h = 0;
  return;
}

void VideoWindow::init ( VideoWindow *parent_window) 
{
  if (lock_level) 
    {tw_error("VideoWindow - illegal while locked");}
  if (parent) 
    parent->remove_callback( this );
  parent = parent_window;

  if (parent && (parent != this)) 
    parent->add_callback ( this );
  update_pos();
  event(VideoEvent::RESIZE);
  event(VideoEvent::REDRAW);
  return;
}

void VideoWindow::locate ( double x1, double x2, 
			   double y1, double y2, 
			   double w1, double w2, 
			   double h1, double h2) 
{
  if (lock_level) {tw_error("VideoWindow - illegal while locked");}
  const_x = x1;
  propr_x = x2;
  const_y = y1;
  propr_y = y2;
  const_w = w1;
  propr_w = w2;
  const_h = h1;
  propr_h = h2;
  update_pos();
  event(VideoEvent::RESIZE);
  event(VideoEvent::REDRAW);
  return;
}

void VideoWindow::deinit() 
{
  STACKTRACE;
  if (lock_level) {tw_error("VideoWindow - illegal while locked");}
  if (parent) {
    parent->remove_callback( this );
    parent = NULL;
  }
  if (callback_list.size()) 
    {tw_error("VideowWindow - deinit illegal while child windows remain");}
}

VideoWindow::~VideoWindow() 
{
  deinit();
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////


void ScrollText(std::vector<std::string>& text, FONT* font, int color, const std::string& backgr, const std::string& align, int speed)
{
  STACKTRACE;
  BITMAP* d_buf = create_bitmap(SCREEN_W, SCREEN_H);
  clear(d_buf);
  copy_buf();

  unsigned int q = 0;
  BITMAP* tmpB = load_bitmap(data_full_path(backgr).c_str(), NULL);
  BITMAP* background = create_bitmap(screen->w, screen->h);
  stretch_blit(tmpB, background, 0, 0, tmpB->w, tmpB->h, 0, 0, background->w, background->h);
  destroy_bitmap(tmpB);
  

  while (1)
    {
      blit(background, d_buf, 0, 0,  0, 0, background->w, background->h);
      for (unsigned int i = 0;i!=text.size();i++)
	{
	  text_mode(-1);
	  if(align == "c")
	    {
	      textout_centre(d_buf, font, text[i].c_str(), screen->w/2, screen->h+20-q/3+i*30,-1);
	    }
	  else if(align == "l")
	    {
	      textout(d_buf, font, text[i].c_str(), 30, screen->h+20-q/3+i*30,-1);
	    }
	  else if(align == "r")
	    {
	      textout_right(d_buf, font, text[i].c_str(), screen->w-30, screen->h+20-q/3+i*30,-1);
	    }
	}
      copy_buf();
      
      HAVE_A_WAIT(10);
      q+=speed;
      
      while (keypressed())
	{ 
	  int k = readkey(); 
	  if(k>>8 == KEY_ESC) 
	    return; 
	} 
      if (q/3 >= text.size()*30+screen->h+140)
	break;
    }
  clear(d_buf);
  copy_buf();	
  destroy_bitmap(d_buf);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
/*------------------------------
		SC2 Unit Conversion
------------------------------*/
int time_ratio; //1000 milliseconds / SC2 framerate
double distance_ratio;

/// \brief SC2 Unit Conversion
void init_sc2_unit_conversion() 
{
}
/// \brief SC2 Unit Conversion
int scale_frames(double value) 
{
  return (int)((value + 1) * time_ratio);
}
/// \brief SC2 Unit Conversion
double scale_turning (double turn_rate) 
{
  return (PI2 / 16) / (turn_rate + 1.0) / time_ratio;
}
/// \brief SC2 Unit Conversion
double scale_velocity (double velocity) 
{
  return velocity * distance_ratio / time_ratio;
}

/// \brief SC2 Unit Conversion
double scale_acceleration (double acceleration, double raw_hotspot_rate) 
{
  return acceleration * distance_ratio / (1 + raw_hotspot_rate) / time_ratio / time_ratio; 
}
/// \brief SC2 Unit Conversion
double scale_range (double range) 
{
  return range * 40;
}

/// \brief show file in pupup dialog
void show_file(std::string file) 
{
  int i;
  char *willy;
  PACKFILE *f;
  f = pack_fopen (file.c_str(), F_READ);
  if (!f) 
    {
      willy = (char*) malloc(file.size()+1);
      sprintf(willy, "Failed to load file \"%s\"", file.c_str());
    }
  else 
    {
      i = file_size(file.c_str());
      willy = (char*)malloc(i+1);
      i = pack_fread (willy, i, f);
      pack_fclose(f);
      willy[i] = 0;
    }
  show_text(willy);
  free(willy);
  return;
}

/// \brief show text in popup dialog
void show_text(std::string text) 
{
  help_dialog[2].dp = (void *) text.c_str();
  help_dialog[2].d1 = 0;
  help_dialog[2].d2 = 0;
  tw_popup_dialog(&videosystem->window, help_dialog, 1);
  return;
}


