/*
  This file is part of "TW-Light" 
  http://tw-light.berlios.de/
  Copyright (C) 2001-2004  TimeWarp development team

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string>
#include <errno.h>
#include <ctype.h>

#include "util/sound.h"

#include <allegro.h>
//#include <alleggl.h>
#ifdef WIN32
#include <allegro/platform/aintwin.h>
#include <winalleg.h>
#endif

#include "libraries/jpgalleg/jpgalleg.h"

#include "other/twconfig.h"
#include "other/dialogs.h"

#include "tests/testmain.h"

// standard includes
#include <iostream>
#include <exception>



extern "C"
{
  void init_ggob(void);
}

#ifdef ALLEGRO_MSVC
#pragma warning (disable:4786)
#endif

#define APPLICATION_NAME "TW-Light"

#include "melee.h"
#include "scp.h"
#include "gui.h"
#include "frame.h"

#include "util/get_time.h"

#include "melee/mview.h"
#include "melee/mcontrol.h"
#include "melee/mcbodies.h"
#include "melee/mgame.h"
#include "melee/mhelpers.h"

#include "melee/mmain.h"
#include "melee/mnet1.h"
#include "util/aastr.h"
#include "melee/mship.h" //remove
#include "melee/mfleet.h"


#include <Python.h>
#include "python/python_class.h"


DATAFILE* g_game_data = NULL;

void GOBIntro();

///the server port to connect to in client mode.  
unsigned serverPort = DEFAULT_SERVER_PORT;
///the server address to connect to in client mode.
std::string serverAddress(DEFAULT_SERVER_ADDRESS);

///the maximum number of clients that can join in server mode.  Any additional
///clients that try to join up won't be able to.  Only meaningful in server mode.
unsigned serverMaxClients = DEFAULT_SERVER_MAX_CLIENTS;

///the default port for a client to listen on.  Normally, with one client per
///machine, you don't have to specify this.  But to run multiple clients on a
///single machine, you should specify a unique port for each client on the same
///machine.  Only meaningful in client mode.
unsigned clientPort = DEFAULT_CLIENT_PORT;

void init_datafile_path();

/// Get SVN revision number from version file (this function work if user does not have subversion)
std::string GetSVNVersion()
{
  std::string tw_svn_version = TW_SVNVERSION;
  if(!(tw_svn_version=="exported"||tw_svn_version == ""))
    return tw_svn_version;
  
  int i;
  char buffy [16000] = {0};
  PACKFILE *f;
  
  f = pack_fopen (data_full_path("version.txt").c_str(), F_READ);
  if (!f)
    {
      return "Unknown";
      tw_error("Unable to retrive version information");
    }
  else 
    {
      i = pack_fread (buffy, 15999, f);
      pack_fclose(f);
      buffy[i] = 0;
    }
  char *s = strchr(buffy, 'r');
  if(!s)
    {
      return "";
    }

  char ver[30] = {0};
  sscanf(s+1, "%s", ver);
  return ver;
}

std::string tw_version() 
{
  STACKTRACE;
  std::string tw_version_string;

  tw_version_string += std::string(APPLICATION_NAME) + " "  + std::string(TW_MAJOR_VERSION) 
    + "r" 
    + GetSVNVersion()
    + " Built: "
    + __DATE__
    + " "
    + __TIME__;
  
  return tw_version_string;
}

/// Blits GUI background bitmap on to a video window
void showTitle(VideoWindow *window = &videosystem->window);

/// MELEE_EX dialog - selects multiplayer games as opposed to standard melee. 
void multiplayer_menu(int i = -1);

/// MELEE_EX dialog - selects alternate games as opposed to standard melee. 
///
///  From here you can access diagnostics (DIAGNOSTICS dialog) and ship info. 
///  (SHIPVIEW dialog) You can also test key jamming from this dialog.
void extended_menu(int i = -1);

/// FLEET dialog - manages fleet compositions for an individual player.
///  \param player Player index indicating which player's fleet to edit.
void edit_fleet(int player);

/*! \brief SHIPVIEW dialog - displays statistics and text information (if available) about the currently installed ships.
  \param si Ship index. By default 0, the first ship in the fleet.
  \param fleet Pointer to a fleet upon which the ship list is built. When 
  this parameter is set to the default value NULL, the reference fleet is 
  used to build the ship list.
*/
//void ship_view_dialog(int si = 0, Fleet *fleet = NULL);

/*! \brief DIAGNOSTICS dialog - displays version number and platform data */
void show_diagnostics();

/*! \brief Opens a screen showing which keys are currently pressed. Here the user may test various key combinations for conflicts. */
void keyjamming_tester();


void play_demo(const char *file_name = "demo.dmo") ;
void play_game(const char *_gametype_name, const std::string& save, Log *_log = NULL) ;
void play_net1client ( const char * address = NULL, int port = -1 ) ;
void play_net1server ( int port = -1 ) ;

int getKey();

Log *new_log (int logtype) 
{
  STACKTRACE;
  union 
  { 
    Log *log; 
    NetLog *netlog; 
  };
  log = NULL;

  switch (logtype) 
    {
    case Log::log_normal: 
      {
	log = new Log();
	log->init();
	return log;
      }
    case Log::log_net1server: 
      {
	netlog = new NetLog();
	netlog->init();
	netlog->type = Log::log_net1server;
	return netlog;
      }
    default: 
      {
	tw_error("that is not a valid log type");
      }
    }
  return NULL;
}

char *detect_gametype( Log *_log ) 
{
  STACKTRACE;
  int ltype;
  _log->unbuffer(Game::channel_init, &ltype, sizeof(int));
  ltype = intel_ordering(ltype);
  int gnamelength;
  _log->unbuffer(Game::channel_init, &gnamelength, sizeof(int));
  gnamelength = intel_ordering(gnamelength);
  if (gnamelength > 1000) 
    {
      tw_error("Game name too long");
      gnamelength = 1000;
    }
  char buffy[1024];
  _log->unbuffer(Game::channel_init, &buffy, gnamelength);
  buffy[gnamelength] = 0;
  _log->reset();
  return strdup(buffy);
}

std::string menuAccept;
std::string menuFocus;
std::string menuDisabled;
std::string menuSpecial;

BITMAP * titlePic     = NULL;

/*! defined in scp.h.  TODO replace these global variables */
std::string titleMusic = "";

/** 
    loads up the title screen and music, and starts playing the background menu music. 
*/
void prepareTitleScreenAssets() 
{
  STACKTRACE;
  titleMusic = data_full_path("music/Robeter-Space-Gods-Loop.ogg");
  tw_sound->play_music(titleMusic);

  menuAccept   = data_full_path("sound/menu/MENUACCEPT.wav");
  menuFocus    = data_full_path("sound/menu/MENUFOCUS.wav");
  menuDisabled = data_full_path("sound/menu/MENUDISABLED.wav");
  menuSpecial  = data_full_path("sound/menu/MENUSPECIAL.wav");

  titlePic = load_bitmap(data_full_path("images/scptitle.jpg").c_str(), NULL);
}

/// clears the screen, and displays a loading message to the user.
void showLoadingScreen() 
{
  STACKTRACE;
  
  std::vector<std::string> waitpics;
  waitpics.push_back(data_full_path("images/bspab.jpg"));
  waitpics.push_back(data_full_path("images/stangrop.jpg"));
  
  acquire_screen(); 
  clear_to_color(screen, 0);
 
  int pic = rand()%waitpics.size();

  BITMAP * temp = load_bitmap(waitpics[pic].c_str(), NULL);
  if(!temp)
    {
      tw_error("Unable to load background!!!\n%s", waitpics[pic].c_str());
    }
  stretch_blit(temp, screen, 0, 0, temp->w, temp->h, 0, 0, screen->w, screen->h);
  destroy_bitmap(temp);
  release_screen();
}

enum {
  DIALOG_CONNECT_BOX = 0,
  DIALOG_CONNECT_TITLE,
  DIALOG_CONNECT_ADDRESS,
  DIALOG_CONNECT_ADDRESS_BOX,
  DIALOG_CONNECT_ADDRESS_EDIT,
  DIALOG_CONNECT_PORT,
  DIALOG_CONNECT_PORT_BOX,
  DIALOG_CONNECT_PORT_EDIT,
  //	DIALOG_CONNECT_TWOLOCALS,
  DIALOG_CONNECT_OK,
  DIALOG_CONNECT_CANCEL,
  DIALOG_CONNECT_BLAH
};

static DIALOG connect_dialog[] = {
  // (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp)
  { d_box_proc,        40,   100,  480,  260,  255,  0,    0,    0,       0,    0,    NULL, NULL, NULL },
  { d_text_proc,       50,   110,  460,  30,   255,  0,    0,    0,       0,    0,    (void *)"Connect to server:" , NULL, NULL },
  { d_text_proc,       50,   140,  460,  30,   255,  0,    0,    0,       0,    0,    (void *)"IP Address" , NULL, NULL },
  { d_box_proc,        48,   168,  464,  34,   255,  0,    0,    0,       0,    0,    NULL, NULL, NULL },
  { d_edit_proc,       50,   170,  460,  30,   255,  0,    0,    0,       75,   0,    dialog_string[0], NULL, NULL },
  { d_text_proc,       50,   220,  460,  30,   255,  0,    0,    0,       0,    0,    (void *)"Port #" , NULL, NULL },
  { d_box_proc,        48,   248,  464,  34,   255,  0,    0,    0,       0,    0,    NULL, NULL, NULL },
  { d_edit_proc,       50,   250,  460,  30,   255,  0,    0,    0,       75,   0,    dialog_string[1], NULL, NULL },
  //{ d_check_proc,      50,   290,  160,  20,   255,  0,    0,    0,       0,    0,    (void *)"Two Local Players", NULL, NULL },
  { d_button_proc,     50,   320,  160,  30,   255,  0,    0,    D_EXIT,  0,    0,    (void *)"Connect", NULL, NULL },
  { d_button_proc,     350,  320,  160,  30,   255,  0,    0,    D_EXIT,  0,    0,    (void *)"Abort", NULL, NULL },
  { d_tw_yield_proc,   0,    0,    0,    0,    255,  0,    0,    0,       0,    0,    NULL, NULL, NULL },
  { NULL,              0,    0,    0,    0,    255,  0,    0,    0,       0,    0,    NULL, NULL, NULL }
};

int connect_menu(VideoWindow *window, char **address, int *port) { STACKTRACE
								     int i = 0;

 if (*address)
   strncpy(dialog_string[0], *address, 70);
 if (port)
   sprintf(dialog_string[1], "%d", *port);

 i = tw_popup_dialog(window, connect_dialog, DIALOG_CONNECT_ADDRESS_EDIT);
 if (i != DIALOG_CONNECT_OK)
   return -1;
 *port = atoi(dialog_string[1]);
 *address = strdup(dialog_string[0]);
 return 0;
}

//#define DIALOG_LISTEN_TWOLOCALS 4
#define DIALOG_LISTEN_OK        4
#define DIALOG_LISTEN_CANCEL    5

static DIALOG listen_dialog[] =
  {
    // (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)     (d1)  (d2)  (dp)
    { d_box_proc,        120,  100,  300,  140,  255,  0,    0,    0,          0,    0,    NULL, NULL, NULL },
    { d_text_proc,       140,  110,  240,  25,   255,  0,    0,    0,          0,    0,    (void*)"Listen on which port?", NULL, NULL },
    { d_box_proc,        178,  138,  124,  29,   255,  0,    0,    0,          0,    0,    NULL, NULL, NULL },
    { d_edit_proc,       180,  140,  120,  25,   255,  0,    0,    0,          5,    0,    dialog_string[1], NULL, NULL },
    { d_button_proc,     140,  200,  120,  30,   255,  0,    0,    D_EXIT,     0,    0,    (void*) "OK", NULL, NULL },
    { d_button_proc,     280,  200,  120,  30,   255,  0,    0,    D_EXIT,     0,    0,    (void*) "Cancel", NULL, NULL },
    { d_tw_yield_proc,   0,    0,    0,    0,    255,  0,    0,    0,          0,    0,    NULL, NULL, NULL },
    { NULL,              0,    0,    0,    0,    255,  0,    0,    0,          0,    0,    NULL, NULL, NULL }
  };

static int listen_menu(VideoWindow *window, int port) 
{
  STACKTRACE
    dialog_string[1][0] = '\0';
  int p = -1;
  sprintf(dialog_string[1], "%d", port);
  while ((p <= 0) || (p > 65535)) 
    {
      if (tw_popup_dialog(window, listen_dialog, 2) != DIALOG_LISTEN_OK) 
	{
	  //game->quit("Quit - game aborted from network \"listen\" menu");
	  return -1;
	}
      p = atoi(dialog_string[1]);
    }
  return p;
}

int is_escape_pressed() 
{
  STACKTRACE;
  poll_keyboard();
  return key[KEY_ESC];
}

void play_game(const char *_gametype_name,  const std::string& save, Log *_log ) 
{
  STACKTRACE;
  char gametype_name[1024];
  char *c;
  Game *new_game = NULL;
  
  showLoadingScreen();
  
  strncpy(gametype_name, _gametype_name, 1000);
  for (c = strchr(gametype_name, '_'); c; c = strchr(c, '_'))
    *c = ' ';

  try 
    {
      if (game) 
	{
	  delete game;
	  game = NULL;
	}
    
      if (!_log) 
	{
	  _log = new Log();
	  _log->init();
	}
      
      GameType *type = gametype(gametype_name);
      if (type)
	new_game = type->new_game();
      else
	tw_error("wait a sec... I can't find that game type");
    
      new_game->preinit();
      new_game->window = new VideoWindow;
      new_game->window->preinit();
      new_game->window->init(&videosystem->window);
      new_game->window->locate(0,0,0,0,0,1,0,1);
      new_game->init(_log, save );
      new_game->play();
      new_game->log->deinit();
      new_game->setGameDone(true);
      
      //damn global variables!
      // inside melee/mgame.cpp, in Game::prepare, ::game is set to the value of the current game
      // inside ais/c_wussiebot.cpp (and some other places), ::game is used
      delete new_game;
      game = NULL;
    }
  
  catch (int i) 
    {
      if (i == -1) 
	throw;
      if (__error_flag & 1) 
	throw;
      if (i != 0) 
	caught_error ("%s %s caught int %d", __FILE__, __LINE__, i);
      if (__error_flag & 1) 
	throw;
    }
  catch (const char *str) 
    {
      if (__error_flag & 1) 
	throw;
      caught_error("message: \"%s\"", str);
      if (__error_flag & 1) 
	throw;
    }
  catch (...) 
    {
      if (__error_flag & 1) 
	throw;
      caught_error("Ack(2)!!!\nAn error occured!\nBut I don't know what error!");
      if (__error_flag & 1) 
	throw;
    }
  
  prepareTitleScreenAssets();
  showTitle();
  return;
}


void play_net1client ( const char *_address, int _port ) 
{
  STACKTRACE;
  NetLog *log = new NetLog();
  log->init();
  log->type = Log::log_net1client;
  
  log->set_all_directions(Log::direction_read);
  log->set_direction(Game::channel_client , Log::direction_write | Log::direction_read | NetLog::direction_immediate);
  log->set_direction(Game::channel_client + Game::_channel_buffered, Log::direction_write | Log::direction_read);

  tw_set_config_file("client.ini");
  char address[128];
  int port, i;
  while (!log->net.isConnected()) {
    if (!_address) strncpy(address, get_config_string("Network", "Address", ""), 127);
    else strncpy(address, _address, 127);
    if (_port == -1) port = get_config_int("Network", "Port", 15515);
    else port = _port;
    char *addressaddress = address;
    if (connect_menu(&videosystem->window, &addressaddress, &port) == -1) 
      return;
    set_config_string("Network", "Address", addressaddress);
    message.out("...");
    i = ((NetLog*)log)->net.connect(addressaddress, port, is_escape_pressed);
    free(addressaddress);
    if (i) 
      {
	//						while (is_escape_pressed());
	while (keypressed()) readkey();
	tw_error("connection failed");
      }
  }
  
  log->net.optimize4latency();
  message.out("connection established");
  
  char *gname = detect_gametype(log);
  log->reset();
  
  play_game(gname, "", log );
  
  free(gname);
  return;
}

void play_net1server( int _port) 
{
  STACKTRACE;
  NetLog *log = new NetLog();
  log->init();
  
  log->type = Log::log_net1server;
  log->set_all_directions(Log::direction_write | Log::direction_read | NetLog::direction_immediate);
  log->set_direction(Game::channel_client , Log::direction_read);
  log->set_direction(Game::channel_client + Game::_channel_buffered, Log::direction_read);
  log->set_direction(Game::channel_server + Game::_channel_buffered, Log::direction_write | Log::direction_read);
  
  tw_set_config_file("client.ini");
  int port;
  while (!log->net.isConnected()) 
    {
      if (_port == -1) port = get_config_int("Network", "Port", 15515);
      else port = _port;
      
      port = listen_menu( &videosystem->window, port );
      if (port == -1) return;
      
      message.out("...");
      log->net.listen(port, is_escape_pressed);
      
    }
  
  log->net.optimize4latency();
  message.out("connection established");
  
  play_game("Melee", "", log);
  
  return;
}



char dialog_string[20][128];

int MAX_PLAYERS = 1;
int MAX_CONFIGURATIONS = 1;
int MAX_TEAMS = 1;

// list box getter functions
char *playerListboxGetter(int index, int *list_size) ;
char *controlListboxGetter(int index, int *list_size) ;
char *viewListboxGetter(int index, int *list_size) ;

FONT *TW_font = NULL;


char **player_type = NULL;
int *player_config = NULL;
int *player_team = NULL;

Control *load_player(int i) 
{
  STACKTRACE;
  char tmp[32];
  Control *r = NULL;

  sprintf (tmp, "Config%d", player_config[i]);
  r = getController(player_type[i], tmp, Game::channel_none);
  if (r)
    {
      r->load("scp.ini", tmp);
    }
  return r;
}


class MainMenu : public BaseClass 
{
public:
  virtual void _event(Event * e);
  virtual void preinit();
  virtual void deinit();
  virtual void init(VideoWindow *parent);
  virtual void doit();
  virtual void enable();
  virtual void disable();
  int state;
  VideoWindow *window;
} mainmenu;

void MainMenu::_event(Event *e) 
{
  STACKTRACE;
  
  if (e->type == Event::VIDEO) 
    {
      if (e->subtype == VideoEvent::REDRAW) if (state & 1) showTitle();
    }
}

void MainMenu::enable() 
{
  STACKTRACE;
		   
  if (!(state & 2)) 
    window->add_callback(this);
  state |= 3;
}

void MainMenu::disable() 
{
  STACKTRACE;	   
  state &=~ 1;
}

void MainMenu::preinit() 
{
  STACKTRACE; 
  window = NULL;
  state = 0;
}

void MainMenu::init(VideoWindow *parent) 
{
  STACKTRACE; 
  if (window) 
    window->init(parent);
  else 
    {
      window = new VideoWindow();
      window->preinit();
      window->init(parent);
    }
}

void MainMenu::deinit() 
{
  STACKTRACE;   
		   
  if (state & 2) 
    {
      window->remove_callback(this);
      window->deinit();
      delete window;
      window = NULL;
    }
}

void MainMenu::doit() 
{
  STACKTRACE;
		   
  int i;
  char tmp[32];
  
  tw_set_config_file("scp.ini");
  if (!player_type) 
    {		
      MAX_PLAYERS        = get_config_int("Limits", "MaxPlayers", 12);
      MAX_CONFIGURATIONS = get_config_int("Limits", "MaxConfigurations", 4);
      MAX_TEAMS          = get_config_int("Limits", "MaxTeams", 6);
      player_type = new char*[MAX_PLAYERS];
      player_config = new int[MAX_PLAYERS];
      player_team   = new int[MAX_PLAYERS];
    }
  for (i = 0; i < MAX_PLAYERS; i += 1) 
    {
      sprintf(tmp, "Player%d", i+1);
      player_type[i] = strdup(get_config_string(tmp, "Type", "Human"));
      player_config[i] = get_config_int (tmp, "Config", i % MAX_CONFIGURATIONS);
      player_team[i] = get_config_int (tmp, "Team", 0);
    }
  
  prepareTitleScreenAssets();
  showTitle();
  enable();
  
  int mainRet;
  do 
    {
      //mainRet = popup_dialog(mainDialog, MAIN_DIALOG_MELEE);
      mainRet = tw_do_dialog(window, mainDialog, MAIN_DIALOG_MELEE);
      switch (mainRet) {
      case MAIN_DIALOG_MELEE:
	disable();
	play_game("Melee", "", NULL);
	enable();
	break;
      case MAIN_DIALOG_MULTIPLAYER:
	disable();
	multiplayer_menu();
	enable();
	break;
      case MAIN_DIALOG_MELEE_EXTENDED:
	disable();
	extended_menu();
	enable();
	break;
      case MAIN_DIALOG_PREFERENCES:
	showTitle();
	options_menu(NULL);
	break;
      case MAIN_DIALOG_HELP:
	show_file(data_full_path("ingame.txt").c_str());
	break;
      case MAIN_DIALOG_NEW_GAME:
	{
	  // Show Intro
	  RGB tmpPal[256];
	  memcpy(tmpPal, palette_color, sizeof(RGB) * 256);
	  set_palette((const RGB*)g_game_data[TW_FONT_PALETTE].dat);
      
	  GOBIntro();
	  videosystem->set_palette(tmpPal);

	  disable();
	  play_game("GOB", "", NULL);
	  enable();
	  break;
	}
	case MAIN_DIALOG_ABOUT:
	{
	  // Show Intro
	  RGB tmpPal[256];
	  memcpy(tmpPal, palette_color, sizeof(RGB) * 256);
	  set_palette((const RGB*)g_game_data[TW_FONT_PALETTE].dat);

	  std::vector<std::string> about;
	  about.push_back("Team");
	  about.push_back("");
	  about.push_back("Programmers:");
	  about.push_back("");
	  about.push_back("  * Yura Semashko (Yurand)");
	  about.push_back("  * Paul Forest (youBastrd)");
	  about.push_back("");
	  about.push_back("Content and Plot manager:");
	  about.push_back("");
	  about.push_back("  * Asaf Hamtzany (UAF)");
	  about.push_back("");
	  about.push_back("Dialog Writers");
	  about.push_back("");
	  about.push_back("  * Pelorki");
	  about.push_back("");
	  about.push_back("Graphic Designers");
	  about.push_back("");
	  about.push_back("  * Rick Bushie (Firehazurd)");
	  about.push_back("");
	  about.push_back("Music");
	  about.push_back("");
	  about.push_back("  * Robeter Productions (www.robeterproductions.com)");
	  about.push_back("  * Mark Forest (Abe Froman)");
	  about.push_back("");
	  about.push_back("Special Thanks:");
	  about.push_back("");
	  about.push_back("  * Orz");
	  about.push_back("  * Tau");
	  about.push_back("  * GeomanNL");
	  about.push_back("  * Officer Flubbo");
	  about.push_back("  * Captain Maim");
	  about.push_back("  * Jumping Peppers");
	  about.push_back("  * Robeter Productions");

	  ScrollText(about,  D_FONT_ENGLISH, -1, "images/background.jpg", "l", 3);

	  videosystem->set_palette(tmpPal);
	  videosystem->redraw();
	  break;
	}
      }
    } while((mainRet != MAIN_DIALOG_EXIT) && (mainRet != -1));
  
}

int tw_main(int argc, char *argv[]);

int main(int argc, char *argv[]) 
{
  int r;
  r = tw_main(argc, argv);
  return r;
}
END_OF_MAIN();

int tw_main(int argc, char *argv[]) 
{ 
  STACKTRACE;

#ifdef WIN32
  char szPath[MAX_PATH];
  GetModuleFileName(NULL, szPath, sizeof(szPath));
  if (strrchr(szPath, '\\')) *strrchr(szPath, '\\') = '\0';
  SetCurrentDirectory(szPath);
#endif

  if(argc==2&&!strcmp(argv[1],"-test"))
    {
      return RunTests();
    }
  
  int i;
  int auto_port = -1;
  const char *auto_play = NULL, *auto_param = NULL;
  
  log_debug(NULL);
  time_t start_time = time(NULL);
  log_debug("Log started at %s\n", asctime(localtime(&start_time)));
  if (allegro_init() < 0)
    tw_error_exit("Allegro initialization failed");
  //install_allegro_gl();
  init_datafile_path();
  create_user_ini();
  python::init();  

  
  g_game_data = load_datafile(data_full_path("gamedata.dat").c_str());

  VideoSystem video;
  videosystem = &video;
  videosystem->preinit();
  
  try 
    {
      init_time();
      init_error();
      
      set_window_title(APPLICATION_NAME);
      tw_set_config_file("client.ini");
      
    
      int screen_width = 640, screen_height = 480, screen_bpp = 32;
      int fullscreen = 0;
      
      auto_unload = get_config_int("System", "AutoUnload", 0);
		
      const AGUP_THEME *theme = agup_theme_by_name(get_config_string("Video", "GuiTheme", "Photon"));
      
      screen_bpp       = get_config_int("Video", "BitsPerPixel", 16);
      screen_width     = get_config_int("Video", "ScreenWidth", 640);
      screen_height    = get_config_int("Video", "ScreenHeight", 480);
      fullscreen       = get_config_int("Video", "FullScreen", false);
      
      SpaceSprite::mip_bias = get_config_int ("View", "Mip_bias", 0);
      SpaceSprite::mip_min = get_config_int ("View", "Mip_min", 0);
      SpaceSprite::mip_max = get_config_int ("View", "Mip_max", 0);
      
      interpolate_frames = get_config_int("View", "InterpolateFrames", 0);
      set_tw_aa_mode(get_config_int("Rendering", "AA_Mode", 0));
      int gamma   = get_config_int("Video", "Gamma", 128);
      set_gamma( gamma );
      
      int inputs = 7;
      bool sound_enabled = true;
      
      // parse command-line arguments
      for (i = 1; i < argc; i += 1) 
	{
	  if (!strcmp(argv[i], "-res") && (argc > i + 2)) 
	    {
	      log_debug("command-line argument -res\n");
	      screen_width = atoi(argv[i+1]);
	      screen_height = atoi(argv[i+2]);
	      i += 2;
	    }
	  else if (!strcmp(argv[i], "-bpp") && (argc > i + 1)) 
	    {
	      log_debug("command-line argument -bpp\n");
	      screen_bpp = atoi(argv[i+1]);
	      i += 1;
	    }
	  else if (!strcmp(argv[i], "-fullscreen")) 
	    {
	      log_debug("command-line argument -fullscreen\n");
	      fullscreen = true;
	    }
	  else if (!strcmp(argv[i], "-window")) 
	    {
	      log_debug("command-line argument -window\n");
	      fullscreen = false;
	    }
	  else if(!strcmp(argv[i], "-nosound")) 
	    {
	      sound_enabled = false;
	    }
	  else if (!strcmp(argv[i], "-nokeyboard")) 
	    {
	      log_debug("command-line argument -nokeyboard\n");
	      inputs &= ~1;
	    }
	  else if (!strcmp(argv[i], "-nomouse")) 
	    {
	      log_debug("command-line argument -nomouse\n");
	      inputs &= ~2;
	    }
	  else if (!strcmp(argv[i], "-nojoystick")) 
	    {
	      log_debug("command-line argument -nojoystick\n");
	      inputs &= ~4;
	    }
	  else if (!strcmp(argv[i], "-noidle")) 
	    {
	      log_debug("command-line argument -noidle\n");
	      _no_idle = 1;
	    }
	  else if (strstr(argv[i], COMMAND_LINE_SERVER_LONG)) 
	    {
	      if (strlen(argv[i]) > strlen(COMMAND_LINE_SERVER_LONG)) 
		{
		  // users can use two formats for the address: an alphanumeric name, like
		  // "localhost", or a dotted ip address, like a.b.c.d .  Further, users
		  // can specify the port here as well, by adding a colon and the number to 
		  // the end of the address.  
		  const char * addressStart = strstr(argv[i], COMMAND_LINE_SERVER_LONG) + strlen(COMMAND_LINE_SERVER_LONG);
		  const char * addressEnd = strstr(addressStart, ":");
		  
		  // user has added a colon, indicating the port.  The port starts at addressEnd
		  // and goes to the end of argv[i]
		  if (addressEnd && strlen(addressEnd) > 1) 
		    { 
		      // this is +1 to skip the ":"
		      serverPort = atoi(addressEnd+1);
		    }
		  serverAddress.clear();
		  serverAddress.append(addressStart, addressEnd-addressStart);
		  
		  log_debug("commandline specified server as : host=[%s], port=[%u]\n", serverAddress.c_str(), serverPort);
		}
	    }
	  else if (strstr(argv[i], COMMAND_LINE_MAX_CLIENTS_LONG)) 
	    {
	      // this is the maximum number of computers that can connect to this program instance. 
	      // Used in server mode only, no effect otherwise.
	      if (strlen(argv[i]) > strlen(COMMAND_LINE_MAX_CLIENTS_LONG)) 
		{
		  const char * portStart = strstr(argv[i], COMMAND_LINE_MAX_CLIENTS_LONG) + strlen(COMMAND_LINE_MAX_CLIENTS_LONG);
		  if (strlen(portStart) > 1) 
		    {
		      serverMaxClients = atoi(portStart);
		    }
		  log_debug("commandline specified max clients as: serverMaxClients=[%u]\n", serverMaxClients);
		}
	    }
	  else if (strstr(argv[i], COMMAND_LINE_CLIENT_PORT_LONG) && (argc > i + 0)) 
	    {
	      // this specifies the port this client will listen on.
	      // Used in client mode only, no effect otherwise.
	      if (strlen(argv[i]) > strlen(COMMAND_LINE_CLIENT_PORT_LONG)) 
		{
		  const char * clientportStart = strstr(argv[i], COMMAND_LINE_CLIENT_PORT_LONG) + strlen(COMMAND_LINE_CLIENT_PORT_LONG);
		  if (strlen(clientportStart) > 1) 
		    {
		      clientPort = atoi(clientportStart);
		    }
		  log_debug("commandline specified client port as : port=[%u]\n", clientPort);
		}
	    }
	  else if (!strcmp(argv[i], "-play") && (argc > i + 2)) 
	    {
	      log_debug("command-line argument -play\n");
	      auto_play = argv[i+1];
	      auto_param = argv[i+2];
	      i += 2;
	      if ((argc > i + 0) && (argv[i][0] != '-')) 
		{
		  auto_port = atoi(argv[i]);
		  i += 1;
		}
	    }
	  else 
	    {
	      log_debug("unrecognized command-line argument:");
	      log_debug(argv[i]);
	      log_debug("\n");
	    }
	}
    
      log_debug("command-line arguments parsed\n");
    
      srand(time(NULL));
      set_color_conversion(COLORCONV_KEEP_TRANS);
    
      videosystem->set_resolution(screen_width, screen_height, screen_bpp, fullscreen);
      register_bitmap_file_type("jpg", load_jpg, NULL);
    
      if(!theme)
	theme = agup_theme_by_name("Photon");
      agup_init(theme);    
      gui_shadow_box_proc = d_agup_shadow_box_proc;
      gui_ctext_proc = d_agup_ctext_proc;
      gui_button_proc = d_agup_button_proc;
      gui_edit_proc = d_agup_edit_proc;
      gui_list_proc = d_agup_list_proc;
      gui_text_list_proc = d_agup_text_list_proc;

      enable_input(inputs);
    
      SoundSystem sound(sound_enabled);
      tw_sound = &sound;
      tw_sound->set_sound_volume(tw_sound->load_sound_volume());
      tw_sound->set_music_volume(tw_sound->load_music_volume());
          

      View *v = NULL;
      v = get_view ( get_config_string("View", "View", NULL) , NULL );
      if (!v) v = get_view ( "Hero", NULL );
      set_view(v);
      init_ships();
      init_fleet();
    
      if (auto_play) {// FIX ME
	if (!strcmp(auto_play, "game")) play_game(auto_param, NULL);
      }
      else {
	mainmenu.preinit();
	mainmenu.init(&videosystem->window);
	mainmenu.doit();
	mainmenu.deinit();
      }

      if(game)
	{
	  delete game;
	  game = NULL;
	}
      disable_input(); 
    }
  catch (int i) {
    if (i == -1) 
      throw;
    if (__error_flag & 1) 
      throw;
    if (i != 0) 
      {
	char t[256];
	sprintf(t, "caught int %d", i);
	tw_error(t);
      }
    if (__error_flag & 1) 
      throw;
  }
  catch (const char *str) {
    if (__error_flag & 1) 
      throw;
    std::string tmp = "message: \"";
    tmp += std::string(str) + "\"";
    tw_error(tmp.c_str());
    if (__error_flag & 1) 
      throw;
  }
  catch (...) {
    if (__error_flag & 1) throw;
    tw_error("Ack!!!\nAn error occured on startup!\nBut I don't know what error!");
    if (__error_flag & 1) throw;
  }
  
  python::cleanup();  
  tw_exit(0);
  return 0;
}



void showTitle(VideoWindow *window) 
{
  STACKTRACE;
  BITMAP *src = titlePic;
  if(!src)
    {
      return;
      tw_error("Unable to open title pic");
    }

  if (!window->surface)
    return;
  window->lock();
  stretch_blit(src, window->surface, 
	       0,0,src->w,src->h,
	       window->x, window->y, window->w, window->h);

  textout_right(screen, font, tw_version().c_str(), 
		screen->w - 0*text_length(font, tw_version().c_str()), screen->h - 1*text_height(font), 
		palette_color[15]);
  window->unlock();
  return;
}


const char *select_game_menu () {
  STACKTRACE;
  select_game_dialog[2].dp3 = game_names;
  tw_set_config_file("client.ini");
  select_game_dialog[2].d1 = get_config_int("Menu", "SelectGame", 0);
  int i = tw_popup_dialog(NULL, select_game_dialog, 2);
  if (i == -1) 
    return NULL;
  else 
    {
      set_config_int("Menu", "SelectGame", select_game_dialog[2].d1);
      return game_names[select_game_dialog[2].d1];
    }
}

// MELEE_EX - dialog function
void multiplayer_menu(int i) 
{
  STACKTRACE;
  
  showTitle();
  if (i == -1)
    i = tw_popup_dialog(NULL, melee_multiplayer_dialog, MELEE_MULTIPLAYER_DIALOG_PLAY_NET1SERVER);
  switch (i) 
    {
    case -1:
    case MELEE_MULTIPLAYER_DIALOG_EXIT: 
      {
	return;
      }
      break;
    case MELEE_MULTIPLAYER_DIALOG_PLAY_NET1SERVER: 
      {
	play_net1server();
      }
      break;
    case MELEE_MULTIPLAYER_DIALOG_PLAY_NET1CLIENT: 
      {
	play_net1client();
      }
      break;
    }
      showTitle();
      return;
}

// MELEE_EX - dialog function
void extended_menu(int i) {
  STACKTRACE;
  
  showTitle();
  if (i == -1)
    i = tw_popup_dialog(NULL, melee_ex_dialog, MELEE_EX_DIALOG_PLAY_GAME);
  switch (i) {
  case -1:
  case MELEE_EX_DIALOG_EXIT: 
    {
      return;
    }
    break;
  case MELEE_EX_DIALOG_PLAY_GAME: 
    {
      const char *gname = select_game_menu();
      if (gname)
	play_game(gname, "", NULL);
    }
    break;
  case MELEE_EX_DIALOG_DIAGNOSTICS: {
    show_diagnostics();
  }
    break;
  case MELEE_EX_DIALOG_LICENSE:
    show_file(COPYING_FILE);
    break;
  case MELEE_EX_DIALOG_SHIPINFO:
    ship_view_dialog(0, reference_fleet);
    break;
  case MELEE_EX_DIALOG_KEYTESTER: {
    keyjamming_tester();
  }
    break;
  }
  showTitle();
  return;
}


// TEAMS - dialog function
void change_teams() {
  STACKTRACE;
  int a, i;
  
  tw_set_config_file("scp.ini");
  
  teamsDialog[TEAMS_DIALOG_PLAYERLIST].d1 = 0;
  
  while (1) 
    {
      dialog_string[0][0] = 0;
      sprintf(dialog_string[0], "Config #");
      
      
      a = tw_do_dialog(NULL, teamsDialog, 0);
      if((a == TEAMS_DIALOG_SELECTCONTROL) || (a == TEAMS_DIALOG_CONTROLLIST)) 
	{
	  player_type[teamsDialog[TEAMS_DIALOG_PLAYERLIST].d1] =
	    control_name[teamsDialog[TEAMS_DIALOG_CONTROLLIST].d1];
	  teamsDialog[TEAMS_DIALOG_PLAYERLIST].d1 += 1;
	}
      else if ((a == TEAMS_DIALOG_SETUP) || (a == TEAMS_DIALOG_PLAYERLIST)) 
	{
	  Control *tmpc = load_player(teamsDialog[TEAMS_DIALOG_PLAYERLIST].d1);
	  if (tmpc) 
	    {
	      showTitle();
	      tmpc->setup();
	      delete tmpc;
	      showTitle();
	    }
	}
      else if (a == TEAMS_DIALOG_TEAM_NUM) 
	{
	  player_team[teamsDialog[TEAMS_DIALOG_PLAYERLIST].d1] += 1;
	  player_team[teamsDialog[TEAMS_DIALOG_PLAYERLIST].d1] %= MAX_TEAMS;
	}
      else if (a == TEAMS_DIALOG_CONFIG_NUM) 
	{
	  player_config[teamsDialog[TEAMS_DIALOG_PLAYERLIST].d1] += 1;
	  player_config[teamsDialog[TEAMS_DIALOG_PLAYERLIST].d1] %= MAX_CONFIGURATIONS;
	}
      else if (a == TEAMS_DIALOG_FLEET) 
	{
	  edit_fleet(teamsDialog[TEAMS_DIALOG_PLAYERLIST].d1);
	  showTitle();
	}
      else break;
    }
  
  tw_set_config_file("scp.ini");
  for (i = 0; i < MAX_PLAYERS; i += 1) 
    {
      sprintf(dialog_string[0], "Player%d", i+1);
      set_config_string (dialog_string[0], "Type", player_type[i]);
      set_config_int (dialog_string[0], "Config", player_config[i]);
      set_config_int (dialog_string[0], "Team", player_team[i]);
    }
  
  return;
}

/*
*** TEAMS dialog section - end
*/




/*
*** FLEET dialog section - begin
*/


char *numeric_string[] = {"Zero", "One", "Two", "Three", "Four", 
			  "Five", "Six", "Seven", "Eight", "Nine", "Ten", "Eleven", 
			  "Twelve"};

char fleetPlayer[18];
char fleetTitleString[100];

int scp_fleet_dialog_text_list_proc(int msg, DIALOG* d, int c);
int scp_fleet_dialog_bitmap_proc(int msg, DIALOG* d, int c);



int d_check_proc_fleeteditor(int msg, DIALOG *d, int c)
{
  if (msg == MSG_CLICK)
    {
		
      /* track the mouse until it is released */
      while (gui_mouse_b()) {
	//			state2 = ((gui_mouse_x() >= d->x) && (gui_mouse_y() >= d->y) &&
	//				(gui_mouse_x() < d->x + d->w) && (gui_mouse_y() < d->y + d->h));
			
	/* let other objects continue to animate */
	broadcast_dialog_message(MSG_IDLE, 0);
      }
		
      /* should we close the dialog? */
      // imo the following mucho better/ simplere than that messy stuff in the allegro routine
      // ... check d_button_proc in guiproc.c in the allegro sources...

      if (d->flags & D_SELECTED)
	d->flags &= ~D_SELECTED;
      else
	d->flags |= D_SELECTED;

      if ( d->flags & D_EXIT)
	return D_CLOSE;

      return D_O_K; 
    }
	
  return d_agup_check_proc(msg, d, 0);
}



bool safeToDrawPreview = false;

// FLEET - dialog function
void edit_fleet(int player) {STACKTRACE
			       char tmp[40];
 char path[80];
 char fleetCostString[80] = "";
 char maxFleetCostString[80] = "";
 bool availableFleetDirty = true;

 static Fleet::SortingMethod sortMethod1 = (Fleet::SortingMethod) Fleet::SORTING_METHOD_DEFAULT,
   sortMethod2 = (Fleet::SortingMethod) Fleet::SORTING_METHOD_DEFAULT;
 static bool sortAscending1 = false,
   sortAscending2 = false;


 sprintf (tmp, "Player%d", player+1);
 Fleet* fleet = new Fleet();
 fleet->load("fleets.ini", tmp);

 if (player + 1 <= 12)
   sprintf(fleetPlayer, "Player %s Fleet", numeric_string[player+1]);
 else sprintf(fleetPlayer, "Player%d Fleet", player+1);
 showTitle();

 int fleetRet;
 int selectedSlot;

 fleetDialog[FLEET_DIALOG_CURRENT_POINTS_VALUE].dp = fleetCostString;
 fleetDialog[FLEET_DIALOG_POINT_LIMIT_BUTTON].dp = maxFleetCostString;
    
 //	// the reference_fleet is used in the list in a hardcoded way, so over"load" it
 //    Fleet *old_reference_fleet = reference_fleet;

 do {
   sprintf(title_str, fleet->getTitle());
   sprintf(fleetTitleString, "%s\n%d points", fleet->getTitle(), fleet->getCost());
        
   fleetDialog[FLEET_DIALOG_FLEET_SHIPS_LIST].dp3 = fleet;
   fleetDialog[FLEET_DIALOG_SORTBY_BUTTON1].dp = Fleet::getSortingMethodName(sortMethod1);
   fleetDialog[FLEET_DIALOG_SORTBY_BUTTON2].dp = Fleet::getSortingMethodName(sortMethod2);

   sprintf(fleetCostString,"%d", fleet->getCost());
   if (fleet->getCost() > fleet->getMaxCost())
     fleetDialog[FLEET_DIALOG_CURRENT_POINTS_VALUE].bg = makecol8(255,0,0);
   else
     fleetDialog[FLEET_DIALOG_CURRENT_POINTS_VALUE].bg = 0;

   sprintf(maxFleetCostString,"%d %s", fleet->getMaxCost(), 
	   Fleet::getFleetCostName(fleet->getMaxCost()));

   if (sortAscending1)
     fleetDialog[FLEET_DIALOG_SORTBY_ASCENDING1].dp = (void *)"^";
   else
     fleetDialog[FLEET_DIALOG_SORTBY_ASCENDING1].dp = (void *)"v";
        
   if (sortAscending2)
     fleetDialog[FLEET_DIALOG_SORTBY_ASCENDING2].dp = (void *)"^";
   else
     fleetDialog[FLEET_DIALOG_SORTBY_ASCENDING2].dp = (void *)"v";

   //if the user has selected a different choice of available ships, regenerate the
   //list of available ships
   if (availableFleetDirty) {
     availableFleetDirty = false;
            
     //clear out the fleet
     reference_fleet->reset();

     for (int c=0; c<num_shiptypes; c++) {
       switch (shiptypes[c].origin) { 
       case SHIP_ORIGIN_TW:
	 if (fleetDialog[FLEET_DIALOG_TW_OFFICIAL_TOGGLE].flags & D_SELECTED)
	   reference_fleet->addShipType(&shiptypes[c]);
	 break;
                
       case SHIP_ORIGIN_UQM:
	 if (fleetDialog[FLEET_DIALOG_TW_EXP_TOGGLE].flags & D_SELECTED)
	   reference_fleet->addShipType(&shiptypes[c]);
	 break;

       case SHIP_ORIGIN_TW_SPECIAL:
	 if (fleetDialog[FLEET_DIALOG_TW_SPECIAL_TOGGLE].flags & D_SELECTED)
	   reference_fleet->addShipType(&shiptypes[c]);
	 break;
       }
     }
     reference_fleet->Sort( sortMethod1, sortAscending1 );
     fleetDialog[FLEET_DIALOG_AVAILABLE_SHIPS_LIST].flags |= D_DIRTY;
   }/**/

   fleetRet = tw_do_dialog(NULL, fleetDialog, -1);

   switch( fleetRet ) {
   case FLEET_DIALOG_AVAILABLE_SHIPS_TEXT: break;
   case FLEET_DIALOG_SHIP_CATAGORIES_TEXT: break;

   case FLEET_DIALOG_TW_OFFICIAL_TOGGLE:
   case FLEET_DIALOG_TW_EXP_TOGGLE:
   case FLEET_DIALOG_TW_SPECIAL_TOGGLE:
     availableFleetDirty = true;
     break;

   case FLEET_DIALOG_SORTBY_TEXT1: break;
   case FLEET_DIALOG_SORTBY_BUTTON1: 
     sortMethod1 = Fleet::cycleSortingMethod(sortMethod1);
     reference_fleet->Sort( sortMethod1, sortAscending1 );
     fleetDialog[FLEET_DIALOG_SORTBY_BUTTON1].dp = Fleet::getSortingMethodName(sortMethod1);
     break;

   case FLEET_DIALOG_SORTBY_ASCENDING1: 
     sortAscending1 = 1 - sortAscending1;
     reference_fleet->Sort( sortMethod1, sortAscending1 );
     if (sortAscending1)
       fleetDialog[FLEET_DIALOG_SORTBY_ASCENDING1].dp = (void *)"^";
     else
       fleetDialog[FLEET_DIALOG_SORTBY_ASCENDING1].dp = (void *)"v";
     break;

   case FLEET_DIALOG_AVAILABLE_SHIPS_LIST:
   case FLEET_DIALOG_ADD_BUTTON: 
     int k;
     k = fleetDialog[FLEET_DIALOG_AVAILABLE_SHIPS_LIST].d1;
     if (k < 0 || k >= reference_fleet->getSize()) {tw_error("invalid ship choice - bug");}

     selectedSlot = fleet->addShipType(reference_fleet->getShipType(k));
     if (selectedSlot != -1)
       fleetDialog[FLEET_DIALOG_FLEET_SHIPS_LIST].d1 = selectedSlot;
               
     break;

   case FLEET_DIALOG_PLAYER_FLEET_BUTTON: break;

   case FLEET_DIALOG_PLAYER_FLEET_TITLE:               
     if(do_dialog(fleet_titleDialog, FLEET_TITLE_DIALOG_BOX) == FLEET_TITLE_DIALOG_OK)
       sprintf(fleet->getTitle(), title_str);
     showTitle();
     break;

   case FLEET_DIALOG_SAVE_BUTTON: 
     sprintf(path, "fleets/");
     if(file_select("Save Fleet", path, "scf")) fleet->save(path, "Fleet");
     showTitle();
     break;

   case FLEET_DIALOG_LOAD_BUTTON: 
     sprintf(path, "fleets/");
     if(file_select("Load Fleet", path, "scf")) fleet->load(path, "Fleet");
     sprintf(title_str, fleet->getTitle());
     sprintf(fleetTitleString, "%s\n%d points", fleet->getTitle(), fleet->getCost());
     showTitle();
     break;

   case FLEET_DIALOG_POINT_LIMIT_TEXT: break;

   case FLEET_DIALOG_POINT_LIMIT_BUTTON:
     fleet->cycleMaxFleetCost();
     break;

   case FLEET_DIALOG_CURRENT_POINTS_TEXT: break;
   case FLEET_DIALOG_CURRENT_POINTS_VALUE: break;
   case FLEET_DIALOG_SORTBY_TEXT2: break;

   case FLEET_DIALOG_SORTBY_BUTTON2: 
     sortMethod2 = Fleet::cycleSortingMethod(sortMethod2);
     fleet->Sort( sortMethod2, sortAscending2 );
     fleetDialog[FLEET_DIALOG_SORTBY_BUTTON2].dp = Fleet::getSortingMethodName(sortMethod2);
     break;

   case FLEET_DIALOG_SORTBY_ASCENDING2: 
     sortAscending2 = 1 - sortAscending2;
     fleet->Sort( sortMethod2, sortAscending2 );
     if (sortAscending2)
       fleetDialog[FLEET_DIALOG_SORTBY_ASCENDING2].dp = (void *)"^";
     else
       fleetDialog[FLEET_DIALOG_SORTBY_ASCENDING2].dp = (void *)"v";
     break;

   case FLEET_DIALOG_ADD_ALL_BUTTON: 
     fleet->addFleet(reference_fleet);
     break;

   case FLEET_DIALOG_CLEAR: 
   case FLEET_DIALOG_FLEET_SHIPS_LIST: 
     fleet->clear_slot(fleetDialog[FLEET_DIALOG_FLEET_SHIPS_LIST].d1);
     if (fleet->getSize() <= 0)
       fleetDialog[FLEET_DIALOG_FLEET_SHIPS_LIST].d1 = 0;
     break;

   case FLEET_DIALOG_CLEARALL: 
     fleet->reset();
     fleetDialog[FLEET_DIALOG_FLEET_SHIPS_LIST].d1 = 0;
     break;

   case FLEET_DIALOG_SHIP_PICTURE_BITMAP: break;
           
   case FLEET_DIALOG_SHIP_SUMMARY_TEXT: break;
   case FLEET_DIALOG_BACK_BUTTON: break;
   case FLEET_DIALOG_HELP_TEXT:/**/
   default:
     ;
   }
   /*if (fleetRet == FLEET_DIALOG_INFO) {
     ship_view_dialog(fleetDialog[FLEET_DIALOG_FLEET_SHIPS_LIST].d1, reference_fleet);
     showTitle();
     }*/

 } while((fleetRet != FLEET_DIALOG_BACK_BUTTON) && (fleetRet != -1));

 fleet->save("fleets.ini", tmp);
 delete fleet;
 showTitle();
}


int scp_fleet_dialog_text_list_proc(int msg, DIALOG* d, int c) {
  
  static int next_anim_time = get_time();
  int old_d1 = d->d1;
  
  int ret = 0;
  
  
  // allow user to select the ships based on keystrokes:
  // select based on the ship's name
  bool shouldConsumeChar = false;
  if (msg == MSG_CHAR) {
    char typed = (char)(0xff & c);
    if (isalnum (typed)) {
      d->d1 = reference_fleet->getNextFleetEntryByCharacter( d->d1, typed);
      shouldConsumeChar = true;
      if (d->d1 != old_d1) {
	
	int size = reference_fleet->getSize();
	int height = (d->h-4) / text_height(font);
	
	ret = D_USED_CHAR; 
	d->flags |= D_DIRTY;
	
	//scroll such that the selection is shown.
	//only change the scroll if the selection is not already shown,
	//and the number of ships in the list is greater than the number
	//of slots that can be shown simultaneously.
	if ( (size > height) &&
	     ( (d->d1 < d->d2) ||
	       (d->d1 >= d->d2 + height))) 
	  {
	    if (d->d1 <= (height/2))
	      d->d2 = 0;
	    else {
	      
	      if (d->d1 >= (size - height))
		d->d2 = (size - height);
	      else {
		d->d2 = d->d1 - height/2;
	      }
	    }
	  }
      }
    }
  }
  ret = d_agup_text_list_proc( msg, d, c );
  
  if (shouldConsumeChar)
    ret = D_USED_CHAR;
  
  static BITMAP* panel = create_bitmap(fleetDialog[FLEET_DIALOG_SHIP_PICTURE_BITMAP].w,
				       fleetDialog[FLEET_DIALOG_SHIP_PICTURE_BITMAP].h);
  fleetDialog[FLEET_DIALOG_SHIP_PICTURE_BITMAP].dp = panel;
  
  static BITMAP * sprite = create_bitmap(fleetDialog[FLEET_DIALOG_SHIP_PICTURE_BITMAP].w,
					 fleetDialog[FLEET_DIALOG_SHIP_PICTURE_BITMAP].h);
  static int rotationFrame = 0;
  
  //selection has changed
  if (d->d1 != old_d1) {
    safeToDrawPreview = false;
    float fractionRotated = 0;
    
    {ShipType* type = reference_fleet->getShipType(old_d1);
    
    if (type && type->data) {
      if (type->data->spriteShip) {
	
	fractionRotated = (float)((float)rotationFrame / (float)(type->data->spriteShip->frames()));
      }
      type->data->unlock();
    }}
    
    rotationFrame = 0;
    
    {ShipType* type = reference_fleet->getShipType(d->d1);
    if (type && type->data) {
      type->data->lock();
      if (type->data->spriteShip)
	rotationFrame = (int)(fractionRotated * type->data->spriteShip->frames());
    }}
  }
  
  if ( ( d->d1 != old_d1 || msg == MSG_START) || 
       (msg == MSG_IDLE && next_anim_time < get_time()) ) {
    safeToDrawPreview = false;
    
    //next_anim_time = get_time() + 50 + rand() % 200;
    next_anim_time = get_time() + 20;
    
    ShipType* type = reference_fleet->getShipType(d->d1);
    
    clear_to_color(sprite, 0);
    
    if (type && type->data && type->data->spriteShip) {
      
      type->data->spriteShip->draw(
				   Vector2(fleetDialog[FLEET_DIALOG_SHIP_PICTURE_BITMAP].w/2,
					   fleetDialog[FLEET_DIALOG_SHIP_PICTURE_BITMAP].h/2) - type->data->spriteShip->size()/2, 
				   type->data->spriteShip->size(), 
				   rotationFrame, sprite 
				   );
      
      rotationFrame++;
      if (rotationFrame >= type->data->spriteShip->frames())
	rotationFrame = 0;
    }
    stretch_blit(sprite, panel, 0, 0, sprite->w, sprite->h, 0, 0, panel->w, panel->h);
    safeToDrawPreview = true;
    
    
    //TODO decide if these next 3 lines should be here
    scare_mouse();
    SEND_MESSAGE(&fleetDialog[FLEET_DIALOG_SHIP_PICTURE_BITMAP], MSG_DRAW, 0);
    unscare_mouse();
  }
  
  return ret;
}

int scp_fleet_dialog_bitmap_proc(int msg, DIALOG* d, int c) {
  //TODO address this: bitmap has to be deleted, but MSG_END does not mean the dialog isn't coming back
  /*if (msg == MSG_END && d->dp) {
    destroy_bitmap( (BITMAP*)d->dp );
    d->dp = NULL;
    }*/

  if ((msg != MSG_DRAW || d->dp) && (safeToDrawPreview) )
    return d_bitmap_proc(msg, d, c);
  return D_O_K;
}

/*
*** FLEET dialog section - end
*/



/*! This is the maximum size of text file the program is willing to read.  This is for security concerns (buffer overflows) */
enum { MAX_SHIP_TEXT_FILE_SIZE=3000 };

/*! This is the maximum size of ini file the program is willing to read.  This is for security concerns (buffer overflows) */
enum { MAX_SHIP_INI_FILE_SIZE=3000 };


/*
*** SHIPVIEW dialog section - begin
*/
void ship_view_dialog(int si, Fleet *fleet) 
{
  int i;
  int r = 0;               // result of tw_do_dialog
  const char *sname;
  char textFile[MAX_SHIP_TEXT_FILE_SIZE] = ""; // ship description contents
  BITMAP *sprite = NULL;

  showLoadingScreen();

  ASSERT(fleet);

  shipviewDialog[SHIPVIEW_DIALOG_LIST].d1 = si;
  shipviewDialog[SHIPVIEW_DIALOG_LIST].dp3 = fleet;
  shipviewDialog[SHIPVIEW_DIALOG_TXTFILE].dp = (char*)"";
  shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp = (char*)"";//this is set later 
  
  // main dialog loop - begin
  while ((r >= 0) && (r != SHIPVIEW_DIALOG_DONE)) 
    {
    
      // update ship selection - begin
      if ((r == 0) || (r == SHIPVIEW_DIALOG_LIST)) 
	{
	  si = shipviewDialog[SHIPVIEW_DIALOG_LIST].d1;
	  sname = shipListboxGetter(si, NULL);
	  if (!sname) 
	    {
	      sprintf(dialog_string[0], "(Null)");
	      sprintf(dialog_string[1], "(Null)");
	    }
	  else 
	    {
	      ShipType *type = fleet->getShipType(si);

	      if (sprite)
		destroy_bitmap(sprite);
	      sprite = NULL;
	      type->data->lock();
	      if (type->data->spriteShip) 
		{
		  sprite = create_bitmap(180, 180);
		  clear_to_color(sprite, 0);
		  type->data->spriteShip->draw( 
					       Vector2(90,90) - type->data->spriteShip->size()/2, 
					       type->data->spriteShip->size(), 
					       0, sprite 
					       );
		}
	      type->data->unlock();
	      
	      // read ship text file contents
	      shipviewDialog[SHIPVIEW_DIALOG_TXTFILE].dp = (char*)textFile;
	      {
		PACKFILE * f = pack_fopen(type->text, F_READ);
		if (!f) {
		  sprintf(textFile, "Failed to load file \"%s\"", type->text);
		}
		else {
		  unsigned long textFileSize = file_size(type->text);
		  if (textFileSize > MAX_SHIP_TEXT_FILE_SIZE)
		    textFileSize = MAX_SHIP_TEXT_FILE_SIZE;
		  i = pack_fread(textFile, textFileSize, f);
		  pack_fclose(f);
		  textFile[i] = '\0';
		}
	      }
	      
	      // read ship ini file contents
	      char * inifile = "";     // ship ini file contents
	      {
		PACKFILE * f = pack_fopen(type->file, F_READ);
		if (!f) {
		  inifile = (char*) malloc(strlen("Failed to load file \"\"") + strlen(type->file) + 1);
		  sprintf(inifile, "Failed to load file \"%s\"", type->file);
		}
		else {
		  unsigned long iniFileSize = file_size(type->file);
		  inifile = (char*) malloc(iniFileSize +1 );
		  i = pack_fread(inifile, iniFileSize, f);
		  pack_fclose(f);
		  inifile[i] = '\0';
		}
	      }
	      
	      // display ship description contents
	      char *c = (char*)malloc( strlen("Name: \n") + strlen(type->name) +
				       strlen("ID: \n")+strlen(type->id) +
				       strlen("Cost: \n")+3+
				       strlen("\n\n\n\nINI file: ()\n")+strlen(type->file) +
				       strlen("-------------------------\n")+strlen(inifile) );
	      char * description = c;
	      
	      c += sprintf(c, "Name: %s\n", type->name);
	      c += sprintf(c, "ID: %s\n", type->id);
	      c += sprintf(c, "Cost: %d\n", type->cost);
	      c += sprintf(c, "\n\n\n\nINI file: (%s)\n", type->file);
	      c += sprintf(c, "-------------------------\n%s", inifile);
				
	      if (shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp &&
		  strlen((char*)(shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp))>0 )
		{
		  free(shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp);
		  shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp = (char*)"";
		}
	      shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp = description;
				
	      if (strlen(inifile) >0)
		free(inifile);
	    }
	  shipviewDialog[SHIPVIEW_DIALOG_PICTURES+0].dp = sprite;
	}

      // change font size - begin
      if (r == SHIPVIEW_DIALOG_FONT) {
	i = shipviewDialog[SHIPVIEW_DIALOG_TWYIELD+1].d1;
	i = (((i/2) + 2) % 3) - 1;
	shipviewDialog[SHIPVIEW_DIALOG_TWYIELD+1].d1 = i*2;
      }
      // change font size - end

      r = tw_do_dialog(NULL, shipviewDialog, SHIPVIEW_DIALOG_LIST);
    }
  // main dialog loop - end

  if (sprite)
    destroy_bitmap(sprite);
  if (shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp &&
      strlen((char*)(shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp))>0 )
    {
      shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp = NULL;
      free(shipviewDialog[SHIPVIEW_DIALOG_DESCRIPTION].dp);
    }
	
  videosystem->redraw();
  return;
}

/*
 * DIAGNOSTICS dialog section - begin
 */


int get_diagnostics_string ( char *dest ) {//returns length of string
  char * tmp = dest;

#	if defined _DEBUG
  tmp += sprintf(tmp, "DEBUGGING BUILD!\n");
#	endif
  tmp += sprintf(tmp, "ALLEGRO (.h) version   = Allegro %s, %s\n", 
		 ALLEGRO_VERSION_STR, ALLEGRO_PLATFORM_STR);
  tmp += sprintf(tmp, "ALLEGRO (.dll) version = %s\n", allegro_id);
  tmp += sprintf(tmp, "Compiler = ");
#	if defined __MINGW32__
  tmp += sprintf(tmp, "MINGW (gcc)\n");
#	elif defined __BORLANDC__
  tmp += sprintf(tmp, "Borland\n");
#	elif defined _MSC_VER
  tmp += sprintf(tmp, "Microsoft Visual C++\n");
#	elif defined DJGPP
  tmp += sprintf(tmp, "DJGPP (gcc)\n");
#	elif defined __GNUC__
  tmp += sprintf(tmp, "gcc\n");
#	else
  tmp += sprintf(tmp, "???\n");
#	endif
  tmp += sprintf(tmp, "Version = %s\n", tw_version().c_str());
  return tmp - dest;
}

// DIAGNOSTICS - dialog function
void show_diagnostics() 
{
  int i;
  char buffy [16000];//fix sometime
  char buffy2[100000];//fix sometime
  char buffy3[16000];//yeah right
  char *tmp;
  PACKFILE *f;
  
  f = pack_fopen (data_full_path("version.txt").c_str(), F_READ);
  if (!f)
    strcpy(buffy, "Failed to load version.txt");
  else {
    i = pack_fread (buffy, 99999, f);
    pack_fclose(f);
    buffy[i] = 0;
  }
  
  diagnostics_dialog[DIAGNOSTICS_DIALOG_VERSION_TXT].dp = (void *) buffy;
  diagnostics_dialog[DIAGNOSTICS_DIALOG_FILES].dp = (void *) buffy2;
  tmp = buffy2;
  
  diagnostics_dialog[DIAGNOSTICS_DIALOG_MAIN].dp = (void *) buffy3;
  tmp = buffy3;
  tmp += get_diagnostics_string( tmp );
    
  tw_popup_dialog(NULL, diagnostics_dialog, 1);
  return;
}


/*
 * DIAGNOSTICS dialog section - end
 */


void keyjamming_tester() 
{
  int i, j = 0;
  char blah[256];
  
  scare_mouse();
  videosystem->window.lock();
  clear_to_color(videosystem->window.surface, 0);
  textprintf(screen, font, 40, 20, palette_color[15], "Press the keys combinations you wish to test");
  textprintf(screen, font, 40, 40, palette_color[15], "When you're finished, press ESCAPE or F10");
  videosystem->window.unlock();
  unscare_mouse();
  
  while (!key[KEY_F10] && !key[KEY_ESC]) 
    {
      if (videosystem->poll_redraw()) 
	{
	  scare_mouse();
	  videosystem->window.lock();
	  clear_to_color(videosystem->window.surface, 0);
	  textprintf(screen, font, 40, 20, palette_color[15], "Press the keys combinations you wish to test");
	  textprintf(screen, font, 40, 40, palette_color[15], "When you're finished, press ESCAPE or F10");
	  videosystem->window.unlock();
	  unscare_mouse();
	}
      rectfill(screen, 50, 60, 500, 60 + 20 * j, palette_color[0]);
      j = 0;
      poll_input();
      for (i = 0; (i < 32767) && (j < 16); i += 1) {
	if (key_pressed(i)) 
	  {
	    key_to_description(i, blah);
	    scare_mouse();
	    acquire_screen();
	    textprintf(screen, font, 50, 60+j*20, palette_color[15], "%s", blah);
	    release_screen();
	    unscare_mouse();
	    j += 1;
	  }
      }
      idle(20);
    }
  showTitle();
  while (key[KEY_F10])
    poll_keyboard();
  clear_keybuf();
  return;
}


char *playerListboxGetter(int index, int *list_size) {
  static char buf[160];
  char *tmp = buf;

  tmp[0] = 0;
  if(index < 0) {
    *list_size = MAX_PLAYERS;
    return NULL;
  } else {
    tmp += sprintf(tmp, "Player%d", index + 1);
    if (index + 1 < 10) tmp += sprintf(tmp, " ");
    tmp += sprintf(tmp, "   %d   %d   %s", player_team[index], player_config[index], player_type[index]);
    if ((strlen(buf) >= 80)) tw_error("playerListboxGetter string too long");
    return buf;
  }
}

char *controlListboxGetter(int index, int *list_size) {
  static char tmp[40];

  tmp[0] = 0;
  if(index < 0) {
    *list_size = num_controls;
    return NULL;
  } else {
    return(control_name[index]);
  }
}

/*
  Path Configuration
*/

std::string datafile_path;
void init_datafile_path()
{
  datafile_path = DATAFILE_PATH;
  if(exists(data_full_path("gamedata.dat").c_str()))
    return;
  datafile_path = "gamedata/";
  if(exists(data_full_path("gamedata.dat").c_str()))
    return;
  tw_error("Unable to locate game data!!!");
}

std::string home_ini_full_path(std::string path)
{
  char * home = getenv("HOME");
  char dest[2040] = {0};
  if(home == NULL)
    {
      if(strstr(path.c_str(), datafile_path.c_str()))
	return path;
      std::string pth = append_filename(dest, datafile_path.c_str(), "default_ini", 2039);
      append_filename(dest, pth.c_str(), path.c_str(), 2039);
      return std::string(dest);
    }
  else
    {
      if(strstr(path.c_str(), home))
	return path;
      return std::string(home) + std::string("/.tw-light/") + path;
    }
}


/// \brief create full data path
std::string data_full_path(std::string path)
{
  char data[2040] = {0};
  std::string ret = append_filename(data, datafile_path.c_str(), path.c_str(), 2039);
  return ret;
}

bool CopyFile(const char * source, const char * target)
{
  FILE * fsrc = fopen(source, "r");
  if(!fsrc)
    return false;

  FILE * ftrg = fopen(target, "w");
  if(!ftrg)
    {
      tw_error("ACK!!! copy file failed!!!");
      fclose(fsrc);
      return false;
    }
  
  unsigned char buffer[1024];
  int readed = 1;
  while(readed!=0)
    {
      readed = fread(buffer, 1, 1024, fsrc);
      int written = fwrite(buffer, 1, readed,ftrg);
      if(readed != written)
	tw_error("ACK!!! copy file failed");
    }
  fclose(fsrc);
  fclose(ftrg);
  return true;
}

void _copy_ship_ini(const char *filename, int attrib, int param)
{
  const char * ship_name = get_filename(filename);
  std::string home_file = home_ini_full_path(std::string("ships/") + ship_name);
  if(!exists(home_file.c_str()))
    CopyFile(filename, home_file.c_str());
  
  tw_set_config_file(home_file);
  int version = get_config_int("Info", "Version", -1);
  if(version<64)
    {
      CopyFile(filename, home_file.c_str());
      tw_set_config_file(home_file);
      std::string ver = GetSVNVersion();
      set_config_string("Info", "Version", ver.c_str());
    }
}

#ifdef WIN32
#define mkdir(x,y) mkdir(x)
#endif

int create_user_ini()
{
  std::string curFile = home_ini_full_path("");
  if(!file_exists(curFile.c_str(), 0, NULL))
    mkdir(curFile.c_str(), 0777);
  
  curFile = home_ini_full_path("client.ini"); 
  if(!exists(curFile.c_str()))
    CopyFile(data_full_path("default_ini/client.ini").c_str(),
	     curFile.c_str());
  
  curFile = home_ini_full_path("fleets.ini"); 
  if(!exists(curFile.c_str()))
    CopyFile(data_full_path("default_ini/fleets.ini").c_str(),
	     curFile.c_str());
  
  curFile = home_ini_full_path("scp.ini"); 
  if(!exists(curFile.c_str()))
    CopyFile(data_full_path("default_ini/scp.ini").c_str(),
	     curFile.c_str());
  
  curFile = home_ini_full_path("server.ini"); 
  if(!exists(curFile.c_str()))
    CopyFile(data_full_path("default_ini/server.ini").c_str(),
	     curFile.c_str());

  curFile = home_ini_full_path("vobject.ini"); 
  if(!exists(curFile.c_str()))
    CopyFile(data_full_path("default_ini/vobject.ini").c_str(),
	     curFile.c_str());
  
  curFile = home_ini_full_path("ships");
  if(!file_exists(curFile.c_str(), 0, NULL))
    mkdir(curFile.c_str(), 0777);
  
  for_each_file( (data_full_path("")+std::string("default_ini/ships/shp*.ini")).c_str(), 
		 FA_ARCH|FA_RDONLY, 
		 _copy_ship_ini, 0);
  return 0;
}


//################################################################################################
//  Intro





void GOBIntro()
{
  BITMAP* d_buf;
  std::vector<std::string> intro_text;

  intro_text.push_back("The Ur-quan are defeated - the destruction");
  intro_text.push_back("of the Sa-matra threw them into chaos,");
  intro_text.push_back("making them easy prey for");
  intro_text.push_back("the New Alliance of Free Stars.");
  intro_text.push_back("Within weeks the NAFS forces managed");
  intro_text.push_back("to destroy most of the fleets of the two ");
  intro_text.push_back("races who terrorized the galaxy");
  intro_text.push_back("for the past 20,000 years!");
  intro_text.push_back("");
  intro_text.push_back("Thanks to captain Zelnick, hope is given");
  intro_text.push_back("to the entire galaxy, and especially to");
  intro_text.push_back("the human race. Now free from their prison");
  intro_text.push_back("under a Slaveshield it's Humanity's time");
  intro_text.push_back("to rebuild.");
  intro_text.push_back("");
  intro_text.push_back("However, it seems there are more troubles");
  intro_text.push_back("ahead. It is time for great space");
  intro_text.push_back("exploration, colonization and piracy.");
  intro_text.push_back("You have to live in this world,");
  intro_text.push_back("young earth captain!!!");

  d_buf = create_bitmap(SCREEN_W, SCREEN_H);
  clear(d_buf);
  copy_buf();

  tw_sound->play_music(data_full_path("music/intro.xm"), -1);
  clear(d_buf);
  BITMAP* logo = load_bitmap(data_full_path("images/3dlogo.jpg").c_str(), NULL);
  stretch_blit(logo, d_buf, 0, 0, logo->w, logo->h, 0, 0, screen->w, screen->h);
  destroy_bitmap(logo);

  textout_centre(d_buf, D_FONT_ENGLISH, "TimeWarp development team", screen->w/2, screen->h-50, -1);
  copy_buf();
  HAVE_A_WAIT(2500);
 
  clear(d_buf);
  textout_centre(d_buf, D_FONT_ENGLISH, "somewhat proudly", screen->w/2, screen->h/2, -1);
  textout_centre(d_buf, D_FONT_ENGLISH, "present", screen->w/2, screen->h/2+40, -1);
  copy_buf();
  HAVE_A_WAIT(2500);
  clear(d_buf);

  BITMAP* pkunk = load_bitmap(data_full_path("images/joses_pkunk.jpg").c_str(), NULL);
  blit(pkunk, d_buf, 0, 0,  (screen->w-pkunk->w)/2, 20, pkunk->w, pkunk->h);
  textout_centre(d_buf, D_FONT_ENGLISH, "TW-Light : Legacies", screen->w/2, pkunk->h+60, -1);
  destroy_bitmap(pkunk);
  copy_buf();
  HAVE_A_WAIT(2500);
  
  ScrollText(intro_text,  D_FONT_ENGLISH, -1, "images/background.jpg", "c", 2);


  clear(d_buf);
  copy_buf();	
  destroy_bitmap(d_buf);
}



