/*
This file is part of "TW-Light" 
                    http://tw-light.berlios.de/
Copyright (C) 2001-2004  TimeWarp development team

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <allegro.h>

#include "melee.h"

#include "scp.h"
#include "frame.h"

#include "melee/mgame.h"
#include "melee/mmain.h"
#include "melee/mview.h"
#include "melee/mcontrol.h"
#include "melee/mcbodies.h"
#include "melee/mshppan.h"
#include "melee/mship.h"
#include "melee/mshot.h"
#include "melee/mlog.h"
#include "melee/manim.h"
#include "melee/mfleet.h"

#include "util/aastr.h"

#include "ggob.h"
#include "sc1ships.h"
#include "sc2ships.h"

#include "other/gup.h"
#include "other/dialogs.h"
#include "other/twconfig.h"

#include "python/python_class.h"

#define gobgame ((GobGame*)game)

///////////////////////////////////////////////////////////////////////
// Save Game stuff
///////////////////////////////////////////////////////////////////////
SaveGobGameInfo* SaveGobGameInfo::getInstance()
{
  static SaveGobGameInfo svg;
  return &svg;
}


void SaveGobGameInfo::updateGame()
{
  gobgame->gobplayer.starbucks    = starbucks;
  gobgame->gobplayer.buckazoids   = buckazoids;
  gobgame->gobplayer.kills        = kills;
  gobgame->gobplayer.ship->pos.x = pos_x;
  gobgame->gobplayer.ship->pos.y = pos_y;
}

void SaveGobGameInfo::updateSave()
{
  starbucks   = gobgame->gobplayer.starbucks;
  buckazoids  = gobgame->gobplayer.buckazoids;
  kills       = gobgame->gobplayer.kills;
  pos_x       = gobgame->gobplayer.ship->pos.x;
  pos_y       = gobgame->gobplayer.ship->pos.y;

  EnemySaveData abc;
  abc.pos_x = 1;
  abc.pos_y = 2;
  tst.push_back(abc);
}

////////////////////////////////////////////////////////////////////////
//				Gob stuff
////////////////////////////////////////////////////////////////////////

int GobAsteroid::handle_damage (SpaceLocation *source, double normal, double direct) 
{
  STACKTRACE;
  if (!exists()) 
    return 0;
  int i = Asteroid::handle_damage(source, normal, direct);
  if (!exists()) 
    {
      GobPlayer *p = gobgame->get_player(source);
      if (p) 
	p->buckazoids += 1;
    }
  return i;
}

void GobAsteroid::death () 
{
  STACKTRACE;
  Animation *a = new Animation(this, pos,
			       explosion, 0, explosion->frames(), time_ratio, get_depth());
  a->match_velocity(this);
  game->add(a);

  game->add ( new GobAsteroid() );
  return;
}

void GobGame::preinit() 
{
  STACKTRACE;
  GobGameBase::preinit();

  gobenemies = 0;
  max_enemies = 0;
  gobenemy = NULL;

  defenderSprite = NULL;
  _player_panel = NULL;
}

void GobGame::prepare()
{
  STACKTRACE;
  GobGameBase::prepare();
}


void GobGame::setGameDone(int done)
{
  STACKTRACE;

    return GobGameBase::setGameDone(done);
}


struct findStar : public std::binary_function<StarSystem*, std::string, bool>
{
  bool operator ()(StarSystem* s, std::string name) const
  {
    if (s->getName() == name)
      return true;
    return false;
  }
};

void GobGame::switch_system(const std::string& name)
{
  STACKTRACE;
  if(name == _name)
    return;

  // remove ship from current system
  item.remove(gobplayer.ship);
  
  std::list<StarSystem*>::iterator sNew = std::find_if(_galaxy._stars.begin(), 
						 _galaxy._stars.end(), 
						 std::bind2nd(findStar(), name));
  if(sNew==_galaxy._stars.end())
    {
      tw_error("Star not found");
    }

  std::list<StarSystem*>::iterator sOld = std::find_if(_galaxy._stars.begin(), 
						 _galaxy._stars.end(), 
						 std::bind2nd(findStar(), _name));
  if(sOld==_galaxy._stars.end())
    {
      tw_error("Star not found");
    }

  // Save current star items  
  (*sOld)->_item = item;

  // Load new items
  item = (*sNew)->_item;
  
  // set new name
  _name = name;
  _galaxy._playerStar = *sNew;

  // add player ship
  item.remove(gobplayer.ship);
  if(gobplayer.ship)
    item.push_back(gobplayer.ship);
}

void GobGame::add_gobplayer(Control *control) 
{
  STACKTRACE;
  gobplayer.init( new_team());
  add_focus(_player_control, _player_control->channel);
  return;
}

void GobGame::AddPanel()
{
  STACKTRACE;
  if (_player_panel) 
    _player_panel->die();
  _player_panel = NULL;
  _player_panel = new ShipPanel(gobplayer.ship);
  _player_panel->always_redraw = true;
  _player_panel->window->init(window);
  
  _player_panel->window->locate(
			0,0.9,
			0,0,
			0,0.1,
			0,0.25
			);
  _player_panel->set_depth(10);
  add(_player_panel);
}

void GobPlayer::died(SpaceLocation *killer) 
{
  STACKTRACE;
  if (upgrade_list[UpgradeIndex::divinefavor]->num && (random()&1)) 
    { //divine favor
      ship->crew = ship->crew_max;
      ship->batt = ship->batt_max;
      ship->translate(random(Vector2(-2048,-2048), Vector2(2048,2048)));
      ship->state = 1;
    }
  else 
    {
      ship = NULL;
      game->setGameDone(GAME_STATE_FINISHED);
      tw_alert("Game Over!\n\nYour ship and its contents are lost!", "Ok");
    }
  return;
}


void GobGame::play_sound (SAMPLE *sample, SpaceLocation *source, int vol, int freq) 
{
  return;
  /*
	double v;
	Vector2 d = source->normal_pos() - space_center;
	d = normalize(d + size/2, size) - size/2;
	v = 1000;
	if (space_zoom > 0.01) v = 500 + space_view_size.x / space_zoom / 4;
	v = 1 + magnitude_sqr(d) / (v*v);
	Game::play_sound(sample, source, iround(vol/v), freq);
  */
}

SpaceSprite* GobGame::LoadStationSprite(std::string datafile, std::string name)
{
  STACKTRACE;
  DATAFILE *tmpdata; 
  tmpdata = load_datafile_object(datafile.c_str(), name.c_str());
  if (!tmpdata) 
    {
      tw_error( (std::string("couldn't find ") + datafile +  " " + name).c_str() );
    }
  SpaceSprite* ss = new SpaceSprite(tmpdata, 1, SpaceSprite::MASKED | SpaceSprite::MIPMAPED, 64);
  unload_datafile_object(tmpdata);
  ss->permanent_phase_shift(8);  
  return ss;
}

void GobGame::init(Log *_log, const std::string& save) 
{
  STACKTRACE;
  GobGameBase::init(_log, save);
  
  size = Vector2(24000, 24000);
  log_file("server.ini");
  max_enemies = get_config_int("Gob", "MaxEnemies", 32);
  gobenemy = (GobEnemy**) malloc(sizeof(GobEnemy*) * max_enemies);

  enemy_team = new_team();
	       

  int server_players;
  tw_set_config_file("client.ini");
  server_players = 1;
  log_int(channel_server, server_players);
  _player_control = create_control(channel_server, "Human", "Config0");

  StarSystem* s = new StarSystem("FakeSystem_123", Vector2(0,0));
  _name = "FakeSystem_123";
  _galaxy.add(s);

  if(!python::exec_file("python/start.py"))
    {
      tw_error("Load statrt script error");
    }
  _galaxy._stars.remove(s);
  delete s;
  
  add ( new RainbowRift() );
  
  next_add_new_enemy_time = 1000;

  this->change_view("Hero");
  view->window->locate(
		       0,0,
		       0,0,
		       0,0.9,
		       0,1
		       );
  return;
}

GobGame::GobGame()
{
  STACKTRACE;
  DATAFILE *tmpdata; 
  tmpdata = load_datafile_object(data_full_path("gob.dat").c_str(), "defender");
  if (!tmpdata) 
    tw_error ("couldn't find gob.dat#defender");
  defenderSprite = new SpaceSprite(tmpdata, 1, SpaceSprite::MASKED | SpaceSprite::MIPMAPED);
  unload_datafile_object(tmpdata);  

  num_planets = 0;
}

GobGame::~GobGame() {
  delete defenderSprite;
  int i;
  for (i = 0; i < gobenemies; i += 1) 
    {
      delete gobenemy[i];
    }
  free(gobenemy);
  return;
}

void GobGame::add_planet_and_station ( int planet_index, 
				       std::string datafile,
				       std::string stationSprite,
				       std::string station_pic_name,
				       std::string station_build_name
				       )
{
  Planet *p = new Planet (size/2, meleedata.planetSprite, planet_index);
  if (num_planets) while (true) 
    {
      SpaceLocation *n;
      n = p->nearest_planet();
      if (!n || (p->distance(n) > 1500)) 
	break;
      p->translate(random(size));
    }
  add ( p );
  
  SpaceSprite *station_sprite = LoadStationSprite(data_full_path(datafile), stationSprite);
  GobStation *gs = new GobStation(station_sprite, p, station_build_name.c_str(), station_pic_name.c_str());
  gs->collide_flag_sameship = ALL_LAYERS;
  gs->collide_flag_sameteam = ALL_LAYERS;
  gs->collide_flag_anyone = ALL_LAYERS;
  add ( gs );
  
  gobgame->planet[gobgame->num_planets] = p;
  gobgame->station[gobgame->num_planets] = gs;
  gobgame->num_planets += 1;
}

void GobGame::fps() 
{
  STACKTRACE;
  GobGameBase::fps();
  
  message.print((int)msecs_per_fps, 15, "System: %s", _name.c_str());
  message.print((int)msecs_per_fps, 15, "enemies: %d", (int)gobenemies);
  message.print((int)msecs_per_fps, 15, "time: %d", (int)(game_time / 1000));

  int i = 0;
      
  if (gobplayer.ship) 
    {
      message.print((int)msecs_per_fps, 15-i, "coordinates: %d x %d", 
		    iround(gobplayer.ship->normal_pos().x), 
		    iround(gobplayer.ship->normal_pos().y));
    }
  message.print((int)msecs_per_fps, 15-i, "starbucks: %d", gobplayer.starbucks);
  message.print((int)msecs_per_fps, 15-i, "buckazoids: %d",gobplayer.buckazoids);
  message.print((int)msecs_per_fps, 15-i, "kills: %d", gobplayer.kills);
  
  return;
}

void GobGame::calculate() 
{
  STACKTRACE;
  
  if (next_add_new_enemy_time <= game_time) 
    {
      next_add_new_enemy_time = game_time;
      int t = 28;
      if ((random() % t) < 4) add_new_enemy();
      int e = gobenemies;
      e -= random() % (1 + game_time / (250 * 1000));
      if (0) ;
      else if (e >=12) 
	next_add_new_enemy_time += 15000;
      else if (e >= 7) 
	next_add_new_enemy_time += 7000;
      else if (e >= 4) 
	next_add_new_enemy_time += 5000;
      else if (e >= 2) 
	next_add_new_enemy_time += 3000;
      else if (e >= 1) 
	next_add_new_enemy_time += 2000;
      else 
	next_add_new_enemy_time += 1000;
    }
  Game::calculate();
  return;
}

int GobGame::get_enemy_index(SpaceLocation *what) 
{
  STACKTRACE;
  int i;
  Ship *s = what->ship;
  if (!s) return -1;
  for (i = 0; i < gobenemies; i += 1) 
    {
      if (gobenemy[i]->ship == s) return i;
    }
  return -1;
}

void GobGame::ship_died(Ship *who, SpaceLocation *source) 
{
  STACKTRACE;
  
  GobPlayer *p = this->get_player(who);
  if (p && (p->ship == who)) 
    { //Player died
      p->died(source);
    }
  int i = get_enemy_index(who);
  if ((i != -1) && (gobenemy[i]->ship == who)) 
    {
      GobEnemy *e = gobenemy[i];
      e->died(source);
      gobenemies -= 1;
      GobEnemy *tmp = gobenemy[gobenemies];
      gobenemy[i] = tmp;
      p = get_player(source);
    }

  GobGameBase::ship_died(who, source);
  return;
}

GobPlayer *GobGame::get_player(SpaceLocation *what) 
{
  STACKTRACE;
  if (what->get_team() == gobplayer.team) 
    return &(gobplayer);
  return NULL;
}

bool GobGame::handle_key(int k)
{
  STACKTRACE;
  switch (k >> 8) 
    {
    case KEY_J: 
      {
	if(_name=="Arix")
	  {
	    switch_system("Beta");
	  }
	else
	  {
	    switch_system("Arix");
	  }
	return true;
      }
      break;
    case KEY_M:
      {
	switch_system(_galaxy.select());
	videosystem->window.redraw();
	return true;
      }
      break;
    default: 
      {
	return GobGameBase::handle_key(k);
      }
      break;
    }
  return false;
}

void GobGame::add_new_enemy() 
{
  STACKTRACE;
  
  static char *enemy_types[] = 
    {
      "thrto", "zfpst", "shosc", "dragr", 
      "kahbo", "ilwsp", 
      "syrpe", "kzedr", "mmrxf", 
      "druma", "earcr", 
      "yehte", "chmav" 
    };
  const int num_enemy_types = sizeof(enemy_types)/sizeof(enemy_types[0]);

  if (gobenemies == max_enemies) 
    return;
  GobEnemy *ge = new GobEnemy();

  int base = game_time / 30 / 1000;
  if (gobenemies >= 4) 
    base += (gobenemies*gobenemies - 10) / 5;
  
  gobenemy[gobenemies] = ge;
  gobenemies += 1;
  base = iround(base / 1.5);
  int e = 99999;
  while (e >= num_enemy_types) {
/*
base	time	low		high

  1		.5		-0.1	3.7
  10	5		2.62	7.47
  50	25		5.89	14.24
 100	50		8.1		17.3
 200	100		11.01	26.49



*/
    e = base;
    e = random() % (e + 2);
    e = random() % (e + 3);
    if (e < pow(2.5*base,0.4) - 1) 
      e = random() % num_enemy_types;
    if (e > sqrt(3*base) + 2) 
      e = random() % (e + 1);
    //if (e > num_enemy_types * 2) e = e % num_enemy_types;
    e = e;
  }
  Ship *ship = create_ship(channel_server, enemy_types[e], "WussieBot", 
			   random(size), random(PI2), enemy_team);

  int sb, bz;
  sb = 1 + e / 4;
  if (sb > 2) 
    sb -= 1;
  bz = (e - 9) / 2;
  if (bz > 1) 
    bz -= 1;
  if (sb < 0) 
    sb = 0;
  if (bz < 0) 
    bz = 0;
  ge->init(ship, sb, bz);
  add(ship->get_ship_phaser());
  
  return;
}

void GobGame::add_stars()
{
  STACKTRACE;
  add(new Stars());
}
void GobGame::add_asteroid()
{
  STACKTRACE;
  add(new GobAsteroid());
}
void GobGame::add_system(std::string name, int x, int y)
{
  STACKTRACE;
  _galaxy.add(new StarSystem(name,Vector2(x,y)));
  gobgame->switch_system(name);
}
void GobGame::add_player(std::string system, int x, int y)
{
  STACKTRACE;
  add_gobplayer(system, Vector2(x,y));
}

void GobEnemy::init(Ship *ship, int kill_starbucks, int kill_buckazoids) 
{
  STACKTRACE;
  this->ship = ship;
  this->starbucks = kill_starbucks;
  this->buckazoids = kill_buckazoids;
  return;
}

void GobEnemy::died(SpaceLocation *what) 
{
  STACKTRACE;
  GobPlayer *p = gobgame->get_player(what);
  if (p) 
    {
      p->starbucks += starbucks;
      p->buckazoids += buckazoids;
      p->kills += 1;
    }
  return;
}

GobPlayer::GobPlayer()
{
  STACKTRACE;
  ship = NULL;
}

GobPlayer::~GobPlayer() 
{
  free (pair_list);
}

void GobPlayer::init(TeamCode team) 
{
  STACKTRACE;
  //  channel = c->channel;
  starbucks = 0;
  buckazoids = 0;
  kills = 0;
  value_starbucks = 0;
  value_buckazoids = 0;
  num_pairs = 0;
  pair_list = NULL;
  ship = NULL;
  total = 0;
  
  this->team = team;
  int i, j;
  for (i = 0; ::upgrade_list[i]; i += 1) ::upgrade_list[i]->index = i;
  upgrade_list = new Upgrade*[i+1];
  upgrade_list[i] = NULL;
  for (j = 0; j < i; j += 1) 
    {
      upgrade_list[j] = ::upgrade_list[j]->duplicate();
      upgrade_list[j]->clear(NULL, NULL, this);
    }
  return;
}

pair *GobPlayer::_get_pair(const char *id) 
{
  STACKTRACE;
  if (!pair_list) 
    return NULL;
  int i;
  for (i = 0; i < num_pairs; i += 1) 
    {
      if (!strcmp(pair_list[i].id, id)) 
	return &pair_list[i];
    }
  return NULL;
}

void GobPlayer::_add_pair(const char *id, int value) 
{
  STACKTRACE;
  if (_get_pair(id)) 
    {
      tw_error("GobPlayer::_add_pair - \"%s\" already exists", id);
      return;
    }
  pair_list = (pair*)realloc(pair_list, sizeof(pair) * (num_pairs+1));
  pair_list[num_pairs].id = strdup(id);
  pair_list[num_pairs].value = value;
  num_pairs += 1;
  return;
}

int GobPlayer::read_pair(const char *id) 
{
  STACKTRACE;
  pair *p = _get_pair(id);
  if (p) 
    return p->value;
  return -1;
}

void GobPlayer::write_pair(const char *id, int value) 
{
  STACKTRACE;
  pair *p = _get_pair(id);
  if (p) 
    p->value = value;
  else 
    _add_pair(id, value);
  return;
}

int GobPlayer::charge (char *name, int price_starbucks, int price_buckazoids) 
{
  char buffy1[512];
  sprintf(buffy1, "Price: %d starbucks plus %d buckazoids", price_starbucks, price_buckazoids);
  if ((starbucks < price_starbucks) || (buckazoids < price_buckazoids)) 
    {
      if (game->is_local(gobgame->_player_control->channel)) 
	alert("You don't have enough.", name, buffy1, "Cancel", NULL, 0, 0);
      return 0;
    }
  int r = 0;
  if (game->is_local(gobgame->_player_control->channel)) 
    r = alert ("Do you wish to make this purchase?", name, buffy1, "&No", "&Yes", 'n', 'y');
  game->log_int(gobgame->_player_control->channel, r);
  if (r == 2) 
    {
      starbucks -= price_starbucks;
      buckazoids -= price_buckazoids;
      return 1;
    }
  return 0;
}

void GobPlayer::new_ship(ShipType *type) 
{
  STACKTRACE;
  Ship *old = ship;
  Vector2 pos = 0;
  double a = 0;
  int i;
  if (old) 
    {
      pos = old->normal_pos();
      a = old->get_angle();
    }
  
  ship = game->create_ship ( type->id, gobgame->_player_control, pos, a, team);    
  gobgame->AddPanel();

  for (i = 0; upgrade_list[i]; i += 1) 
    {
      upgrade_list[i]->clear(old, ship, this);
    }
  if (old) 
    {
      old->die();
      game->add(ship);
    }
  else 
    game->add(ship->get_ship_phaser());
  return;
}

void GobStation::buy_new_ship_menu(GobPlayer *s) 
{
  STACKTRACE;

  char buffy1[512], buffy2[512];
  ShipType *otype = s->ship->type;
  ShipType *ntype = shiptype(_build_type.c_str());
  if (otype == ntype) 
    {
      sprintf (buffy1, "You already have a %s", ntype->name);
      if (game->is_local(gobgame->_player_control->channel)) 
	alert(buffy1, NULL, NULL, "&Cancel", NULL, 'c', 0);
      return;
    }
  int ossb = (s->value_starbucks*3) / 4 + (s->ship->type->cost*1)/1;
  int osbz = (s->value_buckazoids*3) / 4 + (s->ship->type->cost*1)/1;
  int nssb = ntype->cost;
  int nsbz = ntype->cost;
  sprintf (buffy1, "You have a %s worth %d s$ / %d b$", otype->name, ossb, osbz);
  sprintf (buffy2, "A %s costs %d s$ / %d b$", ntype->name, nssb, nsbz);
  if ((nssb <= (ossb + s->starbucks)) && (nsbz <= (osbz + s->buckazoids))) 
    {
      int i = 0;
      if (game->is_local(gobgame->_player_control->channel))
	i = alert(buffy1, buffy2, "Do you wish to buy it?", "Yeah!", "No", 'y', 'n');
      game->log_int(gobgame->_player_control->channel, i);
      if (i == 1) 
	{
	  s->starbucks -= nssb - ossb;
	  s->buckazoids -= nsbz - osbz;
	  s->new_ship(ntype);
	}
    }
  else 
    {
      if (game->is_local(gobgame->_player_control->channel)) 
	alert (buffy1, buffy2, "You don't have enough to buy it", "Cancel", NULL, 0, 0);
    }
  return;
}

GobStation::GobStation ( SpaceSprite *pic, SpaceLocation *orbit_me, const std::string& ship, 
			 const std::string& background) : 
  Orbiter(pic, orbit_me, random() % 200 + 500) 
{
  _build_type = ship;
  _background_pic = background;
  layer = LAYER_CBODIES;
  mass = 99;
}

#define STATION_DIALOG_DEPART  0
#define STATION_DIALOG_UPGRADE 1
#define STATION_DIALOG_NEWSHIP 2
#define STATION_DIALOG_REPAIR  3
static DIALOG station_dialog[] =
{// (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)     (d1)  (d2)  (dp)
  { d_agup_button_proc,     385,  50,   150,  30,   255,  0,    0,    D_EXIT,     0,    0,    (void *)"Depart Station" , NULL, NULL },
  { d_agup_button_proc,     385,  90,   150,  30,   255,  0,    0,    D_EXIT,     0,    0,    (void *)"Upgrade Ship" , NULL, NULL },
  { d_agup_button_proc,     385,  130,  150,  30,   255,  0,    0,    D_EXIT,     0,    0,    (void *)"Buy New Ship" , NULL, NULL },
  { d_agup_button_proc,     385,  170,  150,  30,   255,  0,    0,    D_EXIT,     0,    0,    (void *)"Repair Ship" , NULL, NULL },
  { d_agup_text_proc,       185,  420,  270,  30,   255,  0,    0,    0,          0,    0,    dialog_string[0], NULL, NULL },
  { d_tw_yield_proc,        0,    0,    0,    0,  255,  0,    0,    0,       0,    0,    NULL, NULL, NULL },
  { NULL,              0,    0,    0,    0,    255,  0,    0,    0,          0,    0,    NULL, NULL, NULL }
};
void GobStation::station_screen(GobPlayer *s) 
{
  STACKTRACE;
  BITMAP *background = load_bitmap(data_full_path(_background_pic).c_str(), NULL);
  if (!background) {
    message.print(1000, 15, "%s", _background_pic.c_str());
    tw_error ((std::string("couldn't load station background") + data_full_path(_background_pic).c_str()).c_str());
  }
  game->window->lock();
  aa_set_mode(AA_DITHER);
  aa_stretch_blit(background, game->window->surface, 
		  0,0,background->w,background->h, 
		  game->window->x,game->window->y,game->window->w, game->window->h);
  game->window->unlock();
  while (true) {
    sprintf(dialog_string[0], "%03d Starbucks  %03d Buckazoids", s->starbucks, s->buckazoids);
    int r = 0;
    if (game->is_local(gobgame->_player_control->channel)) 
      r = tw_do_dialog(game->window, station_dialog, STATION_DIALOG_DEPART);
		game->log_int(gobgame->_player_control->channel, r);
		switch (r) {
		case STATION_DIALOG_UPGRADE: {
		  upgrade_menu(this, s);
				aa_set_mode(AA_DITHER);
				aa_stretch_blit(background, game->window->surface, 
						0,0,background->w,background->h, 
						game->window->x,game->window->y,
						game->window->w, game->window->h);
		}
		  break;
		case STATION_DIALOG_NEWSHIP: {
		  buy_new_ship_menu(s);
		}
			break;
		case STATION_DIALOG_REPAIR: {
		  if (s->ship->crew == s->ship->crew_max) {
		    if (game->is_local(gobgame->_player_control->channel)) 
		      alert("You don't need repairs", "", "", "Oh, okay", "I knew that", 0, 0);
		    
		    break;
		  }
		  int p = 0;
		  if (game->is_local(gobgame->_player_control->channel)) 
		    p = alert3("Which would you prefer", "to pay for your repairs", "", "1 &Starbuck", "1 &Buckazoid", "&Nothing!", 's', 'b', 'n');
		  game->log_int(gobgame->_player_control->channel, p);
		  switch (p) {
		  case 1: {
		    if (s->starbucks) {
		      s->starbucks -= 1;
		      s->ship->crew = s->ship->crew_max;
		    }
		    else {
		      if (game->is_local(gobgame->_player_control->channel)) 
			alert("You don't have enough!", NULL, NULL, "&Shit", NULL, 's', 0);
		    }
						}
		    break;
		  case 2: {
		    if (s->buckazoids) {
		      s->buckazoids -= 1;
		      s->ship->crew = s->ship->crew_max;
		    }
		    else {
							if (game->is_local(gobgame->_player_control->channel)) 
							  alert("You don't have enough!", NULL, NULL, "&Shit", NULL, 's', 0);
		    }
		  }
		    break;
		  case 3: {
		    r = STATION_DIALOG_DEPART;
		  }
					break;
		  }
		}
		  break;
			}
		if (r == STATION_DIALOG_DEPART) break;
  }
  return;
}
void GobStation::inflict_damage(SpaceObject *other) {
  STACKTRACE;
	SpaceObject::inflict_damage(other);
	if (!other->isShip()) return;
	GobPlayer *p = gobgame->get_player(other);
	if (!p) return;
	gobgame->pause();
	char buffy[256];
	int a;
	sprintf(buffy, "First visited station %s at time", _build_type.c_str());
	a = p->read_pair(buffy);
	if (a == -1) p->write_pair(buffy, game->game_time);
	sprintf(buffy, "Visited station %s N times", _build_type.c_str());
	a = p->read_pair(buffy);
	if (a == -1) a = 0;
	p->write_pair(buffy, a+1);
	station_screen(p);
	gobgame->unpause();
	return;
	}



int num_upgrade_indexes;
int upgrade_index[999];
GobPlayer *upgrade_list_for;
char *upgradeListboxGetter(int index, int *list_size) {
	static char tmp[150];
	if(index < 0) {
		*list_size = num_upgrade_indexes;
		return NULL;
		}
	int i = upgrade_index[index];
	sprintf(tmp, "%1d %3d s$ / %3d b$  :  %s", upgrade_list_for->upgrade_list[i]->num, upgrade_list_for->upgrade_list[i]->starbucks, upgrade_list_for->upgrade_list[i]->buckazoids, upgrade_list_for->upgrade_list[i]->name);
	return tmp;
	}
#define UPGRADE_DIALOG_EXIT 0
#define UPGRADE_DIALOG_LIST 3
static DIALOG upgrade_dialog[] =
{// (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)     (d1)  (d2)  (dp)
  { d_agup_button_proc,     10,  415,  170,  30,   255,  0,    0,    D_EXIT,     0,    0,    (void *)"Station menu" , NULL, NULL },
  { d_agup_textbox_proc,    20,  40,   250,  40,   255,  0,    0,    D_EXIT,     0,    0,    (void *)"Upgrade Menu", NULL, NULL },
  { d_agup_text_proc,       10,  100,  540,  20,   255,  0,    0,    D_EXIT,     0,    0,    (void *)" # Starbucks Buckazoids Description                     ", NULL, NULL },
  { d_agup_list_proc,       10,  120,  540,  280,  255,  0,    0,    D_EXIT,     0,    0,    (void *) upgradeListboxGetter, NULL, NULL },
  { d_agup_text_proc,       185, 420,  270,  30,   255,  0,    0,    0,          0,    0,    dialog_string[0], NULL, NULL },
  { d_tw_yield_proc,        0,    0,    0,    0,  255,  0,    0,    0,       0,    0,    NULL, NULL, NULL },
  { NULL,              0,    0,    0,    0,   255,  0,    0,    0,          0,    0,    NULL, NULL, NULL }
};

void GobStation::upgrade_menu(GobStation *station, GobPlayer *gs) {
  STACKTRACE;
  int i;
  upgrade_list_for = gs;
  clear_to_color(screen, palette_color[8]);
  while (true) {
    sprintf(dialog_string[0], "%03d Starbucks  %03d Buckazoids", gs->starbucks, gs->buckazoids);
    int j = 0;
    for (i = 0; gs->upgrade_list[i]; i += 1) {
      if (gs->upgrade_list[i]->update(gs->ship, station, gs)) {
	upgrade_index[j] = i;
	j += 1;
      }
    }
    num_upgrade_indexes = j;
    int m = 0;
    if (game->is_local(gobgame->_player_control->channel))
      m = tw_do_dialog(game->window, upgrade_dialog, UPGRADE_DIALOG_EXIT);
    game->log_int(gobgame->_player_control->channel, m);
		if (m == UPGRADE_DIALOG_EXIT) return;
		if (m == UPGRADE_DIALOG_LIST) {
			int i = 0;
			if (game->is_local(gobgame->_player_control->channel))
				i = upgrade_dialog[UPGRADE_DIALOG_LIST].d1;
			game->log_int(gobgame->_player_control->channel, i);
			i = upgrade_index[i];
			Upgrade *u = gs->upgrade_list[i];
			if (gs->charge(u->name, u->starbucks, u->buckazoids)) {
				u->execute(gs->ship, station, gs);
				u->charge(gs);
				}
			}
		}
	return;
	}

GobDefender::GobDefender ( Ship *ship) 
: SpaceObject (ship, ship->normal_pos(), 0, gobgame->defenderSprite)
{
	base_phase = 0;
	next_shoot_time = 0;
	collide_flag_anyone = 0;
}
void GobDefender::calculate() {
  STACKTRACE;
	SpaceObject::calculate();
	if (!ship) {
		die();
		return;
	}
	if (next_shoot_time < gobgame->game_time) {
		SpaceObject *target = NULL;
		Query q;
		q.begin(this, OBJECT_LAYERS &~ bit(LAYER_SHIPS), 300);
		while (q.currento && !target) {
			if (!q.currento->sameTeam(ship)) {
				SpaceLine *l = new PointLaser ( 
					this, palette_color[4], 2, 150, 
					this, q.currento
					);
				add(l);
				if (l->exists()) target = q.currento;
			}
			q.next();
		}
		if (target) {
			next_shoot_time = gobgame->game_time + 400;
		}
	}
	double a = base_phase + (gobgame->game_time % 120000) * ( PI2 / 1000.0) / 6;
	angle = normalize(a,PI2);
	pos = normalize(ship->normal_pos() + 270 * unit_vector ( angle ));
	return;
}

RainbowRift::RainbowRift () 
//: SpaceLocation ( NULL, 12800, 12800, 0) 
: SpaceLocation ( NULL, random(map_size), 0) 
{
	int i;
	collide_flag_sameship = 0;
	collide_flag_sameteam = 0;
	collide_flag_anyone = 0;
	for (i = n*6-6; i < n*6+2; i += 1) {
		p[i] = 75 + random(150.0);
	}
	for (i = 0; i < n; i += 1) {
		squiggle();
	}
	next_time = game->game_time;
	next_time2 = game->game_time;
}
void RainbowRift::animate( Frame *frame ) {
  STACKTRACE;
	Vector2 s;
	s = corner(pos, Vector2(300,300));
	if ((s.x < -500) || (s.x > space_view_size.x + 500) || 
		(s.y < -500) || (s.y > space_view_size.y + 500))
		return;
	int b[n*6+2];
	int i;
	for (i = 0; i < n*6+2; i += 2) { 
		b[i] = iround(s.x + p[i] * space_zoom);
		b[i+1] = iround(s.y + p[i+1] * space_zoom);
	}
	for (i = 0; i < n; i += 1) {
		RGB tc = c[n-i-1];
		int a = tw_color(tc.r, tc.g, tc.b);
		spline ( frame->surface, &b[i*6], a );
	}
	frame->add_box ( 
		iround(s.x - 2), iround(s.y -2), 
		iround(300 * space_zoom+5), iround(300 * space_zoom+5)
		);
	return;
}
void RainbowRift::squiggle() {
  STACKTRACE;
	int i;
	int m = n*6+2;
	for (i = 0; i < m - 6; i += 1) {
		p[i] = p[i+6];
	}
	p[m-6] = p[m-8] * 2 - p[m-10];
	p[m-5] = p[m-7] * 2 - p[m-9];
	p[m-4] = 75 + random(150.0);
	p[m-3] = 75 + random(150.0);
	p[m-2] = 75 + random(150.0);
	p[m-1] = 75 + random(150.0);
	for (i = 0; i < n-1; i += 1) {
		c[i] = c[i+1];
	}
	int r, g, b;
	r = int(game->game_time * 0.5) % 360;
	hsv_to_rgb( r, 1.0, 1.0, &r, &g, &b );
	c[n-1].r = r;
	c[n-1].g = g;
	c[n-1].b = b;
	return;
}
void RainbowRift::calculate() {
  STACKTRACE;
	while (game->game_time > next_time) {
		next_time += 25;
		squiggle();
	}
	while (game->game_time > next_time2) {
		next_time2 += random() % 10000;
		Query q;
		for (q.begin(this, bit(LAYER_SHIPS), 40); q.current; q.next()) {
			GobPlayer *p = gobgame->get_player(q.currento);
			if (q.currento == p->ship) {
				int i = 0;
				i = gobgame->_player_control->choose_ship(game->window, "You found the Rainbow Rift!", reference_fleet);
				game->log_int(gobgame->_player_control->channel, i);
				if (i == -1) 
				  i = random(reference_fleet->getSize());
				game->redraw();
				if (reference_fleet->getShipType(i) == p->ship->type) {
					p->starbucks += random() % 80;
					p->buckazoids += random() % 80;
					game->add(new RainbowRift());
				}
				else {
					p->starbucks += random() % (1+p->value_starbucks);
					p->buckazoids += random() % (1+p->value_buckazoids);
					p->new_ship(reference_fleet->getShipType(i));
				}
				die();
			}
		}
	}
	return;
}


void GobGame::add_gobplayer(const std::string& system, Vector2 pos)
{
  STACKTRACE;
  switch_system(system);
  gobplayer.init(game->new_team());
  add_focus(_player_control, _player_control->channel);
  gobplayer.new_ship(shiptype("supbl"));
  gobplayer.ship->translate(pos);
}

/// Save game 
/// @param file path to saved data
std::string GobGame::Save(const std::string& file)
{
  SaveGobGameInfo::getInstance()->updateSave();
  std::string save_script;
  save_script  = "import twsaveload\n";
  save_script += "data = twsaveload.SaveManager(\"\"\"" + file +"\"\"\")\n";
  save_script += "data.getData()\n";
  save_script += "data.saveSelf()\n";
  python::exec_string(save_script);
  return "";
}

/// Load game 
/// @param file path to saved data
int GobGame::Load(const std::string& file)
{
  std::string load_script;
  load_script  = "import twsaveload\n";
  load_script += "data = twsaveload.SaveManager(\"\"\"" + file +"\"\"\")\n";
  load_script += "data.loadSelf()\n";
  python::exec_string(load_script);
    SaveGobGameInfo::getInstance()->updateGame();
  return 0;
}


REGISTER_GAME(GobGame, "GOB")



/* intended upgrades:

faster marines       == faster Orz Marines, cost 4s
upgrade battle armor == tougher Orz Marines, cost 4s/4b
improve range        == long range Orz cannons, cost 3s
regeneration         == crew regeneration for Orz, cost 10s/25b, only purchasable once
sharper shurikens    == +1 damage for Kohr-Ah blades, cost 5s
faster shurikens     == higher velocity for Kohr-Ah blades, cost 4s
larger corona        == longer range for Kohr-Ah FRIED, cost 15s, only purchasable once
hotter corona        == double damage for Kohr-Ah FRIED, cost 10s, only purchasable once
divine favor         == pkunk respawn, only available from one base, cost 48s/0b, only purchasable once, kept when ship is sold
sentinel system      == Chmmr ZapSats, only available from one base, cost 30s/30b

long range scanners  == can zoom farther out, gives radar, only available from one base, cost 8s/20b

*/
