#include "level.h"
#include "debug.h"
#include "aglhelper.h"
#include "global.h"

#ifndef PI
#define PI 3.14159265358979323846264338327950288419716939937508
#endif

#define HIT_CONST 50
#define DEATH_CONST 200
#define ENDUR_CONST 500

Level::Level(char *fileName) : sizefactor(-3) {

	defaultFont = allegro_gl_convert_allegro_font(font, AGL_FONT_TYPE_TEXTURED, 16.0);//allegro_gl_load_system_font_ex("Veranda", AGL_FONT_TYPE_OUTLINE, 0, 1, 1, 0, 0, 255);
	hit = load_wav("data/hit.wav");
	otherHit = load_wav("data/hit2.wav");
	if (!hit) throw "Could not load data/hit.wav";
	if (!otherHit) throw "Could not load data/hit2.wav";

	playerDying = DEATH_CONST;
	playerX=2.5, playerY=2.5, playerAngle=0;	// random defaults -- should be over-written by the level loader
	playerHealth = 1.0f;
	startHealth = 1.0f;
	beingHit = beingHelped = 0;
	energyLevel = 1;
	energyLevelDuration = 0;

	std::ifstream level(fileName);
	if (!level) throw "Error opening file";

	int numEnemyTypes;
	std::string floorFile, wallFile, bulletFile, portalFile, virusFile, pillFile, energyFile, weakwallFile, ebulletFile;



	level >> floorFile >> wallFile >> numEnemyTypes;

	std::string temp;
	for (int i=0; i<numEnemyTypes; i++) {
		level >> temp;
		Anim enemy;
		enemy.Load(temp.c_str());
		enemies.push_back(enemy);
	}

	level >>bulletFile >> portalFile >> virusFile >> pillFile >> energyFile >> weakwallFile >> ebulletFile;

	std::string theRest;
	while (level >> theRest) {
		levelSequence.push_back(theRest);
	}

	level.close();

	floor = makeTexture(floorFile.c_str());
	wall = makeTexture(wallFile.c_str());
	bullet = makeTexture(bulletFile.c_str());
	portal.Load(portalFile.c_str());
	virus = makeTexture(virusFile.c_str());
	pill.Load(pillFile.c_str());
	energy.Load(energyFile.c_str());
	weakwall = makeTexture(weakwallFile.c_str());
	ebullet = makeTexture(ebulletFile.c_str());
}

void Level::HandleInput()
{

	if (playerDying <= 0) {
		playerHealth = 1;
		playerDying = DEATH_CONST;
		LoadNextMap(true);
	}

	static int timeToBullet = 0;

	poll_keyboard();
	poll_mouse();

	playerAngle-=(key[KEY_RIGHT] - key[KEY_LEFT])*2;
	if (rand()%8==1) playerAngle += ((rand()%101 - 50)/10.0f) * (1-playerHealth) * (1-playerHealth);

	int mickeyx=0, mickeyy=0;
	float factor = -.01f;
	get_mouse_mickeys(&mickeyx, &mickeyy); // get mouse movement
	// And a VERY dirty hack to make sure the mouse doesn't hit the edge of the screen, ever. (Unless you try real hard.) (?) (Don't.)
	{
		position_mouse(SCREEN_W/2, SCREEN_H/2);
		int t1, t2;
		get_mouse_mickeys(&t1, &t2);
	}

	playerAngle+=mickeyx*(get_config_int("config", "Sensitivity", 0)+1)/11.0f;

	float s = sin(playerAngle*(PI/180));
	float c = cos(playerAngle*(PI/180));
	int forward = -(key[KEY_W]||key[KEY_UP]) + (key[KEY_S]||key[KEY_DOWN]);
	int sideways = key[KEY_A] - key[KEY_D];
	float oldx = playerX, oldy = playerY;
	playerX+=(forward)*.1*s - sideways*.1*c;
	playerY-=(forward)*.1*c + sideways*.1*s;
	Collide(oldx, playerX, oldy, playerY);

	if (energyLevel > 0 && timeToBullet-- < 0 && (key[KEY_SPACE] || mouse_b&1)) {
		timeToBullet = 10;
		objects.push_back(Object(BULLET, playerX-.1*s, playerY+.1*c, -.2*s, +.2*c, 20, 0));
		if (energyLevel == 2) {
			objects.push_back(Object(BULLET, playerX-.1*s, playerY+.1*c, -.2*s - .05*c, +.2*c + .05*s, 20, 0));
			objects.push_back(Object(BULLET, playerX-.1*s, playerY+.1*c, -.2*s + .05*c, +.2*c + .05*s, 20, 0));
		}
	}
}

static bool Visible = false;

static void CheckVisibility(BITMAP* bmp, int x, int y, int d)
{
    if(!((Level*)bmp)->IsFloor(x, y))
    {
        Visible = false;
    }
}

void Level::HandleObjects()
{
	for (unsigned int i=0; i<enemies.size(); i++)
		enemies[i].Tick();
	portal.Tick();
	pill.Tick();
	energy.Tick();

	beingHit--;
	beingHelped--;
	if (energyLevelDuration--<=0)
		energyLevel = 1;


	float distToPortalSq = 0;
	bool portalFound = false;

	if (OnVirus(playerX, playerY)) {
		beingHit=HIT_CONST;
		playerHealth-=.004;
	}

	if (playerHealth<=0) {
		playerDying --;
		playerHealth = 0;
	}

	// do "AI" && movement
	for (std::list<Object>::iterator i = objects.begin(); i!=objects.end(); ++i) {
		if (i->type == BULLET || i->type == EBULLET) {
			i->x += i->vx;
			i->y += i->vy;
			int n = toData(int(i->x), int(i->y));
			if (i->duration-- < 0)
				i = objects.erase(i);	// this may skip a bullet every once in a while; not a real problem
			else if (n>-1&&levelData[n].data!=0) {
				i = objects.erase(i);

				if (levelData[n].data == 2)
					levelData[n].data = 0;
				play_sample(otherHit, int(256 * (get_config_int("config", "sound", 0)/10.0f)), 128, 1000, 0);
			}
		}
		else if (i->type == ENEMY) {
			if (--i->stunned<0) {	// stunned monsters don't move
                Visible = true;
                do_line((BITMAP*)this, (int)playerX, (int)playerY,
                    (int)i->x, (int)i->y, 0, CheckVisibility);
                if (Visible)
                {

                    float oldx = i->x, oldy = i->y;
                    float angle = atan2(playerY - i->y, playerX - i->x);
                    float s = cos(angle);
                    float c = sin(angle);
			if (i->enemyNumber == 1)
			{
				s *= .5;
				c *= .5;
			}
                    i->x += .05*s;
                    i->y += .05*c;
                    Collide(oldx, i->x, oldy, i->y);

				if (i->enemyNumber == 1 && i->timeToBullet-- <= 0) {
					i->timeToBullet = 80;
					objects.push_back(Object(EBULLET, i->x+.1*s, i->y+.1*c, .2*s, .2*c, 80, 0));
				}
                }
			}
		}
		else if (i->type == PORTAL) {
			float distSq = (i->x-playerX)*(i->x-playerX) + (i->y-playerY)*(i->y-playerY);
			if (distSq<1&&playerHealth>0) {	// if we're at a portal and not dead, switch to the next map
				set_config_int("config", "Level", get_config_int("config", "Level", 0) + 1);
				LoadNextMap(false);
				return;
			}
			if (!portalFound || distSq < distToPortalSq) {
				portalFound = 1;
				distToPortalSq = distSq;
				closestPortalX = int(i->x);
				closestPortalY = int(i->y);
			}
		}
	}

	// do collisions
	for (std::list<Object>::iterator i = objects.begin(); i!=objects.end(); ++i) {
		if ( fabs(i->x-playerX) < .5 && fabs(i->y-playerY) < .5 ) { // player-object collision
			if (i->type == ENEMY) {
				playerHealth-=.006;
				beingHit = HIT_CONST;
			}
			if (i->type == CHILLPILL&&playerHealth>0) {	// chillpills revive -- but they don't bring back from the grave
				if (pill.getFrameNum() == 0) {
					playerHealth+=.25;
					if (playerHealth>1.0f) playerHealth = 1.0f;
					beingHelped = HIT_CONST;
				}
				else {
					playerHealth-=.10;
					if (playerHealth<0.0f) playerHealth = 0.0f;
					beingHit = HIT_CONST;
				}
				i = objects.erase(i);
			}
			if ( (i->type == ENERGYPILL||i->type == EBULLET) &&playerHealth>0) {
				if (i->type == EBULLET) {
					playerHealth-=.10;
					if (playerHealth<0.0f) playerHealth = 0.0f;
					beingHit = HIT_CONST;
				}
				else if (pill.getFrameNum() == 0) {
					energyLevel = 0;
					energyLevelDuration = ENDUR_CONST;
					beingHit = HIT_CONST;
				}
				else if (pill.getFrameNum() == 1) {
					energyLevel = 2;
					energyLevelDuration = ENDUR_CONST;
					beingHelped = HIT_CONST;
				}
				i = objects.erase(i);
			}
		}

		for (std::list<Object>::iterator j = i; j!=objects.end(); j++) {
			if (j==i) continue;

			if ( fabs(i->x-j->x) < .5 && fabs(i->y-j->y) < .5 ) { // collision
				if (i->type == BULLET && j->type == ENEMY || i->type == ENEMY && j->type == BULLET) { // A bullet hits a monster TODO: EBULLET ADD HERE YEAH
					play_sample(hit, int(256 * (get_config_int("config", "sound", 0)/10.0f)), 128, 1000, 0);
					int h = i->type == ENEMY ? --i->health : --j->health;
					if (h==0) {
						objects.erase(j);
						i = (objects.erase(i));
						if (i==objects.end()) break;
						j = i;
					}
					else {
						if (j->type == BULLET) {
							j=(objects.erase(j))--;
							i->stunned = 20;
						}
						else {
							i = (objects.erase(i));
							if (i==objects.end()) break;
							j=i;
							j->stunned = 20;
						}
					}
				}
			}
		}
	}
}



void Level::InfectNeighbors(int i, int j) {
	for (int x = i-1; x<i+2; x++) {
		for (int y = j-1; y<j+2; y++) {
			if (abs(x-i)==abs(y-j)||rand()%4==1) continue;
			int n = toData(x,y);
			if (n==-1) continue;
			if (!levelData[n].infected&&levelData[n].data==0)
				levelData[n].infected = true;
		}
	}

}

void Level::SpreadVirus() {
	for (int i = 0; i < dimX; i++) {
		for (int j=0; j < dimY; j++) {
			int n = toData(i,j);
			if (n==-1) continue;
			if (levelData[n].infected) {
				if (levelData[n].durationTillSpread-- < 0) {
					InfectNeighbors(i, j);
					levelData[n].durationTillSpread = 100;	// cheap hack to prevent respreading of already-spread tiles; should increase algorithmic efficiency a bit
				}
			}
		}
	}
}

void Level::LoadTextfileMap(char *fileName) {	// abandoned for LoadBitmapMap -- I don't have time to maintain both
	objects.clear();
	std::ifstream level(fileName);

	if (!level) throw "Error opening file";
	level >> dimX >> dimY;
	levelData = std::vector<Tile>(dimX*dimY);

	for (int y=0; y<dimY; y++)
	{
		for (int x=0; x<dimX; x++)
		{
			int n = toData(x, y);
			if (n==-1) continue;
			level >> levelData[n].data;
		}
	}
}

void Level::LoadBitmapMap(const char *fileName) {
	portalExists = false;
	objects.clear();
	BITMAP *level = load_bitmap(fileName, NULL);
	if (!level) throw "Error opening file";
	dimX = level->w;
	dimY = level->h;
	levelData.clear();
	levelData = std::vector<Tile>(dimX*dimY);
	for (int y=0; y<dimY; y++)
	{
		for (int x=0; x<dimX; x++)
		{
			int pix = getpixel(level, x, y);
			int n = toData(x, y);
			if (n==-1) continue;
			if (pix == makecol(0,0,0))	// a more complex read can be done in the future ...
				levelData[n].data = 1;
			else if (pix == makecol(128,128,0))
				levelData[n].data = 2;
			else {
				levelData[n].data = 0;
				if (pix == makecol(255,0,255))
				{
					playerX = x + .5;
					playerY = y + .5;
				}
				if (pix == makecol(255,0,0))
				{
					objects.push_back(Object(ENEMY, x+.5, y+.5, 0));
				}
				if (pix == makecol(128,0,128))
				{
					Object enemyType2(ENEMY, x+.5, y+.5, 0);
					enemyType2.enemyNumber = 1;
					enemyType2.health = 4;
					objects.push_back(enemyType2);
				}
				if (pix == makecol(0,0,255))
				{
					objects.push_back(Object(PORTAL, x+.5, y+.5, 0));
					portalExists = true;
				}
				if (pix == makecol(0, 255, 0))
				{
					levelData[n].infected = true;
				}
				if (pix == makecol(0, 255, 255))
				{
					objects.push_back(Object(CHILLPILL, x+.5, y+.5, 0));
				}
				if (pix == makecol(255, 255, 0))
				{
					objects.push_back(Object(ENERGYPILL, x+.5, y+.5, 0));
				}
			}
		}
	}
	destroy_bitmap(level);
}

void Level::LoadNextMap(int stayback) {
	static std::list<std::string>::iterator current = levelSequence.begin();
	if (stayback == -1) current = levelSequence.begin();
	else if (stayback == 1 && current!=levelSequence.begin()) {
		energyLevel = 1;		// can't keep a powerup through death
		current--;
		playerHealth = startHealth;
	}
	else
		startHealth = playerHealth;
	if (current!=levelSequence.end()) {
		playerAngle=0;
		LoadBitmapMap(current->c_str());
		current++;
	}
	else {
		set_config_int("config", "Level", 0);
		throw "Perhaps congratulations!  You may or may not have won the game.  You may or may not have saved the world.  Possibly good work?";
	}

	playerHealth = 1.0f;	// this is a quick hack to disable the health carryover code; it's painful enough without that.
}

void Level::DebugOut() {
	DODEBUG(
		for (int y=0; y<dimY; y++) {
			for (int x=0; x<dimX; x++) {
				DEBUGOUT(levelData[toData(x,y)].data);
			}
			DEBUGOUT(std::endl);
		}
	)
}

void Level::DrawFloor(int x, int y) {
	int n = toData(x,y);
	if (n == -1) return;	// do not draw anything out of bounds
	if (levelData[n].infected) glBindTexture(GL_TEXTURE_2D, virus);
	else glBindTexture(GL_TEXTURE_2D, floor);
	x++, y++; // oops!, a quick fix
	glBegin(GL_QUADS);
		glTexCoord2i(0, 1); glVertex3f(x*sizefactor, .5*sizefactor,  y*sizefactor);
		glTexCoord2i(0, 0); glVertex3f(x*sizefactor, .5* sizefactor, (y-1)*sizefactor);
		glTexCoord2i(1, 0); glVertex3f((x-1)*sizefactor, .5*sizefactor, (y-1)*sizefactor);
		glTexCoord2i(1, 1); glVertex3f((x-1)*sizefactor, .5*sizefactor, y*sizefactor);
	glEnd();
}

void Level::DrawWall(int x, int y, int data) {
	if (data == 1) glBindTexture(GL_TEXTURE_2D, wall);
	else glBindTexture(GL_TEXTURE_2D, weakwall);
	x++,y++; // oops!, a quick fix
	glBegin(GL_QUADS);
	// back
		glTexCoord2i(0, 0); glVertex3f((x-1)*sizefactor, .5*sizefactor, y*sizefactor);
		glTexCoord2i(1, 0); glVertex3f( x*sizefactor, .5*sizefactor, y*sizefactor);
		glTexCoord2i(1, 1); glVertex3f( x*sizefactor,-.5*sizefactor, y*sizefactor);
		glTexCoord2i(0, 1); glVertex3f((x-1)*sizefactor,-.5*sizefactor, y*sizefactor);

	// front
		glTexCoord2i(0, 0); glVertex3f( x*sizefactor, .5*sizefactor, (y-1)*sizefactor);
		glTexCoord2i(1, 0); glVertex3f((x-1)*sizefactor, .5*sizefactor,  (y-1)*sizefactor);
		glTexCoord2i(1, 1); glVertex3f((x-1)*sizefactor,-.5*sizefactor,  (y-1)*sizefactor);
		glTexCoord2i(0, 1); glVertex3f( x*sizefactor,-.5*sizefactor,  (y-1)*sizefactor);

	// right
		glTexCoord2i(1, 0); glVertex3f( x*sizefactor, .5*sizefactor, (y-1)*sizefactor);
		glTexCoord2i(1, 1); glVertex3f( x*sizefactor,-.5*sizefactor, (y-1)*sizefactor);
		glTexCoord2i(0, 1); glVertex3f( x*sizefactor,-.5*sizefactor, y*sizefactor);
		glTexCoord2i(0, 0); glVertex3f( x*sizefactor, .5*sizefactor, y*sizefactor);

	// left
		glTexCoord2i(1, 0); glVertex3f( (x-1)*sizefactor, .5*sizefactor, y*sizefactor);
		glTexCoord2i(1, 1); glVertex3f( (x-1)*sizefactor,-.5*sizefactor, y*sizefactor);
		glTexCoord2i(0, 1); glVertex3f( (x-1)*sizefactor,-.5*sizefactor, (y-1)*sizefactor);
		glTexCoord2i(0, 0); glVertex3f( (x-1)*sizefactor, .5*sizefactor, (y-1)*sizefactor);
	glEnd();
}

void Level::DrawBig(const Object &e, GLuint tex) {
	glBindTexture(GL_TEXTURE_2D, tex);

	glPushMatrix();

	glTranslatef( (e.x) * sizefactor, 0.0f, (e.y) * sizefactor );
	glRotatef(-playerAngle, 0.0f, 1.0f, 0.0f);

	glBegin(GL_QUADS);
		glTexCoord2i(1,1); glVertex3f((-.5)*sizefactor, -.5*sizefactor, 0);
		glTexCoord2i(0,1); glVertex3f((.5)*sizefactor, -.5*sizefactor, 0);
		glTexCoord2i(0,0); glVertex3f((.5)*sizefactor, .5*sizefactor, 0);
		glTexCoord2i(1,0); glVertex3f((-.5)*sizefactor, .5*sizefactor, 0);
	glEnd();

	glPopMatrix();
}

void Level::DrawSmall(const Object &e, GLuint tex) {
	glBindTexture(GL_TEXTURE_2D, tex);

	glPushMatrix();

	glTranslatef( (e.x) * sizefactor, 0.0f, (e.y) * sizefactor );
	glRotatef(-playerAngle, 0.0f, 1.0f, 0.0f);

	glBegin(GL_QUADS);
		glTexCoord2i(1,1); glVertex3f((-.1)*sizefactor, .3*sizefactor, 0);
		glTexCoord2i(0,1); glVertex3f((.1)*sizefactor, .3*sizefactor, 0);
		glTexCoord2i(0,0); glVertex3f((.1)*sizefactor, .5*sizefactor, 0);
		glTexCoord2i(1,0); glVertex3f((-.1)*sizefactor, .5*sizefactor, 0);
	glEnd();

	glPopMatrix();
}

void Level::DrawBullet(const Object &e) {
	if (e.type == EBULLET) glBindTexture(GL_TEXTURE_2D, ebullet);
	else glBindTexture(GL_TEXTURE_2D, bullet);

	glPushMatrix();

	glTranslatef( (e.x) * sizefactor, 0.0f, (e.y) * sizefactor );
	glRotatef(-playerAngle, 0.0f, 1.0f, 0.0f);

	glBegin(GL_QUADS);
		glTexCoord2i(1,1); glVertex3f((-.1)*sizefactor, -.1*sizefactor, 0);
		glTexCoord2i(0,1); glVertex3f((.1)*sizefactor, -.1*sizefactor, 0);
		glTexCoord2i(0,0); glVertex3f((.1)*sizefactor, .1*sizefactor, 0);
		glTexCoord2i(1,0); glVertex3f((-.1)*sizefactor, .1*sizefactor, 0);
	glEnd();

	glPopMatrix();
}

void Level::DrawPower(GLuint tex) {
	glBindTexture(GL_TEXTURE_2D, tex);

	glPushMatrix();

	glTranslatef(-2,-2,-5);

	glBegin(GL_QUADS);
		glTexCoord2i(1,1); glVertex3f((-.1)*sizefactor, -.1*sizefactor, 0);
		glTexCoord2i(0,1); glVertex3f((.1)*sizefactor, -.1*sizefactor, 0);
		glTexCoord2i(0,0); glVertex3f((.1)*sizefactor, .1*sizefactor, 0);
		glTexCoord2i(1,0); glVertex3f((-.1)*sizefactor, .1*sizefactor, 0);
	glEnd();

	glPopMatrix();
}

void Level::DrawDirectionTip() {

	glPushMatrix();

	float angle = atan2( playerX - closestPortalX, playerY - closestPortalY ) * 180 / PI;
	glTranslatef( (playerX - 1*sin(playerAngle*(PI/180))) * sizefactor , 0.0f, (playerY + 1*cos(playerAngle*(PI/180))) * sizefactor  );
	glRotatef(angle - 90, 0.0f, 1.0f, 0.0f);

	glLineWidth(6.0);
	glColor4f(0,.7,0,1);
	glBegin(GL_LINES);
		glVertex3f(0, -1, 0);
		glVertex3f(1, -1, 0);
		glVertex3f(1, -1, 0);
		glVertex3f(.5,-1, .5);
		glVertex3f(1, -1, 0);
		glVertex3f(.5, -1, -.5);
	glEnd();
	glColor4f(1,1,1,1);

	glPopMatrix();
}

float Level::zDist(float x, float y) {
	float dx = x - playerX, dy = y - playerY;
	float dirx = -sin(playerAngle*(PI/180)), diry = cos(playerAngle*(PI/180));
	return dx*dirx+dy*diry;
}

Level *thisGlobal = NULL;
bool operator< ( Object A, Object B )
{
	return (thisGlobal->zDist(A.x, A.y) > thisGlobal->zDist(B.x, B.y));
}

void Level::DrawHealth() {
	glPushMatrix();

	glTranslatef( (playerX - 1*sin(playerAngle*(PI/180))) * sizefactor , 0.0f, (playerY + 1*cos(playerAngle*(PI/180))) * sizefactor  );
	glRotatef(-playerAngle, 0.0f, 1.0f, 0.0f);

	if (playerHealth<0) playerHealth = 0;

	glBegin(GL_QUADS);
	// background
		glColor4f(0,.2,0,1); glVertex3f((-.4)*sizefactor, -.43*sizefactor, 0);
		glColor4f(.2,0,0,1); glVertex3f((.2)*sizefactor, -.43*sizefactor, 0);
		glColor4f(.2,0,0,1); glVertex3f((.2)*sizefactor, -.4*sizefactor, 0);
		glColor4f(0,.2,0,1); glVertex3f((-.4)*sizefactor, -.4*sizefactor, 0);
	// healthbar
		glColor4f(0,playerHealth,0,1); glVertex3f((-.4+(1-playerHealth)*.6)*sizefactor, -.43*sizefactor, 0);
		glColor4f(1,0,0,1); glVertex3f((.2)*sizefactor, -.43*sizefactor, 0);
		glColor4f(1,0,0,1); glVertex3f((.2)*sizefactor, -.4*sizefactor, 0);
		glColor4f(0,playerHealth,0,1); glVertex3f((-.4+(1-playerHealth)*.6)*sizefactor, -.4*sizefactor, 0);
	glEnd();

	glPopMatrix();
}

void Level::Draw() {
	glPushMatrix();

	float hitColor =  .6-.4*float(beingHit)/HIT_CONST;
	float helpColor = .6-.5*float(beingHelped)/HIT_CONST;
	float deathColor = .6 * (playerDying / float(DEATH_CONST));
	if (playerDying < 0) playerDying = 0;
	if (playerHealth<=0) glColor4f(deathColor, deathColor, deathColor, 1);
	else if (beingHit>0) glColor4f(1, hitColor, hitColor, 1);
	else if (beingHelped>0) glColor4f(helpColor, 1, helpColor, 1);
	else glColor4f(.6,.6,.6,1);


	glRotatef(playerAngle, 0.0f, 1.0f, 0.0f);
	glTranslatef(-playerX*sizefactor,0.0f,-playerY*sizefactor);

	for (int x=0; x<dimX; x++) {
		for (int y=0; y<dimY; y++) {
			int n = toData(x,y);
			if (n==-1) continue;
			if (levelData[n].data == 0) DrawFloor(x, y);
			else DrawWall(x, y, levelData[n].data);
		}
	}

	thisGlobal = this;
	objects.sort();

	for (std::list<Object>::iterator i = objects.begin(); i!=objects.end(); i++) {
		if ( i->type == ENEMY) DrawBig(*i, enemies[i->enemyNumber].Grab());
		else if ( i->type == BULLET || i->type == EBULLET ) DrawBullet(*i);
		else if ( i->type == PORTAL) DrawBig(*i, portal.Grab());
		else if ( i->type == CHILLPILL) DrawSmall(*i, pill.Grab());
		else if ( i->type == ENERGYPILL) DrawSmall(*i, energy.Grab());
	}


	//glDisable(GL_BLEND);
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_TEXTURE_2D);

	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_COLOR);
	if (playerHealth>0) {
		DrawHealth();
		if (portalExists)
			DrawDirectionTip();
	}

	glEnable(GL_TEXTURE_2D);
	//glEnable(GL_BLEND);


	glPopMatrix();

	// now the text
    	glColor4f(1,1,1,1);

	if (energyLevel!=1)
	{
		DrawPower(energy.Grab(energyLevel/2));
	}
	/*
	if (playerHealth>0)
		allegro_gl_printf(defaultFont, -8.5, 8.1, -20.0, makecol(0, 150, 0), "Calmness:");
	if (playerHealth<=0)
		allegro_gl_printf(defaultFont, -3, -.5, -20.0, makecol(255, 0, 0), "The stress got to you.");
*/
	if (playerHealth>0) {
		if (g_english) allegro_gl_printf(defaultFont, -8.5, 8.55, -20.0, makecol(0, 150, 0), "Calmness:");
		else allegro_gl_printf(defaultFont, -8.5, 8.55, -20.0, makecol(0, 150, 0), "CALMNESS:");
	}
	else
		allegro_gl_printf(defaultFont, -3, -.5, -20.0, makecol(255, 0, 0), Translate("The stress got to you."));

	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_DEPTH_TEST);

}

void Level::Collide(float oldx, float &x, float oldy, float &y, float range) {

	//if (Hitting(oldx, oldy, range)) return;	// if the oldx, oldy pair was bad, we really can't do much to fix the new pair.

	if (!Hitting(x,y, range)&&!OnMonster(x,y)) return;

	if (!Hitting(oldx,y, range)&&!OnMonster(oldx,y)) {x = oldx; return;}

	if (!Hitting(x,oldy, range)&&!OnMonster(x,oldy)) {y = oldy; return;}

	x = oldx;
	y = oldy;
}

bool Level::Hitting(float x, float y, float range) {
	if (x<range||y<range||x>=dimX-range+.01||y>=dimY-range+.01) return 1;	// the edges are walls by default
	for (float i = x-range; i<=x+range+.01; i+=.2) {
		for (float j=y-range; j<=y+range+.01; j+=.2) {
			int n = toData(int(i), int(j));
			if (n==-1) return true;
			if (levelData[ n ].data) return true;
		}
	}
	return false;
}

bool Level::OnMonster(float x, float y) {
	for (std::list<Object>::iterator i = objects.begin(); i!=objects.end(); ++i) {
		if ( fabs(i->x-x) < .4 && fabs(i->y-y) < .4 ) { // player-object collision
			if (x==i->x&&y==i->y) continue; 	// there's a good chance this is just the monster intersecting itself
			if (i->type == ENEMY)
				return true;
		}
	}
	if (x!=playerX&&y!=playerY) { // if not the player
		if (fabs(x-playerX) < .4 && fabs(y-playerY) < .4)
			return true;
	}
	return false;
}

bool Level::OnVirus(float x, float y, float range) {
	if (x<range||y<range||x>=dimX-range+.01||y>=dimY-range+.01) return 1;	// the edges are virus by default ... you shouldn't be able to be here anyway
	for (float i = x-range; i<=x+range+.01; i+=.2) {
		for (float j=y-range; j<=y+range+.01; j+=.2) {
			int n = toData(int(i), int(j));
			if (n == -1) return true;	// no sypathy for the out of bounds player!
			if (levelData[ n ].infected) return true;
		}
	}
	return false;
}

int Level::toData(int x, int y) {
	if (x<0||y<0||x>=dimX||y>=dimY) return -1;	// if we're out of bounds, well, everyone has to handle that their own way.
	return y*dimX + x;
}
