/*

Copyright (c) 2003-2005      Arthur Huillet


Permission is hereby granted, free of charge, to any person obtaining a copy of this software and 
associated documentation files (the "Software"), to deal in the Software without restriction, including
 without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is furnished to do so, 
 subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial 
portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
 FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

    */

#include <stdio.h> /*we need it for FILE * */

/** \file objects.h
Definitions of data structures and constants used in the game
*/

#ifndef OBJECTS_H
#define OBJECTS_H

/** The version string
*/
#define VERSION "0.4CVS"

/** \defgroup keybinds_constants Keybindings constants
*/ 
/*@{*/
#define LEFT 1 /**< This constant refers to the action "go left". */
#define RIGHT 2 /**< This constant refers to the action "go right". */
#define JUMP 3 /**< This constant refers to the action "jump". */
#define CROUCH 4 /**< This constant refers to the action "crouch" (not implemented). */
#define FIRE 5 /**< This constant refers to the action "fire" (not implemented). */
#define RUN 6 /**< This constant refers to the action "run". */
#define CH_WEAP 9 /**< This constant refers to the action "change weapon". */
/*@}*/ 


/** \defgroup obj_parameters Objects parameters
*/
/*@{*/
	/** \defgroup pl_obj_parameters Liveplayers parameters
	\ingroup obj_parameters
	*/
	/*@{*/
	#define ZL_ENEMY_CLEVER 1 /**< Enemy is clever or not. This is the "mickey mousse"/"mario brosse" flag. If set, the enemy will target the player and follow him, in the other case he will just wander around.*/
	/*@}*/

	/** \defgroup wl_obj_parameters Walls and livewalls parameters
	\ingroup obj_parameters
	*/
	/*@{*/
	#define WL_IS_INVISIBLE 1 /**< Is the [live]wall invisible.*/

		/** \defgroup lwl_obj_parameters Livewall-only parameters
		\ingroup wl_obj_parameters
		*/
		/*@{*/
		#define LWL_SIZE_VAR 256 /**< Size variation enabled*/
		#define LWL_CENTER_PL 512 /**< Center player on size variation*/
		#define LWL_INVERT_CONTACT 1024 /**< Go the other way on collision with an object that blocks ?*/
		#define LWL_NOMOVE 2048 /**< Moving disabled*/
		/*@}*/
		
	/*@}*/

/*@}*/
	
/** \defgroup	obj_states Objects state flags
*/
/*@{*/
#define EXISTS 1 /**< Tell wether an object exists/is used/whatever. */
#define DIRECTION 2 /**< Used by moving objects to know in what direction the movement takes place.*/
#define NO_LEFT 4 /**< Indicates that the object can't go left (colldet).*/
#define NO_UP 8 /**< Indicates that the object can't go up (colldet).*/
#define NO_DOWN 16 /**< Indicates that the object can't go down (colldet).*/
#define NO_RIGHT 32 /**< Indicates that the object can't go right (colldet).*/

#define IS_GOING_LEFT(X) (!(X->state & DIRECTION)) /**< Macro telling whether an object is going left.*/
#define IS_GOING_RIGHT(X) (X->state & DIRECTION) /**< Macro telling whether an object is going right.*/
#define NOW_GOING_LEFT(X) (X->state &= ~DIRECTION) /**< Macro to make an object go left.*/
#define NOW_GOING_RIGHT(X) (X->state |= DIRECTION) /**< Macro to make an object go right.*/

	/** \defgroup pl_obj_state Liveplayers state flags
	\ingroup obj_states
	*/
	/*@{*/
	#define ZL_PLAYER_IDLE 64 /**< Set when the player is idle */
	#define ZL_PLAYER_JUMPING 128 /**< Is the player currently jumping ?*/
	#define ZL_PLAYER_JPRESSED 256 /**< Is the jump key related to this player currently pressed ?*/
	#define ZL_PLAYER_RUNNING 512 /**< Is the player running ?*/
	#define ZL_PLAYER_INVINCIBLE 1024 /**< Is the player invincible ?*/
	#define ZL_SPRITE_ANIMATE 2048 /**< Must spritecycling be done ?*/
	#define SPR_REVERSE 4096 /**< Cycle sprites in the other direction*/
	#define ZL_PLAYER_LIVE_REFPFORM 8192 /**< Is the player standing on a liveplatform ?*/
	/*@}*/
	
	/** \defgroup lwl_obj_state Livewalls state flags
	\ingroup obj_states
	*/
	/*@{*/
	#define LWL_PAUSE_MVT 128 /**< Is the livewall currently paused ?*/
	#define LWL_REDRAW 256 /**< Must redraw the livewall on next cycle ? (due to sizevar)*/
	#define LWL_SZVAR_X 512 /**< Positive increment of size x*/ 
	#define LWL_SZVAR_Y 1024 /**< Positive increment of size y*/
	/*@}*/
	
	/** \defgroup portal_obj_state Portals state flags
	\ingroup obj_states
	*/
	/*@{*/
	#define INHIBIT_TELEPORTER 2 /**< Portal disabled. It will be displayed in blue.*/
	/*@}*/
/*@}*/

/** Speed divider value. This value, when added to a speed variable, means that the speed is inverted, e.g. 1/ (speed - ZL_DIVIDE_SPD). */
#define ZL_DIVIDE_SPD 128

/** Unused marker for a liveplayer */
#define ZL_PLAYER_UNUSED 0
/** Used by a player marker for a liveplayer */
#define ZL_PLAYER_HUMAN 1
/** Used by an enemy marker for a liveplayer */
#define ZL_PLAYER_ENNEMY 2

/** The part of the screen used for deadzone. The deadzone is the zone within the player can move without making the screen scroll. Its dimensions are expressed in percentage of the total length.*/
#define DEADZONE_RATE 30

/** \defgroup rcode Game return codes
*/
/*@{*/
#define ZL_PLAY_ON 0 /**< Continue playing */
#define ZL_QUIT 1 /**< Quit the game (asked by user) */
#define ZL_ENDLVL 2 /**< End of level reached*/
/*@}*/

#ifdef _WIN32
extern char win32_lockrun; /**< Under Win32, wait in the console before running the game */
#endif

/*do not alter what follows, it is parsed by a script !!!!!!*/
/*do not even touch the comments ! 
YOU HAVE BEEN WARNED */

/** A point.
	*/
typedef struct{
        unsigned int x,y;
        } point;

/** A speed (vector with real coordinates)
	*/
typedef struct{
        float x,y;
        } speed; /*basically, this is a vector*/

/** A star, described by its coordinates and a color
	*/
typedef struct{
        unsigned int x,y;
        short int color;
        } star;

/** The gamescreen, defined by its top left point, its bottom right point, the length and the width. 
	*/
typedef struct{
        point cadrehg,cadrebd;
        unsigned short int L, l;
        } gamescreen;

/** A sprite. A zl_sprite is an element of a linked list of sprites.
	*/
typedef struct zl_sprite_ {
        struct zl_sprite_ * NEXT;
        struct zl_sprite_ * PREV;
        BITMAP * spr;
        } zl_sprite;

/** A spriteset is a group of sprites
	*/
typedef zl_sprite ** sprset;

/** A soundset is a group of samples
	*/
typedef SAMPLE ** sndset;

/** A spriteset entry in the database. This is basically a structure containing a spriteset, a textual key, and the NEXT pointer. 
	*/
typedef struct {
        char key[25];
        sprset dataset;
        struct sprset_entry * NEXT;
        int type;
        } sprset_entry; /*an entry in the db looks like this*/
	
/** A soundset entry in the database. This is basically a structure containing a soundset, a textual key and the NEXT pointer.
	*/
typedef struct {
	char key[25];
	sndset dataset;
	struct sndset_entry * NEXT;
	int type;
	} sndset_entry;

/** Zignals
	*/
typedef struct {
        short int mask; /**< Bitmask listing what zignals were triggered.*/
        int parm[16][2]; /**< Associated parameters*/
        } zignals_t;

/** Effectors of an object
	*/
typedef struct{
        int is_active;
        int armed;
        } effectors_t;

/*do not alter what follows, it is parsed by a script !!!!!!*/
/*do not even touch the comments ! 
YOU HAVE BEEN WARNED */
/*-=AUTOCODE OBJECT LIST BEGINS HERE=-*/

/*-=_liveplayer_=-*/
/** A liveplayer is a human player or an enemy. */
typedef struct liveplayer_{
point refpt; /**< Reference point. The refpt is the top left point of the object, this is the one that is used for every computation. */
unsigned short int xsize; /**< The size of the object on the x axis. */
unsigned short int ysize; /**< The size of the object on the y axis. */
float base_jumping_speed; /**< The jumping speed of the player, not considering inertia and the like. */
float max_jumping_speed; /**< The speed of jumping the player can't exceed. */
short int state; /**< The bitfield of state flags. */
short int parms; /**< The bifield of parameters associated to the object. */
short int trigger_flagmask; /**< The bitfield of state flags that will be triggered by the effector TRIGGER_FLAG. */
float add_x; /**< The float part of the x coordinate of the reference point. */
float add_y; /**< The float part of the y coordinate of the reference point. */
float jtime; /**< The time for which the player has been jumping - jump is time-based in Zlog. */
speed current_spd; /**< The current speed, including inertia, by which the players moves. This is basically its speed vector. */
speed moving_spd; /**< The speed at which the player can move immediately. */
short int life; /**< Health points of the player. */
char name[25]; /**< The name of the player. */
char level_def[150]; /**< The line of the levelfile that defines this object. The purpose is to enable re-reading of this string to restore the object at its initial state. */
char used;  /**< Is this object used, and by what ? */
short int armenb; /**< The weapon number of the player. Currently useless. */
float inert_max_speed_walk; /**< Maximum inertia when walking. */
float inert_max_speed_run; /**< Maxmimum inertia when running. */
zignals_t zignals; /**< The zignals of the object. */
effectors_t effectors; /**< The effectors of the object. */
short int sprite_delay; /**<The time for which the current sprite has been displayed. */
char sprset_key[25]; /**< The name of the spriteset used by the liveplayer. */
char sndset_key[25]; /**< The name of the soundset used by the liveplayer. */
/*-=ENDOFSTRUCT=-*/
/*AUTOCODE_SKIP*/void * ref_pform; /**< The platform or liveplatform on which the player currently stands */
/*AUTOCODE_SKIP*/struct liveplayer_ * target; /**< The current target of the liveplayer.  Positive value is the index of the player, plus one, negative value is the index of the enemy, minus one. */
/*AUTOCODE_SKIP*/zl_sprite * current_sprite; /**< A pointer to the current sprite used by the liveplayer. */
/*AUTOCODE_SKIP*/sprset spriteset; /**< The spriteset used by the liveplayer. */
/*AUTOCODE_SKIP*/sndset soundset; /**< The soundset used by the liveplayer. */
/*AUTOCODE_SKIP*/struct liveplayer_ * NEXT; /**< Next player in the linked list, NULL if tail. */
/*AUTOCODE_SKIP*/struct liveplayer_ *  PREV; /**< Previous player in the list, NULL if head. */
}liveplayer;


/*this is a duplicate that is to be used for autocode. Sorry for the mess*/
/*-=_enemy_=-*/
/*typedef struct{
point refpt; 
unsigned short int xsize;
unsigned short int ysize;
float base_jumping_speed;
float max_jumping_speed;
short int state;
short int parms;
short int trigger_flagmask;
float add_x;
float add_y; 
float jtime;
speed current_spd;
speed moving_spd;
short int life;
char name[25];
char level_def[150];
char used; 
short int armenb;
float inert_max_speed_walk;
float inert_max_speed_run;
zignals_t zignals;
effectors_t effectors;
short int sprite_delay;
char sprset_key[25];
char sndset_key[25];
}liveplayer;*/
/*-=ENDOFSTRUCT=-*/

/*-=_livewall_=-*/
/** A livewall is a platform with special features : it can move, have a variable size, and reacts to environment. */
typedef struct livewall_{
point refpt; /**< Reference point. The refpt is the top left point of the object, this is the one that is used for every computation. */
float add_x; /**< The float part of the x coordinate of the reference point. */
float add_y; /**< The float part of the y coordinate of the reference point. */
unsigned short int xsize; /**< The size of the object on the x axis. */
unsigned short int ysize; /**< The size of the object on the y axis. */
short int state; /**< The bitfield of state flags. */
short int parms; /**< The bifield of parameters associated to the object. */
short int trigger_flagmask; /**< The bitfield of state flags that will be triggered by the effector TRIGGER_FLAG. */
unsigned short int speed; /**< Speed of moving. Can be inverted using ZL_DIVIDE_SPD. */
unsigned short int idling_for; /**< The number of cycles by which the livewall has been idling. */
char dx; /**< Moving instruction for next cycle, x axis. Can be 0 = don't move, 1 = go right, -1 = go left. */
char dy; /**< Moving instruction for next cycle, y axis. Can be 0 = don't move, 1 = go down, -1 = go up. */
point base_mvt; /**< Point where the movement begins (top left point). */
short int mvt_xsize;  /**< The xsize of the movement. */
short int mvt_ysize; /**< The ysize of the movement. */
float resist_inert; /**< Resistance to player inertia. */
speed sizevarspd; /**< Size variation speed, in full variation cycles by game loop. */
point max_size; /**< Maximum size of the object - obviously not really a point.*/
point min_size; /**< Minimum size of the object - obviously not really a point.*/
float size_add_x; /**< The float part of the x size. */
float size_add_y; /**< The float part of the y size. */
zignals_t zignals; /**< Zignals of the object. */
effectors_t effectors; /**< Effectors of the object. */
char template; /**< Template type of the livewall. FIXME : remove. */
char spriteset[25]; /**< The name of the spriteset used by the object. */
/*-=ENDOFSTRUCT=-*/
/*AUTOCODE_SKIP*/BITMAP * sprite; /**< A pointer to the (unique) central sprite of the livewall. */
/*AUTOCODE_SKIP*/BITMAP ** edgeleft; /**< A pointer to the left edge sprite of the livewall. */
/*AUTOCODE_SKIP*/BITMAP ** edgeright; /**< A pointer to the right edge sprite of the livewall. */
/*AUTOCODE_SKIP*/struct livewall_ * NEXT; /**< The next livewall in the linked list, NULL if tail. */
/*AUTOCODE_SKIP*/struct livewall_ * PREV; /**< The previous livewall in the linked list, NULL if head. */
}livewall;


/*-=_wall_=-*/
/** A wall is a standard platform. */
typedef struct wall_{
point refpt;  /**< Reference point. The refpt is the top left point of the object, this is the one that is used for every computation. */
unsigned short int xsize; /**< The size of the object on the x axis. */
unsigned short int ysize; /**< The size of the object on the y axis. */
float resist_inert; /**< Resistance to player inertia. */
short int state; /**< The bitfield of state flags. */
short int parms; /**< The bifield of parameters associated to the object. */
char spriteset[25]; /**< The name of the spriteset used by the object. */
/*-=ENDOFSTRUCT=-*/
/*AUTOCODE_SKIP*/BITMAP * sprite; /**< A pointer to the (unique) sprite of the wall. */
/*AUTOCODE_SKIP*/struct wall_ * NEXT; /**< The next wall in the list. */
/*AUTOCODE_SKIP*/struct wall_ * PREV; /**< The previous wall in the list. */
}wall;


/*-=_killing_line_=-*/
/** A killing line is a line that kills the player when he falls behind it. */
typedef struct kline_{
point position; /**< The position of the left point of the killing line. */
int taille; /**< The size of the kline. */
short int state; /**< The state bitfield for the kline. */
char to_disp; /**< A mere switch to enable displaying of the kline. FIXME : to remove. */
/*AUTOCODE_SKIP*/struct kline_ * NEXT; /**< Next kline*/
/*AUTOCODE_SKIP*/struct kline_ * PREV; /**< Prev kline */
}killing_line;
/*-=ENDOFSTRUCT=-*/

/*-=_portal_=-*/
/** A portal is a teleporter to another teleporter in the same level, or in another level. */
typedef struct portal_{
point position; /**< The top left point of the teleporter. */
unsigned short int xsize; /**< The size, on the x axis, of the teleporter. */
unsigned short int ysize; /**< The size, on the y axis, of the teleporter. */
char state; /**< The state bitfield for the teleporter. */
char id; /**< The level-unique ID of the teleporter. */
short int destination; /**< The destination code of the teleporter, composed as follows : the 8 weakest bits are the destination portal ID, the 8 strongest bits are the destination level ID. */
/*AUTOCODE_SKIP*/struct portal_ * NEXT; 
/*AUTOCODE_SKIP*/struct portal_ * PREV;
}portal;
/*-=ENDOFSTRUCT=-*/

/*-=_madmaker_=-*/
/** */
typedef struct madmaker_ {
point refpt;
unsigned short int speed; /**< Speed of moving. Can be inverted using ZL_DIVIDE_SPD. */
unsigned short int idling_for; /**< The number of cycles by which the livewall has been idling. */
/*AUTOCODE_SKIP*/struct madmaker_ * NEXT; 
}madmaker;
/*-=ENDOFSTRUCT=-*/

/*-=_banana_=-*/
typedef struct banana_ {
point refpt;
/*AUTOCODE_SKIP*/struct banana_ * NEXT; 
/*AUTOCODE_SKIP*/struct banana_ * PREV; 
/*AUTOCODE_SKIP*/sprset spriteset; /**< The spriteset used by the liveplayer. */
}banana;

/*-=ENDOFSTRUCT=-*/
/** The object database. */
/*-=OBJ DATABASE BEGINS=-*/
typedef liveplayer enemy;
typedef struct {
liveplayer * liveplayer_head;
enemy  * enemy_head;
wall * wall_head;
livewall * livewall_head;
killing_line * killing_line_head;
portal * portal_head;
madmaker * madmaker_head;
banana * banana_head;
point screen_pos;
} basededonnees;
/*-=AUTOCODE OBJECT LIST ENDS HERE=-*/


/***************
structures
**************/

gamescreen zlogframe;
basededonnees obj_db; /* = {NULL, NULL, NULL, NULL, NULL, NULL, {0, 0} };*/
liveplayer * cur_player;
/**************
objets globaux
***************/
BITMAP *ecranv[2]; /**< The virtual screen. The two bitmaps are created when page flipping (default), double buffering uses only one. */
BITMAP *game_frame[2]; /**< The sub-bitmaps pointing to the game frame of the virtual screen bitmaps. */
star * stars; /**< The stars of the background. */
int nombre_stars; /**< Number of stars currently drawn in the background. */
int MODE_DEBUG_ACTIF; /**< Tells whether debugging mode is active. */
int CONSOLE_IS_ACTIVE; /**< Tells whether the console is active. */
int perso_to_debug; /**< Tells which player we want additionnal info of. */
int pform_to_debug; /**< Tells which livewall we want additionnal info of. */
short int *pl_kbinds[15]; /**< The keybindings of players. */
short int pl_with_kbinds; /**< The number of players who have keybindings FIXME : check if still used*/
short int pltoggle_ispressed_flag; /**< A bitfield to keep trace of the keys that toggle something. Used to make the toggling user friendly. */
short int lvltoggle_ispressed_flag; /**< A bitfield to keep trace of the keys that toggle something in the leveleditor input loop. */
int delay_to_deadzone; /**< Remaining time before, in deadzone mode, the screen is centered. */

int bpp; /**< The bit per pixel setting. */
int background_offset_x; /**< The offset from which to display the background tile, X axis. */
float add_bgoffsetx, add_bgoffsety; /**< The X and Y float parts of the background offset. */
int background_offset_y; /**< The offset from which to display the background tile, Y axis. */
extern unsigned int RESOL_X; /**< The X resolution of the screen. */
extern unsigned int RESOL_Y; /**< The Y resolution of the screen. */
extern unsigned int GAMESCR_RESOL_X; /**< The width of the game screen. */
extern unsigned int GAMESCR_RESOL_Y; /**< The height of the game screen. */

extern int page_num; /**< The page number to use, when doing pageflipping. */

int cur_level;
MIDI * music;
/** \defgroup color Color Codes
*/
/*@{*/
int RED;
int BLUE;
int GREEN;
int WHITE;
int GREY;
int CYAN;
int YELLOW;
int DARKRED;
/*@}*/

int fontsize; /** The height of the font. */
extern char config_file[250]; /**< Configuration filepath */
extern char campaign_file[250]; /**< Campaign filepath */

extern float gravite; /**< Gravity value */

enum {
	NO_PAGE_FLIP = 0,
	NO_DEADZONE = 1,
	LVLEDIT_MODE = 2,
	LVLEDIT_TESTLEVEL = 3,
};
#define TEST_GLOBAL_STATE(X) ((global_stateflags & (1 << X)))
#define SET_GLOBAL_STATE(X) (global_stateflags |= (1 << X))
#define CLEAR_GLOBAL_STATE(X) (global_stateflags &= ~(1 << X))

extern volatile int global_stateflags; /*defined in zlog.h*/
int alloue_objdb(); /* memory allocation for the objects database - returns number of bytes allocated*/


liveplayer * del_player_from_database(liveplayer * head, liveplayer * todelete);
wall * del_wall_from_database(wall * head, wall * todelete);
livewall * del_livewall_from_database(livewall * head, livewall * todelete);
killing_line * del_killing_line_from_database(killing_line * head, killing_line * todelete);
portal * del_portal_from_database(portal * head, portal * todelete);
banana * del_banana_from_database(banana * head, banana * todelete);
int free_player_database(liveplayer * head);
int free_wall_database(wall * head);
int free_livewall_database(livewall * head);
int free_killing_line_database(killing_line * head);
int free_portal_database(portal * head);
int free_banana_database(banana * head);
liveplayer * add_player_to_database(liveplayer * head, liveplayer * toadd);
wall * add_wall_to_database(wall * head, wall * toadd);
livewall * add_livewall_to_database(livewall * head, livewall * toadd);
killing_line * add_killing_line_to_database(killing_line * head, killing_line * toadd);
portal * add_portal_to_database(portal * head, portal * toadd);
banana * add_banana_to_database(banana * head, banana * toadd);
#define GETNEXT(X) X->NEXT
#define GETPREV(X) X->PREV
#endif
