/*******************************************************************************
 * This is the Engine.  It finally has a sensical name. 
 * It is the main part of the game, and covers the title, game loop,
 * and game over.  It relies on the circle class, but other than that, this
 * file is the entire game.
 ******************************************************************************/
#include "Engine.h"

/**
 * Show the last replay saved
 **/
void Engine::show_replay(char *filename){
    ifstream last_replay;
    last_replay.open(filename, ios::in|ios::binary);
    if (last_replay.is_open() == false)
	return;
    int score;
    last_replay.read((char *)&score, sizeof(int));
    last_replay.read((char *)&replay_circles, sizeof(replay_circles));
    for (int i = 0; i < score + 5; i++){
	last_replay.read((char *)&replay_player[i][0], sizeof(int));
	last_replay.read((char *)&replay_player[i][1], sizeof(int));
    }
    last_replay.close();
    REPLAY = true;
    mainLoop();
}

/**
 * Play a game
 * */
int Engine::play_game(){
    REPLAY = false;
    return mainLoop();
}

/**
 * This is the main loop.  Not surprisingly, it repeats this
 * until the game is over.
 */
int Engine::mainLoop()
{
    init_game();
    while (gameOver == 0)
    {
	int start_tick = clock();
	clear_bitmap(buf);
	blit(bg, buf, 0, 0, 0, 0, SCN_WIDTH, SCN_HEIGHT);

	for (int i = 0; i < NUM_CIRCLES; i++)
	{
	    if (REPLAY == false){

		int x_move, y_move;
		get_mouse_mickeys(&x_move, &y_move);
		current_x = current_x + x_move;
		current_y = current_y + y_move;
		if (current_x < 0)
		    current_x = 0;
		if (current_x > SCN_WIDTH - MOUSE_SIZE)
		    current_x = SCN_WIDTH - MOUSE_SIZE;
		if (current_y < 0)
		    current_y = 0;
		if (current_y > SCN_HEIGHT - MOUSE_SIZE)
		    current_y = SCN_HEIGHT - MOUSE_SIZE;

		position_mouse(SCN_WIDTH / 2, SCN_HEIGHT / 2); 

		replay_player[score][0] = current_x;
		replay_player[score][1] = current_y;
	    } else {
		current_x = replay_player[score][0];
		current_y = replay_player[score][1];
	    }       

	    // Update and draw the circles, and bounce them off the border
	    theCircles[i]->update();
	    theCircles[i]->draw(buf);
	    if (theCircles[i]->getLeft() < 0 || theCircles[i]->getRight() > SCN_WIDTH)
		theCircles[i]->collide(0);
	    if (theCircles[i]->getTop() < 0 || theCircles[i]->getBottom() > SCN_HEIGHT)
		theCircles[i]->collide(1);

	    // Tell the circles where the mouse is.  If they care, they will
	    // act on it.  If they're zombies, they'll ignore it
	    theCircles[i]->lean(current_x, current_y);

	    gameOver += theCircles[i]->testCollide(current_x, current_y, MOUSE_SIZE);
	}

	stretch_sprite(buf, mouseBmp, current_x, current_y, MOUSE_SIZE, MOUSE_SIZE);
	sprintf(scoreString, "Score: %d", score);
	textout_centre_ex(buf, font, scoreString, SCN_WIDTH / 2, 10, 0xffffff, -1);

	levelCounter++;
	score++;

	if (levelCounter > toNextLevel){   
	    levelCounter = 0;
	    NUM_CIRCLES++;
	    if (NUM_CIRCLES > MAX_CIRCLES)
		NUM_CIRCLES = MAX_CIRCLES;

	    if (REPLAY == false){
		replay_circles[NUM_CIRCLES - 1][AI_TYPE] = ZOMBIE;
		replay_circles[NUM_CIRCLES - 1][CIR_X] = rand() % 50 + 10;
		replay_circles[NUM_CIRCLES - 1][CIR_Y] = rand() % 50 + 10;
		replay_circles[NUM_CIRCLES - 1][CIR_SIZE] = rand() % 30 + 20;
		replay_circles[NUM_CIRCLES - 1][CIR_X_VEL] = rand() % 20 - 10;
		replay_circles[NUM_CIRCLES - 1][CIR_Y_VEL] = rand() % 20 - 10;
	    }

	    theCircles[NUM_CIRCLES - 1] = new Circle(replay_circles[NUM_CIRCLES - 1][AI_TYPE], 
		    replay_circles[NUM_CIRCLES - 1][CIR_X], 
		    replay_circles[NUM_CIRCLES - 1][CIR_Y], 
		    replay_circles[NUM_CIRCLES - 1][CIR_SIZE], 
		    replay_circles[NUM_CIRCLES - 1][CIR_X_VEL], 
		    replay_circles[NUM_CIRCLES - 1][CIR_Y_VEL]);
	}

	vsync();
	while (clock() < start_tick + DELAY)
	    ;
	blit(buf, screen, 0, 0, 0, 0, SCN_WIDTH, SCN_HEIGHT);
    }
    return score;
}

/**
 * This just displays the title screen and waits for the any key.
 */
void Engine::intro()
{
    FSOUND_STREAM *intro_music = FSOUND_Stream_Open("Music//authentically_chinese.ogg", FSOUND_HW2D, 0, 0);
    FSOUND_Stream_Play(FSOUND_FREE, intro_music);
    FSOUND_SetVolume(FSOUND_ALL, 128);
    while (keypressed())
	readkey();
    while (!keypressed())
	blit(intro_bmp, screen, 0, 0, 0, 0, SCN_WIDTH, SCN_HEIGHT);
    while (keypressed())
	readkey();
    FSOUND_Stream_Stop(intro_music);
}

/**
 * This returns the background image.
 */
BITMAP *Engine::getBackGround(){   
    BITMAP *foo = load_bitmap("gfx/bg.jpg", NULL);
    if (foo == NULL){
	allegro_message("Could not load gfx/bg.jpg");
	allegro_exit();
    }
    return foo;	
}

/**
 * This is the display for when you lose the game.  It also displays the
 * high scores list, and if the player achieved one, asks for name entry
 * and adds it to the high score table.
 */
int Engine::death(int score)
{
    clear_keybuf();
    char scoreString[64];

    // Save the replay of the game just played
    if (REPLAY == false){
	ofstream last_replay;
	last_replay.open("last_replay.rep", ios::out|ios::binary);
	last_replay.write((char *)&score, sizeof(int));
	last_replay.write((char *)&replay_circles, sizeof(replay_circles));
	for (int i = 0; i < score + 5; i++){
	    last_replay.write((char *)&replay_player[i][0], sizeof(int));
	    last_replay.write((char *)&replay_player[i][1], sizeof(int));
	}
	last_replay.close();
	
	ifstream best_replay;
	best_replay.open("best_replay.rep", ios::in|ios::binary);
	int best_score;
	if (best_replay.is_open()){
	    best_replay.read((char *)&best_score, sizeof(int));
	    best_replay.close();
	} else
	    best_score = 0;
	
	if (score > best_score){
	    last_replay.open("best_replay.rep", ios::out|ios::binary);
	    last_replay.write((char *)&score, sizeof(int));
	    last_replay.write((char *)&replay_circles, sizeof(replay_circles));
	    for (int i = 0; i < score + 5; i++){
		last_replay.write((char *)&replay_player[i][0], sizeof(int));
		last_replay.write((char *)&replay_player[i][1], sizeof(int));
	    }
	    last_replay.close();
	}
    }

    sprintf(scoreString, "Final Score: %d", score);
    blit(entro_bmp, buf, 0, 0, 0, 0, SCN_WIDTH, SCN_HEIGHT);
    textout_centre_ex(buf, font, "Play again?  [Y]es / [N]o / [R]eplay", SCN_WIDTH / 2, SCN_HEIGHT - 30, 0xffffff, -1);
    textout_centre_ex(buf, font, scoreString, SCN_WIDTH / 2, SCN_HEIGHT - 20, 0xffffff, -1);
    blit(buf, screen, 0, 0, 0, 0, SCN_WIDTH, SCN_HEIGHT); 
    while(1){
	if (key[KEY_R]){
	    REPLAY = true;
	    return true;
	}
	if (key[KEY_N])
	    return false;
	if (key[KEY_Y]){
	    REPLAY = false;
	    return true;
	}
    }
}

/**
 * This initializes the allegro library and all of the subsystems of it used
 * in the game
 */
void Engine::init()
{
    REPLAY = false;

    bg = getBackGround();
    mouseBmp = initMouse();	
    buf = create_bitmap(SCN_WIDTH, SCN_HEIGHT);
    intro_bmp = load_bitmap("gfx//intro.jpg", NULL);
    if (intro_bmp == NULL){
	allegro_message("Could not load gfx/intro.jpg");
	allegro_exit();
    }
    entro_bmp = load_bitmap("gfx//entro.jpg", NULL);
    if (entro_bmp == NULL){
	allegro_message("Could not load entro/bg.jpg");
	allegro_exit();
    }
    myGlobals = Globals::Instance();
}

/**
 * Constructor
 */
Engine::Engine(){
    init();	
}

/**
 * Initializes the game.  Done at the beginning of each play.
 */
void Engine::init_game(){
    NUM_CIRCLES = 1;
    levelCounter = 0;
    score = 0;
    gameOver = 0;

    if (REPLAY == false) {
	replay_circles[0][AI_TYPE] = ORBIT;
	replay_circles[0][CIR_X] = rand() % 50 + 10;
	replay_circles[0][CIR_Y] = rand() % 50 + 10;
	replay_circles[0][CIR_SIZE] = rand() % 30 + 20;
	replay_circles[0][CIR_X_VEL] = rand() % 20 - 10;
	replay_circles[0][CIR_Y_VEL] = rand() % 20 - 10;
    }

    theCircles[0] = new Circle(replay_circles[0][AI_TYPE], 
	    replay_circles[0][CIR_X], 
	    replay_circles[0][CIR_Y], 
	    replay_circles[0][CIR_SIZE],
	    replay_circles[0][CIR_X_VEL],	
	    replay_circles[0][CIR_Y_VEL]);


    current_x = SCN_WIDTH - 50;
    current_y = SCN_HEIGHT - 50;	
}


/**
 * Installs the mouse driver and loads the sprite for it
 */
BITMAP *Engine::initMouse()
{
    install_mouse();

    BITMAP *answer = load_bitmap("gfx//MousePointer.bmp", NULL);
    set_mouse_range(0, 0, SCN_WIDTH - MOUSE_SIZE, SCN_HEIGHT - MOUSE_SIZE);
    set_mouse_speed(MOUSE_SPEED, MOUSE_SPEED);
    return answer;
}
