// *******************************************
// Project:     Tetrist
// File:        game
// Desc:        main file for this simple puzzle game
// Author:      el Hamil, elhamil@hotmail.com
// Environment: Dev-C++, Allegro 4, DirectX 8, win98
// Created:     2000-12-31, originally for DOS and mode 13h
// Updates:
//   2003-04-12, converted to Windows with Dev-C++
//   2003-04-13, minor graphics optimizations
// *******************************************


#include "common.h"
#include "global.h"
#include "graphics.h"
#include "font.h"
#include "shape.h"

#include <allegro.h>
#include <winalleg.h>
#include <windows.h>
#include <assert.h>
#include <fstream>
#include <stdlib.h>
#include <stdio.h>


using namespace std;


// variables changed during the flow of the game
uchar	currentShape;
uchar	nextShape;
uchar	angle;
char	posX;
char	posY;
float	dropSpeed;
float	repeatDelay;
uchar	spinKeyState;
ulong	nrLines;
ulong	score;
uchar	isdead;
uchar	runGame;

char*	textbuffer = 0;
uchar*	world      = 0;
short**	worldX     = 0;
short**	worldY     = 0;

LARGE_INTEGER startTime;
LARGE_INTEGER lastTime;
LARGE_INTEGER thisTime;
LARGE_INTEGER deltaTime;
LARGE_INTEGER counterFrequency;
double        inverseFrequency;
double        deltaSeconds   = 0;
ulong         deltaLong      = 0;
long          framesPerSecond= 0;
float         averageFPS     = 0;
LARGE_INTEGER totalTime;
double        totalSeconds   = 0;
ulong         totalFrames    = 0;
uchar         itsTimeToDrop  = 0;
double        droptimer      = 0; // keeps track of the above


void
MessUpCoordinates()
{
	world = new uchar[MAPW * MAPH];
	assert(world);

	// create the silly coordinates for the squares in the playfield
	worldX = new short*[MAPH + 1];
	assert(worldX);
	worldY = new short*[MAPH + 1];
	assert(worldY);

	for (short y = 0; y <= MAPH; y++)
	{
		worldX[y] = new short[MAPW + 1];
		assert(worldX[y]);
		worldY[y] = new short[MAPW + 1];
		assert(worldY[y]);

		char whichway = (rand() % 5) - 2;
		worldY[y][0] = (y << 3) - whichway;
		whichway = NORM(whichway);
		for (short x = 1; x <= MAPW; x++)
		{
			worldY[y][x] = worldY[y][x - 1];
			if (!(rand() % 5))
				worldY[y][x] += whichway;
			if (y)
			{
				if (worldY[y][x] <= worldY[y - 1][x] + 3)
				{
					worldY[y][x] = worldY[y - 1][x] + 4;
					switch (rand() % 3)
					{
						case 0:
							whichway = 0;
							break;
						case 1:
							whichway = -whichway;
							break;
						default:
							break;
					}
				}
			}
		}
	}

	for (short x = 0; x <= MAPW; x++)
	{
		char whichway = (rand() % 5) - 2;
		worldX[0][x] = (x << 3) - whichway;
		whichway = NORM(whichway);
		for (short y = 1; y <= MAPH; y++)
		{
			worldX[y][x] = worldX[y - 1][x];
			if (!(rand() % 5))
				worldX[y][x] += whichway;
			if (x)
			{
				if (worldX[y][x] <= worldX[y][x - 1] + 3)
				{
					worldX[y][x] = worldX[y][x - 1] + 4;
					switch (rand() % 3)
					{
						case 0:
							whichway = 0;
							break;
						case 1:
							whichway = -whichway;
							break;
						default:
							break;
					}
				}
			}
		}
	}
}


void
InitGame()
{
	assert(allegro_init() == 0);

	InitGraphics();

	LoadFont();

	InitShapes();

	// initialize random thingy
	assert(QueryPerformanceCounter(&startTime));
	srand(startTime.QuadPart);

	MessUpCoordinates();

	textbuffer = new char[10];
	assert(textbuffer);

	assert(QueryPerformanceFrequency(&counterFrequency));
	inverseFrequency = 1 / (double)(counterFrequency.QuadPart);

	assert(install_keyboard() == 0);

	runGame = 1;
}


void
ShutdownGame()
{
	delete[] world;
	world = 0;

	delete[] textbuffer;
	textbuffer = 0;

	ShutdownGraphics();
}


void TitleScreen()
{
	ifstream fin("title.raw");

	if (fin.good())
	{
		char color;
		for (long y = 0; y < 200; y++)
		{
			for (long x = 0; x < 320; x++)
			{
				fin.get(color);
				PutPixel(x, y, color - 48);
			}
		}
	}
	else
	{
		ClearVirscr(16843009); // (((((((1 << 8) + 1) << 8) + 1) << 8) + 1)

//		PrintString("T E T R I S T", 115, 80, 1, 0);
//		PrintString("PRESS ENTER TO PLAY", 100, 100, 1, 2);
		textprintf_centre(virscr, font, SCREENW / 2, 80, 0, "T E T R I S T");
		textprintf_centre(virscr, font, SCREENW / 2, 100, 2, "PRESS ENTER TO PLAY");
	}

	fin.close();

	Blit();

	uchar ESCpressed = 1;

	while (1)
	{
		if (ESCpressed)
		{
			if (!(key[KEY_ESC] || key[KEY_Q] || key[KEY_X] || (key[KEY_ALT] && key[KEY_F4])))
				ESCpressed = 0;
		}
		else
		{
			if (key[KEY_ESC] || key[KEY_Q] || key[KEY_X] || (key[KEY_ALT] && key[KEY_F4]))
			{
				runGame = 0;
				break;
			}
		}

		if (key[KEY_ENTER] || key[KEY_SPACE])
		{
			break;
		}
	}
}


void
InitRound()
{
	currentShape = rand() % NRSHAPES;
	nextShape = rand() % NRSHAPES;
	angle = 0;
	score = 0;
	posX = 4;
	posY = 0;
	dropSpeed = 1.5;
	repeatDelay = 0;
	spinKeyState = 0;
	nrLines = 0;
	isdead = 0;
	droptimer = 0;

	// clear the map
	for (short y = 0; y < MAPH; y++)
	{
		for (short x = 0; x < MAPW; x++)
		{
			world[y * MAPW + x] = 0;
		}
	}
}


char
TryPosition(char xpos, char ypos)
{
	// checks if the current shape can exist at the specified position,
	// or if it's already occupied

	for (char y = 0; y < 4; y++)
	{
		for (char x = 0; x < 4; x++)
		{
			if (Shape[currentShape][angle][(y << 2) + x])
			{
				if (xpos + x < 0 || xpos + x > (MAPW - 1))
				{
					return 0;
				}
				if (ypos + y >= 0)
				{
					if (ypos + y >= MAPH)
					{
						return 0;
					}
					if (world[(ypos + y) * MAPW + x + xpos])
					{
						return 0;
					}
				}
			}
		}
	}
	return 1;
}


void
MainGameLoop()
{
	QueryPerformanceCounter(&startTime);
	lastTime = startTime;

	while (!isdead)
	{
		// timer stuff

//		thisTime = uclock();
		QueryPerformanceCounter(&thisTime);
//		deltaTime = thisTime - lastTime;
		deltaTime.QuadPart = thisTime.QuadPart - lastTime.QuadPart;
		lastTime = thisTime;
//		totalTime = thisTime - startTime;
		totalTime.QuadPart = thisTime.QuadPart - startTime.QuadPart;
//		deltaSeconds = deltaTime / (double)UCLOCKS_PER_SEC;
//		totalSeconds = totalTime / (double)UCLOCKS_PER_SEC;
		deltaSeconds = deltaTime.QuadPart * inverseFrequency;
		totalSeconds = totalTime.QuadPart * inverseFrequency;
		deltaLong = (ulong)(deltaSeconds * 0xffffffff);
		totalFrames++;
		framesPerSecond = (long)(1 / deltaSeconds);
		averageFPS = totalFrames / totalSeconds;


		uchar graphicsChanged = 0;


		itsTimeToDrop = (int)(droptimer + (deltaSeconds * dropSpeed));
		if (itsTimeToDrop)
		{
			droptimer = 0;
		}
		droptimer += (deltaSeconds * dropSpeed);

		if (repeatDelay > 0)
		{
			repeatDelay -= deltaSeconds;
		}


		// user input

		char xmove = 0;
		char ymove = 0;

		if (repeatDelay <= 0) // move only if we haven't moved for some time
		{
			if (key[KEY_RIGHT])
			{
				xmove = 1;
			}

			if (key[KEY_LEFT])
			{
				xmove = -1;
			}

			if (key[KEY_DOWN])
			{
				ymove = 1;
				repeatDelay = 0.100;
			}
		}

		if (key[KEY_ENTER] || key[KEY_SPACE])
		{
			spinKeyState = MIN(spinKeyState + 1, 2);
		}
		else
		{
			spinKeyState = 0;
		}

		if (key[KEY_ESC] || key[KEY_Q] || key[KEY_X] || (key[KEY_ALT] && key[KEY_F4]))
		{
			break;
		}

		if (key[KEY_P])
		{
			PALETTE pal;
			get_palette(pal);
			BITMAP* bmp = create_bitmap_ex(8, 320, 200);

			for (long y = 0; y < 200; y++)
			{
				for (long x = 0; x < 320; x++)
				{
					uchar color243 = GetPixel(x, y);
					putpixel(bmp, x, y, color243);
				}
			}

			save_bitmap("scrdump.pcx", bmp, pal);
			destroy_bitmap(bmp);
		}


		// updating everything

		if (itsTimeToDrop)
		{
			ymove = 1;
		}

		if (spinKeyState == 1)
		{
			graphicsChanged = 1;

			uchar oldangle = angle;
			angle = (angle + 1) % 4;

			// make sure we don't spin into a solid place
			if (!TryPosition(posX, posY))
			{
				if (TryPosition(posX - 1, posY))
				{
					posX--;
				}
				else
				{
					if (TryPosition(posX + 1, posY))
					{
						posX++;
					}
					else
					{
						if (TryPosition(posX - 2, posY))
						{
							posX -= 2;
						}
						else
						{
							if (TryPosition(posX + 2, posY))
							{
								posX += 2;
							}
							else
							{
								angle = oldangle;
							}
						}
					}
				}
			}
		}

		if (xmove)
		{
			if (TryPosition(posX + xmove, posY))
			{
				graphicsChanged = 1;
				repeatDelay = 0.100;
				posX += xmove;
			}
		}

		if (ymove)
		{
			graphicsChanged = 1;
			posY++;

			if (!TryPosition(posX, posY))
			{
				posY--;

				// assimilate the current shape into the world...
				for (char y = 0; y < 4; y++)
				{
					for (char x = 0; x < 4; x++)
					{
						if (Shape[currentShape][angle][(y << 2) + x])
						{
							if (posY + y >= 4)
							{
								world[(posY + y) * MAPW + x + posX] = 2;
							}
							else
							{
								isdead = 1;
							}
						}
					}
				}

				currentShape = nextShape;
				nextShape = rand() % NRSHAPES;
				posX = 4;
				posY = 0;
				repeatDelay = 0;
				spinKeyState = 0;
				angle = 0;

				// loop through the world and look for full lines
				for (char y = 0; y < MAPH; y++)
				{
					uchar fullLine = 1;
					for (char x = 0; x < MAPW; x++)
					{
						if (!world[y * MAPW + x])
						{
							fullLine = 0;
							break;
						}
					}

					if (fullLine)
					{
//						cout << "\a";
//						cout.flush();
						MessageBeep(0xFFFFFFFF);

						nrLines++;
//						score;
						if (nrLines / 10 > (nrLines - 1) / 10)
							dropSpeed += 1;

						// move everyhing above down one step
						for (char y2 = y; y2 > 1; y2--)
						{
							for (char x = 0; x < MAPW; x++)
							{
								world[y2 * MAPW + x] = world[(y2 - 1) * MAPW + x];
							}
						}
					}
				}
			}
		}


		// drawing

		if (graphicsChanged) // reduces flickering a lot :D
		{
			ClearVirscr(16843009);

			for (short y = 0; y < MAPH; y++)
			{
				short lastx = worldX[y][0] + PLAYFIELDPOSX;
				short lasty = worldY[y][0] + PLAYFIELDPOSY;
				for (short x = 1; x <= MAPW; x++)
				{
					short thisx = worldX[y][x] + PLAYFIELDPOSX;
					short thisy = worldY[y][x] + PLAYFIELDPOSY;
//					DrawLine(lastx, lasty, thisx, thisy, 245);
//					DrawLine(thisx, thisy, worldX[y + 1][x] + PLAYFIELDPOSX, worldY[y + 1][x] + PLAYFIELDPOSY, 245);
					DrawLine(lastx, lasty, thisx, thisy, 3);
					DrawLine(thisx, thisy, worldX[y + 1][x] + PLAYFIELDPOSX, worldY[y + 1][x] + PLAYFIELDPOSY, 3);
					lastx = thisx;
					lasty = thisy;
				}
			}

			for (short y = 0; y < MAPH; y++)
			{
				for (short x = 0; x < MAPW; x++)
				{
					if (world[y * MAPW + x])
					{
						DrawSquare(x, y, PLAYFIELDPOSX, PLAYFIELDPOSY, world[y * MAPW + x]);
					}
				}
			}

			Shape[nextShape].DrawAsNextShape();

			Shape[currentShape].DrawShape(posX, posY, angle);

			DrawLine(worldX[0][0] + PLAYFIELDPOSX, worldY[0][0] + PLAYFIELDPOSY, worldX[MAPH][0] + PLAYFIELDPOSX, worldY[MAPH][0] + PLAYFIELDPOSY, 2);
			DrawLine(worldX[0][0] + PLAYFIELDPOSX, worldY[0][0] + PLAYFIELDPOSY, worldX[0][MAPW] + PLAYFIELDPOSX, worldY[0][MAPW] + PLAYFIELDPOSY, 2);
			DrawLine(worldX[4][0] + PLAYFIELDPOSX, worldY[4][0] + PLAYFIELDPOSY, worldX[4][MAPW] + PLAYFIELDPOSX, worldY[4][MAPW] + PLAYFIELDPOSY, 2);
			DrawLine(worldX[0][MAPW] + PLAYFIELDPOSX, worldY[0][MAPW] + PLAYFIELDPOSY, worldX[MAPH][MAPW] + PLAYFIELDPOSX, worldY[MAPH][MAPW] + PLAYFIELDPOSY, 2);
			DrawLine(worldX[MAPH][0] + PLAYFIELDPOSX, worldY[MAPH][0] + PLAYFIELDPOSY, worldX[MAPH][MAPW] + PLAYFIELDPOSX, worldY[MAPH][MAPW] + PLAYFIELDPOSY, 2);

			sprintf(textbuffer, "LINES: %d", nrLines);
			PrintString(textbuffer, 2, 2, 1, 2);
			textprintf_right(virscr, font, SCREENW, SCREENH - 10, 3, "Copyright 2000 el Hamil");

			Blit();

		} //end if (graphicsChanged)

		Sleep(1); // play nice with multitasking
	}
}


int
main()
{
	InitGame();

	while (runGame)
	{
		TitleScreen();

		InitRound();

		MainGameLoop();
	}

	ShutdownGame();

	return 0;
}

END_OF_MAIN();

