//============================================================================
// tiledmap.h - Contains basic tiled map related classes
// By Bjrn Lindeijer
//============================================================================

#ifndef _INCLUDED_TILEDMAP_H_
#define _INCLUDED_TILEDMAP_H_

#include <allegro.h>
#include <map>
#include <list>
#include <vector>

using namespace std;


// Point class ===============================================================

class Point {
public:
	Point() {x = y = z = 0;}
	Point(int X, int Y, int Z = 0) {x = X; y = Y; z = Z;}
	Point(Point p, int Z) {x = p.x; y = p.y; z = Z;}
	int x, y, z;
};


// Rectangle class ===========================================================

class Rectangle {
public:
	Rectangle() {x = y = w = h = 0;}
	Rectangle(int X, int Y, int W, int H) {x = X; y = Y; w = W; h = H;}
	void rectToClip(BITMAP *dest);
	void clipToRect(BITMAP *src);
	int x, y, w, h;
};


// TileType ==================================================================
//  An object holding static information about a tile type.

class TileType {
public:
	TileType(BITMAP *tileBitmap, const char *tileName);
	~TileType();

	BITMAP* getBitmap() {return bitmap;}
	char*   getName()   {return name;}

protected:
	BITMAP* bitmap;
	char*   name;
};


// TileRepository ============================================================
//  A tile repository to handle a collection of tile types


// Comparator used for string indexed maps

struct ltstr {
	bool operator()(const char* s1, const char* s2) const {
		return strcmp(s1, s2) < 0;
	}
};

class TileRepository {
public:
	~TileRepository();

	void importDatafile(DATAFILE *dataFile);
	void importBitmap(const char *filename, int tile_w, int tile_h, int tile_spacing, int tiles_in_row);
	void exportBitmap(const char *filename, int tile_w, int tile_h, int tile_spacing, int tiles_in_row);

	TileType* getTileType(const char *tileName);
	vector<TileType*> generateTileArray();
protected:
	map<const char*, TileType*, ltstr> tileTypes;
};


// Tile class ================================================================

class Tile {
public:
	Tile();

	void saveTo(PACKFILE *file);
	void loadFrom(PACKFILE *file, TileRepository *tileRepository);

	void setType(TileType* tileType);
	TileType* getType() {return tileType;}

	bool obstacle;
protected:
	TileType* tileType;
};


// Entity class ==============================================================

class Entity {
public:
	Entity();
	bool visible(BITMAP *dest, Point screenCoords);
	void draw(BITMAP *dest, Point topLeft);

	Point pos;
	BITMAP* bitmap;
};

class EntityP {
public:
	EntityP(Entity *ent) {this->ent = ent;}
	Entity *ent;
	bool operator< (const EntityP& X) const {
		return (ent->pos.y < X.ent->pos.y);
	}
};


// TiledMap class ============================================================
//  Defines a generic tiled map interface and data model.


// Comparator used for determining entity draw order

struct greater_y : public binary_function<Entity*, Entity*, bool> {
	bool operator()(Entity* a, Entity* b) {
		return a->pos.y > b->pos.y;
	}
};

class TiledMap {
public:
	TiledMap();
	virtual ~TiledMap();

	// Map functions
	void resizeTo(int newWidth, int newHeight);
	void saveTo(PACKFILE* file);
	void loadFrom(PACKFILE* file, TileRepository *tileRepository);

	int getWidth()  {return mapWidth;}
	int getHeight() {return mapHeight;}

	// Tile and entity methods
	Tile* getTile(Point tileCoords);
	void addEntity(Entity* entity);
	void removeEntity(Entity* entity);

	// Draw the map
	virtual void setCamera(Point cameraCoords, Rectangle screenRect, bool centerCamera = false, bool modify = true);
	virtual void draw(BITMAP *dest, bool drawObstacle = false) = 0;

	// Coordinate space converters
	virtual Point screenToTile(Point screenCoords);
	virtual Point tileToScreen(Point tileCoords);
	virtual Point screenToMap(Point screenCoords) = 0;
	virtual Point mapToScreen(Point mapCoords) = 0;
	virtual Point mapToTile(Point mapCoords) = 0;
	virtual Point tileToMap(Point tileCoords) = 0;

	virtual Point getMapSize() = 0;

protected:
	// Internal functions
	void drawEntities(BITMAP *dest);

	// Double tile array
	Tile** tileMap;
	int mapWidth, mapHeight;

	// Entity list
	list<Entity*> entities;

	// Camera properties
	Point cameraCoords;
	Rectangle cameraScreenRect;
};


// SquareMap class ===========================================================
//  Provides algorithms for square-tiled maps

class SquareMap : public TiledMap {
public:
	SquareMap(int tileSize);
	SquareMap(int tileWidth, int tileHeight);

	// Draw the map
	virtual void draw(BITMAP *dest, bool drawObstacle = false);

	// Coordinate space converters
	virtual Point screenToMap(Point screenCoords);
	virtual Point mapToScreen(Point mapCoords);
	virtual Point mapToTile(Point mapCoords);
	virtual Point tileToMap(Point tileCoords);

	virtual Point getMapSize();

protected:
	int tileWidth, tileHeight;
};


// IsometricMap class ========================================================
//  Provides algorithms for isometric-tiled maps

class IsometricMap : public TiledMap
{
public:
	IsometricMap(int tileStepX, int tileStepY);

	// Draw the map
	virtual void draw(BITMAP *dest, bool drawObstacle = false);

	// Coordinate space converters
	virtual Point screenToMap(Point screenCoords);
	virtual Point mapToScreen(Point mapCoords);
	virtual Point mapToTile(Point mapCoords);
	virtual Point tileToMap(Point tileCoords);

	virtual Point getMapSize();

protected:
	int tileGridSize;
	int tileStepX, tileStepY;
};


#endif
