//============================================================================
// tiledmap.cpp - Contains implementations of classes defined in tiledmap.h
// By Bjrn Lindeijer
//============================================================================

#include "tiledmap.h"
#include <stdio.h>
#include <allegro.h>
#include <map>
#include <algorithm>


// Rectangle class ===========================================================

void Rectangle::rectToClip(BITMAP *dest)
{
	set_clip(dest, x, y, x + w - 1, y + h - 1);
}

void Rectangle::clipToRect(BITMAP *src)
{
	x = src->cl;
	y = src->ct;
	w = src->cr - src->cl + 1;
	h = src->cb - src->ct + 1;
}


// TileType ==================================================================
//  An object holding static information about a tile type.

TileType::TileType(BITMAP *tileBitmap, const char *tileName)
{
	bitmap = tileBitmap;
	name = (char*)malloc(ustrsizez(tileName));
	ustrcpy(name, tileName);
}

TileType::~TileType()
{
	destroy_bitmap(bitmap);
	free(name);
}


// TileRepository ============================================================
//  A tile repository to handle a collection of tile types

TileRepository::~TileRepository()
{
	// Remove tile types from memory
	map<const char*, TileType*, ltstr>::iterator i;
	while (!tileTypes.empty())
	{
		i = tileTypes.begin();
		TileType* tempTileType = (*i).second;
		tileTypes.erase(i);
		delete tempTileType;
	}
}

void TileRepository::importDatafile(DATAFILE *file)
{
	if (!file) return;

	TileType *tempTileType;
	BITMAP *tempBitmap;

	// Import bitmaps from the datafile
	while (file->type != DAT_END) {
		switch (file->type) {
		case DAT_FILE:
			// Go recursively into nested datafiles
			importDatafile((DATAFILE*)file->dat);
			break;
		case DAT_BITMAP:
			// Create a new tile type and add it to the hash_map
			tempBitmap = create_bitmap(((BITMAP*)file->dat)->w, ((BITMAP*)file->dat)->h);
			blit((BITMAP*)file->dat, tempBitmap, 0, 0, 0, 0, tempBitmap->w, tempBitmap->h);

			tempTileType = new TileType(tempBitmap, get_datafile_property(file, DAT_ID('N','A','M','E')));

			tileTypes.insert(make_pair(tempTileType->getName(), tempTileType));
			break;
		}
		file++;
	}
}

void TileRepository::importBitmap(const char *filename, int tile_w, int tile_h, int tile_spacing, int tiles_in_row)
{
	BITMAP *tileBitmap;
	BITMAP *tempBitmap;
	TileType *tempTileType;
	char tempTilename[256];
	char tempFilename[256];
	PALETTE pal;
	int x, y;

	tileBitmap = load_bitmap(filename, pal);
	if (!tileBitmap) {
		allegro_message("Warning, %s is not a valid time bitmap!", filename);
		return;
	}

	set_palette(pal);
	replace_extension(tempFilename, get_filename(filename), "", 256);

	for (y = 0; y < (tileBitmap->h / tile_h); y++)
	{
		for (x = 0; x < (tileBitmap->w / tile_w); x++)
		{
			// Create a new tile type and add it to the hash_map
			tempBitmap = create_bitmap(tile_w, tile_h);
			blit(
				tileBitmap, tempBitmap,
				x * (tile_w + tile_spacing),
				y * (tile_h + tile_spacing),
				0, 0, tile_w, tile_h
			);
			
			sprintf(tempTilename, "%s%03d", tempFilename, y * tiles_in_row + x);

			tempTileType = new TileType(tempBitmap, tempTilename);
			tileTypes.insert(make_pair(tempTileType->getName(), tempTileType));
		}
	}

	destroy_bitmap(tileBitmap);
}

void TileRepository::exportBitmap(const char *filename, int tile_w, int tile_h, int tile_spacing, int tiles_in_row)
{
	list<TileType*> tiles_to_save;
	map<const char*, TileType*, ltstr>::iterator i;
	list<TileType*>::iterator j;
	char tempTilename[256];
	char tempFilename[256];
	replace_extension(tempFilename, get_filename(filename), "", 256);

	if (!(tiles_in_row > 0 && tile_w > 0 && tile_h > 0)) {
		allegro_message("WARNING: tiles_in_row, tile_w and tile_h must be larger than 0.");
		return;
	}

	for (i = tileTypes.begin(); i != tileTypes.end(); i++) 
	{
		TileType* tempTileType = (*i).second;
		replace_extension(tempTilename, tempTileType->getName(), "", 256);
		if (ustrcmp(tempFilename, tempTilename) == 0) {
			tiles_to_save.push_back(tempTileType);
		}
	}

	if (tiles_to_save.empty()) {
		allegro_message("WARNING: No tiles to save in %s.", filename);
		return;
	}

	BITMAP *tile_bitmap;
	PALETTE pal;

	tile_bitmap = create_bitmap
	(
		tiles_in_row * tile_w,
		(tiles_to_save.size() / tiles_in_row +
		 tiles_to_save.size() % tiles_in_row) * tile_h
	);
	int x = 0;
	int y = 0;

	for (j = tiles_to_save.begin(); j != tiles_to_save.end(); j++) 
	{
		blit((*j)->getBitmap(), tile_bitmap, 0, 0, x * tile_w, y * tile_h, tile_w, tile_h);
		x++;
		if (x == tiles_in_row) {
			y++;
			x = 0;
		}
	}

	get_palette(pal);
	save_bitmap(filename, tile_bitmap, pal);

	destroy_bitmap(tile_bitmap);
}

TileType* TileRepository::getTileType(const char *tileName)
{
	map<const char*, TileType*, ltstr>::iterator found = tileTypes.find(tileName);
	if (found != tileTypes.end()) {
		return (*found).second;
	} else {
		return NULL;
	}
}

vector<TileType*> TileRepository::generateTileArray()
{
	map<const char*, TileType*, ltstr>::iterator i;
	vector<TileType*> tileArray;

	for (i = tileTypes.begin(); i != tileTypes.end(); i++) 
	{
		tileArray.push_back((*i).second);
	}

	return tileArray;
}


// Tile class ================================================================

Tile::Tile()
{
	tileType = NULL;
	obstacle = false;
}

void Tile::saveTo(PACKFILE *file)
{
	// Write tile name to file
	if (tileType) {
		pack_fputs(tileType->getName(), file);
	}
	pack_fputs("\n", file);

	if (obstacle) {
		pack_iputw(1, file);
	} else {
		pack_iputw(0, file);
	}
}

void Tile::loadFrom(PACKFILE *file, TileRepository *tileRepository)
{
	// Load tile name from file and look it up in the tile repository
	char name[32];
	pack_fgets(name, 32, file);
	setType(tileRepository->getTileType(name));

	obstacle = pack_igetw(file) == 1;
}

void Tile::setType(TileType *tileType)
{
	this->tileType = tileType;
}


// Entity class ==============================================================

Entity::Entity()
{
	bitmap = NULL;
}

bool Entity::visible(BITMAP *dest, Point screenCoords)
{
	if (!bitmap) return false;

	int hbw = bitmap->w / 2;
	return !(dest->cl > screenCoords.x + hbw ||
			 dest->cr < screenCoords.x - hbw ||
			 dest->ct > screenCoords.y ||
			 dest->cb < screenCoords.y - bitmap->h);
}

void Entity::draw(BITMAP *dest, Point screenCoords)
{
	if (bitmap) {
		draw_sprite(
			dest,
			bitmap,
			screenCoords.x - bitmap->w / 2,
			screenCoords.y - bitmap->h - pos.z
		);
	}

	//textprintf_centre(dest, font, screenCoords.x, screenCoords.y, makecol(128,128,128), "%i, %i", pos.x, pos.y);
}


// TiledMap class ============================================================
//  Defines a generic tiled map interface and data model.

TiledMap::TiledMap()
{
	mapWidth = 0;
	mapHeight = 0;
	tileMap = NULL;
}

TiledMap::~TiledMap()
{
	// Delete tiles on map
	for (int y = 0; y < mapHeight; y++)
		for (int x = 0; x < mapWidth; x++)
			delete tileMap[x + y * mapWidth];

	delete tileMap;
}

void TiledMap::setCamera(Point cam, Rectangle rect, bool center, bool modify)
{
	if (center) {
		cam.x -= rect.w / 2;
		cam.y -= rect.h / 2;
	}
	if (modify)
	{
		Point mapSize = getMapSize();
		cam.x = MAX(0, MIN(mapSize.x - rect.w, cam.x));
		cam.y = MAX(0, MIN(mapSize.y - rect.h, cam.y));
	}

	cameraCoords = cam;
	cameraScreenRect = rect;
}

void TiledMap::resizeTo(int w, int h)
{
	Tile* *newTileMap = NULL;
	int x, y;

	// Create new map
	newTileMap = (Tile**) malloc(w * h * sizeof(Tile*));
	ASSERT(newTileMap);
	for (y = 0; y < h; y++)
		for (x = 0; x < w; x++)
			newTileMap[x + y * w] = new Tile();

	// Copy old map data
	for (y = 0; y < MIN(h, mapHeight); y++)
		for (x = 0; x < MIN(w, mapWidth); x++)
			newTileMap[x + y * w]->setType(tileMap[x + y * w]->getType());

	// Delete tiles on old map
	for (y = 0; y < mapHeight; y++)
		for (x = 0; x < mapWidth; x++)
			delete tileMap[x + y * w];

	delete tileMap;
	tileMap = newTileMap;
	mapWidth = w;
	mapHeight = h;
}

void TiledMap::saveTo(PACKFILE *file)
{
	// The map header
	pack_iputw(mapWidth, file);
	pack_iputw(mapHeight, file);

	// The tile data
	for (int y = 0; y < mapHeight; y++)
		for (int x = 0; x < mapWidth; x++)
			getTile(Point(x,y))->saveTo(file);

	// Extra newline fixes last tile not loaded.
	pack_fputs("\n", file);
}

void TiledMap::loadFrom(PACKFILE *file, TileRepository *tileRepository)
{
	if (!file) {
		allegro_message("TiledMap::loadFrom - invalid file!");
		return;
	}

	// Load the map header
	int w = pack_igetw(file);
	int h = pack_igetw(file);
	resizeTo(w, h);

	// Load the tile data
	for (int y = 0; y < mapHeight; y++)
		for (int x = 0; x < mapWidth; x++)
			getTile(Point(x,y))->loadFrom(file, tileRepository);
}

Tile *TiledMap::getTile(Point tile)
{
	if (tile.x < 0 || tile.x >= mapWidth ||
		tile.y < 0 || tile.y >= mapHeight)
	{
		return NULL;
	}
	else
	{
		return tileMap[tile.x + tile.y * mapWidth];
	}
}

void TiledMap::addEntity(Entity* entity)
{
	if (entity) entities.push_front(entity);
}

void TiledMap::removeEntity(Entity* entity)
{
	if (entity) entities.remove(entity);
}

void TiledMap::drawEntities(BITMAP *dest)
{
	list<EntityP> visibleEnts;
	list<Entity*>::iterator i;
	list<EntityP>::iterator j;

	for (i = entities.begin(); i != entities.end(); i++)
	{
		if ((*i)->visible(dest, mapToScreen((*i)->pos)))
		{
			visibleEnts.push_back(EntityP((*i)));
		}
	}

	// Sort the visible entities on y value.
	visibleEnts.sort();

	for (j = visibleEnts.begin(); j != visibleEnts.end(); j++) {
		(*j).ent->draw(dest, mapToScreen((*j).ent->pos));
	}

	//textprintf(dest, font, 10, 10, makecol(200,200,200), "%i entities", entities.size());
	//textprintf(dest, font, 10, 30, makecol(200,200,200), "%i drawn entities", visibleEnts.size());
}

Point TiledMap::screenToTile(Point screenCoords)
{
	return mapToTile(screenToMap(screenCoords));
}

Point TiledMap::tileToScreen(Point tileCoords)
{
	return mapToScreen(tileToMap(tileCoords));
}


// SquareMap class ===========================================================
//  Provides algorithms for simple square-tiled maps

SquareMap::SquareMap(int tileSize)
{
	tileWidth = tileSize;
	tileHeight = tileSize;
}

SquareMap::SquareMap(int tileWidth, int tileHeight)
{
	this->tileWidth = tileWidth;
	this->tileHeight = tileHeight;
}

void SquareMap::draw(BITMAP *dest, bool drawObstacle)
{
	Rectangle oldClip;
	TileType *tempTileType;

	oldClip.clipToRect(dest);
	cameraScreenRect.rectToClip(dest);

	// Calculate the part of the map that needs to be drawn (visible part)
	Point start = screenToTile(Point(cameraScreenRect.x, cameraScreenRect.y));
	Point end = screenToTile(Point(cameraScreenRect.x + cameraScreenRect.w - 1, cameraScreenRect.y + cameraScreenRect.h - 1));

	for (int y = start.y; y <= end.y; y++) {
		for (int x = start.x; x <= end.x; x++) {
			tempTileType = tileMap[x + y * mapWidth]->getType();
			if (tempTileType) {
				blit(
					tempTileType->getBitmap(),
					dest,
					0,
					0,
					cameraScreenRect.x - cameraCoords.x + x * tileWidth,
					cameraScreenRect.y - cameraCoords.y + y * tileHeight,
					tempTileType->getBitmap()->w,
					tempTileType->getBitmap()->h
				);
			}
			if (drawObstacle && tileMap[x + y * mapWidth]->obstacle) {
				set_trans_blender(0,0,0,64);
				drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
				rectfill(
					dest,
					cameraScreenRect.x - cameraCoords.x + x * tileWidth,
					cameraScreenRect.y - cameraCoords.y + y * tileHeight,
					cameraScreenRect.x - cameraCoords.x + x * tileWidth  + tileWidth - 1,
					cameraScreenRect.y - cameraCoords.y + y * tileHeight + tileHeight - 1,
					makecol(0,0,0)
				);
				rect(
					dest,
					cameraScreenRect.x - cameraCoords.x + x * tileWidth,
					cameraScreenRect.y - cameraCoords.y + y * tileHeight,
					cameraScreenRect.x - cameraCoords.x + x * tileWidth  + tileWidth - 1,
					cameraScreenRect.y - cameraCoords.y + y * tileHeight + tileHeight - 1,
					makecol(255,0,0)
				);
				drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
			}
		}
	}

	// Now draw the entities
	drawEntities(dest);

	oldClip.rectToClip(dest);
}

Point SquareMap::screenToMap(Point screenCoords)
{
	return Point(
		cameraCoords.x - cameraScreenRect.x + screenCoords.x,
		cameraCoords.y - cameraScreenRect.y + screenCoords.y,
		screenCoords.z
	);
}

Point SquareMap::mapToScreen(Point mapCoords)
{
	return Point(
		cameraScreenRect.x - cameraCoords.x + mapCoords.x,
		cameraScreenRect.y - cameraCoords.y + mapCoords.y,
		mapCoords.z
	);
}

Point SquareMap::mapToTile(Point mapCoords)
{
	return Point(
		MIN(mapWidth - 1, MAX(0, mapCoords.x / tileWidth)),
		MIN(mapHeight - 1, MAX(0, mapCoords.y / tileHeight)),
		mapCoords.z
	);
}

Point SquareMap::tileToMap(Point tileCoords)
{
	return Point(
		(tileCoords.x + 1) * tileWidth - tileWidth / 2,
		(tileCoords.y + 1) * tileHeight,
		tileCoords.z
	);
}

Point SquareMap::getMapSize()
{
	return Point(
		tileWidth  * mapWidth,
		tileHeight * mapHeight
	);
}


// IsometricMap class ========================================================
//  Provides algorithms for isometric-tiled maps

IsometricMap::IsometricMap(int tileStepX, int tileStepY)
{
	this->tileGridSize = tileStepX;
	this->tileStepX = tileStepX;
	this->tileStepY = tileStepY;
}

void IsometricMap::draw(BITMAP *dest, bool drawObstacle)
{
	Rectangle oldClip;
	TileType *tempTileType;
	Point temp, temp2, area;

	oldClip.clipToRect(dest);
	cameraScreenRect.rectToClip(dest);

	temp = screenToTile(Point(cameraScreenRect.x, cameraScreenRect.y));
	area = Point(cameraScreenRect.w / (tileStepX * 2) + 3, cameraScreenRect.h / tileStepY + 3);

	// Move up one row
	temp.x--;

	for (int y = 0; y < area.y; y++) {
		// Initialize temp2 to draw a horizontal line of tiles
		temp2 = temp;

		for (int x = 0; x < area.x; x++) {
			// Check if we are drawing a valid tile
			if (temp2.x >= 0 && temp2.y >= 0 && temp2.x < mapWidth && temp2.y < mapHeight) {
				tempTileType = tileMap[temp2.x + temp2.y * mapWidth]->getType();
			} else {
				tempTileType = NULL;
			}

			// Draw the tile if valid
			if (tempTileType) {
				draw_sprite(
					dest,
					tempTileType->getBitmap(),
					cameraScreenRect.x - cameraCoords.x + (temp2.x - temp2.y - 1) * tileStepX + mapHeight * tileStepX,
					cameraScreenRect.y - cameraCoords.y + (temp2.x + temp2.y    ) * tileStepY
				);
			}

			// Advance to the next tile (to the right)
			temp2.x++; temp2.y--;
		}

		// Advance to the next row
		if ((y & 1) > 0) temp.x++; else temp.y++;
	}	

	/*
	// Draw a red line along the edges of the map
	Point top    = mapToScreen(Point(-1,                      0));
	Point right  = mapToScreen(Point(tileGridSize * mapWidth, 0));
	Point bottom = mapToScreen(Point(tileGridSize * mapWidth, tileGridSize * mapHeight + 1));
	Point left   = mapToScreen(Point(-1,                      tileGridSize * mapHeight + 1));
	line(dest, top.x,    top.y,    right.x,  right.y,  makecol(255,0,0));
	line(dest, right.x,  right.y,  bottom.x, bottom.y, makecol(255,0,0));
	line(dest, bottom.x, bottom.y, left.x,   left.y,   makecol(255,0,0));
	line(dest, left.x,   left.y,   top.x,    top.y,    makecol(255,0,0));
	*/

	// Now draw the entities
	drawEntities(dest);

	oldClip.rectToClip(dest);
}

Point IsometricMap::screenToMap(Point screenCoords)
{
	screenCoords.x = screenCoords.x + cameraCoords.x - cameraScreenRect.x - mapHeight * tileStepX;
	screenCoords.y = screenCoords.y + cameraCoords.y - cameraScreenRect.y;
	return Point(
		screenCoords.y + screenCoords.x / 2,
		screenCoords.y - screenCoords.x / 2,
		screenCoords.z
	);
}

Point IsometricMap::mapToScreen(Point mapCoords)
{
	return Point(
		cameraScreenRect.x - cameraCoords.x + (mapCoords.x - mapCoords.y) + mapHeight * tileStepX,
		cameraScreenRect.y - cameraCoords.y + (mapCoords.x + mapCoords.y) / 2,
		mapCoords.z
	);
}

Point IsometricMap::mapToTile(Point mapCoords)
{
	return Point(
		(mapCoords.x < 0) ? mapCoords.x / tileGridSize - 1 : mapCoords.x / tileGridSize,
		(mapCoords.y < 0) ? mapCoords.y / tileGridSize - 1 : mapCoords.y / tileGridSize,
		mapCoords.z
	);
}

Point IsometricMap::tileToMap(Point tileCoords)
{
	return Point(
		(tileCoords.x + 1) * tileGridSize,
		(tileCoords.y + 1) * tileGridSize,
		tileCoords.z
	);
}

Point IsometricMap::getMapSize()
{
	return Point(
		tileStepX * (mapWidth + mapHeight),
		tileStepY * (mapWidth + mapHeight)
	);
}
