/*
 *  The Moonlight RPG engine  (see readme.txt about version info)
 *  By Bjrn Lindeijer
 *
 ************************************************************************************/

#include "RPG.h"
#include "Engine.h"
#include "Errors.h"
#include "Script.h"
#include <string.h>		// for the string manipulation functions exported to scripts
#include <map>
#include <allegro.h>
#include <seer.h>
#include <stdarg.h>

map<const char*, seerScript, ltstr> scripts;


//===================   The engine header   ==============================================/

char *engine_header =
"#ifndef ENGINE_INCLUDED																\n"
"#define ENGINE_INCLUDED																\n"

"class BITMAP {																			\n"
"	int w, h;																			\n"
"};																						\n"

"class Object																			\n"
"{																						\n"
"public:																				\n"
"	BITMAP* bitmap;																		\n"
"	int x, y, dir;																		\n"
"	int count, tick;																	\n"
"	int speed;																			\n"
"	int type;																			\n"
"	int flags;																			\n"
"   int walking;																		\n"
"   int state;																			\n"
"	int str, agi, def;																	\n"
"	int health, max_health;																\n"
"	int var[10];																		\n"
"	Object* obj[10];																	\n"
"	import void walk(int dir);															\n"
"	import void walk_nocol(int dir);													\n"
"};																						\n"

"struct scArg {																			\n"
"	Object* me;																			\n"
"	Object* obj;																		\n"
"	int a, b, c;																		\n"
"};																						\n"

"import char* strcpy(char*, const char*);												\n"
"import int sprintf(char *buffer, const char *format, ...);								\n"

"import void ex_walk(Object* obj, int dir, int dist);									\n"
"import void ex_wait(int updates);														\n"

"import void show_bitmap_and_wait(BITMAP* bitmap);										\n"
"import void import_tile_bmp(const char* filename);										\n"
"import Object *add_object(int x, int y, const char* name, const char* script);			\n"
"import Object *get_object(const char* name);											\n"
"import void set_player(Object *obj);													\n"
"import void set_camera_target(Object *obj);											\n"
"import BITMAP* get_bitmap(const char* name);											\n"
"import void load_area(const char* area_name);											\n"
"import void set_pgp(const char* name, int value);										\n"
"import int  get_pgp(const char* name);													\n"
"import int random(int nr);																\n"
"import void open_dialog(BITMAP* left, BITMAP* right);									\n"
"import void close_dialog();															\n"
"import void write_text(const char* name, const char* text);							\n"
"import void start_combat(Object* enemy);												\n"
"import void quit_game();																\n"

"#define DIR_UP			0																\n"
"#define DIR_LEFT		1																\n"
"#define DIR_RIGHT		2																\n"
"#define DIR_DOWN		3																\n"

"#define OBJ_STATIC		0																\n"
"#define OBJ_CHAR		1																\n"

"#define FLAG_OBSTACLE	1																\n"
"#define FLAG_ENEMY		2																\n"

"#define WHO_PLAYER		0	// This creature attacked the player						\n"
"#define WHO_ME			1	// We have been attacked by the player						\n"

"#define END_RUNAWAY		0	// The player ran away									\n"
"#define END_VICTORY		1	// The player has won									\n"
"#define END_DEFEAT		2	// The player has been defeated								\n"

"#endif																					\n"
;



//===================   Script helper functions   ========================================/

void compile(seerScript &scrpt, char *filename)
{
	if (scrpt.compileFile(filename)) output_error(1, "%s\n> %s\n", scErrorMsg, scErrorLine);
	if (!scrpt.valid()) output_error(1, "Unable to find %s!\n", filename);
}

void instant(seerInstance &i, seerScript &s)
{
	i.create(s);
	if (scErrorNo) {output_error(1, "%s\n", scErrorMsg);}
	if (!i.valid()) {output_error(1, "Unknown script error\n");}
}

void init_script_externals()
{
	scAdd_Internal_Header("engine_interface", engine_header);

	scAdd_External_Symbol("show_bitmap_and_wait", show_bitmap_and_wait);
	scAdd_External_Symbol("import_tile_bmp", import_tile_bmp);

	scAdd_External_Symbol("strcpy", strcpy);
	scAdd_External_Symbol("sprintf", sprintf);

	scAdd_External_Symbol("ex_walk", ex_walk);
	scAdd_External_Symbol("ex_wait", ex_wait);

	scAdd_External_Symbol("add_object", add_object);
	scAdd_External_Symbol("get_object", get_object);
	scAdd_External_Symbol("set_player", set_player);
	scAdd_External_Symbol("set_camera_target", set_camera_target);

	scAdd_External_Symbol("get_bitmap", get_bitmap);
	scAdd_External_Symbol("load_area", load_area);

	scAdd_External_Symbol("get_pgp", get_pgp);
	scAdd_External_Symbol("set_pgp", set_pgp);

	scAdd_External_Symbol("random", random);

	scAdd_External_Symbol("open_dialog", open_dialog);
	scAdd_External_Symbol("close_dialog", close_dialog);
	scAdd_External_Symbol("write_text", write_text);
	scAdd_External_Symbol("start_combat", start_combat);

	scAdd_External_Symbol("quit_game", quit_game);

	Object::export_script_methods();
}

/* This function will go through all script files and compile and instaniate them.
 * It puts the script instances in the STL map so that they can be easily used.
 */
void prepare_scripts()
{
	output_error(3, "Compiling scripts...\n");

	char searchstr[256] = "scripts/*.sc";
	char foundfile[256];
	char *scriptname;
	al_ffblk finfo;

	fix_filename_slashes(searchstr);

	for (int i = al_findfirst(searchstr, &finfo, FA_ARCH); i == 0; i = al_findnext(&finfo))
	{
		scriptname = new char[256];
		replace_extension(scriptname, finfo.name, "", 256);
		scriptname[strlen(scriptname) - 1] = 0;
		replace_filename(foundfile, searchstr, finfo.name, 256);

		seerScript sc;
		//seerInstance *si = new seerInstance();

		output_error(3, "> \"%s\"\n", scriptname);

		compile(sc, foundfile);
		//instant(*si, sc);
		scripts[scriptname] = sc;

		//sc.free();
	}
}

/* This function can execute any script function in any instaniated script,
 * using the default number of parameters for the script function.
 */
int execute_script(const char* script, char* function, Object* me, Object* obj, int a, int b, int c)
{
	if (exclusive_mode) return 0;

	if (!scripts[script].valid()) {
		output_error(1, "Error: No script named \"%s\" present!\n", script);
		return 0;
	}

	scArg arg;
	arg.a = a;
	arg.b = b;
	arg.c = c;
	arg.me = me;
	arg.obj = obj;
	
	seerScript sc = scripts[script];
	seerInstance si;
	instant(si, sc);

	int symbol_nr = si.symbol(function);
	if (symbol_nr > -1) {
		//output_error(3, "Executing \"%s\" in script \"%s\"...\n", function, script);
		si.vcall(symbol_nr, sizeof(scArg), (int*)&arg);
		//output_error(3, "Finished executing \"%s\" in script \"%s\"...\n", function, script);
	}

	si.free();
	return 1;
}


/* This function is going through the script STL map, freeing script instances.
 */
void clean_scripts()
{
	output_error(3, "Removing scripts...\n");

	map<const char*, seerScript, ltstr>::iterator i;
	while (!scripts.empty())
	{
		i = scripts.begin();

		seerScript sc = (*i).second;
		//sc.free();
		scripts.erase(i);
	}
}



