/*
 *  An object oriented GUI.
 *  By Bjrn Lindeijer
 *
 ************************************************************************************/

#include <list>
#include <allegro.h>
#include "GUI.h"
#include "OOGUI.h"

using namespace std;

extern int gui_color[6];


//====================================================================================
// Branch: Can have multiple Block childs

Branch::Branch():
nr_of_childs(0), padding(0), space(0)
{
	alloc_memory();
}

Branch::Branch(int _padding, int _space):
nr_of_childs(0), padding(_padding), space(_space)
{
	alloc_memory();
}

Branch::Branch(int _min_w, int _min_h, int _hfr, int _vfr, int _padding, int _space):
Block(_min_w, _min_h, _hfr, _vfr),
nr_of_childs(0), padding(_padding), space(_space)
{
	alloc_memory();
}

Branch::~Branch()
{
	for (int i = 0; i < nr_of_childs; i++) delete child[i];
	free(child);
}

int  Branch::message(int msg, int c)
{
	// Send the message to each of the childs, if necessary
	for (int i = 0; i < nr_of_childs; i++) child[i]->message(msg, c);

	return D_O_K;
}

void Branch::calc_min_size()
{
	// Make childs calculate minimum size
	for (int i = 0; i < nr_of_childs; i++) {
		child[i]->calc_min_size();
		min_w = MAX(min_w, child[i]->min_w);
		min_h = MAX(min_h, child[i]->min_h);
	}

	min_w += 2 * padding;
	min_h += 2 * padding;
}

void Branch::become_size(int _x, int _y, int _w, int _h)
{
	// Make childs become the same size
	for (int i = 0; i < nr_of_childs; i++) {
		child[i]->become_size(_x + padding, _y + padding, _w - 2 * padding, _h - 2 * padding);
	}
}

void Branch::draw(BITMAP* bitmap)
{
	// Make childs draw themselves
	for (int i = 0; i < nr_of_childs; i++) child[i]->draw(bitmap);
}

void Branch::add(Block *_child)
{
	// Add child
	child[nr_of_childs] = _child;
	nr_of_childs++;

	// Allocate more memory for childs
	child = (Block**) realloc(child, sizeof(Block*) * (nr_of_childs + 1));
	child[nr_of_childs] = NULL;
}

void Branch::alloc_memory()
{
	// Allocate memory for childs
	child = (Block**) malloc(sizeof(Block*) * (nr_of_childs + 1));
	child[nr_of_childs] = NULL;
}



//====================================================================================
// HBranch: Displays childs horizontally aligned

void HBranch::calc_min_size()
{
	Branch::calc_min_size();

	// Correct the minimum width
	min_w = 2 * padding;
	for (int i = 0; i < nr_of_childs; i++) min_w += child[i]->min_w;
	if (nr_of_childs > 0) min_w += space * (nr_of_childs - 1);
}

void HBranch::become_size(int _x, int _y, int _w, int _h)
{
	if (nr_of_childs > 0) {
		int i;

		// Calculate the amount of free space
		int free_space = _w - min_w;
		int free_pts = 0;
		for (i = 0; i < nr_of_childs; i++) free_pts += child[i]->hfr;

		// Set the position/size of the children
		int x_pos = _x + padding;
		for (i = 0; i < nr_of_childs; i++) {
			int free_usage = 0;
			if (free_pts > 0) free_usage = (child[i]->hfr * free_space) / free_pts;
			else free_usage = free_space / (nr_of_childs - i);
			free_pts -= child[i]->hfr;

			child[i]->become_size(x_pos, _y + padding, child[i]->min_w + free_usage, _h - 2 * padding);
			x_pos += child[i]->min_w + free_usage + space;
			free_space -= free_usage;
		}
	}
}


//====================================================================================
// VBranch: Displays childs vertically aligned

void VBranch::calc_min_size()
{
	Branch::calc_min_size();

	// Correct the minimum height
	min_h = 2 * padding;
	for (int i = 0; i < nr_of_childs; i++) min_h += child[i]->min_h;
	if (nr_of_childs > 0) min_h += space * (nr_of_childs - 1);
}

void VBranch::become_size(int _x, int _y, int _w, int _h)
{
	if (nr_of_childs > 0) {
		int i;

		// Calculate the amount of free space
		int free_space = _h - min_h;
		int free_pts = 0;
		for (i = 0; i < nr_of_childs; i++) free_pts += child[i]->vfr;

		// Set the position/size of the children
		int y_pos = _y + padding;
		for (i = 0; i < nr_of_childs; i++) {
			int free_usage = 0;
			if (free_pts > 0) free_usage = (child[i]->vfr * free_space) / free_pts;
			else free_usage = free_space / (nr_of_childs - i);
			free_pts -= child[i]->vfr;

			child[i]->become_size(_x + padding, y_pos, _w - 2 * padding, child[i]->min_h + free_usage);
			y_pos += child[i]->min_h + free_usage + space;
			free_space -= free_usage;
		}
	}
}


//====================================================================================
// WinThing: A basic block, with appearance

void BlockType::calc_min_size()
{
	Block::calc_min_size();
}

void BlockType::draw(BITMAP* bitmap)
{
	int border;
	set_GUI_color(64,64,64);

	switch (type) {
	case T_DEEP_BORDER:	// 2 pixels
		rect (bitmap, x,         y,         x + w - 1, y + h - 1, gui_color[1]);
		rect (bitmap, x    ,     y    ,     x + w - 2, y + h - 2, gui_color[1]);
		rect (bitmap, x + 1,     y + 1,     x + w - 1, y + h - 1, gui_color[4]);
		border = 2;
		break;
	case T_HIGH_BORDER:	// 3 pixels
		rect (bitmap, x,         y,         x + w - 1, y + h - 1, gui_color[1]);
		rect (bitmap, x,         y,         x + w - 2, y + h - 2, gui_color[4]);
		rect (bitmap, x + 1,     y + 1,     x + w - 2, y + h - 2, gui_color[3]);
		rect (bitmap, x + 2,     y + 2,     x + w - 3, y + h - 3, gui_color[1]);
		rect (bitmap, x + 3,     y + 3,     x + w - 3, y + h - 3, gui_color[4]);
		border = 3;
		break;
	case T_HIGH:
		hline(bitmap, x,         y,         x + w - 2, gui_color[4]);
		hline(bitmap, x,         y + h - 1, x + w - 1, gui_color[1]);
		vline(bitmap, x,         y + 1,     y + h - 2, gui_color[4]);
		vline(bitmap, x + w - 1, y,         y + h - 2, gui_color[1]);
		border = 1;
		break;
	case T_DEEP:
		hline(bitmap, x,         y,         x + w - 2, gui_color[1]);
		hline(bitmap, x,         y + h - 1, x + w - 1, gui_color[4]);
		vline(bitmap, x,         y + 1,     y + h - 2, gui_color[1]);
		vline(bitmap, x + w - 1, y,         y + h - 2, gui_color[4]);
		border = 1;
		break;
	case T_DEEP_DARK:	// darker schadow
		hline(bitmap, x,         y,         x + w - 2, gui_color[0]);
		hline(bitmap, x,         y + h,     x + w - 1, gui_color[4]);
		vline(bitmap, x,         y + 1,     y + h - 2, gui_color[0]);
		vline(bitmap, x + w,     y,         y + h - 2, gui_color[4]);
		border = 1;
		break;
	default: /* W_FLAT */
		rect (bitmap, x,     y,     x + w - 1, y + h - 1, gui_color[3]);
		border = 0;
		break;
	}

	//switch (back) {
	//case 1: rectfill(bitmap, x + border, y + border, x + w - border, y + h - border, gui_color[1]); break;
	rectfill(bitmap, x + border, y + border, x + w - 1 - border, y + h - 1 - border, gui_color[3]);
	//case 3: rectfill(bitmap, x + border, y + border, x + w - border, y + h - border, gui_color[3]); break;
	//}
}



//===================   Button   =====================================================

Button::Button(const char *_text, int _padding):
BlockType(T_HIGH), pressed(false)
{
	padding = _padding;
	text = ustrdup(_text);
}

Button::~Button()
{
	delete text;
}

void Button::calc_min_size()
{
	BlockType::calc_min_size();
	min_w += gui_strlen(text) + 2 * padding;
	min_h += text_height(font) + 2 * padding;
}

void Button::draw(BITMAP *bitmap)
{
	BlockType::draw(bitmap);
	textprintf_centre(bitmap, font, x + (w - 1) / 2, y + padding, gui_color[4], text);
}




//====================================================================================
// Will initialize the size and position of the dialogs and put them on the screen.
// It will then continue to update the block.

int do_block(Block *_block)
{
	_block->calc_min_size();
	_block->become_size(0, 0, SCREEN_W, SCREEN_H);
	_block->draw(screen);

	do {;} while (!key[KEY_Q]);

	delete _block;
	return 0;
}

/*
void set_GUI_color(int r, int g, int b)
{
	gui_color[0] = makecol(0,0,0);
	gui_color[1] = makecol(r / 3, g / 3, b / 3);
	gui_color[2] = makecol(r / 2, g / 2, b / 2);
	gui_color[3] = makecol(r,     g,     b    );
	gui_color[4] = makecol(r * 2, g * 2, b * 2);
	gui_color[5] = makecol(r*2+20, g*2+20, b*2+20);
}
*/
