/*
 *  The Moonlight RPG engine  (see readme.txt about version info)
 *  By Bjrn Lindeijer
 *
 ************************************************************************************/

#include <allegro.h>
#include <seer.h>
#include "Errors.h"
#include "tiledmap.h"
#include "Engine.h"
#include "RPG.h"
#include "Script.h"

map<const char*, Object*, ltstr> objects;
map<const char*, int, ltstr> progress_points;

bool exclusive_mode = false;

Object* player = NULL;
Object* camera_target = NULL;

bool switch_area = false;
char area_name[64];
char map_filename[64];


//====================================================================================

Object::Object(int x, int y, const char* script)
{
	entity = new Entity();
	entity->pos = tiledMap->tileToMap(Point(x,y));
	tiledMap->addEntity(entity);

	type = OBJ_STATIC;
	walking = state = flags = speed = dir = count = tick = 0;
	bitmap = NULL;
	this->x = x;
	this->y = y;
	str = agi = def = 0;
	max_health = health = 0;

	strncpy(this->script, script, 256);
	initialized = false;
	ani_sequence[0] = 0;
	ani_sequence[1] = 1;

	for (int i = 0; i < 10; i++) {
		var[i] = 0;
		obj[i] = NULL;
	}
}

Object::~Object()
{
	tiledMap->removeEntity(entity);
	delete entity;
}

void Object::walk(int dir)
{
	//output_error(3, "\"%s\": Receiving walk command in direction %i.\n", name, dir);

	if (walking == 0) {
		this->dir = dir;
		update_entity();

		// Precalculate where the player is going
		int next_x = x, next_y = y;
		switch (dir) {
		case DIR_UP:    next_y -= 1; break;
		case DIR_DOWN:  next_y += 1; break;
		case DIR_LEFT:  next_x -= 1; break;
		case DIR_RIGHT: next_x += 1; break;
		}

		// Check for map obstacle
		Tile *nextTile = tiledMap->getTile(Point(next_x, next_y));
		if (!nextTile || nextTile->obstacle) {
			update_screen();
			execute_script(script, "bump_into", this, NULL, 0, 0, 0);
			check_stand_on();
			return;
		}

		// Check for object in the way
		map<const char*, Object*, ltstr>::iterator i;
		for (i = objects.begin(); i != objects.end(); i++) {
			Object *obj = ((*i).second);
			if ((flags & FLAG_OBSTACLE) &&
				(obj->flags & FLAG_OBSTACLE) &&
				(obj->x == next_x) &&
				(obj->y == next_y))
			{
				update_screen();
				execute_script(obj->script, "bumped_into", obj, this, 0, 0, 0);
				execute_script(script, "bump_into", this, obj, 0, 0, 0);
				check_stand_on();
				return;
			}
		}

		// No obstacles, so start walking
		walking = 100;
		x = next_x;
		y = next_y;
		if (ani_sequence[1] == 1) ani_sequence[1] = 2;
		else if (ani_sequence[1] == 2) ani_sequence[1] = 1;
	}

	//output_error(3, "End of walk command.\n", name, dir);
}

void Object::walk_nocol(int dir)
{
	//output_error(3, "\"%s\": Receiving walk command in direction %i.\n", name, dir);

	if (walking == 0) {
		this->dir = dir;

		// Precalculate where the player is going
		int next_x = x, next_y = y;
		switch (dir) {
		case DIR_UP:    next_y -= 1; break;
		case DIR_DOWN:  next_y += 1; break;
		case DIR_LEFT:  next_x -= 1; break;
		case DIR_RIGHT: next_x += 1; break;
		}

		// No obstacles, so start walking
		walking = 100;
		x = next_x;
		y = next_y;
		if (ani_sequence[1] == 1) ani_sequence[1] = 2;
		else if (ani_sequence[1] == 2) ani_sequence[1] = 1;
	}
}

void Object::check_stand_on()
{
	// Check if this object is now standing on something

	map<const char*, Object*, ltstr>::iterator i;
	for (i = objects.begin(); i != objects.end(); i++) {
		Object *obj = ((*i).second);
		if ((obj != this) && (obj->x == x) && (obj->y == y)) {
			execute_script(obj->script, "stand_on", obj, this, 0, 0, 0);
		}
	}
}

void Object::update()
{
	if (!initialized) {
		// Call script "init" function
		if (execute_script(script, "init", this, NULL, 0, 0, 0)) {
			initialized = true;
		}
	}

	if (walking > 0) {
		walking -= speed;

		if (walking <= 0) {
			if (walking < 0) walking = 0;
			execute_script(script, "walk_finished", this, NULL, 0, 0, 0);
			// Check if this object is now standing on something
			check_stand_on();
		}
	}

	mapPos = tiledMap->tileToMap(Point(x,y));
	if (walking > 0) {
		switch (dir) {
		case DIR_UP:    mapPos.y += (TILES_H * walking) / 100; break;
		case DIR_DOWN:  mapPos.y -= (TILES_H * walking) / 100; break;
		case DIR_LEFT:  mapPos.x += (TILES_W * walking) / 100; break;
		case DIR_RIGHT: mapPos.x -= (TILES_W * walking) / 100; break;
		}
	}

	update_entity();

	if (tick > 0) {
		count++;
		if (count >= tick) {
			// Call script "tick" function
			execute_script(script, "tick", this, NULL, count, 0, 0);
			count = 0;
		}
	} else {
		count = 0;
	}
}

void Object::update_entity()
{
	if (entity->bitmap) {
		destroy_bitmap(entity->bitmap);
		entity->bitmap = NULL;
	}
	if (bitmap) {
		if (type == OBJ_CHAR) {
			entity->bitmap = create_sub_bitmap(bitmap, ani_sequence[walking / 50] * 17, dir * 33, 16, 32);
		}
		else {
			entity->bitmap = create_sub_bitmap(bitmap, 0, 0, bitmap->w, bitmap->h);
		}
	}

	if (type == OBJ_CHAR) {
		entity->pos = Point(mapPos.x, mapPos.y - 2);
	} else {
		entity->pos = mapPos;
	}
}

void Object::export_script_methods()
{
	scAdd_Member_Symbol("Object.walk", walk);
	scAdd_Member_Symbol("Object.walk_nocol", walk_nocol);
}


//===================   Engine functions   ===========================================

void update_objects()
{
	map<const char*, Object*, ltstr>::iterator i;
	for (i = objects.begin(); i != objects.end(); i++) {
		((*i).second)->update();
	}
}

void switch_area_now()
{
	output_error(3, "Switching to area \"%s\".\n", area_name, map_filename);

	if (!switch_area) return;
	switch_area = false;

	delete tiledMap;
	// Load the new area map
	PACKFILE *file = pack_fopen(map_filename, F_READ_PACKED);
	tiledMap = new SquareMap(TILES_W, TILES_H);
	tiledMap->loadFrom(file, tileRepository);
	pack_fclose(file);

	// Get rid of all the objects and the previous map
	map<const char*, Object*, ltstr>::iterator i;
	while (!objects.empty())
	{
		i = objects.begin();
		Object* temp = (*i).second;
		objects.erase(i);
		if (player != temp) {
			delete temp;
		}
	}

	if (player) {
		objects["player"] = player;
		tiledMap->addEntity(player->entity);
	}

	// Execute the map script
	execute_script(area_name, "prepare_area", NULL, NULL, 0, 0, 0);
}


//===================   Script functions   ===========================================

void show_bitmap_and_wait(BITMAP* bitmap)
{
	if (bitmap) {
		draw_sprite(buffer, bitmap, 0, 0);
		dialog_open = true;
		update_screen();
		dialog_open = false;
	}

	readkey();
}

void import_tile_bmp(const char* filename)
{
	tileRepository->importBitmap(filename, TILES_W, TILES_H, 0, TILES_IN_ROW);
}

// Exclusive functions (no scripts called while executing!)

void ex_walk(Object* obj, int dir, int dist)
{
	exclusive_mode = true;

	int dist_walked = 0;
	while (dist_walked < dist) {
		if (obj->walking == 0) {
			dist_walked++;
			obj->walk_nocol(dir);
		}
		ex_update_game(obj);
	}
	while (obj->walking) ex_update_game(obj);

	exclusive_mode = false;
}

void ex_wait(int updates)
{
	exclusive_mode = true;

	int waited = 0;
	while (waited < updates) {
		while (frames_to_do == 0) {;}
		ex_update_game(NULL);
		waited++;
	}

	exclusive_mode = false;
}





// Object manipulation

Object* add_object(int x, int y, const char* obj_name, const char* script)
{
	char *name = new char[256];
	strncpy(name, obj_name, 256);

	if (objects[name]) {
		output_error(1, "Error: An object with the same name already exists (\"%s\")", name);
		return NULL;
	}

	objects[name] = new Object(x, y, script);
	strncpy(objects[name]->name, name, 256);
	output_error(3, "Object \"%s\" with script \"%s\" added.\n", objects[name]->name, script);

	return objects[name];
}

Object *get_object(const char* name)
{
	//output_error(3, "Searching for object \"%s\"...\n", name);
	return objects[name];
}

void set_player(Object *obj)
{
	player = obj;
}

void set_camera_target(Object *obj)
{
	camera_target = obj;
}


// Retrieving resources

BITMAP* get_bitmap(const char* name)
{
	DATAFILE *found_object = find_datafile_object(bitmap_data, name);
	if (found_object) {
		return (BITMAP*)found_object->dat;
	} else {
		output_error(1, "Error: Cannot find a bitmap called \"%s\"", name);
	}
	return NULL;
}


// Switching to another level

void load_area(const char* new_area_name)
{
	output_error(3, "Preparing switch to area \"%s\"...\n", new_area_name);

	switch_area = true;

	// Generate map and script filenames
	strncpy(area_name, new_area_name, 64);
	sprintf(map_filename, "maps/%s.map", area_name);
}


// Progress handling

void set_pgp(const char* name, int value)
{
	progress_points[name] = value;
}

int get_pgp(const char* name)
{
	//output_error(3, "Progress point \"%s\" currently has value %i.\n", name, progress_points[name]);
	return progress_points[name];
}


// Random

int random(int nr)
{
	return (rand() % nr);
}





void quit_game()
{
	game_end = true;
}