/*
 *  GUI.cpp:  GUI things. Featuring:
 *                 - WinThings, used as basis for any kind of window
 *                 - ButtonThings, automated buttons with label
 *                 - ScrollThings, partly automated scrollbars
 *                 - MenuThings, easy to use pop-up menu's
 *                 - Some standard mouse pointers
 *
 *  By Bjrn Lindeijer
 *
 ************************************************************************************/

#include <allegro.h>
#include "DRS.h"
#include "GUI.h"

extern BITMAP *buffer;

DATAFILE *mouse_pointer = NULL;
int gui_color[6] = {-1,-1,-1,-1,-1,-1};


//===================   WinThing  (it can reprisent anything you want!)   ============


WinThing::WinThing(int ix_min, int iy_min, int ix_max, int iy_max, int itype, int iback)
{
	set_rect(ix_min, iy_min, ix_max, iy_max, itype, iback);
	for (int i = 0; i < 6; i++) {
		if (gui_color[i] < 0) {set_GUI_color(64,64,64);}
	}
}

void WinThing::set_rect(int ix_min, int iy_min, int ix_max, int iy_max, int itype, int iback)
{
	int temp;
	if (x_max < x_min) {temp = x_min; x_min = x_max; x_max = temp;}
	if (y_max < y_min) {temp = y_min; y_min = y_max; y_max = temp;}

	x_min = (ix_min == -99) ? x_min : ix_min;					// When -99, the previous value is preserved
	y_min = (iy_min == -99) ? y_min : iy_min;
	x_max = (ix_max == -99) ? x_max : ix_max;
	y_max = (iy_max == -99) ? y_max : iy_max;
	type  = (itype  == -99) ? type  : itype;
	back  = (iback  == -99) ? back  : iback;
}

void WinThing::draw(BITMAP *bitmap)
{
	if (!bitmap) {bitmap = buffer;}
	int border;

	switch (type) {
	case W_DEEP_BORDER:											// 2 pixels
		rect (bitmap, x_min,     y_min,     x_max,     y_max,     gui_color[1]);
		rect (bitmap, x_min    , y_min    , x_max - 1, y_max - 1, gui_color[1]);
		rect (bitmap, x_min + 1, y_min + 1, x_max,     y_max,     gui_color[4]);
		border = 2;
		break;
	case W_HIGH_BORDER:											// 3 pixels
		rect (bitmap, x_min,     y_min,     x_max,     y_max,     gui_color[1]);
		rect (bitmap, x_min,     y_min,     x_max - 1, y_max - 1, gui_color[4]);
		rect (bitmap, x_min + 1, y_min + 1, x_max - 1, y_max - 1, gui_color[3]);
		rect (bitmap, x_min + 2, y_min + 2, x_max - 2, y_max - 2, gui_color[1]);
		rect (bitmap, x_min + 3, y_min + 3, x_max - 2, y_max - 2, gui_color[4]);
		border = 3;
		break;
	case W_HIGH:
		hline(bitmap, x_min,     y_min,     x_max - 1, gui_color[4]);
		hline(bitmap, x_min,     y_max,     x_max,     gui_color[1]);
		vline(bitmap, x_min,     y_min + 1, y_max - 1, gui_color[4]);
		vline(bitmap, x_max,     y_min,     y_max - 1, gui_color[1]);
		border = 1;
		break;
	case W_DEEP:
		hline(bitmap, x_min,     y_min,     x_max - 1, gui_color[1]);
		hline(bitmap, x_min,     y_max,     x_max,     gui_color[4]);
		vline(bitmap, x_min,     y_min + 1, y_max - 1, gui_color[1]);
		vline(bitmap, x_max,     y_min,     y_max - 1, gui_color[4]);
		border = 1;
		break;
	case W_DEEP_DARK:											// darker schadow
		hline(bitmap, x_min,     y_min,     x_max - 1, gui_color[0]);
		hline(bitmap, x_min,     y_max,     x_max,     gui_color[4]);
		vline(bitmap, x_min,     y_min + 1, y_max - 1, gui_color[0]);
		vline(bitmap, x_max,     y_min,     y_max - 1, gui_color[4]);
		border = 1;
		break;
	default: /* W_FLAT */
		if (back == 0) {
			rect(bitmap, x_min, y_min, x_max, y_max, gui_color[3]);
		}
		border = 0;
		break;
	}
	
	switch (back) {
	case 1: rectfill(bitmap, x_min + border, y_min + border, x_max - border, y_max - border, gui_color[1]); break;
	case 2: rectfill(bitmap, x_min + border, y_min + border, x_max - border, y_max - border, gui_color[2]); break;
	case 3: rectfill(bitmap, x_min + border, y_min + border, x_max - border, y_max - border, gui_color[3]); break;
	}
}

bool WinThing::mouse_on(int border)
{
	if (mouse_x < (x_min + border) || mouse_y < (y_min + border) ||
	    mouse_x > (x_max - border) || mouse_y > (y_max - border)) return false;
	else return true;
}



//===================   ButtonThing   ================================================


ButtonThing::ButtonThing(int ix_min, int iy_min, int ix_max, int iy_max, char *ilabel):
pressed(FALSE),
x_min(ix_min), y_min(iy_min),
x_max(ix_max), y_max(iy_max),
label(ilabel)
{
	w_button = new WinThing(ix_min, iy_min, ix_max, iy_max, 1);
}

ButtonThing::~ButtonThing()
{
	delete w_button;
}

void ButtonThing::set_button(int ix_min, int iy_min, int ix_max, int iy_max, char *ilabel)
{
	x_min = (ix_min == -99) ? x_min : ix_min;
	y_min = (iy_min == -99) ? y_min : iy_min;
	x_max = (ix_max == -99) ? x_max : ix_max;
	y_max = (iy_max == -99) ? y_max : iy_max;
	
	label = (ilabel) ? ilabel : label;

	w_button->set_rect(ix_min, iy_min, ix_max, iy_max, -99);
}

bool ButtonThing::clicked()
{
	pressed = true;
	bool prev_pressed = false;
	do {
		if (mouse_on()) {pressed = true;}
		else            {pressed = false;}

		if (prev_pressed != pressed) {
			draw();
			add_rect(x_min, y_min, x_max, y_max); put_rects();
			prev_pressed = pressed;
		}
	} while (mouse_b & 1);

	pressed = false;
	return prev_pressed;
}

void ButtonThing::draw()
{
	if (pressed) {
		w_button->set_rect(-99, -99, -99, -99, -1);
		w_button->draw();
		textout_centre(buffer, font, label, (x_max + x_min)/2 + 2, (y_min + y_max - text_height(font)) / 2 + 1, gui_color[4]);
	}
	else {
		w_button->set_rect(-99, -99, -99, -99,  1);
		w_button->draw();
		textout_centre(buffer, font, label, (x_max + x_min)/2 + 1, (y_min + y_max - text_height(font)) / 2,     gui_color[4]);
	}

}

bool ButtonThing::mouse_on()
{
	return w_button->mouse_on();
}



//===================   ScrollThing   ================================================


ScrollThing::ScrollThing(int ix_min, int iy_min, int ix_max, int iy_max, int idir, int io_length, int iw_length, int ishift):
grab_dist(0),
dir(idir)
{
	w_scrollbar  = new WinThing(x_min, y_min, x_max, y_max, W_HIGH);
	w_scrollback = new WinThing(x_min, y_min, x_max, y_max, W_FLAT);
	
	set_values(ix_min, iy_min, ix_max, iy_max, io_length, iw_length, ishift);
}

ScrollThing::~ScrollThing()
{
	delete w_scrollbar;
	delete w_scrollback;
}

void ScrollThing::set_values(int ix_min, int iy_min, int ix_max, int iy_max, int io_length, int iw_length, int ishift)
{
	x_min    = (ix_min    == -99) ? x_min : ix_min;		// When -99, the previous value is preserved
	y_min    = (iy_min    == -99) ? y_min : iy_min;
	x_max    = (ix_max    == -99) ? x_max : ix_max;
	y_max    = (iy_max    == -99) ? y_max : iy_max;
	o_length = (io_length == -99) ? o_length : io_length;
	w_length = (iw_length == -99) ? w_length : iw_length;
	shift    = (ishift    == -99) ? shift : ishift;

	w_scrollbar->set_rect (x_min, y_min, x_max, y_max);
	w_scrollback->set_rect(x_min, y_min, x_max, y_max, W_DEEP);

	short int min, max;
	switch (dir) {
	case 0: min = x_min; max = x_max; break;
	default: min = y_min; max = y_max; break;
	}
	
	my_length = max - min + 1;
	bar_size = short((o_length > 0) ? (w_length * my_length) / o_length : my_length);
	if (bar_size > my_length - 1) {
		bar_size = my_length - 1;
		min_scroll = min;
	}
	else {
		min_scroll = short(min + ((shift * my_length) / o_length));
	}
	max_scroll = min_scroll + bar_size;
	
	switch (dir) {
	case 0: w_scrollbar->set_rect(min_scroll, y_min, max_scroll, y_max); break;
	default: w_scrollbar->set_rect(x_min, min_scroll, x_max, max_scroll); break;
	}
}

void ScrollThing::draw()
{
	w_scrollback->draw();
	w_scrollbar->draw();
}

int ScrollThing::handle(bool init)
{
	short int min, max;
	
	switch (dir) {
	case 0:
		if (init) {grab_dist = mouse_x - min_scroll - 1;}
		min_scroll = mouse_x - grab_dist;
		break;
	case 1:
		if (init) {grab_dist = mouse_y - min_scroll - 1;}
		min_scroll = mouse_y - grab_dist;
		break;
	}

	switch (dir) {
	case 0: min = x_min; max = x_max; break;
	default: min = y_min; max = y_max; break;
	}

	min_scroll = (min_scroll < min) ? min : min_scroll;
	min_scroll = (min_scroll > max - bar_size + 1) ? max - bar_size + 1 : min_scroll;

	switch (dir) {
	case 0: shift = min_scroll - min; break;
	default: shift = min_scroll - min; break;
	}

	shift = ((shift) * o_length) / my_length;
	set_values();
	return shift;
}

bool ScrollThing::mouse_on(){
	return w_scrollbar->mouse_on();
}



//===================   MenuThing   ==================================================


MenuThing::MenuThing(int ix, int iy, char *iitems):
x(ix), y(iy),
width(10), height(3),
nr_of_items(0), selected_item(-1)
{
	char *items = new char[256];
	ustrncpy (items, iitems, 256);
	
	// Make a menu from the items string...
	//  - Menu items are seperated by ","
	//  - Horizontal lines can be inserted with ",|,"
	char *token;
	for(token = ustrtok(items, ","); token; token = ustrtok(NULL, ",")) {
		if (nr_of_items < 16) {
			item[nr_of_items] = token;
			if (width < 22 + text_length(font, item[nr_of_items])) {
				width = 22 + text_length(font, item[nr_of_items]);
			}
			height += (ustrcmp(item[nr_of_items], "|") == 0) ? 6 : text_height(font);
			nr_of_items++;
		}
	}

	x = (x + width  > SCREEN_W - 1) ? x - width  : x;
	y = (y + height > SCREEN_H - 1) ? y - height : y;
	w_back = new WinThing(x, y, x + width, y + height);
	background = create_bitmap(width + 1, height + 1);
}

MenuThing::~MenuThing()
{
	delete w_back;
	destroy_bitmap(background);
}

void MenuThing::locate(int ix, int iy)
{
	x = ix; y = iy;
	x = (x + width  > SCREEN_W - 1) ? x - width  : x;
	y = (y + height > SCREEN_H - 1) ? y - height : y;
	w_back->set_rect(x, y, x + width, y + height);
}

int  MenuThing::handle()
{
	int prev_selected_item = selected_item = -1;
	
	do {;} while (mouse_b & 2);
	blit(buffer, background, x, y, 0, 0, width + 1, height + 1);
	draw(); put_rects();
	
	do {
		// First, just track mouse and try to give an interactive feel...
		do {
			selected_item = find_selected_item();
			if (prev_selected_item != selected_item) {
				draw(); put_rects();
				prev_selected_item = selected_item;
			}
		} while (!((mouse_b & 1) || (mouse_b & 2)));
		
		// Check whether the click was outside the menu, in which case we close it.
		if (selected_item == -1) {
			blit(background, buffer, 0, 0, x, y, width + 1, height + 1);
			add_rect(x, y, x + width, y + height); put_rects();
			return selected_item;
		}
		
		// Keep tracking the mouse to get the final choice...
		do {
			selected_item = find_selected_item();
			if (prev_selected_item != selected_item) {
				draw(); put_rects();
				prev_selected_item = selected_item;
			}
		} while (((mouse_b & 1) || (mouse_b & 2)));
	} while (selected_item < 0);
	
	blit(background, buffer, 0, 0, x, y, width + 1, height + 1);
	add_rect(x, y, x + width, y + height); put_rects();
	return selected_item;
} 

int MenuThing::find_selected_item()
{
	int y_plus = 0, nr = 0, nr_of_hlines = 0;
	WinThing item_rect(x, y, x + width, y + 13);

	do {
		if (ustrcmp(item[nr], "|") == 0) {
			item_rect.set_rect(-99, y + y_plus, -99, y + y_plus + 9);
			if (item_rect.mouse_on(2)) {return -2;}
			y_plus += 6; nr_of_hlines++;
		}
		else {
			item_rect.set_rect(-99, y + y_plus, -99, y + y_plus + text_height(font) + 3);
			if (item_rect.mouse_on(2)) {return nr - nr_of_hlines;}
			y_plus += text_height(font);
		}
		nr++;
	} while (nr < nr_of_items);
	
	return -1;
}

void MenuThing::draw()
{
	w_back->draw();

	int y_plus = 0, nr = 0, nr_of_hlines = 0;

	do {
		if (ustrcmp(item[nr], "|") == 0) {
			hline(buffer, x + 4, y + 4 + y_plus, x + width - 4, gui_color[1]);
			hline(buffer, x + 4, y + 5 + y_plus, x + width - 4, gui_color[4]);
			y_plus += 6; nr_of_hlines++;
		}
		else {
			if ((nr - nr_of_hlines) == selected_item) {
				rectfill(buffer, x + 2, y + 2 + y_plus, x + width - 2, y + 1 + y_plus + text_height(font), gui_color[4]);
				textout(buffer, font, item[nr], x + 12, y + y_plus + 1, gui_color[1]);
			}
			else {
				//textout(buffer, font, item[nr], x + 13, y + y_plus + 2, gui_color[4]);
				//textout(buffer, font, item[nr], x + 12, y + y_plus + 1, gui_color[1]);
				textout(buffer, font, item[nr], x + 12, y + y_plus + 1, gui_color[4]);
			}
			y_plus += text_height(font);
		}
		nr++;
	} while (nr < nr_of_items);
	
	add_rect(x, y, x + width, y + height);
}


//===================   Initialize GUI   =============================================


void init_GUI() {
	DATAFILE *temp = load_datafile_object("GUI.dat", "SmallFont");
	if (temp) {font = ((FONT *)temp->dat);}
}



//===================   Standard mouse pointers   ====================================


void set_GUI_mouse(int pointer_type)
{
	show_mouse(NULL);
	if (mouse_pointer) unload_datafile_object(mouse_pointer);
	
	switch (pointer_type) {
	case P_FILE:
		mouse_pointer = load_datafile_object("GUI.dat", "_MS_FILE_BMP");
		if (mouse_pointer) set_mouse_sprite((BITMAP *)mouse_pointer->dat);
		set_mouse_sprite_focus(5, 6);
		break;
	default: /* P_STANDARD */
		mouse_pointer = load_datafile_object("GUI.dat", "_MS_STD_BMP");
		if (mouse_pointer) set_mouse_sprite((BITMAP *)mouse_pointer->dat);
	}
	
	if (!mouse_pointer) set_mouse_sprite(NULL);
	
	show_mouse(screen);
}



//===================   Global GUI functions   =======================================


void set_GUI_color(int r, int g, int b)
{
	gui_color[0] = makecol(0,0,0);
	gui_color[1] = makecol(r / 3, g / 3, b / 3);
	gui_color[2] = makecol(r / 2, g / 2, b / 2);
	gui_color[3] = makecol(r,     g,     b    );
	gui_color[4] = makecol(r * 2, g * 2, b * 2);
	gui_color[5] = makecol(r*2+20, g*2+20, b*2+20);
}