/*
 *  EditTiles.cpp:	Classes of the tile-related parts of RPG Edit
 *  By Bjrn Lindeijer
 *
 ************************************************************************************/

#include <vector>
#include <allegro.h>
#include "Errors.h"
#include "DRS.h"
#include "GUI.h"
#include "tiledmap.h"
#include "Editor.h"

using namespace std;

extern int gui_color[];

extern Tile_Edit_Window   *window_tile_edit;
extern All_Tiles_Window   *window_all_tiles;
extern Tiled_Map_Window   *window_tiled_map;
extern Palet_Window       *window_palet;
/*
extern Tile_Select_Window *window_tile_select;
*/

extern volatile int updates_to_do;
extern BITMAP *buffer;
extern int selected_color;

extern TileRepository *tileRepository;
extern TileType *selectedTile;

extern char status_message[];



//===================   Tile_Edit_Window   ===========================================


Tile_Edit_Window::Tile_Edit_Window(int ix_min, int iy_min, int ix_max, int iy_max)
{
	x_min = ix_min; x_max = ix_max;
	y_min = iy_min; y_max = iy_max;

	w_main = new WinThing   (x_min,     y_min,      x_max,                   y_max);
	w_zoom = new WinThing   (x_min + 2, y_min + 2,  x_min + (TILES_W*5) + 2, y_min + (TILES_H*5) + 2, -1, 2);
	b_grid = new ButtonThing(x_min + 2, y_max - 15, x_min + (TILES_W*5) + 2, y_max - 2, "Grid");

	grid = true; b_grid->pressed = true;
	changed = true;
}

Tile_Edit_Window::~Tile_Edit_Window()
{
	delete w_main;
	delete w_zoom;
	delete b_grid;
}

void Tile_Edit_Window::user_input()
{
	if (mouse_b & 1) {								// --> You are pressing the left mouse button,
		if (b_grid->mouse_on()) {						  // ---> hovering above grid button:
			grid = !grid;
			b_grid->pressed = !b_grid->pressed;
			draw(); put_rects();
			do {;} while (mouse_b & 1);
			updates_to_do = 0;
		}
		else if (selectedTile && w_zoom->mouse_on()) {
			// Draw in the selected color
			int x_pos = (mouse_x - (x_min + 3)) / 5;
			int y_pos = (mouse_y - (y_min + 3)) / 5;
			x_pos = (x_pos < 0) ? 0 : (x_pos > TILES_W - 1) ? TILES_W - 1 : x_pos;
			y_pos = (y_pos < 0) ? 0 : (y_pos > TILES_H - 1) ? TILES_H - 1 : y_pos;

			BITMAP* tileBitmap = selectedTile->getBitmap();
			if (tileBitmap) {
				if (selected_color != getpixel(tileBitmap, x_pos, y_pos)) {
					putpixel(tileBitmap, x_pos, y_pos, selected_color);
					window_tile_edit->changed = true;
					window_all_tiles->changed = true;
					window_tiled_map->changed = true;
				}
			}

		}
	}
	else if (mouse_b & 2) {
		if (selectedTile && w_zoom->mouse_on()) {
			// Select the color beneath the mouse
			int x_pos = (mouse_x - (x_min + 3)) / 5;
			int y_pos = (mouse_y - (y_min + 3)) / 5;
			x_pos = (x_pos < 0) ? 0 : (x_pos > TILES_W - 1) ? TILES_W - 1 : x_pos;
			y_pos = (y_pos < 0) ? 0 : (y_pos > TILES_H - 1) ? TILES_H - 1 : y_pos;

			BITMAP* tileBitmap = selectedTile->getBitmap();
			if (tileBitmap) {
				if (selected_color != getpixel(tileBitmap, x_pos, y_pos)) {
					selected_color = getpixel(tileBitmap, x_pos, y_pos);
					window_palet->changed = true;
				}
			}
		}
	}
}

void Tile_Edit_Window::draw()
{
	w_main->draw();
	w_zoom->draw();
	b_grid->draw();

	if (selectedTile) {
		stretch_sprite(buffer, selectedTile->getBitmap(), x_min + 3, y_min + 3, (TILES_W*5)-1, (TILES_H*5)-1);
	}

	if (grid) {
		int i;
		for (i = 1; i < TILES_W; i++)
			vline(buffer, x_min + 2 + (i * 5), y_min + 3, y_min + (TILES_H*5) + 1/*y_max - 18*/, makecol(15, 15, 20));
		for (i = 1; i < TILES_H; i++)
			hline(buffer, x_min + 3, y_min + 2 + (i * 5), x_min + (TILES_W*5) + 1, makecol(15, 15, 20));
	}

	// HACK dinges
	window_palet->draw();

	add_rect(x_min, y_min, x_max, y_max);
	changed = false;
}



//===================   All_Tiles_Window   ===========================================


All_Tiles_Window::All_Tiles_Window(int ix_min, int iy_min, int ix_max, int iy_max)
{
	x_min = ix_min; x_max = ix_max;
	y_min = iy_min; y_max = iy_max;
	y_mov = 0;

	int main_width = (x_max - 8) - (x_min + 1) - 2;
	tiles_in_row = main_width / (TILES_W + 1);

	vector<TileType*> tiles = tileRepository->generateTileArray();

	w_main1     =    new WinThing(x_min,     y_min,     x_max - 7, y_max,      W_FLAT, 0);
	w_main2     =    new WinThing(x_min + 1, y_min + 1, x_max - 8, y_max - 1,  W_DEEP, 2);
	s_scrollbar = new ScrollThing(x_max - 6, y_min,     x_max,     y_max,      1,
		(tiles.size() / tiles_in_row) * (TILES_H + 1) + 1, (y_max - 2) - (y_min + 1), y_mov);

	changed = true;
	active = true;
}

All_Tiles_Window::~All_Tiles_Window()
{
	delete w_main1;
	delete w_main2;
	delete s_scrollbar;
}

void All_Tiles_Window::resize_to(int ix_min, int iy_min, int ix_max, int iy_max)
{
	x_min = (ix_min == -99) ? x_min : ix_min;
	y_min = (iy_min == -99) ? y_min : iy_min;
	x_max = (ix_max == -99) ? x_max : ix_max;
	y_max = (iy_max == -99) ? y_max : iy_max;

	int main_width = (x_max - 8) - (x_min + 1) - 2;
	tiles_in_row = main_width / (TILES_W + 1);

	vector<TileType*> tiles = tileRepository->generateTileArray();

	w_main1->set_rect      (x_min,     y_min,     x_max - 7, y_max);
	w_main2->set_rect      (x_min + 1, y_min + 1, x_max - 8, y_max - 1);
	s_scrollbar->set_values(x_max - 6, y_min,     x_max,     y_max,
		(tiles.size() / tiles_in_row)*(TILES_H + 1) + 1, (y_max-2) - (y_min+1));
	
	changed = true;
}

void All_Tiles_Window::user_input()
{
	if (active)
	{
		if (w_main2->mouse_on(1))
		{
			vector<TileType*> tiles = tileRepository->generateTileArray();

			int x = (mouse_x - x_min - 3        ) / (TILES_W+1);
			int y = (mouse_y - y_min - 3 + y_mov) / (TILES_H+1);

			x = (x < 0) ? 0 : (x > tiles_in_row - 1) ? tiles_in_row - 1 : x;
			y = (y < 0) ? 0 : y;

			int tile_nr = y * tiles_in_row + x;

			if (tile_nr < (int)tiles.size() && tiles[tile_nr]) {
				sprintf(status_message, "(%s)", tiles[tile_nr]->getName());
			}

			if (mouse_b & (1 | 2))
			{
				if (tile_nr < (int)tiles.size() && tiles[tile_nr] != selectedTile) {
					selectedTile = tiles[tile_nr];
					changed = true;
					window_tile_edit->changed = true;
				}
			}
		}
		else if (s_scrollbar->mouse_on()) {
			if (mouse_b & (1 | 2))
			{
				int prev_y_mov = y_mov;
				s_scrollbar->handle(1);
				do {
					y_mov = s_scrollbar->handle(0);
					
					if (prev_y_mov != y_mov) {
						prev_y_mov = y_mov;
						draw(); put_rects();
					}
				} while (mouse_b & 1);
				updates_to_do = 0;
			}
		}
	}
}

void All_Tiles_Window::draw()
{
	if (active) {
		vector<TileType*> tiles = tileRepository->generateTileArray();

		w_main1->draw();
		w_main2->draw();
		s_scrollbar->set_values(-99, -99, -99, -99, (tiles.size() / tiles_in_row) * (TILES_H + 1) + 1, -99, y_mov);
		s_scrollbar->draw();
		
		int x, y;
		WinThing tile_rect(x_min, y_min, x_min, y_min, 0, 0);
		
		set_clip(buffer, x_min + 2, y_min + 2, x_max - 7, y_max - 2);
		for (int i = 0; i < (int)tiles.size(); i++) {
			x = i % tiles_in_row; y = i / tiles_in_row;
			
			if (tiles[i] == selectedTile) {
				tile_rect.set_rect(x_min + x*(TILES_W+1)+2, y_min + y*(TILES_H+1)+2 - y_mov,
					x_min + x*(TILES_W+1)+2 + (TILES_W + 1), y_min + y*(TILES_H+1)+2 + (TILES_H + 1) - y_mov, -2);
				tile_rect.draw();
			}
			
			draw_sprite(buffer, tiles[i]->getBitmap(),
				x_min + (TILES_W+1)*x + 3, y_min + (TILES_H+1)*y + 3 - y_mov);
		}
		set_clip(buffer, 0, 0, buffer->w - 1, buffer->h - 1);
		
		add_rect(x_min, y_min - 18, x_max, y_max);
		changed = false;
	}
}



//===================   Palet_Window   ===============================================


Palet_Window::Palet_Window(int ix_min, int iy_min, int ix_max, int iy_max)
{
	x_min = ix_min; x_max = ix_max;
	y_min = iy_min; y_max = iy_max;

	w_main = new WinThing(x_min, y_min, x_max, y_max);
	w_color = new WinThing(x_min + 2, y_min + 2, x_max - 2, y_min + 22, W_DEEP, 0);

	rgb_to_hsv(r, g, b, &h, &s, &v);

	s_r = new ScrollThing(x_min + 2, y_min + 26, x_max - 10, y_min + 32, 0, 283, 28, r);
	s_g = new ScrollThing(x_min + 2, y_min + 36, x_max - 10, y_min + 42, 0, 283, 28, g);
	s_b = new ScrollThing(x_min + 2, y_min + 46, x_max - 10, y_min + 52, 0, 283, 28, b);
	s_v = new ScrollThing(x_min + 2, y_min + 66, x_max - 10, y_min + 72, 0, 283, 28, v * 255);

	w_r = new WinThing(x_max - 8, y_min + 26, x_max - 2, y_min + 32, W_DEEP, 0);
	w_g = new WinThing(x_max - 8, y_min + 36, x_max - 2, y_min + 42, W_DEEP, 0);
	w_b = new WinThing(x_max - 8, y_min + 46, x_max - 2, y_min + 52, W_DEEP, 0);

	changed = true;
}

Palet_Window::~Palet_Window()
{
	delete w_main;
	delete w_color;
	delete s_r;
	delete s_g;
	delete s_b;
	delete s_v;
	delete w_r;
	delete w_g;
	delete w_b;
}


void Palet_Window::user_input()
{
	if (mouse_b & 1)
	{
		if (s_r->mouse_on()) {
			int prev_r = getr(selected_color);
			s_r->handle(1);
			do {
				r = s_r->handle(0);
				r = (r < 0) ? 0 : (r > 255) ? 255 : r;
				if (prev_r != r) {
					selected_color = makecol(r, getg(selected_color), getb(selected_color));
					prev_r = r;
					draw(); put_rects();
				}
			} while (mouse_b & 1);
			updates_to_do = 0;
		}
		else if (s_g->mouse_on()) {
			int prev_g = getg(selected_color);
			s_g->handle(1);
			do {
				g = s_g->handle(0);
				g = (g < 0) ? 0 : (g > 255) ? 255 : g;
				if (prev_g != g) {
					selected_color = makecol(getr(selected_color), g, getb(selected_color));
					prev_g = g;
					draw(); put_rects();
				}
			} while (mouse_b & 1);
			updates_to_do = 0;
		}
		else if (s_b->mouse_on()) {
			int prev_b = getb(selected_color);
			s_b->handle(1);
			do {
				b = s_b->handle(0);
				b = (b < 0) ? 0 : (b > 255) ? 255 : b;
				if (prev_b != b) {
					selected_color = makecol(getr(selected_color), getg(selected_color), b);
					prev_b = b;
					draw(); put_rects();
				}
			} while (mouse_b & 1);
			updates_to_do = 0;
		}
		else if (s_v->mouse_on()) {
			int prev_r = getr(selected_color);
			int prev_g = getg(selected_color);
			int prev_b = getb(selected_color);
			int prev_vi, vi;
			float prev_h, prev_s, prev_v;
			rgb_to_hsv(prev_r, prev_g, prev_b, &prev_h, &prev_s, &prev_v);
			vi = (int)(prev_v * 255);
			prev_vi = vi;

			s_v->handle(1);
			do {
				vi = s_v->handle(0);
				vi = (vi < 0) ? 0 : (vi > 255) ? 255 : vi;
				if (prev_vi != vi) {
					hsv_to_rgb(prev_h, prev_s, (float)vi / 255.0, &prev_r, &prev_g, &prev_b);

					selected_color = makecol(prev_r, prev_g, prev_b);
					prev_vi = vi;
					draw(); put_rects();
				}
			} while (mouse_b & 1);
			updates_to_do = 0;
		}
	}
}

void Palet_Window::draw()
{
	r = getr(selected_color);
	g = getg(selected_color);
	b = getb(selected_color);

	// Calculate value
	//void hsv_to_rgb(float h, float s, float v, int *r, int *g, int *b);
	rgb_to_hsv(r, g, b, &h, &s, &v);

	//w_main->draw();

	s_r->set_values(-99, -99, -99, -99, -99, -99, r);
	s_g->set_values(-99, -99, -99, -99, -99, -99, g);
	s_b->set_values(-99, -99, -99, -99, -99, -99, b);
	s_v->set_values(-99, -99, -99, -99, -99, -99, v * 255);
	s_r->draw();
	s_g->draw();
	s_b->draw();
	s_v->draw();
	w_r->draw();
	w_g->draw();
	w_b->draw();
	rectfill(buffer, x_max - 7, y_min + 27, x_max - 3, y_min + 31, makecol(r,0,0));
	rectfill(buffer, x_max - 7, y_min + 37, x_max - 3, y_min + 41, makecol(0,g,0));
	rectfill(buffer, x_max - 7, y_min + 47, x_max - 3, y_min + 51, makecol(0,0,b));

	w_color->draw();
	rectfill(buffer, x_min + 3, y_min + 3,  x_max - 3, y_min + 21, selected_color);

	add_rect(x_min, y_min, x_max, y_max);
	changed = false;
}



//===================   Tile_Select_Window   =========================================

/*
Tile_Select_Window::Tile_Select_Window(int ix_min, int iy_min, int ix_max, int iy_max)
{
	x_min = ix_min; x_max = ix_max;
	y_min = iy_min; y_max = iy_max;
	
	w_main      = new WinThing   (x_min,     y_min + 18, x_max - 7, y_max, 2);
	w_menu      = new WinThing   (x_min,     y_min,      x_max,     y_min + 17);
	b_all_tiles = new ButtonThing(x_min + 5, y_min + 2,  x_max - 5, y_min + 15, "Show all tiles");
	s_scrollbar = new ScrollThing(x_max - 6, y_min + 18, x_max,     y_max, 1, 0, (y_max-3) - (y_min+18), 0);
	popup_menu  = new MenuThing  (x_min,     y_min, "Remove tile,Add selected tile");
	window_all_tiles->resize_to  (-99,       y_min + 18,   -99,       -99);
	
	for (int i = 0; i < MAX_GROUPS; i++) {
		group_label[i] = new char[16];
		ustrncpy(group_label[i], "untitled", 16);
		group_tiles[i] = 0;
		group_open[i]  = false;
		w_group_border[i] = new WinThing(x_min + 5, -99, x_max - 13, -99, 3, 2);
		b_group[i]     = new ButtonThing(x_min + 8, -99, x_max - 15, -99, group_label[i]);
	}

	load_groups("standard.ord");
	sel_group = -1; sel_tile = -1;

	calculate_values();
	changed = true;
}

Tile_Select_Window::~Tile_Select_Window()
{
	delete w_main;
	delete w_menu;
	delete b_all_tiles;
	delete s_scrollbar;
	delete popup_menu;
	for (int i = 0; i < MAX_GROUPS; i++) {
		delete group_label[i];
		delete w_group_border[i];
		delete b_group[i];
	}
}

void Tile_Select_Window::user_input()
{
	if ((key[KEY_PGDN]) && (y_mov < (length - ((y_max-3) - (y_min+18)))) ) {y_mov++; changed = true;}
	if ((key[KEY_PGUP]) && (y_mov > 0))                                    {y_mov--; changed = true;}

	if (mouse_b & 1) {						// --> You are pressing the left mouse button
		if (w_main->mouse_on(3)) {
			for (int i = 0; i < MAX_GROUPS; i++) {
				if (b_group[i]->mouse_on()) {			// hovering above a group button
					set_clip(buffer, x_min + 2, y_min + 18, x_max - 9, y_max - 2);
					if (b_group[i]->clicked()) {
						group_open[i] = !group_open[i];
					}
					set_clip(buffer, 0, 0, buffer->w - 1, buffer->h - 1);
					changed = true;
					i = MAX_GROUPS;
					updates_to_do = 0;
				}
				else if (group_open[i]) {				// the group is open, so you might be selecting a tile
					int height = (group_tiles[i] / tiles_in_row) * (TILES_H + 1) + 1;
					height = (group_tiles[i] % tiles_in_row > 0) ? height + (TILES_H + 1) : height;
					
					if (!(mouse_x < x_centre ||
						  mouse_x > x_centre + tiles_in_row * (TILES_W+1) - 1 ||
						  mouse_y < group_y[i] + 15 || mouse_y > group_y[i] + 13 + height))
					{
						int x = (mouse_x - x_centre)          / (TILES_W + 1);
						int y = (mouse_y - (group_y[i] + 15)) / (TILES_H + 1);
						int tile = y * tiles_in_row + x;
						if (tile < group_tiles[i]) {
							if (sel_group != i || sel_tile != tile) {
								sel_group = i; sel_tile = tile;
								changed = true;
							}

							int tile_nr = group_tile[i][tile];
							if (tile_nr != selected_tile) {
								selected_tile = tile_nr;
								window_tile_edit->changed = true;
								if (window_all_tiles->active) {window_all_tiles->changed = true;}
							}
						}
						i = MAX_GROUPS;
					}
				}
			}
		}
		else if (s_scrollbar->mouse_on()) {				// hovering above the scrollbar
			int prev_y_mov = y_mov;
			s_scrollbar->handle(1);
			do {
				y_mov = s_scrollbar->handle(0);

				if (prev_y_mov != y_mov) {
					prev_y_mov = y_mov;
					draw(); put_rects();
				}
			} while (mouse_b & 1);
			updates_to_do = 0;
		}
		else if (b_all_tiles->mouse_on()) {
			if (b_all_tiles->clicked()) {
				if (window_all_tiles->active) {
					all_tiles_off();
				}
				else {
					window_all_tiles->active = true;
					window_all_tiles->changed = true;
					b_all_tiles->set_button(-99,-99,-99,-99, "Hide all tiles");
					w_menu->set_rect(x_min - 284);
					window_tiled_map->resize_to(-99, -99, x_min - 285, -99);
					changed = true;
				}
			}
		}
	}
	if (mouse_b & 2) {						// --> You are pressing the right mouse button
		if (w_main->mouse_on(3)) {
			for (int i = 0; i < MAX_GROUPS; i++) {
				if (group_open[i]) {				// the group is open, so you might be clicking a tile
					int height = (group_tiles[i]/tiles_in_row) * 17 + 1;
					height = (group_tiles[i]%tiles_in_row > 0)? height + 17 : height;
					
					if (w_group_border[i]->mouse_on(1)) {
						int x = (mouse_x - x_centre)          / (TILES_W+1);
						int y = (mouse_y - (group_y[i] + 15)) / (TILES_H+1);
						x = (x < 0) ? 0 : (x > tiles_in_row - 1) ? tiles_in_row - 1 : x;
						int tile = y * tiles_in_row + x;
						if (sel_group != i || sel_tile != tile) {
							if (tile < group_tiles[i]) {
								sel_group = i; sel_tile = tile;
								draw();
								put_rects();
							}
						}

						popup_menu->locate(mouse_x - 1, mouse_y - 1);
						switch (popup_menu->handle()) {
						case 0: remove_tile(i, tile); break;
						case 1: add_tile(i, tile, selected_tile); break;
						}
						updates_to_do = 0;
						i = MAX_GROUPS;
					}
				}
			}
		}
	}
}

void Tile_Select_Window::all_tiles_off() {
	window_all_tiles->active = false;
	b_all_tiles->set_button(-99,-99,-99,-99, "Show all tiles");
	w_menu->set_rect(x_min);
	window_tiled_map->resize_to(-99, -99, x_min - 1, -99);
	changed = true;
}

void Tile_Select_Window::draw()
{
	WinThing tile_rect(x_min, y_min, x_min, y_min, 0, 0);

	calculate_values();
	w_menu->draw();
	b_all_tiles->draw();
	w_main->draw();

	int height;

	set_clip(buffer, x_min + 2, y_min + 21, x_max - 9, y_max - 3);
	for (int i = 0; i < MAX_GROUPS; i++) {
		if (group_open[i] && group_tiles[i] > 0) {
			height = (group_tiles[i] / tiles_in_row) * (TILES_H + 1) + 1;
			height = (group_tiles[i] % tiles_in_row > 0) ? height + (TILES_H + 1) : height;
		}
		else {
			height = 0;
		}

		if (group_y[i] < y_max - 2) {
			b_group[i]->set_button     (-99, group_y[i],     -99, group_y[i] + 14, group_label[i]);
			w_group_border[i]->set_rect(-99, group_y[i] + 6, -99, group_y[i] + 18 + height);
			w_group_border[i]->draw();

			b_group[i]->draw();

			if (group_open[i] && group_tiles[i] > 0) {
				hline(buffer, x_min + 11, group_y[i] + 7, x_min + 15, gui_color[0]);
				int x, y;
				for (int nr=0; nr<group_tiles[i]; nr++) {
					x = (nr % tiles_in_row);
					y = (nr / tiles_in_row);
					int tile_nr = group_tile[i][nr];
					if (tile_bmp[tile_nr]) {
						if (sel_group == i && sel_tile == nr) {
							tile_rect.set_rect(
								x * (TILES_W + 1) + x_centre - 1,
								y * (TILES_H + 1) + group_y[i] + 16,
								x * (TILES_W + 1) + x_centre - 1 + (TILES_W + 1),
								y * (TILES_H + 1) + group_y[i] + 16 + (TILES_H + 1),
								-2
							);
							tile_rect.draw();
						}

						draw_sprite(buffer, tile_bmp[tile_nr],
							x*(TILES_W+1) + x_centre, y*(TILES_H+1) + group_y[i] + 17);
					}
				}
			}
			else {
				hline(buffer, x_min + 11, group_y[i] + 7, x_min + 15, gui_color[0]);
				vline(buffer, x_min + 13, group_y[i] + 5, group_y[i] + 9, gui_color[0]);
			}
		}
	}
	set_clip(buffer, 0, 0, buffer->w - 1, buffer->h - 1);

	s_scrollbar->set_values(-99, -99, -99, -99, length, -99, y_mov);
	s_scrollbar->draw();

	add_rect(x_min, y_min, x_max, y_max);
	changed = false;
}

void Tile_Select_Window::calculate_values()
{
	tiles_in_row = ((x_max - 16) - (x_min + 8)) / (TILES_W + 1);
	x_centre =    (((x_max - 15) - (x_min + 7)) - (tiles_in_row * (TILES_W + 1) - 1)) / 2 + x_min + 8;

	// length of virtual window
	int height;
	length = 4;
	for (int i = 0; i < MAX_GROUPS; i++) {
		if (group_open[i] && group_tiles[i] > 0) {
			height = (group_tiles[i] / tiles_in_row) * (TILES_H + 1) + 1;
			height = (group_tiles[i] % tiles_in_row > 0) ? height + (TILES_H + 1) : height;
		}
		else {
			height = 0;
		}

		group_y[i] = length + y_min + 20 - y_mov;
		length += 22 + height;
	}
	length++;
	int prev_y_mov =  y_mov;  correct_y_mov();
	if (prev_y_mov != y_mov) {calculate_values();}
}

void Tile_Select_Window::correct_y_mov()
{
	if (y_mov < 0) {y_mov = 0;}
	if (y_mov > (length - ((y_max-3) - (y_min+18)))) {y_mov = (length - ((y_max-3) - (y_min+18)));}
	if (length < ((y_max-3) - (y_min+18))) {y_mov = 0;}
}

void Tile_Select_Window::remove_tile(int group_nr, int tile_nr)
{
	if (group_nr < 0 || group_nr >= MAX_GROUPS) {output_error(1, "Error, cannot remove tile from group nr. %i (out of range)", group_nr);}
	if (tile_nr < 0 || tile_nr >= MAX_TILES_IN_GROUP) {return;}

	for (int i = tile_nr; i < MAX_TILES_IN_GROUP - 1; i++) {
		group_tile[group_nr][i] = group_tile[group_nr][i+1];
	}
	group_tile[group_nr][MAX_TILES_IN_GROUP - 1] = 0;
	group_tiles[group_nr]--;

	changed = true;
}

void Tile_Select_Window::add_tile(int group_nr, int tile_nr, int add_tile)
{
	if (group_nr < 0 || group_nr >= MAX_GROUPS) {output_error(1, "Error, cannot add tile to group nr. %i (out of range)", group_nr);}
	if (group_tiles[group_nr] == MAX_TILES_IN_GROUP) {output_error(0, "Error, group %i is full", group_nr); return;}

	tile_nr = (tile_nr < 0) ? 0 : tile_nr;
	tile_nr = (tile_nr > group_tiles[group_nr]) ? group_tiles[group_nr] : tile_nr;

	if (tile_nr < group_tiles[group_nr]) {
		for (int i = MAX_TILES_IN_GROUP - 1; i >= tile_nr; i--) {
			group_tile[group_nr][i] = group_tile[group_nr][i-1];
		}
	}
	group_tile[group_nr][tile_nr] = add_tile;
	group_tiles[group_nr]++;

	changed = true;
}

int Tile_Select_Window::save_groups(char *filename)
{
	PACKFILE *file = pack_fopen(filename, F_WRITE_NOPACK);
	if (file == NULL) return errno;
	char version_var[8] = "0.34";

	pack_fwrite(&version_var[0], sizeof(version_var), file);

	for (int group_nr = 0; group_nr < MAX_GROUPS; group_nr++) {
		if (group_tiles[group_nr] > 0) {
			pack_iputw(group_nr, file);
			pack_fwrite(group_label[group_nr], sizeof(group_label), file);
			pack_iputw(group_tiles[group_nr], file);
			for (int tile_nr = 0; tile_nr < group_tiles[group_nr]; tile_nr++) {
				pack_iputw(group_tile[group_nr][tile_nr], file);
			}
		}
	}

	pack_fclose(file);
	return 0;
}

int Tile_Select_Window::load_groups(char *filename)
{
	int group_nr;
	
	PACKFILE *file = pack_fopen(filename, F_READ_PACKED);
	if (file == NULL) return errno;
	char version_var[8] = "";
	
	pack_fread(&version_var[0], sizeof(version_var), file);
	
	while (!pack_feof(file)) {
		group_nr = pack_igetw(file);
		pack_fread(group_label[group_nr], sizeof(group_label), file);
		group_tiles[group_nr] = pack_igetw(file);
		for (int tile_nr = 0; tile_nr < group_tiles[group_nr]; tile_nr++) {
			group_tile[group_nr][tile_nr] = pack_igetw(file);
		}
	}

	pack_fclose(file);
	return 0;
}
*/

