/*
 *  The Moonlight RPG engine  (see readme.txt about version info)
 *  By Bjrn Lindeijer
 *
 ************************************************************************************/

#include "RPG.h"
#include "DRS.h"
#include "Errors.h"
#include "MIDI.h"
#include "tiledmap.h"
#include "GUI.h"
#include "Engine.h"
#include "Script.h"
#include <allegro.h>
#include <seer.h>

volatile int frames_to_do;
volatile int fps, fps_counter;

void handle_game_time()	{frames_to_do++;}
END_OF_FUNCTION(handle_game_time);
void handle_fps_counter() {fps = fps_counter; fps_counter = 0;}
END_OF_FUNCTION(handle_fps_counter);


int inkey;							// Last pressed key

bool game_end = false;
bool bVSync = false;
bool bStretch = true;
char filename[256];
DATAFILE *interface_graphics;
DATAFILE *bitmap_data;

TiledMap *tiledMap;
TileRepository *tileRepository;
TileType *selectedTile = NULL;

BITMAP *buffer;

int main()
{
	init_engine();

	clear(screen);
	output_error(3, "All finished, initializing game...\n");
	execute_script("game", "init", NULL, NULL, 0, 0, 0);

	do {
		// Handle user input
		if (frames_to_do > 0) {
			handle_input();
			frames_to_do--;
		}

		// If less than 2 extra frames are waiting, take time to display current frame
		if (frames_to_do < 2) {draw_screen();}
		else if (frames_to_do > 10) {frames_to_do = frames_to_do % 10;}
	} while (!game_end);

	exit_program();
	return 0;
}
END_OF_MAIN();


void ex_update_game(Object* excl_obj)
{
	//if (frames_to_do > 1) frames_to_do = 1;

	// Handle user input
	if (frames_to_do > 0) {
		if (excl_obj) excl_obj->update();

		map<const char*, Object*, ltstr>::iterator i;
		for (i = objects.begin(); i != objects.end(); i++) {
			((*i).second)->update_entity();
		}

		frames_to_do--;
		draw_screen();
	}

	// If less than 2 extra frames are waiting, take time to display current frame
	//if (frames_to_do < 2) {draw_screen();}
}

void init_engine()
{
	// Allegro initialisation
	allegro_init();
	install_keyboard();
	install_timer();

	open_log_file();

	// SeeR initialisation
	output_error(3, "Initializing SeeR scripting engine...\n");
	scInit_SeeR();
	scOpen_Debug("seer.dbg");
	init_script_externals();
	prepare_scripts();

	set_config_file("RPG.cfg");
	int driver = GFX_AUTODETECT_WINDOWED;
	if (get_config_int("video", "fullscreen", 1)) {
		driver = GFX_AUTODETECT_FULLSCREEN;
	}
	bVSync = get_config_int("video", "vsync", 0);
	bStretch = get_config_int("video", "stretch", 1);

	// Screen initialisation
	int width, height;
	if (bStretch) {
		width = 640; height = 480;
	} else {
		width = 320; height = 240;
	}
	output_error(3, "Setting screen mode (%ix%ix16)...\n", width, height);
	set_color_depth(16);
	if (set_gfx_mode(driver, width, height, 0, 0) < 0) {
		output_error(3, "Failed, trying screen mode (%ix%ix15)...\n", width, height);
		set_color_depth(15);
		if (set_gfx_mode(driver, width, height, 0, 0) < 0) {
			output_error(1, "%s\n", allegro_error);
		}
	}
	text_mode(-1);

	textprintf_centre(screen, font, SCREEN_W/2, SCREEN_H/2, makecol(255,0,0), "LOADING...");

	output_error(3, "Creating screen buffer...\n");
	buffer = create_bitmap(320, 240);
	clear(buffer);

	output_error(3, "Creating tile repository...\n");
	tileRepository = new TileRepository();

	output_error(3, "Loading gui.dat...\n");
	interface_graphics = load_datafile("gui.dat");
	if (!interface_graphics) {output_error(1, "Error while loading: gui.dat");}

	DATAFILE *temp = find_datafile_object(interface_graphics, "SmallFont");
	if (temp) {font = ((FONT *)temp->dat);}

	output_error(3, "Loading bitmaps.dat...\n");
	bitmap_data = load_datafile("bitmaps.dat");
	if (!bitmap_data) {output_error(1, "Error while loading: bitmaps.dat");}

	if (init_midi()) output_error(3, "Warning: can't initialize MIDI\n");
	else {
		output_error(3, "MIDI initialized\n");
		//if (!ld_midi("music/somemidi.mid")) play_midi();
	}

	output_error(3, "Installing timers...\n");
	LOCK_VARIABLE(frames_to_do);			// Game speed handler
	LOCK_FUNCTION(handle_game_time);
	LOCK_VARIABLE(fps);						// Frames per second counter
	LOCK_VARIABLE(fps_counter);
	LOCK_FUNCTION(handle_fps_counter);
	install_int_ex(handle_game_time, BPS_TO_TIMER(LPS));
	install_int_ex(handle_fps_counter, BPS_TO_TIMER(1));
}


void wait_for_update()
{
	frames_to_do = 0;
	while (frames_to_do == 0) {;}
}


void handle_input()
{
	if (switch_area) {
		switch_area_now();
	}

	if (key[KEY_ESC]) {game_end = true;}

	if (keypressed()) {inkey = (readkey() >> 8);}
	else              {inkey = 0;}

	if (!dialog_open) {
		if (player && !exclusive_mode) {
			if (key[KEY_UP]    && player->walking == 0) { player->walk(DIR_UP); }
			if (key[KEY_LEFT]  && player->walking == 0) { player->walk(DIR_LEFT); }
			if (key[KEY_RIGHT] && player->walking == 0) { player->walk(DIR_RIGHT); }
			if (key[KEY_DOWN]  && player->walking == 0) { player->walk(DIR_DOWN); }

			if (inkey == KEY_SPACE && player->walking == 0) {
				int pos_x = player->x;
				int pos_y = player->y;

				switch (player->dir) {
				case DIR_UP:    pos_y -= 1; break;
				case DIR_DOWN:  pos_y += 1; break;
				case DIR_LEFT:  pos_x -= 1; break;
				case DIR_RIGHT: pos_x += 1; break;
				}

				map<const char*, Object*, ltstr>::iterator i;
				for (i = objects.begin(); i != objects.end(); i++) {
					Object *obj = ((*i).second);
					if ((obj->x == pos_x) && (obj->y == pos_y)) {
						execute_script(obj->script, "activate", obj, player, 0, 0, 0);
					}
				}
			}
		}

		update_objects();
	}
}


void draw_screen()
{
	if (!dialog_open) {
		if (camera_target) {
			tiledMap->setCamera(
				camera_target->entity->pos,
				Rectangle(0, 0, buffer->w, buffer->h),
				true, true
			);
		}

		clear_bitmap(buffer);
		tiledMap->draw(buffer);

		textprintf_centre(buffer, font, buffer->w / 2 + 1, 12, makecol(0,0,0)      , "%i fps", fps);
		textprintf_centre(buffer, font, buffer->w / 2,     11, makecol(128,128,150), "%i fps", fps);
	}

	update_screen();

	fps_counter++;
}

void update_screen()
{
	if (bVSync) vsync();
	if (bStretch) {
		stretch_blit(buffer, screen, 0, 0, buffer->w, buffer->h, 0, 0, SCREEN_W, SCREEN_H);
	} else {
		blit(buffer, screen, 0, 0, 0, 0, buffer->w, buffer->h);
	}
}


void exit_program()
{
	output_error(3, "Game ended\n");

	delete tiledMap;

	output_error(3, "Removing timers...\n");
	remove_int(handle_fps_counter);
	remove_int(handle_game_time);

	output_error(3, "Unloading datafiles...\n");
	unload_datafile(interface_graphics);
	unload_datafile(bitmap_data);

	end_midi();
	output_error(3, "Destroying screen buffer...\n");
	destroy_bitmap(buffer);
	clean_scripts();

	output_error(3, "Closing log file. Bye bye!\n");
	close_log_file();
}
