// Curlian
// (c) 2004-5 Peter Hull 
/// v.1.1


#include <math.h>
#include <allegro.h>
#ifdef _WIN32
#include <winalleg.h>
double GetNow() {
  return double(GetTickCount())/1000.0;
}
#else
#include <sys/time.h>
double GetNow() {
  static struct timeval tv;
  gettimeofday(&tv, NULL);
  return double(tv.tv_sec)+tv.tv_usec/1.0e6;
}
#endif
#include "curlian.h"

int rnd(int a, int b); // Return a random number between a & b
bool WaitUntil(double when); // Wait a certain time, interrupt with mouse click.

struct ColourItems {
  int white;
  int orange;
  struct ca {int c; int a; } particles[256];
} Colour;


class GrfObj {
protected:
  GrfObj(int ix, int iy, int iw, int ih, BITMAP* ii) 
    : x(ix), y(iy), w(iw), h(ih), img(ii) { tlast=0.0;}
  int x, y, w, h;
  double tlast;
  BITMAP* img;
public:
  int getX() const {return x;}
  int getY() const {return y;}
  virtual bool update(double)=0;
  virtual void draw(BITMAP* s);	
  bool hit(const GrfObj& other);
};

class Item : public GrfObj {
public:
  Item(int ix, int iy, BITMAP* ii) : GrfObj(ix,iy,ii->w,ii->h,ii) {active=true;}
  bool update(double) { return false;}
  bool isActive() const {return active;}
  void setActive(bool b) {active=b;}
  void moveTo(int cx, int cy, bool ac=true) {x=cx; y=cy; setActive(ac);}
protected:
  bool active;
};
class ScoreHint : public GrfObj {
	public:
	ScoreHint();
	void moveTo(int ix, int iy, int sc);
	bool update(double);
	void draw(BITMAP*);
	bool isActive() const {return active;}
	protected:
	int score;
	double endtime;
	bool active;
};
class Ship : public GrfObj {
public:
  Ship(int ix, int iy, BITMAP* ii) : GrfObj(ix,iy,ii->w,ii->h,ii) {
    px=ix;
    py=iy;
    rad=0.0;
    ang=0.0;
    angv=0.1;
    recalcPos(320, 240);
    imageangle=0;
    tlast=-1.0;
    nextpart=0.0;
  }
  bool update(double);
  void draw(BITMAP* s);
  void recalcPos(int nx, int ny);
  bool flyAway(double);
  bool slowDown(double);
  bool crashOff(double);
  void getXYAvoid(int& x, int& y);
protected:
  int px, py;
  double ang, rad, angv;
  fixed imageangle;
  double nextpart;
};

class Particle {
public:
  Particle();
  void init(int x, int y, double v, double ang);
  void update(double);
  void draw(BITMAP*);
  bool isActive() const {return active;}
  void reset() {active=false;}
protected:
  static const double LIFE0;
  double lifetime;
  double x, y;
  double dx, dy;
  bool active;
};
class ParticleManager {
 public:
  ParticleManager(int count=100);
  ~ParticleManager();
  void update(double);
  void draw(BITMAP* s);
  void request(int x, int y, double v, double ang, int count=1);
  void reset();
 protected:
  Particle* pts;
  int count;
  int next;
  double tlast;
} *ThePMgr;

DATAFILE* dat;
//Graphics
BITMAP* b_s;
BITMAP* b_i;
BITMAP* b_r;
BITMAP* b_t;
BITMAP* b_c;
BITMAP* b_l;
// Fonts
FONT* score_font;
// sounds
SAMPLE* swish;
SAMPLE* collect;
SAMPLE* explode;
// Other
BITMAP* osb;

int hiscore=0;
int score=0;

struct Level {
  int stars, rocks, pins, timelimit;
  bool addrocks, windin;
  const char* msg;
};

extern Level levels[];
  
void LoadResources() {
	dat=load_datafile("curlian.dat");
	set_palette(static_cast<RGB*>(dat[DAT_TitlePal].dat));
	unload_datafile(dat);
	dat=load_datafile("curlian.dat");
	b_s=static_cast<BITMAP*>(dat[DAT_Ship].dat);
	b_i=static_cast<BITMAP*>(dat[DAT_Star].dat);
	b_r=static_cast<BITMAP*>(dat[DAT_Rock].dat);
	b_t=static_cast<BITMAP*>(dat[DAT_Title].dat);
	b_c=static_cast<BITMAP*>(dat[DAT_Cursor].dat);
	b_l=static_cast<BITMAP*>(dat[DAT_Telltale].dat);

	swish=static_cast<SAMPLE*>(dat[DAT_Swish].dat);
	collect=static_cast<SAMPLE*>(dat[DAT_Collect].dat);
	explode=static_cast<SAMPLE*>(dat[DAT_Explode].dat);
	score_font=static_cast<FONT*>( dat[DAT_ScoreFont].dat);
	ThePMgr=new ParticleManager(100);
	osb=create_bitmap(640, 480);
	// Hi score
	hiscore=get_config_int("Curlian", "hiscore", 0);
	// Do the colours
	Colour.white=makecol(255,255,255);
	Colour.orange=makecol(240, 128, 0);
	float r=255.0;
	float g=255.0;
	float b=255.0;
	for (int i=0; i<256; ++i) {
	  int a=i>128 ? 511-2*i : 255;
	  Colour.particles[i].c=makecol(int(r),int(g),int(b));
	  Colour.particles[i].a=a;
	  r*=0.99;
	  g*=0.95;
	  b*=0.90;
	}
}
void UnloadResources() {
	unload_datafile(dat);
	destroy_bitmap(osb);
	set_config_int("Curlian", "hiscore", hiscore);
	delete ThePMgr;
}
enum GameStatus {
  Running, Quit, Dead, Won, TimeOut
 };
struct MenuOpt {
  int x, y, r;
  int opt;
  bool contains(int px, int py) {
    return (px-x)*(px-x)+(py-y)*(py-y)<r*r;
  }
};

// Show Menu screen and return
// 1: play
// 2: options
// 3: quit
MenuOpt MainMenu[]={
  { 320, 334, 32, 1},
  { 522, 400, 32, 3},
  { 115, 400, 32, 2},
};

void DisplayScores(BITMAP* osb) {
	textprintf(osb, score_font, 0, 0, Colour.orange, "Score: %d", score);
	textprintf(osb, score_font, 530, 0, Colour.orange, "Hi: %d", hiscore);
}

int ShowSplash() {
  int opt;
  double nw=0.0;
  while(true) {
    double tt=GetNow();
    opt=0;
    int x=mouse_x, y=mouse_y;
    blit(b_t, osb, 0, 0, 0, 0, b_t->w, b_t->h);
    ThePMgr->draw(osb);
    draw_sprite(osb, b_c, x-b_c->w/2, y-b_c->h/2);
	DisplayScores(osb);
	blit(osb, screen, 0, 0, 0, 0, 640, 480);
    for (int i=0; i<3; ++i) {
      MenuOpt& m=MainMenu[i];
      if (m.contains(x, y)) {
	opt=m.opt;
	x=m.x+int(double(m.r)*cos(tt*2*M_PI));
	y=m.y+int(double(m.r)*sin(tt*2*M_PI));
    }
   }
   if (tt>=nw) {
     ThePMgr->request(x, y, 10., tt*2.0*M_PI);
     nw=tt+0.05;
   }
   if (opt!=0 && mouse_b) 
     break;
   if (key[KEY_ESC]) {
     opt=3;
     break;
   }
   ThePMgr->update(tt);
 }
 ThePMgr->reset();
 return opt;
}
void DoOptions() {
  clear_bitmap(screen);
  textout_centre(screen, score_font, "Made with Allegro", 320, 240, Colour.white);
  textout_centre(screen, score_font, "http://alleg.sourceforge.net/", 320, 300, Colour.white);
  double wt=GetNow()+5.0;
  while(GetNow()<wt) {
    rest(100);
  }
}

int main()
{
  allegro_init();
  set_color_depth(16);
  set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0);
  install_keyboard();
  install_mouse();
  install_sound(DIGI_AUTODETECT, MIDI_NONE, 0);
  install_timer();
  srand(int(GetNow()));
  LoadResources();
  text_mode(-1);
  // Score hints 
  ScoreHint hints[10];
  while (true) {
    int ch=ShowSplash();
    if (ch==3) break;
    if (ch==2) DoOptions();
    if (ch==1) {
      Level* currentlevel=&levels[0];
      int lives=3;
      Ship ship(320, 100, b_s);
      Item* stars[10];
      Item* rocks[10];
      for (int i=0; i<10; ++i) {
	stars[i]=new Item(0, 0, b_i);
	rocks[i]=new Item(0, 0, b_r); 
      }
      score=0;
      while (currentlevel->stars>0) {
	int active_stars=currentlevel->stars;
	int active_rocks=currentlevel->rocks;
	int pins=currentlevel->pins;
	// reposition all the stars and rocks
		bool mdown=false;
	double wt=GetNow()+5.0;
	clear_bitmap(screen);
	textprintf_centre(screen, score_font, 320, 240, Colour.white, "Level %d. %s", currentlevel-levels+1,currentlevel->msg ?  currentlevel->msg: "");
	DisplayScores(screen);

	for (int i=0; i<10; ++i) {
	  int x, y;
	  ship.getXYAvoid(x, y);
	  stars[i]->moveTo(x, y, i<active_stars);
	  ship.getXYAvoid(x, y);
	  rocks[i]->moveTo(x,y, i<active_rocks);
	}
	WaitUntil(wt);
	GameStatus status=Running;
	double endtime=GetNow()+double(currentlevel->timelimit);
	int smult=10;
	while(status==Running) {
	  double tt=GetNow();
	  if (mouse_b) {
	    if (!mdown) {
	      if (score>0) --score;
	      ship.recalcPos(mouse_x, mouse_y);
	      if (currentlevel->addrocks && active_rocks<10) rocks[active_rocks++]->moveTo(mouse_x, mouse_y);
	      --pins;
	      smult=10;
	    }
	    mdown=true;
	  }
	  else {
	    mdown=false;
	  }
	  ship.update(tt);
	  ThePMgr->update(tt);
	  clear(osb);
	  ThePMgr->draw(osb);
	  ship.draw(osb);
	  for(int i=0; i<10; ++i) {
	    Item& star=*stars[i];
	    if (star.isActive()) {
	      star.draw(osb);
	      if (ship.hit(star)) {
		play_sample(collect, 255, 128, 1000, 0);
		score+=smult;
		for (int j=0; j<10; ++j) 
			if (!hints[j].isActive()) {
				hints[j].moveTo(star.getX(), star.getY(), smult);
				break;
			}
		smult+=10;
		star.setActive(false);
		if (--active_stars==0) {status=Won;}
	      }
	    }
	  }
	  for(int i=0; i<10; ++i) {
	    Item& rock=*rocks[i];
	    if (rock.isActive()) {
	      rock.draw(osb);
	      if (ship.hit(rock)) {
		play_sample(explode, 255, 128, 1000, 0);
		status=Dead;
		rock.setActive(false);
	      }
	    }
	  }
	  for (int i=0; i<10 ; ++i) {
 		    if (hints[i].isActive()) {
 		    		hints[i].update(tt);
 		    		hints[i].draw(osb);
 		    }
	  }

	  if (key[KEY_ESC]) {status=Quit;}
	  if (currentlevel->timelimit>0 && endtime<tt) {status=TimeOut;}
	  draw_sprite(osb, b_c, mouse_x-b_c->w/2, mouse_y-b_c->h/2);
	
	  DisplayScores(osb);
	  if (currentlevel->pins>0) textprintf(osb, score_font, 410, 0, Colour.white, "Pins: %d", pins);
	  if (currentlevel->timelimit>0) textprintf(osb, score_font, 160, 0, Colour.white, "Time: %d",int(endtime-tt));
	  textprintf(osb, score_font, 300, 0, Colour.white, "Lives: %d", lives);
	  if (key[KEY_F1]) save_bitmap("shot.bmp", osb, 0);
	  blit(osb, screen, 0, 0, 0, 0, 640, 480);
	  if (score>hiscore) hiscore=score;
	} // end of play loop 
	// End-of-level sequence (killed, timeout, quit, won, ..)
	bool (Ship::*func)(double);
	char* msg=0;
	switch (status) {
	case Dead: 
	  func=&Ship::crashOff;
	  msg="Crash!";
	  break;
	case TimeOut:
	  msg="Time Out!";
	  func=&Ship::slowDown;
	  break;
	case Quit:
	  func=&Ship::slowDown;
	  break;
	case Won:
	  msg="Well Done!";
	  func=&Ship::flyAway;
	  break;
	case Running:
	  // Should never happen
	  func=0;
	  break;
	}
	while(true) {
	  double tt=GetNow();
	  if (!(ship.*func)(tt)) break;
	  ThePMgr->update(tt);
	  clear(osb);
	  ThePMgr->draw(osb);
	  ship.draw(osb);
	  for(int i=0; i<10; ++i) {
	    Item& star=*stars[i];
	    if (star.isActive()) {
	      star.draw(osb);
	    }
	  }
	  for(int i=0; i<10; ++i) {
	    Item& rock=*rocks[i];
	    if (rock.isActive()) {
	      rock.draw(osb);
	    }
	  }
	  if (msg) textout_centre(osb, score_font, msg, 320, 240, Colour.white);
	 	DisplayScores(osb);
	blit(osb, screen, 0, 0, 0, 0, 640, 480);
	} // end of run-out sequence
	ThePMgr->reset();
	if (status==Quit) break;
	// Wait a bit
	WaitUntil(GetNow()+5.0);
	if (status==Dead || status == TimeOut) {
	  if (--lives<0) break;
	  ship.recalcPos(320,240);
	}
	if (status==Won) ++currentlevel;
      } // end of all levels
    }
  }
  UnloadResources();
  set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
  allegro_exit();
  
  return 0;
}
END_OF_MAIN();

void Ship::recalcPos(int nx, int ny) {
  double dx=double(x-nx);
  double dy=double(y-ny);
  double rr=(dx*dx+dy*dy);
  if (rr<400.0) return;
  rad=sqrt(rr);
  double dotp=(dx*double(x-px)+dy*double(y-py))*angv;
  angv=500.0/rad;
  ang=atan2(dy, dx);
  if (dotp<0.0) angv=-angv;
  px=nx;
  py=ny;
}

void Ship::getXYAvoid(int& x, int& y) {
  int amax=int(rad)+w;
  int amin=int(rad)-w;
  amin*=amin;
  amax*=amax;
  int r2;
  do {
    x=rnd(0,640);
    y=rnd(40,480);
    r2=(x-px)*(x-px)+(y-py)*(y-py);
  } while (r2>amin && r2<amax);
}

bool Ship::update(double now) {
  if (tlast<0.0) tlast=now;
  ang+=angv*(now-tlast);
  tlast=now;
  x=px+int(rad*cos(ang));
  y=py+int(rad*sin(ang));
  if (ang>2*M_PI) {
    ang-=2*M_PI;
    play_sample(swish, 255, 128, 1000, 0);
  }
  if (ang<-2*M_PI) {
    ang+=2*M_PI;
    play_sample(swish, 255, 128, 1000, 0);
  }
  // Draw particles
  if (now>nextpart) {
    ThePMgr->request(x, y, 100, ang+M_PI/2.0);
    nextpart=now+0.02;
  }
  return true;
}

bool Ship::slowDown(double now) {
  angv-=(now-tlast)*angv;
  update(now);
  return (angv>0.1 || angv<-0.1);
}

bool Ship::flyAway(double now) {
  x=px+int(rad*cos(ang)-(now-tlast)*sin(ang)*(angv*rad));
  y=py+int(rad*sin(ang)+(now-tlast)*cos(ang)*(angv*rad));
  if (now>nextpart) {
    ThePMgr->request(x, y, 100, ang+M_PI/2.0);
    nextpart=now+0.02;
  }
  return x>=-60 && x<700 && y>=-60 && y<540;
}
bool Ship::crashOff(double now) {
  ang+=0.5;
  y+=int((now-tlast)*(now-tlast));
  if (now>nextpart) {
    ThePMgr->request(x, y, 100, ang+M_PI/2.0,10);
    nextpart=now+0.02;
  }
  return x>=-60 && x<700 && y>=-60 && y<540;
}

void Ship::draw(BITMAP* o) {
  //line(o, px, py, x,y, Colour.white);
  int clipx, clipy;
  bool offscreen;
  if (y<0) {
    offscreen=true;
    clipy=0;
    clipx=(clipy-py)*(x-px)/(y-py)+px;
  }
  else if (y>=470) {
    offscreen=true;
    clipy=470;
    clipx=(clipy-py)*(x-px)/(y-py)+px;
  }
  else {
    clipx=x;
    clipy=y;
    offscreen=false;
  }
  if (clipx<0) {
    offscreen=true;
    clipy=(0-px)*(clipy-py)/(clipx-px)+py;
    clipx=0;
  }
  else if (clipx>=630) {
    offscreen=true;
    clipy=(630-px)*(clipy-py)/(clipx-px)+py;
    clipx=630;
  }
  if (offscreen) {
    rotate_sprite(o, b_l, clipx, clipy, ftofix(ang/M_PI*128.0));
  }
  else
    {
      rotate_sprite(o, img, x-w/2, y-h/2, ftofix(ang/M_PI*128.0+(angv>0.0 ? 64.0 : -64.0)));
    }
}
 
void GrfObj::draw(BITMAP* o) {
  draw_sprite(o, img, x-w/2, y-h/2);
}
 
bool GrfObj::hit(const GrfObj& other) {
	int dx=x-other.x;
	int dy=y-other.y;
	int jw=(w+other.w)*4/10;
	// Absolute rubbish! But...
	return dx*dx+dy*dy<jw*jw;
  //return x-w/2<other.x+other.w/2 && x+w/2>other.x-other.w/2 
    //&& y-h/2<other.y+other.h/2 && y+h/2>other.y-other.h/2;
}
const double Particle::LIFE0=1.0;
Particle::Particle() {
  active=0;
}
void Particle::init(int ix, int iy, double v, double a) {
  x=double(ix);
  y=double(iy);
  lifetime=LIFE0;
  dx=v*cos(a);
  dy=v*sin(a);
  active=true;
}
void Particle::update(double delta) {
  if (!active) return;
  x+=dx*delta;
  y+=dy*delta;
  lifetime-=delta;
  active=lifetime>0.0 && x>=0.0 && y>=0.0 && x<=640.0 && y<480.0;
}
void Particle::draw(BITMAP* s) {
  if (active) {
    int d=6-int(lifetime/LIFE0*5.0);
    if (d<2) d=2;
    ColourItems::ca& cc= Colour.particles[255-int(lifetime/LIFE0*255.0)];
    set_trans_blender(0,0,0,cc.a);
    circlefill(s, int(x), int(y), d, cc.c);
    //putpixel(s, int(x),int(y), makecol(r,g,b));
  }
}

ParticleManager::ParticleManager(int cc) {
  count=cc;
  pts=new Particle[cc];
  next=0;
}
ParticleManager::~ParticleManager() {
  delete[] pts;
}
void ParticleManager::request(int ix, int iy, double vv, double ang, int ct) {
  for (int r=0; r<count; ++r) {
    if (ct<=0) return;
    if (!pts[next].isActive()) {
      vv+=double(rand())/double(RAND_MAX)*5.0-2.5;
      ang+=double(rand())/double(RAND_MAX)*0.5-0.25;
      pts[next].init(ix, iy, vv, ang);
      --ct;
    }
    ++next;
    if (next>=count) next=0;
  }
}


void ParticleManager::update(double t) {
  double delta=t-tlast;
  tlast=t;
  for (int i=0; i<count; ++i) 
    pts[i].update(delta);
}
void ParticleManager::draw(BITMAP* s)  {
  drawing_mode(DRAW_MODE_TRANS, 0, 0, 0);
  for (int i=0; i<count; ++i) 
    pts[i].draw(s);
  drawing_mode(DRAW_MODE_SOLID, 0, 0, 0);
}
void ParticleManager::reset() {
  for (int i=0; i<count; ++i)
    pts[i].reset();
}

/* 
Level info
1-3 'Practice' - capture 1, 5, 9 stars
4-6 Time trial
7-9 Limited pins

10-18 With rocks
19-27 More rocks
28-36 Add your own rocks
*/

Level levels[] ={
  // stars, rocks, pins, timelimit, addrocks, windin
  {1, 0, 0, 0, false, false, "Collect one"},
  {5, 0, 0, 0, false, false, "Collect five"},
  {9, 0, 0, 0, false, false, "Collect nine"},
  {1, 0, 0, 10, false, false, "Time limit"},
  {5, 0, 0, 15, false, false, 0},
  {9, 0, 0, 20, false, false, 0},
  {1, 0, 3, 0, false, false, "Pin limit"},
  {5, 0, 9, 0, false, false, 0},
  {9, 0, 11, 0, false, false, 0},

  {1, 2, 0, 0, false, false, "Beware rocks"},
  {5, 2, 0, 0, false, false, 0},
  {9, 2, 0, 0, false, false, 0},
  {1, 2, 0, 10, false, false, "Time limit"},
  {5, 2, 0, 20, false, false, 0},
  {9, 2, 0, 30, false, false, 0},
  {1, 2, 3, 0, false, false, "Pin limit"},
  {5, 2, 9, 0, false, false, 0},
  {9, 2, 11, 0, false, false, 0},

  {1, 5, 0, 0, false, false, "More rocks"},
  {5, 5, 0, 0, false, false, 0},
  {9, 5, 0, 0, false, false, 0},
  {1, 5, 0, 10, false, false, "Time limit"},
  {5, 5, 0, 20, false, false, 0},
  {9, 5, 0, 30, false, false, 0},
  {1, 5, 3, 0, false, false, "Pin limit"}, 
  {5, 5, 9, 0, false, false, 0},
  {9, 5, 11, 0, false, false, 0},

  {1, 0, 0, 0, true, false, "Add rocks"},
  {5, 0, 0, 0, true, false, 0},
  {9, 0, 0, 0, true, false, 0},
  {1, 0, 0, 10, true, false, 0},
  {5, 0, 0, 15, true, false, 0},
  {9, 0, 0, 20, true, false, 0},
  {1, 0, 3, 0, true, false, 0},
  {5, 0, 9, 0, true, false, 0},
  {9, 0, 11, 0, true, false, 0},

  {0, 0, 0, 0, false, false, 0}
};

int rnd(int a, int b) {
  return a+int(rand()/(double(RAND_MAX)/(b-a)));
}

bool WaitUntil(double when) {
  bool held=true;
  while (GetNow()<when) {
    rest(100);
    if (held) {
      if (mouse_b==0) held=false;
    }
    else {
      if (mouse_b!=0 || key[KEY_ESC])
	return true;
    }
  }
  return false;
}

ScoreHint::ScoreHint() :GrfObj(0, 0, 32, 10, 0) {
	active=false;
}
void ScoreHint::moveTo(int ix, int iy, int sc) {
	score=sc;
	x=ix;
	y=iy;
	endtime=-1.0;
	active=true;
}

bool ScoreHint::update(double now) {
	if (endtime<0.0) {
		endtime=now+1.0;
	}
	if (now>=endtime) {
		active=false;
		return false;
	}
	else
		return true;
}

void ScoreHint::draw(BITMAP* s) {
	textprintf_centre(s, score_font, x, y--, Colour.white, "%d", score);
}


