
/*
 *	CTilemap.cpp
 *
 *	Implimentation of CTilemap class.
 *	CTilemap is a class that loads, manages, renders,
 *	etc tilemaps.  
 *	Note, a tilemap needs to be more than 1 layer, as
 *	the first one (layer 0) is used for tile map 
 *	collision detection...
 *
 */

#include <fstream>
#include <allegro.h>

#include "CTileset.h"		// CTileset class header
#include "CViewport.h"		// CViewport class header
#include "CTilemap.h"		// CTilemap class header
#include "CBaseObject.h"
#include "CEnemy.h"
#include "CItem.h"
#include "Game.h"

// this string is written in the header of each tilemap file to
// properly identify it
const char ID_STRING[] = "TILEMAP FILE - ESKIMO NINJA 2002";



// CTilemap()
// default class constructor
CTilemap::CTilemap()
{
	// reset all variables
	m_iWidth = 0;
	m_iHeight = 0;
	m_iLayers = 0;
	m_iPlayerStartX = 0;
	m_iPlayerStartY = 0;

	m_iTileWidth = TILE_SIZE;
	m_iTileHeight = TILE_SIZE;

	// set pointer to NULL
	m_iMap = NULL;

}





// ~CTilemap()
// default class destructor
CTilemap::~CTilemap()
{

}





// CTilemap(char)
// overloaded constructor, load a map file
CTilemap::CTilemap(char *cFilename)
{
	// reset all variables
	m_iWidth = 0;
	m_iHeight = 0;
	m_iLayers = 0;
	m_iPlayerStartX = 0;
	m_iPlayerStartY = 0;

	m_iTileWidth = TILE_SIZE;
	m_iTileHeight = TILE_SIZE;

	// set pointer to NULL
	m_iMap = NULL;

	// now load the map file
	Load(cFilename);

}





// CTilemap(int,int,int)
// overloaded constructor, create a new map file
CTilemap::CTilemap(int iLayers, int iWidth, int iHeight)
{
	// reset all variables
	m_iWidth = 0;
	m_iHeight = 0;
	m_iLayers = 0;
	m_iPlayerStartX = 0;
	m_iPlayerStartY = 0;

	// set pointer to NULL
	m_iMap = NULL;

	// set the tilesize
	m_iTileWidth = 16;
	m_iTileHeight = 16;

	// now create the new map
	New(iLayers, iWidth, iHeight);

}





// Unload()
// unloads the map, return false on error
bool CTilemap::Unload()
{
	// check if there is a map in memory
	if(Exists())
	{
		// map exists, free it
		MemFree();		// free the memory
		m_iMap = NULL;	// set pointer to NULL
	}
	else
		return false;

	return true;
}






// Exists()
// helper function, returns 'true' if a map is loaded into memory
bool CTilemap::Exists()
{
	// check if the map has been allocated
	if(m_iMap == NULL)
	{
		// hasn't been allocated, return false
		return false;
	}

	return true;
}





// MemAlloc(int,int,int)
// allocate enough memory for the tilemap, return 'false' on error
bool CTilemap::MemAlloc(int iLayers, int iWidth, int iHeight)
{
	// make sure none of the elements <= 0
	if((iWidth <= 0) || (iHeight <= 0) || (iLayers <= 0))
	{
		// can't make a map with < 1 for anything, return false
		return false;
	}

	// check if the level data is clear
	if(Exists())
	{
		// it does, so free it
		MemFree();
	}

	// now allocate memory for each map layer
	m_iMap = new int**[iLayers];

	// iterate through each layer
	for(int l = 0; l < iLayers; l++)
	{
		// and allocate each x element
		m_iMap[l] = new int*[iWidth];
		
		// iterate through each x element
		for(int x = 0; x < iWidth; x++)
		{
			// and allocate each y element
			m_iMap[l][x] = new int[iHeight];
		}
	}

	// make sure memory allocation was succesful
	if(!Exists())
	{
		// map still hasn't been allocated, return false
		return false;
	}

	return true;
}





// MemFree()
// free any memory used by the tilemap, return false on error
bool CTilemap::MemFree()
{
	// first, make sure map isn't already clear
	if(!Exists())
	{
		// doesn't exist, return false
		return false;
	}

	// now free memory used
	delete m_iMap;		// free the memory
	m_iMap = NULL;		// set pointer to null

	return true;
}





// MemResize(int,int,int)
// resize the amount memory allocated for the map
bool CTilemap::MemResize(int iLayers, int iWidth, int iHeight)
{
	// make sure none of the elements == 0
	if((iWidth == 0) || (iHeight == 0) || (iLayers == 0))
	{
		// can't make a map with 0 for anything, return false
		return false;
	}


	// create a temporary map to hold the current data
	int ***iTempMap;

	// allocate memory for each map layer
	iTempMap = new int**[iLayers];

	// iterate through each layer
	for(int l = 0; l < iLayers; l++)
	{
		// and allocate each x element
		iTempMap[l] = new int*[iWidth];
		
		// iterate through each x element
		for(int x = 0; x < iWidth; x++)
		{
			// and allocate each y element
			iTempMap[l][x] = new int[iHeight];
		}
	}

	// copy all the data across that will be saved (if map is resized too small,
	// data will be lost, if too big then new cells will be set to '0')
	for(l = 0; l < iLayers; l++)
	{
		for(int x =  0; x < iWidth; x++)
		{
			for(int y = 0; y < iHeight; y++)
			{
				// make sure this cell is within the original map
				if((l < m_iLayers) && (x < m_iWidth) && (y < m_iHeight))
				{
					// cell is within original map, copy the data
					iTempMap[l][x][y] = m_iMap[l][x][y];
				}
				else
				{
					// cell is not within map, set new cell to 0
					iTempMap[l][x][y] = 0;
				}
			}
		}
	}

	// now get rid of the old map
	MemFree();

	// and allocate enough memory for the new one
	MemAlloc(iLayers, iWidth, iHeight);

	// make sure the new map has been allocated
	if(!Exists())
	{
		// error map couldn't be allocated, free temp map and return false
		delete iTempMap;
		return false;
	}

	// copy the data back across
	for(l = 0; l < iLayers; l++)
	{
		for(int x = 0; x < iWidth; x++)
		{
			for(int y = 0; y < iHeight; y++)
			{
				// copy the map data
				m_iMap[l][x][y] = iTempMap[l][x][y];
			}
		}
	}

	// finally, free the temporary map
	delete iTempMap;

	return true;
}





// New(int,int,int)
// create a new map with the given dimentions, returns false on error
bool CTilemap::New(int iLayers, int iWidth, int iHeight)
{
	// first check if theres already a map in memory
	if(Exists())
	{
		// there is, so free it
		MemFree();
	}

	// allocate memory for the new map, check for error
	if(!MemAlloc(iLayers, iWidth, iHeight))
	{
		// error in memory allocation, return false
		return false;
	}

	// everything went ok, save the map paramaters
	m_iWidth = iWidth;
	m_iHeight = iHeight;
	m_iLayers = iLayers;

	// using tile dimentions, work out how many are visible on screen
	SetScreenSize(SCREEN_W,SCREEN_H);

	// set the maximum scrolling values for viewport
	Viewport.SetLimits((m_iWidth * m_iTileWidth) - (m_iNumTilesWidth * m_iTileWidth),
					   (m_iHeight * m_iTileHeight) - (m_iNumTilesHeight * m_iTileHeight));

	return true;
}





/*
bool CMap::Load(char *cFilename)
{
	// make sure the filename passed is valid before trying to open it
	if(!cFilename)
	{
		WriteLog("*** ERROR : Bad filename given to CMap::Load() ***!\n");
		return false;
	}

	// load the file
	FILE *File;						// the map file
    File = fopen(cFilename, "rb");	// open the map file
													 
	// if the file could not be loaded, return an error
	if(!File)
	{
		WriteLog("*** ERROR : Couldn't load map file '%s' in CMap::Load()! ***\n", cFilename);
		return false;
	}

	// load the tileset filenames
//	fread(m_cFloorTilesetName, sizeof(char), FILENAME_LENGTH, File);
//	fread(m_cWallTilesetName, sizeof(char), FILENAME_LENGTH, File);

	// now load the map array
	for(int l = 0; l < NUM_LAYERS; l++)
		for(int x = 0; x < MAP_WIDTH; x++)
			for(int y = 0; y < MAP_HEIGHT; y++)
			{
				// read the tile
				fread(&m_iMap[l][x][y], sizeof(int), 1, File);
			}

	// close the file
	fclose(File);

	// and load the correct tilesets
//	LoadTilesets();

	return true;
}

bool CMap::Save(char *cFilename)
{
	// make sure the filename passed is valid before trying to save it
	if(!cFilename)
	{
		WriteLog("*** ERROR : Bad filename given to CMap::Save() ***!\n");
		return false;
	}
	
	// save the file
	FILE *File;						// the map file
	File = fopen(cFilename, "wb");	// open the map file
													 
	// if the file could not be saved, return an error
	if(!File)
	{
		WriteLog("*** ERROR : Couldn't save map file '%s' in CMap::Save()! ***\n", cFilename);
		return false;
	}

	// save the tileset filenames
//	fwrite(m_cFloorTilesetName, sizeof(char), FILENAME_LENGTH, File);
//	fwrite(m_cWallTilesetName, sizeof(char), FILENAME_LENGTH, File);

	// now save the map array
	for(int l = 0; l < NUM_LAYERS; l++)
		for(int x = 0; x < MAP_WIDTH; x++)
			for(int y = 0; y < MAP_HEIGHT; y++)
			{
				// save the tile
				fwrite(&m_iMap[l][x][y], sizeof(int), 1, File);
			}

	// close the file
	fclose(File);

	return true;
}*/

// Load(char)
// load a tilemap from a file, return false on error
bool CTilemap::Load(char *cFilename)
{
	// make sure the filename is valid
	if(!cFilename)
		return false;

	// make sure the map exists
	if(!Exists())
	{
		// no map to save, return error
		return false;
	}

	// load the file
/*	std::ifstream File(cFilename, std::ios::in);
													 
	// if the file could not be loaded, return an error
	if(File.bad())
	{
		// display an error message and exit
		return false;
	}

	// load in the map dimentions
	File >> m_iWidth >> m_iHeight >> m_iLayers;

	// load the tile size
	File >> m_iTileWidth >> m_iTileHeight;
	
	// resize the map to given dimentions
	//Resize(m_iLayers, m_iWidth, m_iHeight);

	// load the name of the tileset
	// ...

	// now load the map array
	for(int l = 0; l < m_iLayers; l++)
	{
		for(int x = 0; x < m_iWidth; x++)
		{
			for(int y = 0; y < m_iHeight; y++)
			{
				// write this map tile
				File >> m_iMap[l][x][y];
			}
		}
	}

	// load the tileset given in the map file
	// ...*/

	FILE *File;						// the map file
    File = fopen(cFilename, "rb");	// open the map file
													 
	// if the file could not be loaded, return an error
	if(!File)
	{
		return false;
	}

	// load the tileset filenames
	fread(&m_iWidth, sizeof(int), 1, File);
	fread(&m_iHeight, sizeof(int), 1, File);
	fread(&m_iLayers, sizeof(int), 1, File);
	fread(&m_iTileWidth, sizeof(int), 1, File);
	fread(&m_iTileHeight, sizeof(int), 1, File);
	fread(&m_iPlayerStartX, sizeof(int), 1, File);
	fread(&m_iPlayerStartY, sizeof(int), 1, File);

	fread(&m_iRespawnPosX, sizeof(int), 1, File);
	fread(&m_iRespawnPosY, sizeof(int), 1, File);

	New(m_iLayers, m_iWidth, m_iHeight);

	// now load the map array
	for(int l = 0; l < m_iLayers; l++)
		for(int x = 0; x < m_iWidth; x++)
			for(int y = 0; y < m_iHeight; y++)
			{
				// read the tile
				fread(&m_iMap[l][x][y], sizeof(int), 1, File);
			}

	for(int i = 0; i < MAX_ENEMYS; i++)
	{
		Enemys[i].SetActive(false);
		Enemys[i].Reset();
	}

	LoadObjects(File);

	// close the file
	fclose(File);

	// set the maximum scrolling values for viewport
	Viewport.SetLimits(m_iWidth * m_iTileWidth - SCREEN_W, m_iHeight * m_iTileHeight - SCREEN_H);

	// set scroll around player start point
	//Viewport.SetScroll(m_iPlayerStartX - SCREEN_W/2, m_iPlayerStartY - SCREEN_H/2);
	Viewport.SetScroll(GetPlayerStartX()-(SCREEN_W/2),GetPlayerStartY()-(SCREEN_H/2)-100);

	return true;
}





// Save(char)
// save the tilemap to a file, return false on error
bool CTilemap::Save(char *cFilename)
{
	// make sure the filename is valid
	if(!cFilename)
		return false;

	// make sure the map exists
	if(!Exists())
	{
		// no map to save, return error
		return false;
	}

/*	// create the save file
	std::ofstream File(cFilename, std::ios::out);

	// if the file could not be created, return an error
	if(File.bad())
	{
		// display an error message and exit
		return false;
	}

	// write in the map dimentions
	File << m_iWidth << m_iHeight << m_iLayers;

	// write the tile size
	File << m_iTileWidth << m_iTileHeight;

	// write the name of the tileset
	// ...

	// now write the map array
	for(int l = 0; l < m_iLayers; l++)
	{
		for(int x = 0; x < m_iWidth; x++)
		{
			for(int y = 0; y < m_iHeight; y++)
			{
				// write this map tile
				File << m_iMap[l][x][y];
			}
		}
	}*/

	// save the file
	FILE *File;						// the map file
	File = fopen(cFilename, "wb");	// open the map file
													 
	// if the file could not be saved, return an error
	if(!File)
	{
		//WriteLog("*** ERROR : Couldn't save map file '%s' in CMap::Save()! ***\n", cFilename);
		return false;
	}

	fwrite(&m_iWidth, sizeof(int), 1, File);
	fwrite(&m_iHeight, sizeof(int), 1, File);
	fwrite(&m_iLayers, sizeof(int), 1, File);
	fwrite(&m_iTileWidth, sizeof(int), 1, File);
	fwrite(&m_iTileHeight, sizeof(int), 1, File);
	fwrite(&m_iPlayerStartX, sizeof(int), 1, File);
	fwrite(&m_iPlayerStartY, sizeof(int), 1, File);

	fwrite(&m_iRespawnPosX, sizeof(int), 1, File);
	fwrite(&m_iRespawnPosY, sizeof(int), 1, File);

	// now save the map array
	for(int l = 0; l < m_iLayers; l++)
		for(int x = 0; x < m_iWidth; x++)
			for(int y = 0; y < m_iHeight; y++)
			{
				// save the tile
				fwrite(&m_iMap[l][x][y], sizeof(int), 1, File);
			}

	SaveObjects(File);

	// close the file
	fclose(File);

	return true;
}





// Resize(int,int,int)
// resize the map, data may be lost if map is shrunk in size
bool CTilemap::Resize(int iLayers, int iWidth, int iHeight)
{
	// check a map actually exists
	if(!Exists())
	{
		// map doesn't exist, return false
		return false;
	}

	// resize the map, check for errors
	if(!MemResize(iLayers, iWidth, iHeight))
	{
		// error occured, return false
		return false;
	}

	// everything went ok, save the map paramaters
	m_iWidth = iWidth;
	m_iHeight = iHeight;
	m_iLayers = iLayers;

	// using tile dimentions, work out how many are visible on screen
	SetScreenSize(SCREEN_W,SCREEN_H);

	// set the maximum scrolling values for viewport
	Viewport.SetLimits((m_iWidth * m_iTileWidth) - (m_iNumTilesWidth * m_iTileWidth),
					   (m_iHeight * m_iTileHeight) - (m_iNumTilesHeight * m_iTileHeight));

	return true;
}





// LoadTileset(char*)
// load a tileset from disk, return false on error
bool CTilemap::LoadTileset(char *cFilename)
{
	// make sure filename is valid
	if(!cFilename)
	{
		// bad filename, return false
		return false;
	}

	// load the tileset, check for error
	if(!Tileset.Load(cFilename))
	{
		// error loading tileset, return false
		return false;
	}

	return true;
}





// UnloadTileset()
// unload a previously loaded tileset
bool CTilemap::UnloadTileset()
{
	// make sure a tileset exists
	if(!Tileset.Exists())
	{
		// no tileset to unload, return false
		return false;
	}

	// unload the tileset
	Tileset.Unload();

	return true;
}





// SetScreenSize(int,int)
// using the screen size work out how many tiles are visible, so we know how many to 
// draw in each rendering loop
void CTilemap::SetScreenSize(int iScreenWidth, int iScreenHeight)
{
	// these two variables hold how many tiles fit on the screen
	m_iNumTilesWidth = iScreenWidth / m_iTileWidth;
	m_iNumTilesHeight = iScreenHeight / m_iTileHeight;
}



#include <stdio.h>

// Draw(BITMAP)
// render the tilemap to the specified bitmap
bool CTilemap::Draw(BITMAP *Bitmap)
{
	// make sure the map exists
	if(!Exists())
	{
		// map doesn't exist, return false
		return false;
	}

	// now get the current scroll values so we can determine which tiles need
	// to be drawn
	Vector2D ScrollValues = Viewport.GetScroll();

	// the top and left coordinates are the current scroll values / tile size
	int StartX = ScrollValues.x / m_iTileWidth;
	int StartY = ScrollValues.y / m_iTileHeight;

	// the bottom and right coorinates are current scroll values + num tiles on screen
	// need to make sure the end coordinates don't go over the map edge
	int EndX = StartX + m_iNumTilesWidth;
	if(EndX != m_iWidth)	// allow an extra tile to be drawn on map edge 
		EndX++;

	int EndY = StartY + m_iNumTilesHeight;
	if(EndY != m_iHeight)
		EndY++;

	// iterate through each visible map element and draw it
	for(int l = 0; l < m_iLayers; l++)
	{
		for(int x = StartX; x < EndX; x++)
		{
			if(x >= m_iWidth)
				break;

			for(int y = StartY; y < EndY; y++)
			{
				// check if this cell has a tile (value > 0)
				if(m_iMap[l][x][y] > 0)
				{
					if(y >= m_iHeight)
						break;

					if(l > 0)
						draw_sprite(Bitmap, (BITMAP*)Tileset.GetTile(m_iMap[l][x][y]), 
							(x * m_iTileWidth) - ScrollValues.x, 
							(y * m_iTileHeight) - ScrollValues.y);

					if(m_bDrawData)
					{
						if(m_iMap[0][x][y] > 0)
						{
							int col;
							if(m_iMap[0][x][y] == TILE_TYPE_BLOCKED)
								col = makecol(255,255,255);
							else if(m_iMap[0][x][y] == TILE_TYPE_DAMAGE)
								col = makecol(255,0,0);
							else if(m_iMap[0][x][y] == TILE_TYPE_BOUNCE)
								col = makecol(255,255,0);
							else if(m_iMap[0][x][y] == TILE_TYPE_WIN)
								col = makecol(0,255,255);

							line(Bitmap,
								 (x * m_iTileWidth) - ScrollValues.x, 
								 (y * m_iTileHeight) - ScrollValues.y,
								 (x * m_iTileWidth) - ScrollValues.x + TILE_SIZE-1, 
								 (y * m_iTileHeight) - ScrollValues.y + TILE_SIZE-1,
								 col);
							line(Bitmap,
								 (x * m_iTileWidth) - ScrollValues.x + TILE_SIZE-1, 
								 (y * m_iTileHeight) - ScrollValues.y,
								 (x * m_iTileWidth) - ScrollValues.x, 
								 (y * m_iTileHeight) - ScrollValues.y + TILE_SIZE-1,
								 col);
						}
					}
				}
			}
		}
	}

	if(m_bDrawData)
	{
		// draw start pos
		rect(Bitmap, m_iPlayerStartX - ScrollValues.x, 
			m_iPlayerStartY - ScrollValues.y, 
			m_iPlayerStartX - ScrollValues.x + TILE_SIZE, 
			m_iPlayerStartY - ScrollValues.y + TILE_SIZE, 
			makecol(0,128,0));
		rect(Bitmap, m_iPlayerStartX - ScrollValues.x+1, 
			m_iPlayerStartY - ScrollValues.y+1, 
			m_iPlayerStartX - ScrollValues.x + TILE_SIZE-1, 
			m_iPlayerStartY - ScrollValues.y + TILE_SIZE-1, 
			makecol(0,255,0));
		rect(Bitmap, m_iPlayerStartX - ScrollValues.x+2, 
			m_iPlayerStartY - ScrollValues.y+2, 
			m_iPlayerStartX - ScrollValues.x + TILE_SIZE-2, 
			m_iPlayerStartY - ScrollValues.y + TILE_SIZE-2, 
			makecol(0,128,0));

		// draw respawn pos
		rect(Bitmap, m_iRespawnPosX - ScrollValues.x, 
			m_iRespawnPosY - ScrollValues.y, 
			m_iRespawnPosX - ScrollValues.x + TILE_SIZE, 
			m_iRespawnPosY - ScrollValues.y + TILE_SIZE, 
			makecol(128,0,0));
		rect(Bitmap, m_iRespawnPosX - ScrollValues.x+1, 
			m_iRespawnPosY - ScrollValues.y+1, 
			m_iRespawnPosX - ScrollValues.x + TILE_SIZE-1, 
			m_iRespawnPosY - ScrollValues.y + TILE_SIZE-1, 
			makecol(255,0,0));
		rect(Bitmap, m_iRespawnPosX - ScrollValues.x+2, 
			m_iRespawnPosY - ScrollValues.y+2, 
			m_iRespawnPosX - ScrollValues.x + TILE_SIZE-2, 
			m_iRespawnPosY - ScrollValues.y + TILE_SIZE-2, 
			makecol(128,0,0));

		// temp stuff to display various map data
		char cString[50];
		Vector2D Scroll = Viewport.GetScroll();
		sprintf(cString,"ScrollX - %d / %d", Scroll.x/m_iTileWidth, m_iWidth);
		textout(Bitmap,font,cString, 0, 15, makecol(255,255,255));
		sprintf(cString,"ScrollY - %d / %d", Scroll.y/m_iTileHeight, m_iHeight);
		textout(Bitmap,font,cString, 0, 30, makecol(255,255,255));
		sprintf(cString,"NumTiles# - %d / %d", m_iNumTilesWidth, m_iNumTilesHeight);
		textout(Bitmap,font,cString, 0, 45, makecol(255,255,255));
	}
	else
	{
		if(m_bHasHitRespawn)
		{
			// draw respawn pos
			rectfill(Bitmap, m_iRespawnPosX - ScrollValues.x+2, 
				m_iRespawnPosY - ScrollValues.y+2, 
				m_iRespawnPosX - ScrollValues.x + TILE_SIZE-2, 
				m_iRespawnPosY - ScrollValues.y + TILE_SIZE-2, 
				makecol(0,96,0));
			rect(Bitmap, m_iRespawnPosX - ScrollValues.x, 
				m_iRespawnPosY - ScrollValues.y, 
				m_iRespawnPosX - ScrollValues.x + TILE_SIZE, 
				m_iRespawnPosY - ScrollValues.y + TILE_SIZE, 
				makecol(0,128,0));
			rect(Bitmap, m_iRespawnPosX - ScrollValues.x+1, 
				m_iRespawnPosY - ScrollValues.y+1, 
				m_iRespawnPosX - ScrollValues.x + TILE_SIZE-1, 
				m_iRespawnPosY - ScrollValues.y + TILE_SIZE-1, 
				makecol(0,255,0));
			rect(Bitmap, m_iRespawnPosX - ScrollValues.x+2, 
				m_iRespawnPosY - ScrollValues.y+2, 
				m_iRespawnPosX - ScrollValues.x + TILE_SIZE-2, 
				m_iRespawnPosY - ScrollValues.y + TILE_SIZE-2, 
				makecol(0,255,0));
		}
		else
		{
			// draw respawn pos
			rectfill(Bitmap, m_iRespawnPosX - ScrollValues.x+2, 
				m_iRespawnPosY - ScrollValues.y+2, 
				m_iRespawnPosX - ScrollValues.x + TILE_SIZE-2, 
				m_iRespawnPosY - ScrollValues.y + TILE_SIZE-2, 
				makecol(0,0,96));
			rect(Bitmap, m_iRespawnPosX - ScrollValues.x, 
				m_iRespawnPosY - ScrollValues.y, 
				m_iRespawnPosX - ScrollValues.x + TILE_SIZE, 
				m_iRespawnPosY - ScrollValues.y + TILE_SIZE, 
				makecol(0,0,128));
			rect(Bitmap, m_iRespawnPosX - ScrollValues.x+1, 
				m_iRespawnPosY - ScrollValues.y+1, 
				m_iRespawnPosX - ScrollValues.x + TILE_SIZE-1, 
				m_iRespawnPosY - ScrollValues.y + TILE_SIZE-1, 
				makecol(0,0,255));
			rect(Bitmap, m_iRespawnPosX - ScrollValues.x+2, 
				m_iRespawnPosY - ScrollValues.y+2, 
				m_iRespawnPosX - ScrollValues.x + TILE_SIZE-2, 
				m_iRespawnPosY - ScrollValues.y + TILE_SIZE-2, 
				makecol(0,0,128));
		}
	}

	return true;
}





// SetTile(int,int,int, int)
// set specified tile to specified value, returns false if out of map bounds
bool CTilemap::SetTile(int iLayer, int iTileX, int iTileY, int iValue)
{
	// check if any values given are negative
	if((iLayer < 1) || (iTileX < 0) || (iTileY < 0) || (iValue < 0))
	{
		// error, can't have any negative values, return false
		return false;
	}

	// check if tile given is out of map range
	if((iLayer >= m_iLayers) || (iTileX >= m_iWidth) || (iTileY >= m_iHeight))
	{
		// tile outside of map boundaries, return false
		return false;
	}

	// no errors, set the tile's value
	m_iMap[iLayer][iTileX][iTileY] = iValue;

	return true;
}





// GetTile(int,int,int)
// return the value of the specified tile, or -1 if error
int CTilemap::GetTile(int iLayer, int iTileX, int iTileY)
{
	// check if any values given are negative
	if((iLayer < 1) || (iTileX < 0) || (iTileY < 0))
	{
		// error, can't have any negative values, return false
		return 0;
	}

	// check if tile given is out of map range
	if((iLayer >= m_iLayers) || (iTileX >= m_iWidth) || (iTileY >= m_iHeight))
	{
		// tile outside of map boundaries, return false
		return 0;
	}

	// no errors, return the tile's value
	return m_iMap[iLayer][iTileX][iTileY];
}





// SetTileType(int,int,int)
// set the tile type (layer 0) to specified value, return false on error
bool CTilemap::SetTileType(int iTileX, int iTileY, int iValue)
{
	// check if any values given are negative
	if((iTileX < 0) || (iTileY < 0) || (iValue < 0))
	{
		// error, can't have any negative values, return false
		return false;
	}

	// check if tile given is out of map range
	if((iTileX >= m_iWidth) || (iTileY >= m_iHeight))
	{
		// tile outside of map boundaries, return false
		return false;
	}

	// no errors, set the tile's value
	m_iMap[0][iTileX][iTileY] = iValue;

	return true;
}





// GetTileType(int,int)
// get tile type (layer 0) of specified tile
int CTilemap::GetTileType(int iTileX, int iTileY)
{
	// check if any values given are negative
	if((iTileX < 0) || (iTileY < 0))
	{
		// error, can't have any negative values, return false
		return 0;
	}

	// check if tile given is out of map range
	if((iTileX >= m_iWidth) || (iTileY >= m_iHeight))
	{
		// tile outside of map boundaries, return false
		return 0;
	}

	// no errors, return the tile's value
	return m_iMap[0][iTileX][iTileY];
}





// Scroll(int,int)
// scroll the map (basically call this function in the CViewport class)
bool CTilemap::Scroll(int iDirection, int iSpeed)
{
	// call the function in CViewport
	if(!Viewport.Scroll(iDirection,iSpeed))
	{
		// error in scroll, return false
		return false;
	}

	return true;
}





// ClearTo(int)
// clear all layers and tiles of the map to given value
bool CTilemap::ClearTo(int iValue)
{
	// make sure the map is valid
	if(!Exists())
	{
		// no map to clear, return false
		return false;
	}

	// make sure a valid value is given
	if(iValue < 0)
	{
		// cannot have negative tile value, return false
		return false;
	}

	// no errors, now clear each tile of the map to given value
	for(int l = 0; l < m_iLayers; l++)
	{
		for(int x = 0; x < m_iWidth; x++)
		{
			for(int y = 0; y < m_iHeight; y++)
			{
				// clear the current tile
				m_iMap[l][x][y] = iValue;
			}
		}
	}

	return true;
}





// ClearLayer(int,int)
// clear iLayer to iValue
bool CTilemap::ClearLayer(int iLayer, int iValue)
{
	// make sure the map is valid
	if(!Exists())
	{
		// no map to clear, return false
		return false;
	}

	// make sure given layer is valid
	if((iLayer < 0) || (iLayer >= m_iLayers))
	{
		// invalid layer
		return false;
	}

	// make sure a valid value is given
	if(iValue < 0)
	{
		// cannot have negative tile value, return false
		return false;
	}

	// no errors, now clear each tile of the layer to given value
	for(int x = 0; x < m_iWidth; x++)
	{
		for(int y = 0; y < m_iHeight; y++)
		{
			// clear the current tile
			m_iMap[iLayer][x][y] = iValue;
		}
	}
	
	return true;
}





// MapPos(int,int)
// return the coords converted into position on map
Vector2D CTilemap::MapPos(int iPosX, int iPosY)
{
	// create a temporary variable initialised to 0
	Vector2D vPosition = {0, 0};

	// check the position is within the map
	if((iPosX < 0) || (iPosX > m_iWidth * m_iTileWidth) || (iPosY < 0) || (iPosY > m_iHeight * m_iTileHeight))
	{
		// error return vector set to 0
		return vPosition;
	}

	// no errors, work out the coordinates

	// first get the current scrolling coordinates
	Vector2D vScroll = Viewport.GetScroll();

	// work out each position
	vPosition.x = (iPosX + vScroll.x) / m_iTileWidth;
	vPosition.y = (iPosY + vScroll.y) / m_iTileHeight;

	// return the new position
	return vPosition;
}





// NumTiles()
// return number of tiles in tileset
int CTilemap::NumTiles()
{
	return Tileset.NumTiles();
}








/********************************************************************************

		SaveObjects()

	save all external data into map save file

********************************************************************************/

bool CTilemap::SaveObjects(FILE *File)
{										 
	if(!File)
		return false;

	// save list of enemys
	for(int i = 0; i < MAX_ENEMYS; i++)
	{
		if(Enemys[i].GetActive())
		{
			bool b = Enemys[i].GetActive();
			bool d = Enemys[i].GetDirX();
			float PosX = Enemys[i].GetPosX();
			float PosY = Enemys[i].GetPosY();
			int Type = Enemys[i].GetEnemyType();
			
			fwrite(&b, sizeof(bool), 1, File);
			fwrite(&d, sizeof(bool), 1, File);
			fwrite(&PosX, sizeof(float), 1, File);
			fwrite(&PosY, sizeof(float), 1, File);
			fwrite(&Type, sizeof(int), 1, File);
			
		}
		else
		{
			bool b = false;
			bool d = false;
			float PosX = 0;
			float PosY = 0;
			int Type = 0;
			
			fwrite(&b, sizeof(bool), 1, File);
			fwrite(&d, sizeof(bool), 1, File);
			fwrite(&PosX, sizeof(float), 1, File);
			fwrite(&PosY, sizeof(float), 1, File);
			fwrite(&Type, sizeof(int), 1, File);
			
		}
	}

	// save list of items
	for(i = 0; i < MAX_ITEMS; i++)
	{
		if(Items[i].GetActive())
		{
			bool b = Items[i].GetActive();
			float PosX = Items[i].GetPosX();
			float PosY = Items[i].GetPosY();
			int Type = Items[i].GetItemType();
			
			fwrite(&b, sizeof(bool), 1, File);
			fwrite(&PosX, sizeof(float), 1, File);
			fwrite(&PosY, sizeof(float), 1, File);
			fwrite(&Type, sizeof(int), 1, File);
			
		}
		else
		{
			bool b = false;
			float PosX = 0;
			float PosY = 0;
			int Type = 0;
			
			fwrite(&b, sizeof(bool), 1, File);
			fwrite(&PosX, sizeof(float), 1, File);
			fwrite(&PosY, sizeof(float), 1, File);
			fwrite(&Type, sizeof(int), 1, File);
			
		}
	}

	return true;
}





/********************************************************************************

		LoadObjects()

	load all external data from map save file

********************************************************************************/

bool CTilemap::LoadObjects(FILE *File)
{										 
	if(!File)
		return false;

	// load list of enemys
	for(int i = 0; i < MAX_ENEMYS; i++)
	{
		bool b;
		float PosX;
		float PosY;
		int Type;
		bool d;

		fread(&b, sizeof(bool), 1, File);
		fread(&d, sizeof(bool), 1, File);
		fread(&PosX, sizeof(float), 1, File);
		fread(&PosY, sizeof(float), 1, File);
		fread(&Type, sizeof(int), 1, File);
			
		if(b)	
			AddEnemy(PosX, PosY, Type, d);
	}

	// load list of Items
	for(i = 0; i < MAX_ITEMS; i++)
	{
		bool b;
		float PosX;
		float PosY;
		int Type;

		fread(&b, sizeof(bool), 1, File);
		fread(&PosX, sizeof(float), 1, File);
		fread(&PosY, sizeof(float), 1, File);
		fread(&Type, sizeof(int), 1, File);
			
		if(b)	
			AddItem(PosX, PosY, Type);
	}

	return true;
}


