
/*
 *	Main.cpp
 *
 *	Main program file for game.  Initialises Allegro
 *	library, and runs main game loop.
 *
 */

#include <fstream>
#include <time.h>
#include <stdio.h>
#include <string.h>
#include <allegro.h>
#include "Main.h"
#include "CViewport.h"
#include "CTileset.h"
#include "CTilemap.h"
#include "CBaseObject.h"
#include "CEnemy.h"
#include "Game.h"
#include "Menus.h"

const int SCREEN_WIDTH   = 640;		// screen dimentions
const int SCREEN_HEIGHT  = 480;
const int SCREEN_BPP     = 32;		// colour depth of screen
const int GAME_SPEED     = 30;		// speed of game in frames per second

const char GAME_NAME[]    = "Stickdude!!!1";		// game name to write in log file
const char LOGFILE_NAME[] = "data//log.txt";	// name of log to write to
const char WINDOW_NAME[]  = "Stickdude!!!1";		// name of window title

bool Fullscreen = true;				// true if game should run in fullscreen
bool QuitGame   = false;			// flag for quitting game
bool QuitProgram = false;

BITMAP *g_Buffer = NULL;				// main game backbuffer

volatile int TargetGameTime = 0;	// used in the game timer
int FPS;							// counter for FPS
int CurrentFPS;						// the FPS for this second

FILE *Log;							// file handle to log

// needed for getting current time
time_t Seconds;						// calender time
struct tm *Date;					// pointer to broken down time structure

// list of the names of each month
char Months[12][10] = { "January",
					"February",
					"March",
					"April",
					"May",
					"June",
					"July",
					"August",
					"September",
					"October",
					"November",
					"December" };



/********************************************************************************

		InitLog()

	Initialise log file system ready for use.
	Thanks to Paul Frazee's (The Rainmaker) tutorial on nehe.gamedev.net for
	the basis of this.

********************************************************************************/

void InitLog()
{
	Seconds = time(NULL);						// get the calendar time						
	Date = localtime(&Seconds);					// convert to broken-down time

	Log=NULL;

	Log = fopen(LOGFILE_NAME,"w");			// Open The File, Clear Contents

	if(!Log)	
		allegro_message("Couldn't open logfile!");
	else
	{
		WriteLog("Logfile for '%s'.  (c) 2003 Eskimo Ninja\n", GAME_NAME);
		WriteLog("************************************************\n");
		WriteLog("Log started at %d:%d:%d ", Date->tm_hour, Date->tm_min, Date->tm_sec);
		WriteLog("on %d of %s, %d.\n", Date->tm_mday, Months[Date->tm_mon], Date->tm_year+1900);
		WriteLog("Allegro Version : %s\n\n", &allegro_id);
	}

}





/********************************************************************************

		WriteLog()

	Writes given text into a log file, saving it automatically.  The log file to
	write in is defined above as 'LOGFILE_NAME'.
	Thanks to Paul Frazee's (The Rainmaker) tutorial on nehe.gamedev.net for
	the basis of this.

********************************************************************************/

inline void WriteLog(char *text,...)
{
	// Open The Log File For Writing
	if(!Log) 
		Log = fopen(LOGFILE_NAME,"a");

	if(!Log)				// If There Was An Error, Stop
		return;

	// Create The g_Buffer
	va_list arglist;
	char buff[1024];
	memset(buff,0,sizeof(buff));

	va_start(arglist,text);
	vsprintf(buff,text,arglist);
	va_end(arglist);
	fprintf(Log,buff);			// Write To The File

	// Close The Log
	if(Log!=NULL) 
	{ 
		fclose(Log); 
		Log = NULL; 
	}
}





/********************************************************************************

		FPScounter()

	used for counting the FPS, called every second to update the FPS value

********************************************************************************/

inline void FPScounter(void)
{    
	CurrentFPS = FPS;
	FPS = 0;
}





/********************************************************************************

		IncrementFPSCounter()

	must be called each full game drawing loop to update the FPS

********************************************************************************/

inline void IncrementFPSCounter()
{
	FPS++;	// increment the counter
}





/********************************************************************************

		DrawFPS()

	draw the FPS counter to the specified bitmap

********************************************************************************/

inline void DrawFPS(BITMAP *Bitmap)
{
	textprintf(Bitmap,font,0,0,makecol(255,255,255),"FPS : %d", CurrentFPS);
}





/********************************************************************************

		IncrementTargetGameTime()

	timer for the game to keep it at a constant speed

********************************************************************************/

void IncrementTargetGameTime()
{
	TargetGameTime++;
}
END_OF_FUNCTION(IncrementTargetGameTime);	// timer locking code





/********************************************************************************

		GameMain()

	main Game loop, runs game at set speed

********************************************************************************/

bool GameMain()
{
	int ActualGameTime = 0;  // used in game timer
	bool OkToDraw = false;	

	WriteLog("Entering GameMain()\n");

	// initialise game, return false on error
	if(InitGame())
	{
		WriteLog("	InitGame() successful\n");
	}
	else
	{
		WriteLog("*** ERROR : Error in InitGame()! Exiting... ***\n");
		return false;
	}

	WriteLog("	Entering main game loop\n");

	DoIntro();

	while(!QuitProgram)
	{
		if(!DoMainMenu())
		{
			QuitGame = true;
			QuitProgram = true;
		}
		else
		{
			QuitGame = false;
			QuitProgram = false;
		}

		while(!QuitGame)  // this is set to TRUE when user wants to quit
		{
			while(ActualGameTime < TargetGameTime)
			{
				// perform the logic loop, return false on error
				if(!GameLogicLoop())
				{
					WriteLog("*** ERROR : Error in GameLogicLoop()! Exiting...***\n");
					return false;
				}

				ActualGameTime++;	// increment game timer
				OkToDraw = TRUE;	// set the draw flag to true
			}

			if(OkToDraw)
			{
				// perform the game drawing loop, return false on error
				if(!GameGraphicsLoop())
				{
					WriteLog("*** ERROR : Error in GameGraphicsLoop()! Exiting... ***\n");
					return false;
				}

				OkToDraw = FALSE;	// set draw flag to false for next loop
				IncrementFPSCounter();	// incrementer the FPS counter
			}
		}
	}

	WriteLog("	Leaving main game loop\n");

	// shutdown game, return false on error
	if(ShutdownGame())
	{
		WriteLog("	ShutdownGame() successful\n");
	}
	else
	{
		WriteLog("*** ERROR : Error in ShutdownGame()! Exiting... ***\n");
		return false;
	}


	WriteLog("Leaving GameMain()\n\n");

	return true;
}





/********************************************************************************

		InitAllegro()

	init all allegro functions and modules needed

********************************************************************************/

bool InitAllegro()
{
	// some stuff needed in InitLog() can only be done after calling
	// allegro_init(), so the order of doing the process and logging 
	// it here is a little strange
	allegro_init();		// init allegro

	InitLog();			// init the log system so game progress can be recorded

	WriteLog("Entering InitAllegro()\n");
	WriteLog("	Allegro library opened\n");

	install_keyboard();	// install keyboard hander so keys can be read
	WriteLog("	Keyboard handler installed\n");
	install_timer();	// install functions for using timers
	WriteLog("	Timer handler installed\n");
	install_mouse();	// as above, but for mouse
	WriteLog("	Mouse handler installed\n");
	install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, 0);	// install sound
	WriteLog("	Sound handler installed\n");

	text_mode(-1);		// text output will have a transparent background

	set_color_depth(SCREEN_BPP);	// set the correct colour depth

	if(Fullscreen == true)
	{
		// set a fullscreen mode

		// set the graphics mode, if cannot set then use SAFE mode
		if(set_gfx_mode(GFX_AUTODETECT,SCREEN_WIDTH,SCREEN_HEIGHT,0,0))  
		{
			// set_gfx_mode failed, set the safest resolution possible
			allegro_message("Error setting default screen mode %dx%d at %d! Switching to SAFE_MODE 320x200",SCREEN_WIDTH,SCREEN_HEIGHT,SCREEN_BPP);

			if(set_gfx_mode(GFX_SAFE,SCREEN_WIDTH,SCREEN_HEIGHT,0,0))
			{
				// not even sure if its possible to fail that, but just in case
				allegro_message("Unable to set any screen resolution!  Quiting...");
				return false;
			}
		}
	}
	else
	{
		// set a windowed mode

		// set the graphics mode, if cannot set then use SAFE mode
		if(set_gfx_mode(GFX_AUTODETECT_WINDOWED,SCREEN_WIDTH,SCREEN_HEIGHT,0,0))  
		{
			// set_gfx_mode failed, set the safest resolution possible
			allegro_message("Error setting default screen mode %dx%d at %d (windowed)! Switching to SAFE_MODE 320x200",SCREEN_WIDTH,SCREEN_HEIGHT,SCREEN_BPP);

			if(set_gfx_mode(GFX_SAFE,SCREEN_WIDTH,SCREEN_HEIGHT,0,0))
			{
				// not even sure if its possible to fail that, but just in case
				allegro_message("Unable to set any screen resolution!  Quiting...");
				return false;
			}
		}
	}

	if(Fullscreen)
		WriteLog("	Graphics mode set at %dx%dx%d in fullscreen mode\n", SCREEN_W, SCREEN_H, SCREEN_BPP);
	else
		WriteLog("	Graphics mode set at %dx%dx%d in windowed mode\n", SCREEN_W, SCREEN_H, SCREEN_BPP);

	 // this is the main backbuffer for drawing
	g_Buffer = create_bitmap(SCREEN_W,SCREEN_H);  

	if(!g_Buffer)	// check for errors in creation			 
	{
		// theres been an error creating bitmap, return an error
		WriteLog("*** ERROR : Couldn't create main backbuffer!  Exiting... ***\n");
		return false;
	}

	clear_bitmap(g_Buffer); // bitmap created, clear any garbage from it
	WriteLog("	Backbuffer created at %dx%dx%d\n", SCREEN_W, SCREEN_H, SCREEN_BPP);

	srand(time(NULL));   // seed the random number generator

	// these are to allow the game timer to work properly
	LOCK_VARIABLE(TargetGameTime);
	LOCK_FUNCTION(IncrementTargetGameTime);

	// set the game speed to that defined in main.h
	install_int_ex(IncrementTargetGameTime,BPS_TO_TIMER(GAME_SPEED));

	// initialise main game timer
	LOCK_VARIABLE (FPS); 
	LOCK_FUNCTION (FPScounter);
	install_int (FPScounter, 1000);
	WriteLog("	Timer installed\n");

	// setup transparency
	set_trans_blender(0,0,0,128);
	WriteLog("	Transparency Enabled\n");

	// set the window title
	set_window_title((char*)WINDOW_NAME);

	WriteLog("Leaving InitAllegro()\n\n");

	return true;
}





/********************************************************************************

		ShutdownAllegro()

	closes the allegro library

********************************************************************************/

bool ShutdownAllegro()
{
	WriteLog("Entering ShutdownAllegro()\n");

	destroy_bitmap(g_Buffer);  // destroy the double buffer
	g_Buffer = NULL;
	WriteLog("	Backbuffer destroyed\n");
//	allegro_exit();			// clear allegro from memory
	WriteLog("	Allegro library closed\n");

	WriteLog("Leaving ShutdownAllegro()\n\n");

	return true;
}





/********************************************************************************

		main()

	take a wild guess what this does

********************************************************************************/

int main()
{
	// initialise allegro
	if(!InitAllegro())
	{
		WriteLog("*** ERROR : Error in InitAllegro() ***\n");
		ShutdownAllegro();
		return 1;
	}

	// now enter the main game loop
	if(!GameMain())
	{
		WriteLog("*** ERROR : Error in GameMain() ***\n");
		ShutdownAllegro();
		return 1;
	}

	// shutdown allegro functions 
	if(!ShutdownAllegro())
	{
		WriteLog("*** ERROR : Error in ShutdownAllegro() ***\n");
		return 1;
	}

	// write the time logging ended
	WriteLog("No errors in game detected by log.\n");
	Seconds = time(NULL);						// get the calendar time						
	Date = localtime(&Seconds);					// convert to broken-down time
	WriteLog("Log ended at %d:%d:%d ", Date->tm_hour, Date->tm_min, Date->tm_sec);
	WriteLog("on %d of %s, %d.\n\n", Date->tm_mday, Months[Date->tm_mon], Date->tm_year+1900);
	free(Log);
	Log = NULL;

	// and get the hell outta here
	return 0;
}
END_OF_MAIN();
