
/*
 *	CTilemap.h
 *
 *	Interface for CTilemap class.
 *	CTilemap is a class that loads, manages, renders,
 *	etc tilemaps.  
 *	Note, a tilemap needs to be more than 1 layer, as
 *	the first one (layer 0) is used for tile map 
 *	collision detection...
 *
 */

// watch for multiple inclusions
#ifndef CTILEMAP_HEADER
#define CTILEMAP_HEADER



// different tile types
#define TILE_TYPE_CLEAR		0
#define TILE_TYPE_BLOCKED	1
#define TILE_TYPE_DAMAGE	2
#define TILE_TYPE_BOUNCE	3
#define TILE_TYPE_WIN		4



#define TILE_SIZE		32		// dimentions of each tile
#define NUM_TILES		46
#define NUM_TILE_TYPES	5

//#define NUM_ENEMYS		100



// this string is written in the header of each tilemap file to
// properly identify it
extern const char ID_STRING[];



class CTilemap
{

private:

	int m_iWidth;					// width of map in tiles
	int m_iHeight;					// height of map in tiles
	int m_iLayers;					// layers of tiles in map

	int m_iNumTilesWidth;			// number of tiles visible on screen at any time
	int m_iNumTilesHeight;

	int m_iTileWidth;				// dimentions in pixels of each map tile
	int m_iTileHeight;

	float m_iPlayerStartX;			// player start position
	float m_iPlayerStartY;	

	float m_iRespawnPosX;			// player restart pos if they reach it
	float m_iRespawnPosY;
	bool m_bHasHitRespawn;		// set to true when hit

	int ***m_iMap;					// dynamic map array

	
	CTileset Tileset;				// map tileset

	bool m_bDrawData;				

	bool MemAlloc(int iLayers, int iWidth, int iHeight);	// allocate enough memory for the map
	bool MemFree();											// free any memory used by the tilemap
	bool MemResize(int iLayers, int iWidth, int iHeight);	// resize the memory for the map

	void SetScreenSize(int iScreenWidth, int iScreenHeight);	// work out how many tiles are visible on screen at once

	bool SaveObjects(FILE *f);
	bool LoadObjects(FILE *f);


public:

	CTilemap();						// default constructor
	~CTilemap();					// default destructor

	CTilemap(char *cFilename);						// overloaded constructor, load a file
	CTilemap(int iLayers, int iWidth, int iHeight);	// overloaded constructor, create new map
	
	bool Exists();					// helper function, returns 'true' if map has been allocated

	bool New(int iLayers, int iWidth, int iHeight);		// create a new map
	bool Load(char *cFilename);							// load a map file
	bool Save(char *cFilename);							// save a map file
	bool Resize(int iLayers, int iWidth, int iHeight);	// resize the map

	bool LoadTileset(char *cFilename);		// load a map tileset
	bool UnloadTileset();					// unload the tileset

	bool Unload();							// unload the current map

	bool Draw(BITMAP *Bitmap);		// draw the tilemap on specified bitmap

	bool SetTile(int iLayer, int iTileX, int iTileY, int iValue);	// set specified tile to specified value
	int  GetTile(int iLayer, int iTileX, int iTileY);				// return specified tile's value

	bool SetTileType(int iTileX, int iTileY, int iValue);			// set tile's 'type' (layer 0)
	int GetTileType(int iTileX, int iTileY);						// return tile's 'type'

	int NumTiles();				// returns number of tiles in current tileset

	bool Scroll(int iDirection, int iSpeed);			// scroll the tilemap 

	bool ClearTo(int iValue);				// clear the entire map to value
	bool ClearLayer(int iLayer, int iValue);// clear given layer to value	

	void SetPlayerStartPos(int x, int y)	{ m_iPlayerStartX = x; m_iPlayerStartY = y;}
	int GetPlayerStartX()					{ return m_iPlayerStartX;			}
	int GetPlayerStartY()					{ return m_iPlayerStartY;			}

	void SetRespawnPos(int x, int y)		{ m_iRespawnPosX = x; m_iRespawnPosY = y;	}
	int GetRespawnPosX()					{ return m_iRespawnPosX;			}
	int GetRespawnPosY()					{ return m_iRespawnPosY;			}
	bool GetHasHitRespawn()					{ return m_bHasHitRespawn;			}
	void SetHasHitRespawn(bool b)			{ m_bHasHitRespawn = b;				}

	void DrawMapData(bool b)				{ m_bDrawData = b;					}
	
	Vector2D MapPos(int iPosX, int iPosY);	// convert position given to map coordinates

	CViewport Viewport;				// the map viewport

};


#endif