// SARC.CPP -- Spades Artificially Reasoning Computer
// By Chris Hiszpanski, 08/16/2001

#include <allegro.h>
#include <iostream.h>
#include <string.h>
#include "carddeck.h"

/// GLOBAL DEFINITIONS //////////////////////////
struct card { suits suit; int value; };                 				// defines a card

/// FUNCTION DECLARATIONS ///////////////////////
void  DrawCard(BITMAP *bmp, DATAFILE *cards, int x, int y, suits suit, int value);
int   BidInput();
int   CheckMouseOnCard(int cards_x_coor[], int mouse_xc, int mouse_yc);
bool  AllPlayed(suits suit, int value, card cards_played[], int num_cards_played);
card  SarcAi(Hand Temp_Hand, card cards_played[], suits &suit_led, bool played[], int turn, bool void_spades[], 
             bool void_diamonds[], bool void_clubs[], bool void_hearts[], int num_cards_played, bool &spades_broken);
void  AddCardPlayed(suits suit, int value, card cards_played[]);
int   EvalTrickWin(int turn, card cards_played[], int num_cards_played);
int   Score(int tricks[], int bid[], int pair, int score);
void  Exit(DATAFILE *cards);

int main()
{    
  /// VARIBLE DEFINITIONS ///////////////////////
  int      i, x = 53, x_coor = 0, y_coor = 0, temp_x = 0, temp_y = 0, save_1 = 0, save_2 = 0, temp_b = 0, 
           min = 53, max = 606, num_cards_played = 0, score_1 = 0, score_2 = 0, turn = 0;
  int      cards_x_coor[26], plays[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
  int      tricks[4] = { 0, 0, 0, 0 };							// number of tricks taken by each player
  int      bid[4] = { 0, 0, 0, 0 };							// players' bids
  int      carrying_card = 14;								// 14 is no card
  card     Card_Played, cards_played[52];						// tracking of cards played
  CardDeck Deck;                                        				// creates unshuffled card deck
  char     bid_str[5];									// for text conversion of bid integer
  suits    suit_led = BLANK;								// suit of card led

  /// FLAG DEFINITIONS //////////////////////////
  bool void_spades[4] = { false, false, false, false };					// tracks voids when someone trumps on some suit
  bool void_diamonds[4] = { false, false, false, false };
  bool void_clubs[4] = { false, false, false, false };
  bool void_hearts[4] = { false, false, false, false };
  bool played[4] = { false, false, false, false };					// tracks who has already played in a trick
  bool spades_broken = false;
  
  /// CREATE HANDS //////////////////////////////
  Hand Hand_1, Hand_2, Hand_3, Hand_4, Temp_Hand;

  /// START GRAPHICS ////////////////////////////
  allegro_init();
  install_keyboard();
  install_mouse();
  install_timer();
  
  set_color_depth(15);									// established 15 or 16 bit color depth
  if (set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0) != 0) {
    set_color_depth(16);
    set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640, 480, 0, 0);
  }

  set_window_title("SARC -- Spades Artificially Reasoning Computer");			// window title
  
  BITMAP *bmp = create_bitmap(640, 480);                				// create a screen buffer
  BITMAP *hidden_area = create_bitmap(73, 97);						// stores area behind a card while moving
  BITMAP *behind_mouse = create_bitmap(17,22);
  DATAFILE *cards = load_datafile("#");	         					// set pointer and load graphics file

  while (!key[KEY_ESC] && score_1 < 500 && score_2 < 500) {				// repeat rounds until someone wins


    /// RESET VARIBLES //////////////////////////

    x = 53;
    min = 53;
    max = 606;
    num_cards_played = 0;
    carrying_card = 14;
    spades_broken = false;
    suit_led = CLUBS;									// clubs must be led first

    for (i = 0; i < 8; i++) {
      plays[i] = 0;
    }

    for (i = 0; i < 4; i++) {
      tricks[i] = 0;
    }

    for (i = 0; i < 4; i++) {
      bid[i] = 0;
    }

    Deck.Shuffle();
  
    Hand_1.SetHand(Deck, 0);
    Hand_2.SetHand(Deck, 1);
    Hand_3.SetHand(Deck, 2);
    Hand_4.SetHand(Deck, 3);

    for (i = 0; i < 4; i++) {
      void_spades[i] = false;
      void_clubs[i] = false;
      void_diamonds[i] = false;
      void_spades[i] = false;
      played[i] = false;
    }

    for (i = 0; i < 52; i++) {                            				// setup the cards played
      cards_played[i].suit = BLANK;
      cards_played[i].value = 0;
    }

    Hand_1.OrgHand ();                                    				// organize by value and suit
    Hand_2.OrgHand ();
    Hand_3.OrgHand ();
    Hand_4.OrgHand ();


    /// DRAW INITIAL GRAPHICS TO BUFFER /////////

    acquire_screen();									// lock screen manuall for rapid directx drawing
    acquire_bitmap(bmp);								// lock buffer manually for rapid directx drawing
    clear_to_color(bmp, makecol(41,115,214));           				// make blue screen

    for (i = 0; i < 12; i++) {                            				// draw cards
      DrawCard(bmp, cards, x, 370, Hand_1.CardSuit(i), Hand_1.CardValue(i));
      cards_x_coor[(i * 2)] = x;
      if (Hand_1.CardSuit(i) != Hand_1.CardSuit(i + 1)) {
        cards_x_coor[(i * 2) + 1] = x + 73;
        x += 100;                                         				// seperate suits
      } else {
        x += 20;
        cards_x_coor[(i * 2) + 1] = x;
      }
    }                                                     				// draw last card
    DrawCard(bmp, cards, x, 370, Hand_1.CardSuit(12), Hand_1.CardValue(12));
    cards_x_coor[24] = x;
    cards_x_coor[25] = x + 73;

    masked_blit((BITMAP *)cards[52].dat, bmp, 0, 0, 10, 174, 73, 129);			// draw computer player graphics
    masked_blit((BITMAP *)cards[52].dat, bmp, 0, 0, 555, 174, 73, 129);
    masked_blit((BITMAP *)cards[52].dat, bmp, 0, 0, 284, 10, 73, 129);

    text_mode(-1);
    rectfill(bmp, 20, 20, 255, 92, makecol(255,255,204));				// draw score box
    rect(bmp, 20, 20, 255, 92, makecol(0,0,0));						// score box text
    textout(bmp, font, "Alvin", 22, 37, makecol(0,0,0));
    textout_centre(bmp, font, "Alvin", 47, 305, makecol(255,192,192));
    textout(bmp, font, "Theodore", 22, 52, makecol(0,0,0));
    textout_centre(bmp, font, "Simon", 320, 141, makecol(255,255,192));
    textout(bmp, font, "Simon", 22, 67, makecol(0,0,0));
    textout_centre(bmp, font, "Theodore", 592, 305, makecol(192,255,192));
    textout(bmp, font, "Human", 22, 82, makecol(0,0,0));
    textout(bmp, font, "Bid", 100, 22, makecol(0,0,0));
    textout(bmp, font, "Tricks", 140, 22, makecol(0,0,0));
    textout(bmp, font, "Score", 205, 22, makecol(0,0,0));

    hline(bmp, 21, 32, 254, makecol(255,192,0));					// score box divisions
    hline(bmp, 21, 62, 254, makecol(255,192,0));
    vline(bmp, 90, 21, 91, makecol(255,192,0));
    vline(bmp, 130, 21, 91, makecol(255,192,0));
    vline(bmp, 195, 21, 91, makecol(255,192,0));

    textout(bmp, font, "0", 140, 45, makecol(0,0,0));					// initial number of tricks
    textout(bmp, font, "0", 140, 75, makecol(0,0,0));
    itoa(score_2, bid_str, 10);								// initial score
    textout(bmp, font, bid_str, 205, 45, makecol(0,0,0));
    itoa(score_1, bid_str, 10);
    textout(bmp, font, bid_str, 205, 75, makecol(0,0,0));

    blit(bmp, screen, 0, 0, 0, 0, 648, 480);              				// display everything before drawing gui for bid input
    release_bitmap(bmp);
    release_screen();

    /// BIDDING /////////////////////////////////

    bid[0] = BidInput();								// human player's bid

    if (bid[0] == 15) {									// esc key pressed
      Exit(cards);
      return 0;
    }

    bid[1] = Hand_2.Bid();								// computer players' bids
    bid[2] = Hand_3.Bid();
    bid[3] = Hand_4.Bid();

    itoa(bid[1] + bid[3], bid_str, 10);							// output bids
    textout(bmp, font, bid_str, 100, 45, makecol(0,0,0));
    itoa(bid[0] + bid[2], bid_str, 10);
    textout(bmp, font, bid_str, 100, 75, makecol(0,0,0));


    //        x1   y1   x2   y2
    rect(bmp, 200, 213, 274, 311, makecol(192,192,192));   				// draw rectangles
    rect(bmp, 283, 160, 357, 258, makecol(192,192,192));
    rect(bmp, 366, 213, 440, 311, makecol(192,192,192));
    rect(bmp, 283, 267, 357, 365, makecol(192,192,192));


    /// SET VARS BEFORE LOOP ////////////////////

    turn = 1;                                            				// 2 of clubs leads
  
    if (Hand_2.LowClub() == 2) {
      turn = 2;
    }

    if (Hand_3.LowClub() == 2) {
      turn = 3;
    }

    if (Hand_4.LowClub() == 2) {
      turn = 4;
    }


  /// MAIN LOOP /////////////////////////////////

    while (!key[KEY_ESC] && num_cards_played != 52) {                                  	// while no esc key

      acquire_screen();
      acquire_bitmap(bmp);

      x_coor = mouse_x - 5;                                       			// save mouse coordinates to prevent debris
      y_coor = mouse_y;
      temp_b = mouse_b;

      if (played[0] == true && played[1] == true && played[2] == true && played[3] == true) {
        rest(2000);
        turn = EvalTrickWin(turn, cards_played, num_cards_played);
        tricks[turn - 1]++;                                       			// increment number of tricks taken
        suit_led = BLANK;
        played[0] = false;                                        			// set that no one has played in the next trick
        played[1] = false;
        played[2] = false;
        played[3] = false;

        rectfill(bmp, 201, 214, 273, 310, makecol(41,115,214)); 			// clear cards
        rectfill(bmp, 284, 161, 356, 257, makecol(41,115,214));
        rectfill(bmp, 367, 214, 439, 310, makecol(41,115,214));
        rectfill(bmp, 284, 268, 356, 364, makecol(41,115,214));

        rectfill(bmp, 138, 43, 158, 61, makecol(255,255,204));    			// trick updating
        rectfill(bmp, 138, 73, 158, 91, makecol(255,255,204));
        itoa(tricks[1] + tricks[3], bid_str, 10);
        textout(bmp, font, bid_str, 140, 45, makecol(0,0,0));
        itoa(tricks[0] + tricks[2], bid_str, 10);
        textout(bmp, font, bid_str, 140, 75, makecol(0,0,0));

        num_cards_played += 4;                                    			// update numbers of cards player
      }

      if (turn == 4 && num_cards_played != 52) {					// sarc ai 4 plays
        rest(500);
        Temp_Hand = Hand_4;
        Temp_Hand.RuleHash(num_cards_played, suit_led, spades_broken);
        Card_Played = SarcAi(Temp_Hand, cards_played, suit_led, played, 4, void_spades, void_diamonds, void_clubs, void_hearts, num_cards_played, spades_broken);
        DrawCard(bmp, cards, 367, 214, Card_Played.suit, Card_Played.value);
        AddCardPlayed(Card_Played.suit, Card_Played.value, cards_played);
        Hand_4.RemoveCard(Card_Played.suit, Card_Played.value);      
        turn = 1;
        played[3] = true;
      }

      if (turn == 3 && num_cards_played != 52) {					// sarc ai 3 plays
        rest(500);
        Temp_Hand = Hand_3;
        Temp_Hand.RuleHash(num_cards_played, suit_led, spades_broken);
        Card_Played = SarcAi(Temp_Hand, cards_played, suit_led, played, 3, void_spades, void_diamonds, void_clubs, void_hearts, num_cards_played, spades_broken);
        DrawCard(bmp, cards, 284, 161, Card_Played.suit, Card_Played.value);
        AddCardPlayed(Card_Played.suit, Card_Played.value, cards_played);
        Hand_3.RemoveCard(Card_Played.suit, Card_Played.value);
        turn = 4;
        played[2] = true;
      }
    
      if (turn == 2 && num_cards_played != 52) {					// sarc ai 2 plays
        rest(500);
        Temp_Hand = Hand_2;
        Temp_Hand.RuleHash(num_cards_played, suit_led, spades_broken);
        Card_Played = SarcAi(Temp_Hand, cards_played, suit_led, played, 2, void_spades, void_diamonds, void_clubs, void_hearts, num_cards_played, spades_broken);
        DrawCard(bmp, cards, 201, 214, Card_Played.suit, Card_Played.value);
        AddCardPlayed(Card_Played.suit, Card_Played.value, cards_played);
        Hand_2.RemoveCard(Card_Played.suit, Card_Played.value);
        turn = 3;
        played[1] = true;
      }


      /// HUMAN PLAYER'S TURN ///////////////////

      if (turn == 1) {
        Temp_Hand = Hand_1;                                         			// check if legal to play card
        Temp_Hand.RuleHash(num_cards_played, suit_led, spades_broken);    
        i = CheckMouseOnCard(cards_x_coor, x_coor, y_coor);
        if (Temp_Hand.FindCard(Hand_1[i].suit, Hand_1[i].value) == -1) {
          i = 14;
        }

        if (mouse_b &1 && carrying_card == 14 && i != 14) {				// begin dragging card
          x = 53;

          temp_x = x_coor - cards_x_coor[i * 2];
          temp_y = y_coor - 370;

          save_1 = cards_x_coor[i * 2 + 1];
          save_2 = cards_x_coor[i * 2];
      
          cards_x_coor[i * 2 + 1] = 0;                              			// set a card as moving
          cards_x_coor[i * 2] = 0;
          carrying_card = i;
                                                          				// clear previous set of cards
          rectfill(bmp, 40, 370, 610, 467, makecol(41,115,214));
          for (i = 0; i < 13; i++) {                                			// redraw Cards
            if (cards_x_coor[i * 2] != 0) {                         			// redisplay all cards except the moving one
              DrawCard(bmp, cards, cards_x_coor[i * 2], 370, Hand_1.CardSuit(i), Hand_1.CardValue(i));
            }
          }
        }

        if (carrying_card != 14 && !temp_b &1 && turn) {				// card dropped in the box
          if (x_coor < 358 && x_coor > 283 && y_coor > 267 && y_coor < 366 ) {
            DrawCard(bmp, cards, 284, 268, Hand_1.CardSuit(carrying_card), Hand_1.CardValue(carrying_card));

            x = 53;
        
            for (i = 0; i < 13; i++) {                              			// condense cards (remove blanks)
              if (cards_x_coor[i * 2] != 0) {
                int n = i + 1;
                while (n < 13 && cards_x_coor[n * 2] == 0) {
                  n++;
                }

                if (n >= 13 || Hand_1.CardSuit(i) != Hand_1.CardSuit(n)) {
                  cards_x_coor[i * 2] = x;
                  cards_x_coor[i * 2 + 1] = x + 73;
                  x += 100;
                } else {
                  cards_x_coor[i * 2] = x;
                  x += 20;
                  cards_x_coor[i * 2 + 1] = x - 1;
                }
              }
            }

            i = 0;                                                  			// center cards
            while (i < 13 && cards_x_coor[i * 2] == 0) {
              i++;
            }
            min = cards_x_coor[i * 2];


            i = 12;
            while (i >= 0 && cards_x_coor[i * 2 + 1] == 0) {
              i--;
            }
            max = cards_x_coor[i * 2 + 1];


            x = 320 - (max - min) / 2 - 53;
            for (i = 0; i < 26; i++) {
              if (cards_x_coor[i] != 0) {
                cards_x_coor[i] += x;
              }
            }
        
            rectfill(bmp, 53, 370, 610, 467, makecol(41,115,214));

            for (i = 0; i < 13; i++) {                            			// redraw Cards
              if (cards_x_coor[i * 2] != 0) {                     			// redisplay all cards except the moving one
                DrawCard(bmp, cards, cards_x_coor[i * 2], 370, Hand_1.CardSuit(i), Hand_1.CardValue(i));
              }
            }
 
            if (suit_led == BLANK) {                              			// set leading suit if leading
              suit_led = Hand_1.CardSuit(carrying_card);
            }
            if (suit_led == SPADES) {
              spades_broken = true;
            }
        
            AddCardPlayed(Hand_1.CardSuit(carrying_card), Hand_1.CardValue(carrying_card), cards_played);
            Hand_1.RemoveCard(Hand_1.CardSuit(carrying_card), Hand_1.CardValue(carrying_card));
            carrying_card = 14;
            turn = 2;
            played[0] = true;

          } else {									// card dropped outside of box
            cards_x_coor[carrying_card * 2 + 1] = save_1;
            cards_x_coor[carrying_card * 2] = save_2;
            rectfill(bmp, 53, 370, 593, 467, makecol(41,115,214));
            for (i = 0; i < 13; i++) {                            			// redraw Cards
              if (cards_x_coor[i * 2] != 0) {                     			// redisplay all cards except the moving one
                DrawCard(bmp, cards, cards_x_coor[i * 2], 370, Hand_1.CardSuit(i), Hand_1.CardValue(i));
              }
            }
            carrying_card = 14;
          }
        }
      }


      /// DRAWING ROUTINE ///////////////////////

      if (carrying_card != 14 && temp_b &1) {
        blit(bmp, hidden_area, x_coor - temp_x, y_coor - temp_y, 0, 0, 73, 97);     	// backup area that moving card overwrites
        DrawCard(bmp, cards, x_coor - temp_x, y_coor - temp_y, Hand_1.CardSuit(carrying_card), Hand_1.CardValue(carrying_card));
      }

      blit(bmp, behind_mouse, x_coor - 5, y_coor, 0, 0, 17, 22);      			// save what is behind mouse and display mouse
      masked_blit((BITMAP *)cards[53].dat, bmp, 0, 0, x_coor - 5, y_coor, 17, 22); 	// show mouse

      blit(bmp, screen, 0, 0, 0, 0, 640, 480);                    			// copy buffer to screen

      blit(behind_mouse, bmp, 0, 0, x_coor - 5, y_coor, 17, 22);      			// redraw what was behind mouse

      if (carrying_card != 14 && temp_b &1) {
        blit(hidden_area, bmp, 0, 0, x_coor - temp_x, y_coor - temp_y, 73, 97);     	// redraw what was behind card
      }

      release_bitmap(bmp);
      release_screen();
    }

    score_1 = Score(tricks, bid, 0, score_1);						// update scores at end of round
    score_2 = Score(tricks, bid, 1, score_2);
  }

  if (score_2 > score_1 && score_2 >= 500) {						// If game over
    clear_to_color(bmp, makecol(255,255,255));
    for (temp_b = 0; temp_b < 5000; temp_b++) {
      textout(bmp, font, "You are victorious!", rand()%640, rand()%480, makecol(41,115,214));
      blit(bmp, screen, 0, 0, 0, 0, 640, 480);
    }
  }

  Exit(cards);
  return 0;
}

END_OF_MAIN();





/// FUNCTIONS ///////////////////////////////////

// Draws a card with a suit and value at (x,y) to buffer
void DrawCard (BITMAP *bmp, DATAFILE *cards, int x, int y, suits suit, int value)
{
  int card_index = (value - 1) * 4;

  if (suit == HEARTS) {
    card_index -= 2;
  } else {
    if (suit == CLUBS) {
      card_index -= 4;
    } else {
      if (suit == DIAMONDS) {
        card_index -= 3;
      } else {
        if (suit == SPADES) {
          card_index -= 1;
        }
      }
    }
  }
  
  blit((BITMAP *)cards[card_index].dat, bmp, 0, 0, x, y, 73, 97);
}

// Allows the player to input bid and returns that value
int BidInput()
{
    int i;

    DIALOG bid_box[] =
    {
    //(dialog proc)        x    y    w    h    fg    bg                      key          flags   d1      d2   dp   dp2 dp3//
      { d_shadow_box_proc, 180, 200, 280, 80,  0,    makecol(255,255,204), 0,           0,      0,      0,   NULL, NULL, NULL },
      { d_ctext_proc,      320, 202, 0,   0,   0,    makecol(255,255,204), 0,           0,      0,      0,   (void *)"Please select your bid:", NULL, NULL },
      { d_radio_proc,      182, 217, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"Nil", NULL, NULL },
      { d_radio_proc,      261, 217, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"1", NULL, NULL },
      { d_radio_proc,      300, 217, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"2", NULL, NULL },
      { d_radio_proc,      339, 217, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"3", NULL, NULL },
      { d_radio_proc,      378, 217, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"4", NULL, NULL },
      { d_radio_proc,      417, 217, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"5", NULL, NULL },
      { d_radio_proc,      182, 237, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"6", NULL, NULL },
      { d_radio_proc,      221, 237, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"7", NULL, NULL },
      { d_radio_proc,      261, 237, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"8", NULL, NULL },
      { d_radio_proc,      300, 237, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"9", NULL, NULL },
      { d_radio_proc,      339, 237, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"10", NULL, NULL },
      { d_radio_proc,      378, 237, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"11", NULL, NULL },
      { d_radio_proc,      417, 237, 15,  15,  0,    makecol(255,255,255), 0,           0,      0,   	1,   (void *)"12", NULL, NULL },
      { d_button_proc,     300, 262, 40,  15,  0,    makecol(255,255,255), KEY_ENTER,   D_EXIT, 0,      0,   (void *)"OK", NULL, NULL },
      { 0,              0,   0,   0,   0,   0,    0,                      0,           	0,      0,      0,   0, NULL, NULL }
    };

  do {
    popup_dialog(bid_box, -1);

    i = 2;

    while (bid_box[i].flags != D_SELECTED && i < 15) {
      i++;
    }

  } while (i == 15 && !key[KEY_ESC]);

  return i;
}

// Returns index (0 - 12) of card which mouse is over. Returns 14 if not over a card
int CheckMouseOnCard(int cards_x_coor[], int mouse_xc, int mouse_yc)
{
  if (mouse_yc > 370 && mouse_yc < 467 && mouse_xc > 40 && mouse_xc < 593) {
    for (int i = 1; i < 26; i += 2) {
      if (mouse_xc <= cards_x_coor[i] && mouse_xc >= cards_x_coor[i - 1]) {
        return ((i - 1) / 2);
      }
    }
  }
  return 14;
}

// SARC Artificial Intelligence: Returns card best to play
card SarcAi(Hand Temp_Hand, card cards_played[], suits &suit_led, bool played[], int turn, bool void_spades[], bool void_diamonds[], bool void_clubs[], bool void_hearts[], int num_cards_played, bool &spades_broken)
{
  card Current_Play;
  int index, highest_spade;
  bool flag, flag2;

  if (suit_led == SPADES) {
    // If all cards above highest played, play highest
    if (AllPlayed(SPADES, Temp_Hand.HighCard(SPADES), cards_played, num_cards_played) && !Temp_Hand.VoidSuit(SPADES)) {
      Current_Play.suit = SPADES;
      Current_Play.value = Temp_Hand.HighCard(SPADES);
      spades_broken = true;
      return Current_Play;
    } else {
      if (Temp_Hand.VoidSuit(SPADES)) {
        void_spades[turn - 1] = true;
        if (!Temp_Hand.VoidSuit(HEARTS)) {
          Current_Play.suit = HEARTS;
          Current_Play.value = Temp_Hand.LowestCard(HEARTS);
          return Current_Play;
        } else {
          if (!Temp_Hand.VoidSuit(CLUBS)) {
            Current_Play.suit = CLUBS;
            Current_Play.value = Temp_Hand.LowestCard(CLUBS);
            return Current_Play;
          } else {
            Current_Play.suit = DIAMONDS;
            Current_Play.value = Temp_Hand.LowestCard(DIAMONDS);
            return Current_Play;
          }
        }
      } else {
        Current_Play.suit = SPADES;
        Current_Play.value = Temp_Hand.LowestCard(SPADES);
        spades_broken = true;
        return Current_Play;
      }
    }
  }

  if (suit_led == DIAMONDS) {
    if (!Temp_Hand.VoidSuit(DIAMONDS) && AllPlayed(DIAMONDS, Temp_Hand.HighCard(DIAMONDS), cards_played, num_cards_played)) {
      Current_Play.suit = DIAMONDS;
      Current_Play.value = Temp_Hand.HighCard(DIAMONDS);
      return Current_Play;
    } else {
      if (Temp_Hand.VoidSuit(DIAMONDS)) {
        void_diamonds[turn - 1] = true;
        
        for (index = 0, flag = false; index < 4; index++) {
          if (index != turn - 1 && !played[index] && void_diamonds[index] && !void_spades[index]) {
            flag = true;
          }
        }

        highest_spade = 1;                               				// Has a higher spade been played?
        for (index = num_cards_played; cards_played[index].suit != BLANK; index++) {    // num_cards_played is updated at the end of the trick but cards_played updated at end of turn
          if (cards_played[index].suit == SPADES && cards_played[index].value > highest_spade) {
            highest_spade = cards_played[index].value;
          }
        }

        flag2 = false;
        if (highest_spade > Temp_Hand.HighCard(SPADES)) {
          flag2 = true;
        }

											// The following two if statements check whether partner already has highest trump
        if (index - num_cards_played == 3 && cards_played[num_cards_played + 1].suit == SPADES && cards_played[num_cards_played + 1].value == highest_spade) {
          flag2 = true;
        }

        if (index - num_cards_played == 2 && cards_played[num_cards_played].suit == SPADES && cards_played[num_cards_played].value == highest_spade) {
          flag2 = true;
        }

        if (!Temp_Hand.VoidSuit(SPADES) && ((!flag && !flag2) || (Temp_Hand.VoidSuit(CLUBS) && Temp_Hand.VoidSuit(HEARTS)))) {
          Current_Play.suit = SPADES;
          if (highest_spade == 1) {
            Current_Play.value = Temp_Hand.LowestCard(SPADES);
          } else {
            Current_Play.value = Temp_Hand.HighCard(SPADES);
          }
          spades_broken = true;
          return Current_Play;
        } else {
          if (!Temp_Hand.VoidSuit(HEARTS)) {
            Current_Play.suit = HEARTS;
            Current_Play.value = Temp_Hand.LowestCard(HEARTS);
            return Current_Play;
          } else {
            if (!Temp_Hand.VoidSuit(CLUBS)) {
              Current_Play.suit = CLUBS;
              Current_Play.value = Temp_Hand.LowestCard(CLUBS);
              return Current_Play;
            } else {
              Current_Play.suit = SPADES;
              Current_Play.value = Temp_Hand.LowestCard(SPADES);
              spades_broken = true;
              return Current_Play;
            }
          }
        }
      } else {
        Current_Play.suit = DIAMONDS;
        Current_Play.value = Temp_Hand.LowestCard(DIAMONDS);
        return Current_Play;
      }
    }
  }

  if (suit_led == CLUBS) {
    if (!Temp_Hand.VoidSuit(CLUBS) && AllPlayed(CLUBS, Temp_Hand.HighCard(CLUBS), cards_played, num_cards_played)) {
      Current_Play.suit = CLUBS;
      Current_Play.value = Temp_Hand.HighCard(CLUBS);
      return Current_Play;
    } else {
      if (Temp_Hand.VoidSuit(CLUBS) == true) {
        void_clubs[turn - 1] = true;

        for (index = 0, flag = false; index < 4; index++) {
          if (index != turn - 1 && !played[index] && void_diamonds[index] && !void_spades[index]) {
            flag = true;
          }
        }
        
        highest_spade = 1;                               				// Has a higher spade been played?
        for (index = num_cards_played; cards_played[index].suit != BLANK; index++) {
          if (cards_played[index].suit == SPADES && cards_played[index].value > highest_spade) {
            highest_spade = cards_played[index].value;
          }
        }

        flag2 = false;
        if (highest_spade > Temp_Hand.HighCard(SPADES)) {
          flag2 = true;
        }

											// The following two if statements check whether partner already has highest trump
        if (index - num_cards_played == 3 && cards_played[num_cards_played + 1].suit == SPADES && cards_played[num_cards_played + 1].value == highest_spade) {
          flag2 = true;
        }

        if (index - num_cards_played == 2 && cards_played[num_cards_played].suit == SPADES && cards_played[num_cards_played].value == highest_spade) {
          flag2 = true;
        }

        if (!Temp_Hand.VoidSuit(SPADES) && ((!flag && !flag2) || (Temp_Hand.VoidSuit(DIAMONDS) && Temp_Hand.VoidSuit(HEARTS)))) {
          Current_Play.suit = SPADES;
          if (highest_spade == 1) {
            Current_Play.value = Temp_Hand.LowestCard(SPADES);
          } else {
            Current_Play.value = Temp_Hand.HighCard(SPADES);
          }
          spades_broken = true;
          return Current_Play;
        } else {
          if (!Temp_Hand.VoidSuit(HEARTS)) {
            Current_Play.suit = HEARTS;
            Current_Play.value = Temp_Hand.LowestCard(HEARTS);
            return Current_Play;
          } else {
            if (!Temp_Hand.VoidSuit(DIAMONDS)) {
              Current_Play.suit = DIAMONDS;
              Current_Play.value = Temp_Hand.LowestCard(DIAMONDS);
              return Current_Play;
            } else {
              Current_Play.suit = SPADES;
              Current_Play.value = Temp_Hand.LowestCard(SPADES);
              spades_broken = true;
              return Current_Play;
            }
          }
        }
      } else {
        Current_Play.suit = CLUBS;
        Current_Play.value = Temp_Hand.LowestCard(CLUBS);
        return Current_Play;
      }
    }
  }
  
  if (suit_led == HEARTS) {
    if (!Temp_Hand.VoidSuit(HEARTS) && AllPlayed(HEARTS, Temp_Hand.HighCard(HEARTS), cards_played, num_cards_played)) {
      Current_Play.suit = HEARTS;
      Current_Play.value = Temp_Hand.HighCard(HEARTS);
      return Current_Play;
    } else {
      if (Temp_Hand.VoidSuit(HEARTS)) {
        void_hearts[turn - 1] = true;

        for (index = 0, flag = false; index < 4; index++) {
          if (index != turn - 1 && !played[index] && void_diamonds[index] && !void_spades[index]) {
            flag = true;
          }
        }

        highest_spade = 1;                               				// Has a higher spade been played?
        for (index = num_cards_played; cards_played[index].suit != BLANK; index++) {
          if (cards_played[index].suit == SPADES && cards_played[index].value > highest_spade) {
            highest_spade = cards_played[index].value;
          }
        }

        flag2 = false;
        if (highest_spade > Temp_Hand.HighCard(SPADES)) {
          flag2 = true;
        }

											// The following two if statements check whether partner already has highest trump
        if (index - num_cards_played == 3 && cards_played[num_cards_played + 1].suit == SPADES && cards_played[num_cards_played + 1].value == highest_spade) {
          flag2 = true;
        }

        if (index - num_cards_played == 2 && cards_played[num_cards_played].suit == SPADES && cards_played[num_cards_played].value == highest_spade) {
          flag2 = true;
        }

        if (!Temp_Hand.VoidSuit(SPADES) && ((!flag && !flag2) || (Temp_Hand.VoidSuit(DIAMONDS) && Temp_Hand.VoidSuit(CLUBS)))) {
          Current_Play.suit = SPADES;
          if (highest_spade == 1) {
            Current_Play.value = Temp_Hand.LowestCard(SPADES);
          } else {
            Current_Play.value = Temp_Hand.HighCard(SPADES);
          }
          spades_broken = true;
          return Current_Play;
        } else {
          if (!Temp_Hand.VoidSuit(CLUBS)) {
            Current_Play.suit = CLUBS;
            Current_Play.value = Temp_Hand.LowestCard(CLUBS);
            return Current_Play;
          } else {
            if (!Temp_Hand.VoidSuit(DIAMONDS)) {
              Current_Play.suit = DIAMONDS;
              Current_Play.value = Temp_Hand.LowestCard(DIAMONDS);
              return Current_Play;
            } else {
              Current_Play.suit = SPADES;
              Current_Play.value = Temp_Hand.LowestCard(SPADES);
              spades_broken = true;
              return Current_Play;
            }
          }
        }
      } else {
        Current_Play.suit = HEARTS;
        Current_Play.value = Temp_Hand.LowestCard(HEARTS);
        return Current_Play;
      }
    }
  }

  // The 'default' case where suit is BLANK
  for (index = 0, flag = false; index < 4; index++) {					// Can anyone be expected to trump? If so, save highest heart
    if (index != turn - 1 && void_hearts[index] && !void_spades[index]) {       	// Note: turn range 1 - 4; index range 0 - 3
      flag = true;									// By setting flag, high card play prevented
    }
  }
  if (!Temp_Hand.VoidSuit(HEARTS) && flag == false && AllPlayed(HEARTS, Temp_Hand.HighCard(HEARTS), cards_played, num_cards_played)) {
    Current_Play.suit = HEARTS;
    Current_Play.value = Temp_Hand.HighCard(HEARTS);
    suit_led = HEARTS;
    return Current_Play;
  }

  for (index = 0, flag = false; index < 4; index++) {					// Can anyone be expected to trump? If so, save highest club
    if (index != turn - 1 && void_clubs[index] && !void_spades[index]) {
      flag = true;
    }
  }
  if (!Temp_Hand.VoidSuit(CLUBS) && flag == false && AllPlayed(CLUBS, Temp_Hand.HighCard(CLUBS), cards_played, num_cards_played)) {
    Current_Play.suit = CLUBS;
    Current_Play.value = Temp_Hand.HighCard(CLUBS);
    suit_led = CLUBS;
    return Current_Play;
  }

  for (index = 0, flag = false; index < 4; index++) {					// Can anyone be expected to trump? If so, save highest diamond
    if (index != turn - 1 && void_diamonds[index] && !void_spades[index]) {
      flag = true;
    }
  }
  if (!Temp_Hand.VoidSuit(DIAMONDS) && flag == false && AllPlayed(DIAMONDS, Temp_Hand.HighCard(DIAMONDS), cards_played, num_cards_played)) {
    Current_Play.suit = DIAMONDS;
    Current_Play.value = Temp_Hand.HighCard(DIAMONDS);
    suit_led = DIAMONDS;
    return Current_Play;
  }

  // Play highest spade if broken and other suits can be trumped and if have highest spade in the game at this point
  if (spades_broken == true && !Temp_Hand.VoidSuit(SPADES) && AllPlayed(SPADES, Temp_Hand.HighCard(SPADES), cards_played, num_cards_played)) {
    Current_Play.suit = SPADES;
    Current_Play.value = Temp_Hand.HighCard(SPADES);
    suit_led = SPADES;
    spades_broken = true;
    return Current_Play;
  }
  if (!Temp_Hand.VoidSuit(HEARTS)) {							// If no sure wins, lead lowest cards in order
    Current_Play.suit = HEARTS;
    Current_Play.value = Temp_Hand.LowestCard(HEARTS);
    suit_led = HEARTS;
    return Current_Play;
  }
  if (!Temp_Hand.VoidSuit(CLUBS)) {
    Current_Play.suit = CLUBS;
    Current_Play.value = Temp_Hand.LowestCard(CLUBS);
    suit_led = CLUBS;
    return Current_Play;
  }
  if (!Temp_Hand.VoidSuit(DIAMONDS)) {
    Current_Play.suit = DIAMONDS;
    Current_Play.value = Temp_Hand.LowestCard(DIAMONDS);
    suit_led = DIAMONDS;
    return Current_Play;
  }
  if (!Temp_Hand.VoidSuit(SPADES)) {							// If void everything else, lead trump suit
    Current_Play.suit = SPADES;
    Current_Play.value = Temp_Hand.LowestCard(SPADES);
    suit_led = SPADES;
    spades_broken = true;
    return Current_Play;
  }

  // Will never reach the following statement but here to make the compiler happy
  return Current_Play;
}

// Returns true if all the cards in a certain suit above a certain value have
// been played. Else, returns false.
bool AllPlayed(suits asuit, int avalue, card cards_played[], int num_cards_played)
{
  bool flag = false;

  for (int i = 14; i > avalue; i--) {
    flag = false;
    for (int j = 0; j < num_cards_played; j++) {
      if (cards_played[j].suit == asuit && cards_played[j].value == i) {
        flag = true;
      }
    }

    if (flag == false) {
      return false;
    }
  }
  return true;
}

// Adds played card to next available position in cards_played array
void AddCardPlayed(suits suit, int value, card cards_played[])
{
  int index = 0;
  while (cards_played[index].suit != BLANK) {
    index++;
  }
  cards_played[index].suit = suit;
  cards_played[index].value = value;
}

// Return turn of who won the trick
int EvalTrickWin(int turn, card cards_played[], int num_cards_played)
{
  suits suit_led = cards_played[num_cards_played].suit;
  int highest_index = num_cards_played;
  int highest_value = 20 * cards_played[highest_index].suit + cards_played[highest_index].value;
  int current_value;
  int winner;
  
  for (int i = num_cards_played + 1; i < num_cards_played + 4; i++) {
    if (cards_played[i].suit == suit_led || cards_played[i].suit == SPADES) {
      current_value = 20 * cards_played[i].suit + cards_played[i].value;
      if (current_value > highest_value) {
        highest_value = current_value;
        highest_index = i;
      }
    }
  }
  
  winner = turn + (highest_index - num_cards_played);
  if (winner > 4) {
     winner -= 4;
  }
  return winner;
}

// Returns the score from the end of a round
int Score(int tricks[], int bid[], int pair, int score)
{
  int bids = bid[pair] + bid[pair + 2];
  int trick = tricks[pair] + tricks[pair + 2];

  // Nil bids
  if (bid[pair] == 0) {
    if (tricks[pair] == 0) {
      score += 100;
    } else {
      score -= 100;
    }
  }

  if (bid[pair + 2] == 0) {
    if (tricks[pair + 2] == 0) {
      score += 100;
    } else {
      score -= 100;
    }
  }
  
  // exact number of tricks
  if (bids == trick) {
    score += (10 * bids);
  }

  int prev_score = score;

  // too many tricks
  if (bids < trick) {
    score += (10 * bids + (trick - bids));
  }

  // 10 overbook subtraction
  if ((prev_score % 10 > score % 10) || (trick - bids >= 10)) {
    score -= 100;
  }

  // not enough tricks
  if ( bids > trick) {
    score -= (10 * bids);
  }

  return score;
}

// Performs shutdown commands
void Exit(DATAFILE *cards) {
  unload_datafile(cards);                               				// Done drawing cards
  set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);                   				// Switch back to text mode
  allegro_exit();                                       				// Shuts-off graphics and mouse
}