/* GIF Loader
   by Paul Bartrum
   Animation by Connelly Barnes */

#include <stdlib.h>
#include <stdio.h>
#include <allegro.h>
#include <string.h>


#define MEMORY_BITMAP 0
#define VIDEO_BITMAP  1
#define SYSTEM_BITMAP 2
#define HAS_ANIMATE_FLAG(bmp) ((bmp)->clip == 2)
#define SET_ANIMATE_FLAG(bmp) ((bmp)->clip = 2)
#define CLEAR_ANIMATE_FLAG(bmp) ((bmp)->clip = 0)

#if (ALLEGRO_DATE<20000000)
#define ALLEGRO_312
#else
#define ALLEGRO_WIP
#endif

#ifdef ALLEGRO_WIP
/* An internal Allegro function, needed to find proper color conversion */
int _color_load_depth(int depth, int hasalpha);
#else
int _color_load_depth(int depth);
#endif

/* Allegro 3.12 doesn't use 'const char *' when it should.  Use ALLEGRO_CHAR *
   in place of 'const char *' to fix this problem */
#ifdef ALLEGRO_WIP
#define ALLEGRO_CHAR const char
#else
#define ALLEGRO_CHAR char
#endif

struct LZW_STRING
{
	short base;
  char n;
	short length;
};
PACKFILE *gif_file;
int global_empty_string, curr_bit_size, bit_overflow;
int bit_pos, data_pos, data_len, gif_entire, gif_code;
int gif_cc, string_length, global_i, bit_size;
unsigned char string[4096];
struct LZW_STRING gif_str[4096];
BITMAP *global_bmp;
int image_x, image_y, image_w, image_h, global_x, global_y;
int interlace;
int t_color = -1, _disposal, gif_registered = 0;

void clear_table(void)
{
   global_empty_string = gif_cc + 2;
	curr_bit_size = bit_size + 1;
	bit_overflow = 0;
}

void get_code(void)
{
	if(bit_pos + curr_bit_size > 8) {
    if(data_pos >= data_len) { data_len = pack_getc(gif_file); data_pos = 0; }
    gif_entire = (pack_getc(gif_file) << 8) + gif_entire;
		data_pos ++;
	}
	if(bit_pos + curr_bit_size > 16) {
    if(data_pos >= data_len) { data_len = pack_getc(gif_file); data_pos = 0; }
    gif_entire = (pack_getc(gif_file) << 16) + gif_entire;
		data_pos ++;
	}
  gif_code = (gif_entire >> bit_pos) & ((1 << curr_bit_size) - 1);
	if(bit_pos + curr_bit_size > 8)
    gif_entire >>= 8;
	if(bit_pos + curr_bit_size > 16)
    gif_entire >>= 8;
	bit_pos = (bit_pos + curr_bit_size) % 8;
	if(bit_pos == 0) {
    if(data_pos >= data_len) { data_len = pack_getc(gif_file); data_pos = 0; }
    gif_entire = pack_getc(gif_file);
		data_pos ++;
	}
}


void get_string(int num)
{
  if(num < gif_cc)
	{
		string_length = 1;
    string[0] = gif_str[num].n;
	}
	else
	{
    global_i = gif_str[num].length;
    string_length = global_i;
    while(global_i > 0)
		{
      global_i --;
      string[global_i] = gif_str[num].n;
      num = gif_str[num].base;
		}
		/* if(num != -1) **-{[ERROR]}-** */
	}
}

void output_string(void)
{
  for(global_i = 0; global_i < string_length; global_i ++)
	{
    if (t_color >= 0) {
      if (string[global_i] == t_color) {
        /* do nothing */
      } else if (string[global_i] == 0) {
        putpixel(global_bmp, global_x, global_y, t_color);
      } else {
        putpixel(global_bmp, global_x, global_y, string[global_i]);
      }
    } else {
      putpixel(global_bmp, global_x, global_y, string[global_i]);
    }
    global_x ++;
    if(global_x >= image_x + image_w)
		{
      global_x = image_x;
      global_y += interlace;
			if(interlace)
			{
        if(global_y >= image_y + image_h)
				{
          if(interlace == 8 && (global_y - image_y) % 8 == 0) {
						interlace = 8;
            global_y = image_y + 4;
					}
          else if(interlace == 8  && (global_y - image_y) % 8 == 4) {
						interlace = 4;
            global_y = image_y + 2;
					}
					else if(interlace == 4) {
						interlace = 2;
            global_y = image_y + 1;
					}
				}
			}
		}
   }
}


/* Returns size of bitmap structure, needed because Allegro uses ugly dynamic
   structure sizing. */
#define bmp_struct_size(bmp) (sizeof(BITMAP) + sizeof(char *) * (bmp)->h)
/* load_gif:
    Loads a 2-256 colour GIF file onto a bitmap, returning the bitmap
    structure and storing the pallete data in the specified pallete (this
    should be an array of at least 256 RGB structures).

    Now also binds an ANIMATION into a global linked list so the returned
    BITMAP struct can be updated behind your back - creating animation...

    Be careful to NOT call destroy_bitmap() on the returned bitmap.
 */

BITMAP *load_gif_ex(ALLEGRO_CHAR *, RGB *, int);

BITMAP *load_gif(ALLEGRO_CHAR *filename, RGB *pal)
{
  return load_gif_ex(filename, pal, MEMORY_BITMAP);
}

BITMAP *load_video_gif(ALLEGRO_CHAR *filename, RGB *pal)
{
  return load_gif_ex(filename, pal, VIDEO_BITMAP);
}

#ifdef ALLEGRO_WIP
BITMAP *load_system_gif(ALLEGRO_CHAR *filename, RGB *pal) {
  return load_gif_ex(filename, pal, SYSTEM_BITMAP);
}
#else
BITMAP *load_system_gif(ALLEGRO_CHAR *filename, RGB *pal) {
  return load_gif_ex(filename, pal, VIDEO_BITMAP);
}
#endif

BITMAP *load_gif_ex(ALLEGRO_CHAR *filename, RGB *pal, int bmp_type)
{
  int width, height, depth;
  int old, ocolor, yval, xval, pcolor;
  int dest_depth;
  PALETTE swap_pal;
  BITMAP *frame = NULL;
  int current_delay = 0;
  int temp_pal = 0;

  _disposal = 0;

  if (!pal) {
    pal = (RGB *) malloc(sizeof(RGB) * 256);
    temp_pal = 1;
  }

  gif_file = pack_fopen(filename, F_READ);
  if (!gif_file) /* can't open file */
		return NULL;

  global_i  = pack_mgetw(gif_file) << 8;
  global_i += pack_getc(gif_file);
  if(global_i != 0x474946) /* is it really a GIF? */
	{
    pack_fclose(gif_file);
		return NULL;
	}
  pack_fseek(gif_file, 3); /* skip version */

  width = pack_igetw(gif_file);
  height = pack_igetw(gif_file);

  global_i = pack_getc(gif_file);
  if(global_i & 128) /* no global colour table? */
    depth = (global_i & 7) + 1;
	else
		depth = 0;

  pack_fseek(gif_file, 2); /* skip background colour and aspect ratio */

	if(pal && depth) /* only read palette if pal and depth are not 0 */
	{
    for(global_i = 0; global_i < (1 << depth); global_i ++)
		{
      pal[global_i].r = pack_getc(gif_file) / 4;
      pal[global_i].g = pack_getc(gif_file) / 4;
      pal[global_i].b = pack_getc(gif_file) / 4;
		}
	}
	else
		if(depth)
      pack_fseek(gif_file, (1 << depth) * 3);

  global_bmp = 0;
  t_color = -1;
  do
	{
    global_i = pack_getc(gif_file);
    if (pack_feof(gif_file)) { global_i = 0x3B; }
    switch(global_i)
		{
			case 0x2C: /* Image Descriptor */
        image_x = pack_igetw(gif_file);
        image_y = pack_igetw(gif_file); /* individual image dimensions */
        image_w = pack_igetw(gif_file);
        image_h = pack_igetw(gif_file);

        if (!global_bmp) {
          global_bmp = create_bitmap_ex(8, width, height);
          if (global_bmp == NULL) {
            pack_fclose(gif_file);
            return NULL;
          }
          clear_to_color(global_bmp, bitmap_mask_color(global_bmp));
        }

        if (_disposal == 2 || _disposal == 3) {
          set_clip(global_bmp, image_x, image_y, image_x+image_w-1, image_y+image_h-1);
          clear_to_color(global_bmp, bitmap_mask_color(global_bmp));
        } else {
          set_clip(global_bmp, 0, 0, global_bmp->w - 1, global_bmp->h - 1);
        }

        global_i = pack_getc(gif_file);
        if(global_i & 64)
					interlace = 8;
				else
					interlace = 1;

        if(global_i & 128)
				{
          /* FIXME: Per-frame palettes will be butchered */
          depth = (global_i & 7) + 1;
          if(pal)
					{
            for(global_i = 0; global_i < (1 << depth); global_i ++)
						{
              pal[global_i].r = pack_getc(gif_file) / 4;
              pal[global_i].g = pack_getc(gif_file) / 4;
              pal[global_i].b = pack_getc(gif_file) / 4;
						}
					}
					else
            pack_fseek(gif_file, (1 << depth) * 3);
				}

				/* lzw stream starts now */
        bit_size = pack_getc(gif_file);
        gif_cc = 1 << bit_size;

				/* initialise string table */
        for(global_i = 0; global_i < gif_cc; global_i ++)
				{
          gif_str[global_i].base = -1;
          gif_str[global_i].n = global_i;
          gif_str[global_i].length = 1;
				}

				/* initialise the variables */
				bit_pos = 0;
        data_len = pack_getc(gif_file); data_pos = 0;
        gif_entire = pack_getc(gif_file); data_pos ++;
        string_length = 0; global_x = image_x; global_y = image_y;

        /* starting gif_code */
				clear_table();
				get_code();
        if(gif_code == gif_cc)
					get_code();
        get_string(gif_code);
				output_string();
        old = gif_code;

				while(TRUE)
				{
					get_code();

          if(gif_code == gif_cc)
					{
            /* starting gif_code */
						clear_table();
						get_code();
            get_string(gif_code);
						output_string();
            old = gif_code;
					}
          else if(gif_code == gif_cc + 1)
					{
						break;
					}
          else if(gif_code < global_empty_string)
					{
            get_string(gif_code);
						output_string();

						if(bit_overflow == 0) {
              gif_str[global_empty_string].base = old;
              gif_str[global_empty_string].n = string[0];
              gif_str[global_empty_string].length = gif_str[old].length + 1;
              global_empty_string ++;
              if(global_empty_string == (1 << curr_bit_size))
								curr_bit_size ++;
							if(curr_bit_size == 13) {
								curr_bit_size = 12;
								bit_overflow = 1;
							}
						}

            old = gif_code;
					}
					else
					{
						get_string(old);
            string[gif_str[old].length] = string[0];
						string_length ++;

						if(bit_overflow == 0) {
              gif_str[global_empty_string].base = old;
              gif_str[global_empty_string].n = string[0];
              gif_str[global_empty_string].length = gif_str[old].length + 1;
              global_empty_string ++;
              if(global_empty_string == (1 << curr_bit_size))
								curr_bit_size ++;
							if(curr_bit_size == 13) {
								curr_bit_size = 12;
								bit_overflow = 1;
							}
						}

						output_string();
            old = gif_code;
					}
				}

        if (pal) {
          memcpy((void *) swap_pal, (void *) pal, sizeof(RGB) * 256);
          if (t_color >= 0) {
            swap_pal[t_color] = swap_pal[0];
            swap_pal[0].r = 63;
            swap_pal[0].g = 0;
            swap_pal[0].b = 63;
          }
        }

        if (bmp_type == MEMORY_BITMAP) {
          /* convert to correct colour depth */
#ifdef ALLEGRO_WIP
          dest_depth = _color_load_depth(8, 0);
#else
          dest_depth = _color_load_depth(8);
#endif

          if (!dest_depth) { dest_depth = 8; }

          /* copy global (drawing) bitmap to more permanent memory. */
          frame = create_bitmap_ex(dest_depth, global_bmp->w, global_bmp->h);
        } else if (bmp_type == VIDEO_BITMAP) {
          frame = create_video_bitmap(global_bmp->w, global_bmp->h);
#ifdef ALLEGRO_WIP
        } else if (bmp_type == SYSTEM_BITMAP) {
          frame = create_system_bitmap(global_bmp->w, global_bmp->h);
#endif
        }

        if (!frame) {
          pack_fclose(gif_file);
          destroy_bitmap(frame);
          destroy_bitmap(global_bmp);
          return NULL;
        }
       
        select_palette(swap_pal);
        blit(global_bmp, frame, 0, 0, 0, 0, global_bmp->w, global_bmp->h);
        unselect_palette();
		break;
			case 0x21: /* Extension Introducer */
        global_i = pack_getc(gif_file);
        if(global_i == 0xF9) /* Graphic Control Extension */
				{
          pack_fseek(gif_file, 1); /* skip size (it's 4) */
          global_i = pack_getc(gif_file);
          current_delay = pack_igetw(gif_file);

          if(global_i & 1) /* is transparency enabled? */
					{
            ocolor = t_color;
            t_color = pack_getc(gif_file); /* transparent colour */

            /* was transparency just enabled?
               if so, swap around transparent colors */
            if (global_bmp && ocolor < 0) {
              for (yval = 0; yval < global_bmp->h; yval++) {
                for (xval = 0; xval < global_bmp->w; xval++) {
                  pcolor = _getpixel(global_bmp, xval, yval);
                  if (pcolor == 0) { _putpixel(global_bmp, xval, yval, t_color); }
                }
              }
            }
					}
          else
          {
            t_color = -1;
            pack_fseek(gif_file, 1);
          }

          _disposal = (global_i >> 2) & 7;

				}
        global_i = pack_getc(gif_file);
        while(global_i) /* skip Data Sub-blocks */
				{
          pack_fseek(gif_file, global_i);
          global_i = pack_getc(gif_file);
				}
				break;
			case 0x3B: /* Trailer - end of data */
        pack_fclose(gif_file);

          if (temp_pal) { free((void *) pal); }
          destroy_bitmap(global_bmp);

          return frame;
        }
	} while(TRUE);

	/* this is never executed but DJGPP complains if you leave it out */
	return NULL;
}




