#include <math.h>
#include <stdlib.h>
#include "allegro.h"
#include "rnd.h"
#include "warp.h"


#define SPRINGINESS 0.01


WARP *create_warp(int w, int h) {
 int x, y;

 WARP *warp = malloc(sizeof(WARP) + h*sizeof(WARP_POINT *));

 warp->w = w;
 warp->h = h;

 for (y=0; y<h; y++) {
  warp->line[y] = malloc(w*sizeof(WARP_POINT));

  for (x=0; x<w; x++) {
   warp->line[y][x].x = RND(16) - 8;
   warp->line[y][x].y = RND(16) - 8;
   warp->line[y][x].xv = 0;
   warp->line[y][x].yv = 0;
  }
 }

 return warp;
}


void propagate_warp(WARP *warp) {
 int x, y;

/* Helper macro for constructing warpers for different parts of the array.
   This is necessary because we need to access points adjacent to each
   point. Separate loops are used for the edges to avoid array overruns. */
#define WARPER(BEGIN_Y, BEGIN_X, low, mid, high, vel)   \
                                                        \
BEGIN_Y {                                               \
 BEGIN_X {                                              \
  float v;                                              \
                                                        \
  /* Calculate force on this point in the warp array */ \
  v = (high) - 2*(mid) + (low);                         \
                                                        \
  /* Adjust speed of point */                           \
  warp->line[y][x].vel += v * SPRINGINESS;              \
 }                                                       \
}

 /* Horizontal speeds, omitting left and right edges */
 WARPER( for (y=0; y < warp->h; y++)   ,
         for (x=1; x < warp->w-1; x++) ,
         warp->line[y][x-1].x, warp->line[y][x].x, warp->line[y][x+1].x, xv);

 /* Horizontal speeds at left edge */
 WARPER( for (y=0; y < warp->h; y++)   ,
         x = 0;                        ,
         0                   , warp->line[y][x].x, warp->line[y][x+1].x, xv);

 /* Horizontal speeds at right edge */
 WARPER( for (y=0; y < warp->h; y++)   ,
         x = warp->w-1;                ,
         warp->line[y][x-1].x, warp->line[y][x].x, 0                   , xv);

 /* Vertical speeds, omitting top and bottom edges */
 WARPER( for (y=1; y < warp->h-1; y++) ,
         for (x=0; x < warp->w; x++)   ,
         warp->line[y-1][x].y, warp->line[y][x].y, warp->line[y+1][x].y, yv);

 /* Vertical speeds at top edge */
 WARPER( y = 0;                        ,
         for (x=0; x < warp->w; x++)   ,
         0                   , warp->line[y][x].y, warp->line[y+1][x].y, yv);

 /* Vertical speeds at bottom edge */
 WARPER( y = warp->h-1;                ,
         for (x=0; x < warp->w; x++)   ,
         warp->line[y-1][x].y, warp->line[y][x].y, 0                   , yv);

 /* Finally, we add the velocities to the positions */
 for (y=0; y < warp->h; y++) {
  for (x=0; x < warp->w; x++) {
   warp->line[y][x].x += warp->line[y][x].xv;
   warp->line[y][x].y += warp->line[y][x].yv;
  }
 }
}


/* This retrieves a point from the warp array, applying interpolation
   and correctly handling points outside the array.
*/
void get_warp_position(WARP *warp, float x, float y, float *xp, float *yp) {
 int xi, yi;

 {
  float f;

  f = floor(x);
  x -= f;
  xi = f;

  f = floor(y);
  y -= f;
  yi = f;
 }

 {
  float x_xl, y_xl, x_xh, y_xh;
  float x_yl, y_yl, x_yh, y_yh;

  /* Interpolate top-left and top-right points */
  if (yi >= 0 && yi < warp->h) {

   if (xi >= 0 && xi < warp->w) {
    x_xl = warp->line[yi][xi].x;
    y_xl = warp->line[yi][xi].y;
   } else {
    x_xl = 0;
    y_xl = 0;
   }

   if (xi >= -1 && xi < warp->w-1) {
    x_xh = warp->line[yi][xi+1].x;
    y_xh = warp->line[yi][xi+1].y;
   } else {
    x_xh = 0;
    y_xh = 0;
   }

   x_yl = x_xl + (x_xh-x_xl) * x;
   y_yl = y_xl + (y_xh-y_xl) * x;

  } else {
   x_yl = 0;
   y_yl = 0;
  }

  /* Interpolate bottom-left and bottom-right points */
  yi++;

  if (yi >= 0 && yi < warp->h) {

   if (xi >= 0 && xi < warp->w) {
    x_xl = warp->line[yi][xi].x;
    y_xl = warp->line[yi][xi].y;
   } else {
    x_xl = 0;
    y_xl = 0;
   }

   if (xi >= -1 && xi < warp->w-1) {
    x_xh = warp->line[yi][xi+1].x;
    y_xh = warp->line[yi][xi+1].y;
   } else {
    x_xh = 0;
    y_xh = 0;
   }

   x_yh = x_xl + (x_xh-x_xl) * x;
   y_yh = y_xl + (y_xh-y_xl) * x;

  } else {
   x_yh = 0;
   y_yh = 0;
  }

  /* Interpolate top and bottom interpolated values */
  *xp = x_yl + (x_yh-x_yl) * y;
  *yp = y_yl + (y_yh-y_yl) * y;
 }
}


void destroy_warp(WARP *warp) {
 int y;

 for (y=0; y < warp->h; y++) free(warp->line[y]);

 free(warp);
}
