#include "grid.h"
#include "dataLevels.h"
#include "dataGfx.h"


Grid::Grid() : Object(4, 4, 0, 0), nBricks(0), currentLevel(-1) {
	w = 16;
	h = 20;
	blocks.reserve(w*h);
	for (int i=0; i<w*h; ++i) {
		blocks.push_back(Block());
	}
	
	int bx = 0;
	int by = 0;
	int bw = blocks[0].Width();
	int bh = blocks[0].Height();
	for (vector<Block>::iterator i = blocks.begin(); i != blocks.end(); ++i) {
		i->Create(x + bx, y + by, 1, NULL);
		bx += bw;
		if (bx >= w * bw) {
			bx = 0;
			by += bh;
		}
	}
}


Grid::~Grid() {
}


void Grid::Update() {
	for (vector<Block>::iterator i = blocks.begin(); i != blocks.end(); ++i) {
		i->Update();
	}
}


void Grid::Draw(BITMAP *buffer) {
	for (vector<Block>::iterator i = blocks.begin(); i != blocks.end(); ++i) {
		i->Draw(buffer);
	}
}


void Grid::CountBricks() {
	nBricks = 0;
	for (vector<Block>::iterator i = blocks.begin(); i != blocks.end(); ++i) {
		if (i->IsBrick()) {
			++nBricks;
		}
	}
}


bool Grid::Load(int level, DATAFILE *dataGfx, DATAFILE *dataLevels) {
	char name[32];
	usprintf(name, "LEVEL%02d", level+1);
	DATAFILE *gridDat = find_datafile_object(dataLevels, name);
	if (!gridDat) {
		return false;
	}
	
	char *grid = (char *)gridDat->dat;

	//char *grid = (char *)dataLevels[level + LEVEL01].dat;

	int bx = 0;
	int by = 0;
	int bw = blocks[0].Width();
	int bh = blocks[0].Height();

	for (int i=0; i<w*h; ++i) {
		blocks[i].Reset();
		blocks[i].Create(x + bx, y + by, grid[i]-1, (grid[i] != 0) ? (BITMAP *)dataGfx[grid[i]-1+BLOCK0].dat : NULL);

		bx += bw;
		if (bx >= w * bw) {
			bx = 0;
			by += bh;
		}
	}
	
	CountBricks();
	return true;
}


bool Grid::FirstLevel(DATAFILE *dataGfx, DATAFILE *dataLevels) {
	currentLevel = 0;
	return Load(currentLevel, dataGfx, dataLevels);
}


bool Grid::NextLevel(DATAFILE *dataGfx, DATAFILE *dataLevels) {
	++currentLevel;
	if (!Load(currentLevel, dataGfx, dataLevels)) {
		--currentLevel;
		return false;
	}
	else {
		return true;
	}
}


bool Grid::CollisionDetection(Ball *b, int &destroyedBricks, int &destroyedIndex) {
	// get ball position (relative to the grid) and speed
	float bx = b->GetX() - x;
	float by = b->GetY() - y;
	float bvx = b->GetVx();
	float bvy = b->GetVy();
	int bw = b->Size();
	bool ret = false;
	
	int blockW = blocks[0].Width();
	int blockH = blocks[0].Height();
	
	// get ball direction
	int dir = bvy < 0 ? (bvx >= 0 ? 0 : 1) : (bvx >= 0 ? 2 : 3);

	// get grid coordinates of the 4 corners of the ball
	int i[4], j[4];
	i[0] = (int)bx/blockW;		j[0] = (int)by/blockH;
	i[1] = (int)(bx+bw)/blockW;	j[1] = (int)by/blockH;
	i[2] = (int)(bx+bw)/blockW;	j[2] = (int)(by+bw)/blockH;
	i[3] = (int)bx/blockW;		j[3] = (int)(by+bw)/blockH;

	// handle each direction separately
	bool p3 = true;		// do we have to check the third point as well?
	float c;
	bool changeDir = false;
	int fieldSize = w*h;
	int index;
	switch (dir) {
	case 0:
		index = j[0]*w + i[0];
		if (index < fieldSize && !blocks[index].Dead()) {
			b->BounceY();
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			p3 = false;
			changeDir = true;
			ret = true;
		}
		
		index = j[2]*w + i[2];
		if (index < fieldSize && !blocks[index].Dead()) {
			b->BounceX();
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			p3 = false;
			changeDir = true;
			ret = true;
		}

		index = j[1]*w + i[1];
		if (index < fieldSize && p3) {
			if (blocks[index].Dead()) return ret;
			c = bx + bw + (bvx/bvy)*((j[1]+1)*blockH - by);
			if (c >= i[1]*blockW) {
				b->BounceY();
			}
			else {
				b->BounceX();
			}
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			changeDir = true;
			ret = true;
		}
		break;
	case 1:
		index = j[1]*w + i[1];
		if (index < fieldSize && !blocks[index].Dead()) {
			b->BounceY();
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			p3 = false;
			changeDir = true;
			ret = true;
		}

		index = j[3]*w + i[3];
		if (index < fieldSize && !blocks[index].Dead()) {
			b->BounceX();
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			p3 = false;
			changeDir = true;
			ret = true;
		}
		
		index = j[0]*w + i[0];
		if (index < fieldSize && p3) {
			if (blocks[index].Dead()) return ret;
			c = bx + (bvx/bvy)*((j[1]+1)*blockH - by);
			if (c >= i[0]*blockW) {
				b->BounceY();
			}
			else {
				b->BounceX();
			}
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			changeDir = true;
			ret = true;
		}
		break;
	case 2:
		index = j[3]*w + i[3];
		if (index < fieldSize && !blocks[index].Dead()) {
			b->BounceY();
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			p3 = false;
			changeDir = true;
			ret = true;
		}

		index = j[1]*w + i[1];
		if (index < fieldSize && !blocks[index].Dead()) {
			b->BounceX();
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			p3 = false;
			changeDir = true;
			ret = true;
		}

		index = j[2]*w + i[2];
		if (index < fieldSize && p3) {
			if (blocks[index].Dead()) return ret;
			c = bx + bw + (bvx/bvy)*((j[2])*blockH - by - bw);
			if (c >= i[2]*blockW) {
				b->BounceY();
			}
			else {
				b->BounceX();
			}
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			changeDir = true;
			ret = true;
		}
		break;
	case 3:
		index = j[2]*w + i[2];
		if (index < fieldSize && !blocks[index].Dead()) {
			b->BounceY();
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			p3 = false;
			changeDir = true;
			ret = true;
		}

		index = j[0]*w + i[0];
		if (index < fieldSize && !blocks[index].Dead()) {
			b->BounceX();
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			p3 = false;
			changeDir = true;
			ret = true;
		}

		index = j[3]*w + i[3];
		if (index < fieldSize && p3) {
			if (blocks[index].Dead()) return ret;
			c = bx + (bvx/bvy)*((j[3])*blockH - by - bw);
			if (c >= i[3]*blockW) {
				b->BounceY();
			}
			else {
				b->BounceX();
			}
			if (blocks[index].Hit()) {
				++destroyedBricks;
				destroyedIndex = index;
			}
			changeDir = true;
			ret = true;
		}
		break;
	};

	if (!changeDir) {
		b->SetVx(bvx);
		b->SetVy(bvy);
	}
	else {
		//b->SetY(by - bvy + y);
		//b->SetX(bx + (bw>>1) - bvx + x);
	}
	
	nBricks -= destroyedBricks;
	return ret;
}


bool Grid::CollisionDetection(Bullet *b, int &destroyedBricks, int &destroyedIndex) {
	bool ret = false;
	
	// get bullet position (relative to the grid)
	int bx = b->GetX() - x;
	int by = b->GetY() - y;
	
	// get grid coordinates of the 4 corners of the ball
	int i = bx/BlockWidth();
	int j = by/BlockHeight();

	int index = j*w + i;
	if (index < w*h && !blocks[index].Dead()) {
		if (blocks[index].Hit()) {
			++destroyedBricks;
			destroyedIndex = index;
		}
		
		ret = true;
	}

	nBricks -= destroyedBricks;
	return ret;
}


bool Grid::IsWall(int bx, int by) {
	// get bullet position (relative to the grid)
	bx = bx - x;
	by = by - y;
	
	// get grid coordinates of the 4 corners of the ball
	int i = bx/BlockWidth();
	int j = by/BlockHeight();

	int index = j*w + i;
	if (index < w*h && blocks[index].Unbreakable()) {
		return true;
	}
	else {
		return false;
	}
}


bool Grid::IsHit(int bx, int by) {
	// get bullet position (relative to the grid)
	bx = bx - x;
	by = by - y;
	
	// get grid coordinates of the 4 corners of the ball
	int i = bx/BlockWidth();
	int j = by/BlockHeight();

	int index = j*w + i;
	if (index < w*h && blocks[index].IsHit()) {
		return true;
	}
	else {
		return false;
	}
}
