#include "paranoid.h"
#include "params.h"
#include "dataGfx.h"
#include "dataLevels.h"
#include "dataSounds.h"
#include "dataMusic.h"
#include "dataFont.h"
#include "global.h"
#include "screenshot.h"
#include <math.h>


RGB_MAP Paranoid::rgbTable;

Paranoid::Paranoid()
	:Game(),
	nBalls(0),
	score(),
	lives(3),
	shotDelay(0),
	endLevelDelay(0),
	musicPlayerDelay(0),
	scrollingText(NULL),
	pressingSpace(false),
	sinescroller(NULL),
	gameState(DOING_MAIN_MENU),
	transitionCounter(24),
	maxTransitionCounter(24),
	mainMenu(NULL),
	optionsMenu(NULL),
	creditsMenu(NULL),
	levelClearedMenu(NULL),
	hscMenu(NULL),
	gameOverMenu(NULL),
	hscEditMenu(NULL),
	helpMenu(NULL),
	shortcutsMenu(NULL),
	pauseMenu(NULL),
	finishedMenu(NULL),
	titleBitmap(NULL),
	hscTable(NULL),
	tvscreen(NULL)
{
	usprintf(gameTitle, "Paranoid");
	dataGfx = NULL;
	dataLevels = NULL;
	dataSounds = NULL;
	dataFont = NULL;
	soundVolume = 255;
}


Paranoid::~Paranoid() {
	if (dataGfx) {
		unload_datafile(dataGfx);
	}
	
	if (dataLevels) {
		unload_datafile(dataLevels);
	}
	
	if (dataSounds) {
		unload_datafile(dataSounds);
	}
	
	if (dataFont) {
		unload_datafile(dataFont);
	}
	
	for (list<Ball *>::iterator i = balls.begin(); i != balls.end();) {
		delete *i;
		*i = NULL;
		i = balls.erase(i);
	}
	nBalls = 0;

	for (list<Powerup *>::iterator i = powerups.begin(); i != powerups.end();) {
		delete *i;
		*i = NULL;
		i = powerups.erase(i);
	}

	for (list<Bullet *>::iterator i = bullets.begin(); i != bullets.end();) {
		delete *i;
		*i = NULL;
		i = bullets.erase(i);
	}
	
	if (scrollingText) {
		delete scrollingText;
		scrollingText = NULL;
	}
	
	if (mainMenu) {
		delete mainMenu;
		mainMenu = NULL;
	}
	
	if (optionsMenu) {
		delete optionsMenu;
		optionsMenu = NULL;
	}

	if (creditsMenu) {
		delete creditsMenu;
		creditsMenu = NULL;
	}
	
	if (levelClearedMenu) {
		delete levelClearedMenu;
		levelClearedMenu = NULL;
	}
	
	if (gameOverMenu) {
		delete gameOverMenu;
		gameOverMenu = NULL;
	}
	
	if (hscEditMenu) {
		delete hscEditMenu;
		hscEditMenu = NULL;
	}
	
	if (helpMenu) {
		delete helpMenu;
		helpMenu = NULL;
	}
	
	if (shortcutsMenu) {
		delete shortcutsMenu;
		shortcutsMenu = NULL;
	}
	
	if (pauseMenu) {
		delete pauseMenu;
		pauseMenu = NULL;
	}

	if (finishedMenu) {
		delete finishedMenu;
		finishedMenu = NULL;
	}

	if (titleBitmap) {
		destroy_bitmap(titleBitmap);
		titleBitmap = NULL;
	}
	
	if (hscTable) {
		delete hscTable;
		hscTable = NULL;
	}
	
	mPlayer.Stop();
	
	if (sinescroller) {
		delete sinescroller;
		sinescroller = NULL;
	}
}


void Paranoid::DestroyDoubleBuffer() {
	Game::DestroyDoubleBuffer();
	
	if (tvscreen) {
		destroy_bitmap(tvscreen);
		tvscreen = NULL;
	}
}


Error Paranoid::Init() {
	Error e = Game::Init();
	if (e) {
		return e;
	}
	
	// other specific initialization goes here
	dataGfx = load_datafile(Params::dataGfx);
	if (!dataGfx) {
		return Error(Error::NOBITMAP);
	}

	dataFont = load_datafile(Params::dataFont);
	if (!dataFont) {
		return Error(Error::NOBITMAP);
	}
	font = (FONT *)dataFont[FNT6x6].dat;
	
	set_palette(*(PALETTE *)dataGfx[PAL].dat);
	create_rgb_table(&rgbTable, *(PALETTE *)dataGfx[PAL].dat, NULL);
	rgb_map = &rgbTable;
	
	BITMAP *back = (BITMAP *)dataGfx[SPRITE_BACK].dat;
	paddle.SetSprite((BITMAP *)dataGfx[SPRITE_PADDLE].dat);
	paddle.SetRange(4, back->w - 4);
	paddle.SetW(24);
	paddle.SetY(back->h-8-1);
	
	dataLevels = load_datafile(Params::dataLevels);
	if (!dataLevels) {
		return Error(Error::NOBITMAP);
	}
	grid.Load(0, dataGfx, dataLevels);
	
	dataSounds = load_datafile(Params::dataSounds);
	if (!dataSounds) {
		return Error(Error::UNKNOWN);
	}

	score.Setup(Params::screenWidth-4, Params::screenHeight-8, font);
	scrollingText = new ScrollText(52, Params::screenHeight-8, 11, font, " PARANOID  - version 1.0 - Copyright (c) 2003 by MIRAN AMON", " LEVEL 1");
	Powerup::Init();
	
	Menu::back = (BITMAP *)dataGfx[SPRITE_MENU].dat;
	mainMenu = new MainMenu;
	optionsMenu = new OptionsMenu;
	creditsMenu = new CreditsMenu;
	levelClearedMenu = new LevelClearedMenu;
	gameOverMenu = new GameOverMenu;
	hscEditMenu = new HscEditMenu;
	helpMenu = new HelpMenu(dataGfx);
	shortcutsMenu = new ShortcutsMenu;
	pauseMenu = new PauseMenu;
	finishedMenu = new FinishedMenu;
	
	titleBitmap = create_bitmap(text_length(font, "PARANOID")+1, text_height(font)+1);
	clear(titleBitmap);
	text_mode(-1);
	textout(titleBitmap, font, "PARANOID", 1, 1, makecol(120,120,120));
	textout(titleBitmap, font, "PARANOID", 0, 0, makecol(200,200,200));
	
	hscTable = new Highscore("paranoid.hsc");

	sinescroller = new SineScroller("PARANOID V1.0 - AN AMAZING BREAKOUT GAME WITH 40 LEVELS OF PURE FUN - USE ARROW KEYS TO CONTROL THE PADDLE AND SPACE TO RELEASE THE BALL AND FIRE BULLETS - KEYPAD UP AND DOWN ARROWS CHANGE MUSIC VOLUME, PGUP AND PGDN CHANGE SOUND VOLUME, LEFT AND RIGHT ARROWS SELECT TRACKS AND THE 5 KEY REPLAYS THE CURRENT TRACK - COPYRIGHT (c) 2003 BY MIRAN AMON - ");
	DrawBackground();
	mainMenu->Start(tvscreen);

	soundVolume = Params::soundVolume;
	atexit(&dumb_exit);
	dumb_register_stdfiles();
	dumb_resampling_quality = DUMB_RQ_ALIASING;
	
	mPlayer.LoadPlaylist(Params::dataMusic);
	mPlayer.SetLoop(1);
	mPlayer.SetRandom(1);
	mPlayer.SetVolume(Params::musicVolume);
	mPlayer.Play(0);
	
	rest(500);
	return Error(Error::NONE);
}


Error Paranoid::InitDoubleBuffer() {
	tvscreen = create_bitmap(Params::screenWidth, Params::screenHeight);
	if (!tvscreen) {
		return Error(Error::MEMORY);
	}
	else {
		clear(tvscreen);
		return Game::InitDoubleBuffer();
	}
}


void Paranoid::NewBall() {
	Ball *ball = new Ball;
	ball->SetSprite((BITMAP *)dataGfx[SPRITE_BALL].dat);
	ball->SetRange(4, Params::screenWidth-4, 4, Params::screenHeight - 9 - paddle.Height());
	ball->SetX(20);
	ball->SetY(20);
	balls.push_back(ball);
	paddle.GiveBall(ball);
	++nBalls;
}


void Paranoid::ExtraBalls() {
	Ball *ball = new Ball;
	ball->SetSprite((BITMAP *)dataGfx[SPRITE_BALL].dat);
	ball->SetRange(4, Params::screenWidth-4, 4, Params::screenHeight - 9 - paddle.Height());
	
	Ball *parent = *(--balls.end());
	float x = parent->GetX();
	float y = parent->GetY();
	ball->SetX(x);
	ball->SetY(y);
	float vx = parent->GetVy()/2.0f;
	if (vx > 0.0f) {
		if (vx < Params::ballMinVx) {
			vx = Params::ballMinVx;
		}
	}
	else if (vx > -Params::ballMinVx) {
		vx = -Params::ballMinVx;
	}
	float vy = sqrt(3.0f - vx*vx);
	if (vy > 0.0f) {
		if (vy < Params::ballMinVy) {
			vy = Params::ballMinVy;
		}
	}
	else if (vy > -Params::ballMinVy) {
		vy = -Params::ballMinVy;
	}
	ball->Launch(-vx, -vy);
	balls.push_back(ball);
	++nBalls;
}


void Paranoid::MakePowerup(int x, int y) {
	int r = Powerup::RandomPowerup();
	if (r != -1) {
		Powerup *p = new Powerup(x, y, r, (BITMAP *)dataGfx[POWERUP00+r].dat);
		powerups.push_back(p);
	}
}


void Paranoid::UpdateBalls() {
	for (list<Ball *>::iterator i = balls.begin(); i != balls.end();) {
		Ball *b = *i;
		b->Update();
		switch (b->HitWall()) {
			case 1:
			case 2:
				PlaySample(SMP_WALL);
				break;
			case 3:
				PlaySample(SMP_WALL);
				break;
		};
		if (b->Dead()) {
			if (b->GetY() > Params::screenHeight) {
				delete b;
				b = NULL;
				i = balls.erase(i);
				--nBalls;
				if (!nBalls && !grid.Clear()) {
					--lives;
					if (lives <= 0) {
						PlaySample(SMP_GAMEOVER);
						GameOver();
					}
					else {
						PlaySample(SMP_LOSELIFE);
						paddle.Reset();
					}
				}
			}
		}
		else {
			// check if it bounces of the paddle
			if (b->GetY() + b->Size() >= paddle.GetY()) {
				if (b->GetX() >= paddle.GetX() - b->Size() && b->GetX() <= paddle.GetX() + paddle.Width()) {
					if (paddle.BounceBall(b)) {
						PlaySample(SMP_WALL);
					}
				}
			}
			
			// check if it hits a brick
			int destroyedBricks = 0;
			int destroyedIndex = 0;
			if (grid.CollisionDetection(b, destroyedBricks, destroyedIndex)) {
				if (!destroyedBricks) {
					PlaySample(SMP_BLOCK);
				}
			}
			score += destroyedBricks*100;
			CheckForNewLife(destroyedBricks*100);
			if (destroyedBricks > 0) {
				PlaySample(SMP_CLEAR);
				MakePowerup(grid.GetX() + 3 + grid.BlockWidth()*(destroyedIndex%grid.Width()), grid.GetY() + grid.BlockHeight() + grid.BlockHeight()*destroyedIndex/grid.Width());
			}
			++i;
		}
	}
}


void Paranoid::NewBullet() {
	if (shotDelay) return;

	switch (paddle.Bullets()) {
		case 1: {
			Bullet *b = new Bullet((int)(paddle.GetX() + (paddle.Width()>>1)), (int)paddle.GetY(), (BITMAP *)dataGfx[SPRITE_BULLET].dat);
			bullets.push_back(b);
			PlaySample(SMP_SHOT);
		}	break;
		case 2: {
			Bullet *b = new Bullet((int)(paddle.GetX() + (paddle.Width()/3)), (int)paddle.GetY(), (BITMAP *)dataGfx[SPRITE_BULLET].dat);
			bullets.push_back(b);
			b = new Bullet((int)(paddle.GetX() + ((paddle.Width()/3)<<1)), (int)paddle.GetY(), (BITMAP *)dataGfx[SPRITE_BULLET].dat);
			bullets.push_back(b);
			PlaySample(SMP_SHOT);
		}	break;
		case 3: {
			Bullet *b = new Bullet((int)(paddle.GetX() + (paddle.Width()>>2)), (int)paddle.GetY(), (BITMAP *)dataGfx[SPRITE_BULLET].dat);
			bullets.push_back(b);
			b = new Bullet((int)(paddle.GetX() + (paddle.Width()>>1)), (int)paddle.GetY(), (BITMAP *)dataGfx[SPRITE_BULLET].dat);
			bullets.push_back(b);
			b = new Bullet((int)(paddle.GetX() + 3*(paddle.Width()>>2)), (int)paddle.GetY(), (BITMAP *)dataGfx[SPRITE_BULLET].dat);
			bullets.push_back(b);
			PlaySample(SMP_SHOT);
		}	break;
	};
	
	shotDelay = 4;
}


void Paranoid::UpdateBullets() {
	for (list<Bullet *>::iterator i = bullets.begin(); i != bullets.end();) {
		Bullet *b = *i;
		b->Update();
		if (b->GetY() < 4 - b->Height() || grid.IsWall(b->GetX(), b->GetY()) || grid.IsHit(b->GetX(), b->GetY())) {
			delete b;
			b = NULL;
			i = bullets.erase(i);
		}
		else {
			// check if it hits a brick
			int destroyedBricks = 0;
			int destroyedIndex = 0;
			if (grid.CollisionDetection(b, destroyedBricks, destroyedIndex)) {
				if (!destroyedBricks) {
					PlaySample(SMP_BLOCK);
				}
			}
			score += destroyedBricks*100;
			CheckForNewLife(destroyedBricks*100);
			if (destroyedBricks > 0) {
				PlaySample(SMP_CLEAR);
				MakePowerup(grid.GetX() + 3 + grid.BlockWidth()*(destroyedIndex%grid.Width()), grid.GetY() + grid.BlockHeight() + grid.BlockHeight()*destroyedIndex/grid.Width());
				delete b;
				b = NULL;
				i = bullets.erase(i);
			}
			else {
				++i;
			}
		}
	}
}


void Paranoid::ChangeBallSpeed(float factor) {
	for (list<Ball *>::iterator i = balls.begin(); i != balls.end(); ++i) {
		(*i)->ChangeSpeed(factor);
	}
}


void Paranoid::UpdatePowerups() {
	for (list<Powerup *>::iterator i = powerups.begin(); i != powerups.end();) {
		Powerup *p = *i;
		p->Update();
		if (p->GetY() > paddle.GetY() - p->Height() + 3) {
			if (p->GetX() >= paddle.GetX() - (p->Width()>>1) && p->GetX() <= paddle.GetX() + paddle.Width() - (p->Width()>>1)) {
				//handle powerup
				switch (p->GetType()) {
					case Powerup::SIZEUP:
						paddle.Grow();
						PlaySample(SMP_POWERUP);
						break;
					case Powerup::SIZEDOWN:
						paddle.Shrink();
						PlaySample(SMP_POWERDOWN);
						break;
					case Powerup::SPEEDUP:
						ChangeBallSpeed(1.2f);
						PlaySample(SMP_POWERDOWN);
						break;
					case Powerup::SPEEDDOWN:
						ChangeBallSpeed(0.9f);
						PlaySample(SMP_POWERUP);
						break;
					case Powerup::MONEY:
						score += 250;
						if (!CheckForNewLife(250)) {
							PlaySample(SMP_POWERUP);
						}
						break;
					case Powerup::MONEYBIG:
						score += 2500;
						if (!CheckForNewLife(2500)) {
							PlaySample(SMP_POWERUP);
						}
						break;
					case Powerup::GLUE:
						paddle.MakeSticky();
						PlaySample(SMP_POWERUP);
						break;
					case Powerup::MULTIBALL:
						ExtraBalls();
						PlaySample(SMP_POWERUP);
						break;
					case Powerup::SHOOT:
						paddle.GiveBullet();
						PlaySample(SMP_POWERUP);
						break;
					case Powerup::EXTRALIFE:
						++lives;
						PlaySample(SMP_EXTRALIFE);
						break;
				};
				
				delete p;
				p = NULL;
				i = powerups.erase(i);
			}
			else if (p->GetY() >= paddle.GetY() + paddle.Height()) {
				delete p;
				p = NULL;
				i = powerups.erase(i);
			}
			else {
				++i;
			}
		}
		else {
			++i;
		}
	}
}


bool Paranoid::CheckForNewLife(int delta) {
	int s = score.Get();
	int olds = s - delta;
	if (s%25000 < olds%25000) {
		++lives;
		PlaySample(SMP_EXTRALIFE);
		return true;
	}
	
	return false;
}


bool Paranoid::UpdateGame() {
	if (transitionCounter == 0) {
		if (nBalls < 1) NewBall();
		if (key[KEY_SPACE]) {
			if (!pressingSpace) {
				NewBullet();
			}
			
			pressingSpace = true;
		}
		else {
			pressingSpace = false;
		}
		if (shotDelay > 0) --shotDelay;
		paddle.Update();
		UpdateBalls();
		UpdatePowerups();
		UpdateBullets();
		grid.Update();
		if (grid.Clear()) {
			RemoveBalls();
			if (!endLevelDelay) {
				endLevelDelay = maxTransitionCounter;
			}
			else {
				--endLevelDelay;
				if (endLevelDelay <= 0 && powerups.empty() && score.Done()) {
					endLevelDelay = 0;
					NextLevel();
					PlaySample(SMP_LEVELDONE);
				}
			}
		}
		score.Update();
		scrollingText->Update();

		if (!musicPlayerDelay) {
			if (key[KEY_Q]) {
				RemoveBalls();
				--lives;
				if (lives <= 0) {
					PlaySample(SMP_GAMEOVER);
					GameOver();
				}
				else {
					PlaySample(SMP_LOSELIFE);
					NewBall();
					paddle.LaunchBall();
				}
				musicPlayerDelay = 20;
			}
		}
	}
	
	return key[KEY_ESC];
}


bool Paranoid::Update() {
	switch (gameState) {
		case PLAYING_GAME:
			if (UpdateGame()) {
				DrawBackground();
				mainMenu->Start(tvscreen);
				gameState = DOING_MAIN_MENU;
				transitionCounter = maxTransitionCounter;
				mPlayer.Play(0);
			}
			else {
				if (key[KEY_P]) {
					pauseMenu->Start(tvscreen);
					gameState = DOING_PAUSE_MENU;
				}
			}
			break;
		
		case DOING_MAIN_MENU:
			if (mainMenu->Update()) {
				mainMenu->Stop(tvscreen);
				switch (mainMenu->State()) {
					case MainMenu::MENU_EXIT:
						gameState = DOING_EXIT_SCREEN;
						transitionCounter = maxTransitionCounter;
						break;
					
					case MainMenu::MENU_NEW_GAME:
						NewGame();
					case MainMenu::MENU_RESUME_GAME:
						transitionCounter = maxTransitionCounter;
						gameState = PLAYING_GAME;
						mPlayer.Next();
						break;
					
					case MainMenu::MENU_OPTIONS:
						optionsMenu->Start(tvscreen);
						gameState = DOING_OPTIONS_MENU;
						break;
					
					case MainMenu::MENU_HELP:
						helpMenu->Start(tvscreen);
						gameState = DOING_HELP_MENU;
						break;
					
					case MainMenu::MENU_SHORTCUTS:
						shortcutsMenu->Start(tvscreen);
						gameState = DOING_SHORTCUTS_MENU;
						break;
					
					case MainMenu::MENU_CREDITS:
						creditsMenu->Start(tvscreen);
						gameState = DOING_CREDITS_MENU;
						break;
					
					case MainMenu::MENU_SCORES:
						hscMenu = new HighscoreMenu(hscTable);
						hscMenu->Start(tvscreen);
						gameState = DOING_HSC_MENU;
						break;
					
					default:
						transitionCounter = maxTransitionCounter;
						gameState = PLAYING_GAME;
						break;
				};
			}
			break;

		case DOING_OPTIONS_MENU:
			if (optionsMenu->Update()) {
				if (optionsMenu->State() == OptionsMenu::MENU_BACK) {
					optionsMenu->Stop(tvscreen);
					mainMenu->Start(tvscreen);
					gameState = DOING_MAIN_MENU;
				}
			}
			soundVolume = Params::soundVolume;
			mPlayer.SetVolume(Params::musicVolume);
			break;

		case DOING_HELP_MENU:
			if (helpMenu->Update()) {
				helpMenu->Stop(tvscreen);
				mainMenu->Start(tvscreen);
				gameState = DOING_MAIN_MENU;
			}
			break;

		case DOING_SHORTCUTS_MENU:
			if (shortcutsMenu->Update()) {
				shortcutsMenu->Stop(tvscreen);
				mainMenu->Start(tvscreen);
				gameState = DOING_MAIN_MENU;
			}
			break;

		case DOING_CREDITS_MENU:
			if (creditsMenu->Update()) {
				creditsMenu->Stop(tvscreen);
				mainMenu->Start(tvscreen);
				gameState = DOING_MAIN_MENU;
			}
			break;

		case DOING_LEVEL_CLEARED_MENU:
			if (levelClearedMenu->Update()) {
				levelClearedMenu->Stop(tvscreen);
				transitionCounter = maxTransitionCounter;
				gameState = PLAYING_GAME;
				mPlayer.Next();
			}
			break;

		case DOING_HSC_MENU2:
		case DOING_HSC_MENU:
			if (hscMenu->Update()) {
				hscMenu->Stop(tvscreen);
				delete hscMenu;
				hscMenu = NULL;
				if (gameState == DOING_HSC_MENU2) {
					transitionCounter = maxTransitionCounter;
				}
				mainMenu->Start(tvscreen);
				gameState = DOING_MAIN_MENU;
			}
			break;

		case DOING_GAME_OVER_MENU:
			if (gameOverMenu->Update()) {
				gameOverMenu->Stop(tvscreen);
				mainMenu->Start(tvscreen);
				transitionCounter = maxTransitionCounter;
				gameState = DOING_MAIN_MENU;
				mPlayer.Play(0);
			}
			break;

		case DOING_GAME_OVER_MENU2:
			if (gameOverMenu->Update()) {
				gameOverMenu->Stop(tvscreen);
				hscEditMenu->Start(tvscreen);
				gameState = DOING_HSC_EDIT_MENU;
			}
			break;

		case DOING_FINISHED_MENU:
			if (finishedMenu->Update()) {
				finishedMenu->Stop(tvscreen);
				mainMenu->Start(tvscreen);
				transitionCounter = maxTransitionCounter;
				gameState = DOING_MAIN_MENU;
				mPlayer.Play(0);
			}
			break;

		case DOING_FINISHED_MENU2:
			if (finishedMenu->Update()) {
				finishedMenu->Stop(tvscreen);
				hscEditMenu->Start(tvscreen);
				gameState = DOING_HSC_EDIT_MENU;
			}
			break;

		case DOING_HSC_EDIT_MENU:
			if (hscEditMenu->Update()) {
				hscEditMenu->Stop(tvscreen);
				hscTable->Add(hscEditMenu->GetText(), score.Get(), grid.Level());
				grid.FirstLevel(dataGfx, dataLevels);
				lives = 3;
				score.Reset();
				hscMenu = new HighscoreMenu(hscTable);
				hscMenu->Start(tvscreen);
				gameState = DOING_HSC_MENU2;
				mPlayer.Play(0);
			}
			break;

		case DOING_PAUSE_MENU:
			if (pauseMenu->Update()) {
				pauseMenu->Stop(tvscreen);
				gameState = PLAYING_GAME;
			}
			break;
		
		case DOING_EXIT_SCREEN:
			if (keypressed() && !transitionCounter) {
				mPlayer.Stop();
				fade_out(2);
				return true;
			}
			else {
				clear_keybuf();
			}
			break;
	};

	if (transitionCounter) {
		--transitionCounter;
	}

	if (!musicPlayerDelay) {
		if (key[KEY_8_PAD] && Params::musicVolume < 255) {
			Params::musicVolume++;
			mPlayer.SetVolume(Params::musicVolume);
		} else
		if (key[KEY_2_PAD] && Params::musicVolume > 0) {
			Params::musicVolume--;
			mPlayer.SetVolume(Params::musicVolume);
		}

		if (key[KEY_9_PAD] && soundVolume < 255) {
			soundVolume++;
		} else
		if (key[KEY_3_PAD] && soundVolume > 0) {
			soundVolume--;
		}

		if (key[KEY_4_PAD]) {
			mPlayer.Previous();
			musicPlayerDelay = 20;
		} else
		if (key[KEY_6_PAD]) {
			mPlayer.Next();
			musicPlayerDelay = 20;
		} else
		if (key[KEY_5_PAD]) {
			mPlayer.Play();
			musicPlayerDelay = 20;
		}
		
		if (key[KEY_F]) {
			Params::showFPS = Params::showFPS ? 0 : 1;
			musicPlayerDelay = 20;
		}
		if (key[KEY_U]) {
			Params::unlimitedFPS = Params::unlimitedFPS ? 0 : 1;
			musicPlayerDelay = 20;
		}
		if (key[KEY_V]) {
			Params::vsync = Params::vsync ? 0 : 1;
			musicPlayerDelay = 20;
		}
		if (key[KEY_Y] || key[KEY_Z]) {
			Params::yield = Params::yield ? 0 : 1;
			musicPlayerDelay = 20;
		}
		if (key[KEY_M]) {
			Params::updateMethod++;
			Params::updateMethod%=4;
			clear(buffer);
			clear(screen);
			musicPlayerDelay = 20;
		}
		if (key[KEY_PRTSCR]) {
			Screenshot::Take(buffer);
			musicPlayerDelay = 20;
		}
	}
	else {
		--musicPlayerDelay;
	}

	mPlayer.Update();
	sinescroller->Update();
	Game::Update();
	return false;
}


void Paranoid::DrawGame() {
	BITMAP *back = (BITMAP *)dataGfx[SPRITE_BACK].dat;
	blit(back, tvscreen, 0, 0, 0, 0, back->w, back->h-9);
	for (list<Bullet *>::iterator i = bullets.begin(); i != bullets.end(); ++i) {
		(*i)->Draw(tvscreen);
	}
	grid.Draw(tvscreen);
	for (list<Powerup *>::iterator i = powerups.begin(); i != powerups.end(); ++i) {
		(*i)->Draw(tvscreen);
	}
	paddle.Draw(tvscreen);
	for (list<Ball *>::iterator i = balls.begin(); i != balls.end(); ++i) {
		(*i)->Draw(tvscreen);
	}
	
	text_mode(-1);
	char buf[6];
	blit(back, tvscreen, 0, back->h-9, 0, back->h-9, back->w, 10);
	textout(tvscreen, font, "LIFE", 6, Params::screenHeight-8, makecol(64,64,64));
	usprintf(buf, "%d", lives);
	textout(tvscreen, font, buf, 34, Params::screenHeight-8, makecol(96,96,96));
	scrollingText->Draw(tvscreen);
	textout_right(tvscreen, font, "SCORE", Params::screenWidth-50, Params::screenHeight-8, makecol(64,64,64));
	score.Draw(tvscreen);
	if (Params::showFPS) {
		textprintf(tvscreen, font, 4, 4, makecol(255,255,255), "FPS = %d", fps->Get());
	}
}


void Paranoid::Draw() {
	switch (gameState) {
		case PLAYING_GAME:
			DrawGame();
			break;
		
		case DOING_MAIN_MENU:
			DrawBackground();
			mainMenu->Draw(tvscreen);
			break;
		
		case DOING_OPTIONS_MENU:
			DrawBackground();
			optionsMenu->Draw(tvscreen);
			break;
		
		case DOING_HELP_MENU:
			DrawBackground();
			helpMenu->Draw(tvscreen);
			break;
		
		case DOING_SHORTCUTS_MENU:
			DrawBackground();
			shortcutsMenu->Draw(tvscreen);
			break;
		
		case DOING_CREDITS_MENU:
			DrawBackground();
			creditsMenu->Draw(tvscreen);
			break;
		
		case DOING_LEVEL_CLEARED_MENU:
			levelClearedMenu->Draw(tvscreen);
			break;
		
		case DOING_HSC_MENU:
			DrawBackground();
		case DOING_HSC_MENU2:
			hscMenu->Draw(tvscreen);
			break;

		case DOING_GAME_OVER_MENU2:
		case DOING_GAME_OVER_MENU:
			gameOverMenu->Draw(tvscreen);
			break;

		case DOING_FINISHED_MENU2:
		case DOING_FINISHED_MENU:
			finishedMenu->Draw(tvscreen);
			break;

		case DOING_HSC_EDIT_MENU:
			hscEditMenu->Draw(tvscreen);
			break;

		case DOING_PAUSE_MENU:
			pauseMenu->Draw(tvscreen);
			break;
		
		case DOING_EXIT_SCREEN:
			DrawExitScreen();
			break;
	};
}


void Paranoid::DrawDoubleBuffer() {
	switch (Params::updateMethod) {
		case 0:
			RadarBlit(tvscreen, buffer);
			Game::DrawDoubleBuffer();
			break;
		case 1:
			TVBlit(tvscreen, buffer);
			Game::DrawDoubleBuffer();
			break;
		case 2:
			StretchBlit(tvscreen, buffer);
			Game::DrawDoubleBuffer();
			break;
		default:
			blit(tvscreen, screen, 0, 0, (SCREEN_W - tvscreen->w)>>1, (SCREEN_H - tvscreen->h)>>1, tvscreen->w, tvscreen->h);
			break;
	};
}


void Paranoid::StretchBlit(BITMAP *src, BITMAP *dest) {
	int startY;
	int destH;
	
	startY = transitionCounter * src->h / maxTransitionCounter;
	destH = src->h - startY;

	int factor = MIN(dest->w/src->w, dest->h/src->h);
	//stretch_blit(tvscreen, screen, 0, 0, tvscreen->w, tvscreen->h, (SCREEN_W - (factor*tvscreen->w))>>1, (SCREEN_H - (factor*tvscreen->h))>>1, factor*tvscreen->w, factor*tvscreen->h);

	stretch_blit(src, dest, 0, startY, src->w, destH, (SCREEN_W - (factor*src->w))>>1, (SCREEN_H - (factor*src->h))>>1, factor*src->w, factor*destH);
}

void Paranoid::RadarBlit(BITMAP *src, BITMAP *dest) {
	int startY;
	int destH;
	
	startY = transitionCounter * src->h / maxTransitionCounter;
	destH = src->h;
	
	int i,j=0;
	for (int y=startY; y<destH; ++y) {
		i = (y%2) + (dest->w - 3*src->w)>>1;
		for (int x=0; x<src->w; ++x) {
			//int col = getpixel(src, x, y);
			int col = ((unsigned char *)src->line[y])[x];
			int r = getr8(col);
			int g = getg8(col);
			int b = getb8(col);
			
			//vline(dest, i, j, j+2, makecol(0,MAX(r,g),0));
			col = makecol8(0, MAX(r,g), 0);
			dest->line[j][i] = col;
			dest->line[j+1][i] = col;
			dest->line[j+2][i] = col;
			
			
			//vline(dest, i+2, j, j+2, makecol(0,MAX(b,g),0));
			col = makecol8(0, MAX(b,g), 0);
			i += 2;
			dest->line[j][i] = col;
			dest->line[j+1][i] = col;
			dest->line[j+2][i] = col;
			
			++i;
		}
		j += 3;
	}
}


void Paranoid::TVBlit(BITMAP *src, BITMAP *dest) {
	int startY;
	int destH;
	
	startY = transitionCounter * src->h / maxTransitionCounter;
	destH = src->h;

	int i,j=0;
	for (int y=startY; y<destH; ++y) {
		i = (dest->w - 3*src->w)>>1;
		for (int x=0; x<src->w; ++x) {
			int col = ((unsigned char *)src->line[y])[x];
			int r = getr8(col);
			int g = getg8(col);
			int b = getb8(col);
			
			//vline(dest, i, j, j+2, makecol(r,0,0));
			col = makecol8(r,0,0);
			dest->line[j][i] = col;
			dest->line[j+1][i] = col;
			dest->line[j+2][i] = col;
			++i;
			
			//vline(dest, i+1, j, j+2, makecol(0,g,0));
			col = makecol8(0,g,0);
			dest->line[j][i] = col;
			dest->line[j+1][i] = col;
			dest->line[j+2][i] = col;
			++i;
			
			//vline(dest, i+2, j, j+2, makecol(0,0,b));
			col = makecol8(0,0,b);
			dest->line[j][i] = col;
			dest->line[j+1][i] = col;
			dest->line[j+2][i] = col;
			++i;
		}
		j += 3;
	}
}


void Paranoid::NewGame() {
	lives = 3;
	if (!grid.FirstLevel(dataGfx, dataLevels)) {
		allegro_message("Couldn't load any levels!");
		allegro_exit();
	}
	Reset();
	score.Reset();
}


void Paranoid::NextLevel() {
	if (!grid.NextLevel(dataGfx, dataLevels)) {
		GameFinished();
	}
	else {
		char buf[16];
		usprintf(buf, "LEVEL %d", grid.Level());
		scrollingText->SetText2(buf);
		Reset();
		levelClearedMenu->Start(tvscreen);
		gameState = DOING_LEVEL_CLEARED_MENU;
	}
}


void Paranoid::GameFinished() {
	Reset();

	if (hscTable->CanAdd(score.Get())) {
		finishedMenu->Start(tvscreen);
		gameState = DOING_FINISHED_MENU2;
	}
	else {
		finishedMenu->Start(tvscreen);
		gameState = DOING_FINISHED_MENU;
		grid.FirstLevel(dataGfx, dataLevels);
		lives = 3;
		score.Reset();
	}
}


void Paranoid::GameOver() {
	Reset();
	
	if (hscTable->CanAdd(score.Get())) {
		gameOverMenu->Start(tvscreen);
		gameState = DOING_GAME_OVER_MENU2;
	}
	else {
		gameOverMenu->Start(tvscreen);
		gameState = DOING_GAME_OVER_MENU;
		grid.FirstLevel(dataGfx, dataLevels);
		lives = 3;
		score.Reset();
	}
}


void Paranoid::Reset() {
	paddle.Reset();
	RemoveBalls();

	for (list<Powerup *>::iterator pi = powerups.begin(); pi != powerups.end(); ++pi) {
		delete *pi;
		*pi = NULL;
	}
	powerups.clear();

	for (list<Bullet *>::iterator ti = bullets.begin(); ti != bullets.end(); ++ti) {
		delete *ti;
		*ti = NULL;
	}
	bullets.clear();

	NewBall();
}


void Paranoid::RemoveBalls() {
	for (list<Ball *>::iterator bi = balls.begin(); bi != balls.end(); ++bi) {
		delete *bi;
		*bi = NULL;
	}
	nBalls = 0;
	balls.clear();
}


void Paranoid::DrawBackground() {
	BITMAP *tile = (BITMAP *)dataGfx[BLOCK4].dat;
	int dx = tile->w;
	int dy = tile->h;
	
	int x, y;
	for (y=0; y<tvscreen->h; y += dy) {
		x = ((y/dy)%2) ? -(dx>>1) : 0;
		for (; x<tvscreen->w; x += dx) {
			blit(tile, tvscreen, 0, 0, x, y, dx, dy);
		}
	}
	
	stretch_sprite(tvscreen, titleBitmap, (tvscreen->w - (titleBitmap->w<<2))>>1, 4, titleBitmap->w<<2, 5*titleBitmap->h);
	sinescroller->Draw(tvscreen);
}


void Paranoid::DrawExitScreen() {
	clear_to_color(tvscreen, makecol(0,0,0));
	
	int x = tvscreen->w>>1;
	textout_centre(tvscreen, font, "THANK YOU FOR PLAYING!", x, 50, makecol(220,220,220));
	textout_centre(tvscreen, font, "PARANOID V1.0", x, 100, makecol(160,160,160));
	textout_centre(tvscreen, font, "COPYRIGHT (c) 2003 BY MIRAN AMON", x, 112, makecol(160,160,160));
	textout_centre(tvscreen, font, "THIS GAME WAS MADE WITH ALLEGRO", x, 140, makecol(100,100,100));
}
