#include "menuitem.h"
#include "global.h"
#include "dataSounds.h"


int MenuItem::maxCounter = 20;
int MenuItem::maxKeyCounter = 8;


MenuItem::MenuItem() : currentState(NORMAL), stateCounter(0), keyCounter(0), colFactor(1.0f), dCF(-0.02f) {
}


MenuItem::~MenuItem() {
}


void MenuItem::Update() {
	if (stateCounter) {
		--stateCounter;
	}

	if (keyCounter) {
		--keyCounter;
	}
	
	colFactor += dCF;
	if (colFactor > 1.0f) {
		colFactor = 1.0f;
		dCF = -dCF;
	}
	else if (colFactor < 0.8f) {
		colFactor = 0.8f;
		dCF = -dCF;
	}
}


void MenuItem::Draw(BITMAP *bmp, int x, int y, int w, int h) {
	int col = stateCounter*48/maxCounter;
	if (currentState == SELECTED) {
		col = 48 - col;
		col += 64;
		col = (int)(col*colFactor);
	}
	else {
		col += 64;
	}
	rectfill(bmp, x, y, x+w, y+h, makecol(col,col,col));
	DrawContent(bmp, x, y, w, h);
}


void MenuItem::Select() {
	currentState = SELECTED;
	stateCounter = maxCounter;
}


void MenuItem::Deselect() {
	currentState = NORMAL;
	stateCounter = maxCounter;
}


void MenuItem::Enable() {
	currentState = NORMAL;
}


void MenuItem::Disable() {
	currentState = DISABLED;
}


int MenuItem::GetHeight() {
	return text_height(font) + 2;
}


///////////////////////////////////////////////////////////////////////////////


MenuText::MenuText(const char *t) : MenuItem(), text(NULL) {
	text = new char[ustrsizez(t)];
	ustrcpy(text, t);
}


MenuText::~MenuText() {
	if (text) {
		delete [] text;
		text = NULL;
	}
}


void MenuText::DrawContent(BITMAP *bmp, int x, int y, int w, int h) {
	h = h;
	int col = (currentState == DISABLED) ? 180 : 224;
	textout_centre(bmp, font, text, x + (w>>1), y+1, makecol(col,col,col));
}


int MenuText::GetWidth() {
	return text_length(font, text);
}


///////////////////////////////////////////////////////////////////////////////


MenuToggle::MenuToggle(const char *t, bool on) : MenuText(t) {
	this->on = on;
}


MenuToggle::~MenuToggle() {
}


void MenuToggle::DrawContent(BITMAP *bmp, int x, int y, int w, int h) {
	h = h;
	int col = (currentState == DISABLED) ? 180 : 224;
	int w1 = text_length(font, text);
	int w2 = text_length(font, " - OFF");
	textout(bmp, font, text, x + ((w - w1 - w2)>>1), y+1, makecol(col,col,col));
	textout(bmp, font, on ? " - ON" : " - OFF", x + ((w + w1 - w2)>>1), y+1, makecol(col,col,col));
}


int MenuToggle::GetWidth() {
	return text_length(font, text) + text_length(font, " - OFF");
}


int MenuToggle::AcceptInput() {
	if (keyCounter == 0) {
		if (key[KEY_SPACE] || key[KEY_ENTER] || key[KEY_LEFT] || key[KEY_RIGHT]) {
			on = !on;
			keyCounter = 2*maxKeyCounter;
			PlaySample(SMP_CLEAR);
			return on ? 1 : 0;
		}
	}
	
	return -1;
}


///////////////////////////////////////////////////////////////////////////////


MenuEnum::MenuEnum(const char *t, int s, int max) : MenuText(t), sel(s), maxSel(max) {
}


MenuEnum::~MenuEnum() {
}


void MenuEnum::DrawContent(BITMAP *bmp, int x, int y, int w, int h) {
	h = h;
	int col = (currentState == DISABLED) ? 180 : 224;
	int w1 = text_length(font, text);
	char buf[6];
	usprintf(buf, " - %d", sel);
	int w2 = text_length(font, buf);
	textout(bmp, font, text, x + ((w - w1 - w2)>>1), y+1, makecol(col,col,col));
	textout(bmp, font, buf, x + ((w + w1 - w2)>>1), y+1, makecol(col,col,col));
}


int MenuEnum::GetWidth() {
	return text_length(font, text) + text_length(font, " - 0");
}


int MenuEnum::AcceptInput() {
	if (keyCounter == 0) {
		if (key[KEY_SPACE] || key[KEY_ENTER] || key[KEY_RIGHT]) {
			++sel;
			sel %= maxSel;
			keyCounter = 2*maxKeyCounter;
			PlaySample(SMP_CLEAR);
			return sel;
		}
		if (key[KEY_LEFT]) {
			--sel;
			sel = sel < 0 ? maxSel - 1 : sel;
			keyCounter = 2*maxKeyCounter;
			PlaySample(SMP_CLEAR);
			return sel;
		}
	}
	
	return -1;
}


///////////////////////////////////////////////////////////////////////////////


MenuSlider::MenuSlider(const char *t, int s, int max, int w) : MenuEnum(t, s, max), width(w) {
}


MenuSlider::~MenuSlider() {
}


void MenuSlider::DrawContent(BITMAP *bmp, int x, int y, int w, int h) {
	h = h;
	int col = (currentState == DISABLED) ? 180 : 224;
	int w1 = text_length(font, text);
	int w2 = width;
	int w3 = sel*w2/(maxSel-1);
	textout(bmp, font, text, x + ((w - w1 - w2)>>1), y+1, makecol(col,col,col));
	rectfill(bmp, x + ((w + w1 - w2)>>1) + w3, y+1, x + ((w + w1 - w2)>>1) + w2, y+text_height(font), makecol(3*col>>2, 3*col>>2, 3*col>>2));
	rectfill(bmp, x + ((w + w1 - w2)>>1), y+1, x + ((w + w1 - w2)>>1) + w3, y+text_height(font), makecol(col, col, col));
}


int MenuSlider::GetWidth() {
	return width;
}


int MenuSlider::AcceptInput() {
	if (keyCounter == 0) {
		if ((key[KEY_SPACE] || key[KEY_ENTER] || key[KEY_RIGHT]) && sel < maxSel-1) {
			++sel;
			keyCounter = maxKeyCounter;
			PlaySample(SMP_CLEAR);
			return sel;
		}
		if (key[KEY_LEFT] && sel > 0) {
			--sel;
			keyCounter = maxKeyCounter;
			PlaySample(SMP_CLEAR);
			return sel;
		}
	}
	
	return -1;
}


///////////////////////////////////////////////////////////////////////////////


MenuEdit::MenuEdit(const char *t) : MenuItem() {
	if (t) {
		ustrncpy(text, t, 8);
		text[8] = 0;
		cur = ustrsize(text);
	}
	else {
		text[0] = 0;
		text[8] = 0;
		cur = 0;
	}
}


MenuEdit::~MenuEdit() {
}


void MenuEdit::DrawContent(BITMAP *bmp, int x, int y, int w, int h) {
	int col = (currentState == DISABLED) ? 180 : 224;
	int w2 = text_length(font, "1234");
	rectfill(bmp, x + (w>>1) - w2, y+1, x + (w>>1) + w2 - 1, y + h - 1, makecol(64,64,64));
	textout_centre(bmp, font, text, x + (w>>1), y+1, makecol(col,col,col));
	if (currentState != DISABLED && (counter/10) % 2) {
		vline(bmp, x + ((w + GetWidth())>>1), y+1, y+h-2, makecol(col,col,col));
	}
}


int MenuEdit::GetWidth() {
	return text_length(font, text);
}


int MenuEdit::AcceptInput() {
	counter++;
	counter %= 20;
		
	if (keyCounter == 0) {
		if (key[KEY_BACKSPACE]) {
			if (cur > 0) {
				--cur;
				text[cur] = 0;
				keyCounter = maxKeyCounter;
				PlaySample(SMP_BLOCK);
			}
		}
		
		else if (keypressed()) {
			if (cur < 8) {
				int c = readkey() & 0xFF;
				if (c >= 'a' && c <= 'z') {
					c -= 32;
				}
				if ((c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9')) {
					text[cur] = c;
					++cur;
					keyCounter = maxKeyCounter;
					PlaySample(SMP_BLOCK);
				}
			}
			clear_keybuf();
		}
	}
	
	return -1;
}


///////////////////////////////////////////////////////////////////////////////


MenuBitmap::MenuBitmap(const char *t, BITMAP *b1, BITMAP *b2) : MenuText(t), icon1(b1), icon2(b2) {
}


void MenuBitmap::DrawContent(BITMAP *bmp, int x, int y, int w, int h) {
	h = h;
	int iw = GetWidth();
	int col = (currentState == DISABLED) ? 180 : 224;
	
	int cx = x + ((w - iw)>>1);
	if (icon1) {
		draw_sprite(bmp, icon1, cx + (icon2 ? 0 : (8 - (icon1->w>>1))), y + ((h - icon1->h)>>1));
		cx += icon1->w + 2;
	}
	if (icon2) {
		draw_sprite(bmp, icon2, cx, y + ((h - icon2->h)>>1));
		cx += icon2->w + 2;
	}
	textout(bmp, font, text, x + ((w - iw)>>1) + 24, y+1, makecol(col,col,col));
}


int MenuBitmap::GetWidth() {
	return 132;
}
