#include "game.h"
#include "params.h"

#ifndef ALLEGRO_DOS
static bool closed = false;
static void closehook(void) { closed = true; }
#else
#define closed 0
#endif


#ifdef ALLEGRO_WINDOWS
#include <winalleg.h>
#define YIELD() Sleep(1)
#else
#ifdef ALLEGRO_UNIX
#include <sys/time.h>
static void YIELD(void)
{
	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = 1;
	select(0, NULL, NULL, NULL, &tv);
}
#else
#define YIELD() yield_timeslice()
#endif
#endif


static volatile int timer = 0;
static void timer_f() {
	++timer;
} END_OF_STATIC_FUNCTION(timer_f);


Game::Game() : buffer(NULL) {
	usprintf(gameTitle, "mygame");
}


Game::~Game() {
	DestroyDoubleBuffer();
	
	remove_int(timer_f);
	char buf[40];
	usprintf(buf, "%s.ini", gameTitle);
	Params::Save(buf);
}


void Game::DestroyDoubleBuffer() {
	if (buffer) {
		destroy_bitmap(buffer);
		buffer = NULL;
	}
	
	delete fps;
}


Error Game::Init() {
	if (allegro_init() != 0) return Error(Error::ALLEGRO);
	if (install_keyboard() != 0) return Error(Error::KEYBOARD);
	if (install_timer() != 0) return Error(Error::TIMER);
	if (install_mouse() < 0) return Error(Error::MOUSE);
	//if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL) != 0) return Error(Error::SOUND);
	install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL);
	set_volume(-1, -1);
	char buf[40];
	usprintf(buf, "%s.ini", gameTitle);
	Params::Load(buf);
#ifndef ALLEGRO_DOS
	set_window_title(gameTitle);
	set_window_close_hook(&closehook);
#endif
	set_display_switch_mode(SWITCH_BACKGROUND);
	set_color_depth(Params::bpp);
	request_refresh_rate(Params::refresh);
	if (set_gfx_mode(Params::fullscreen ? GFX_AUTODETECT : GFX_AUTODETECT_WINDOWED, Params::w, Params::h, 0, 0) != 0) {
		int alternative;
		switch (Params::bpp) {
			case 8:		alternative = 0;	break;
			case 15:	alternative = 16;	break;
			case 16:	alternative = 15;	break;
			case 24:	alternative = 32;	break;
			case 32:	alternative = 24;	break;
			default:	alternative = 16;	break;
		}
		set_color_depth(alternative);
		if (set_gfx_mode(Params::fullscreen ? GFX_AUTODETECT : GFX_AUTODETECT_WINDOWED, Params::w, Params::h, 0, 0) != 0) {
			return Error(Error::GFX);
		}
		Params::bpp = alternative;
	}
	
	LOCK_VARIABLE(timer);
	LOCK_FUNCTION(timer_f);
	if (install_int_ex(timer_f, BPS_TO_TIMER(Params::fps)) < 0) return Error(Error::TIMER);

	text_mode(-1);
	srand((unsigned)time(NULL));

	fps = new FPS(Params::fps);
	return InitDoubleBuffer();
}


Error Game::InitDoubleBuffer() {
	buffer = create_bitmap(SCREEN_W, SCREEN_H);
	if (!buffer) {
		return Error(Error::MEMORY);
	}
	else {
		clear(buffer);
		return Error(Error::NONE);
	}
}


void Game::Run() {
	bool done = false;
	bool needToRedraw = true;

	timer = 0;
	while (!done) {
		while (timer && !done) {
			--timer;
			done = Update();
			needToRedraw = true;
			fps->Tick();
		}
		
		if (needToRedraw || Params::unlimitedFPS) {
			Draw();
			DrawDoubleBuffer();
			fps->Frame();
			needToRedraw = false;
		}
		
		done |= closed;
		if (Params::yield) {
			YIELD();
		}
	}
}


bool Game::Update() {
	return key[KEY_ESC];
}


void Game::Draw() {
	if (Params::showFPS) {
		fps->Draw(buffer);
	}
}


void Game::DrawDoubleBuffer() {
	if (Params::vsync) {
		vsync();
	}
	blit(buffer, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
}
