/*
 *  Packdude game engine
 *  Copyright (C) 2003  Jaan Pullerits
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------
 * SOUND FX functions
 * 
 * 	Created by JP on 3rd March 2003
 *
 */

#include <stdio.h>
#include "fmod.h"

static int sound = -1;

static struct {
	FSOUND_SAMPLE *dat;
	int active;
	int channel;
} sndpl[32];


/* sound_load:
 */
int sound_load(int num, char *fname)
{
	if(!cfg_get("sound_sfx"))
		return 1;
	
	if (num > 31 || num < 0) {
		fprintf(stderr,
			"Sample load to slot higher than 31 or lower than 0 requested!\n");
		return 0;
	}

	sndpl[num].dat = FSOUND_Sample_Load(num, fname, FSOUND_NORMAL, 0);

	if (sndpl == NULL) {
		fprintf(stderr, "Sample load from '%s' failed!\n", fname);
		return 0;
	}

	sndpl[num].active = 1;

	return 1;
}


/* sound_free:
 */
int sound_free(int num)
{
	if (num > 31 || num < 0) {
		fprintf(stderr,
			"Sample free to slot higher than 31 or lower than 0 requested!\n");
		return 0;
	}

	if (!sndpl[num].active)
		return 1;

	FSOUND_Sample_Free(sndpl[num].dat);
	sndpl[num].active = 0;

	return 1;
}


/* sound_clear:
 */
void sound_clear(void)
{
	int i;

	for (i = 0; i < 32; i++)
		sound_free(i);
}


/* sound_play:
 */
int sound_play(int num)
{
	if(!cfg_get("sound_sfx"))
		return 1;
	
	if (num > 31 || num < 0) {
		fprintf(stderr,
			"Sample play from slot higher than 31 or lower than 0 requested!\n");
		return 0;
	}

	if (!sndpl[num].active) {
		fprintf(stderr,
			"Sample play from unallocated slot requested!\n");
		return 0;
	}

	if ((sndpl[num].channel =
	     FSOUND_PlaySound(FSOUND_FREE, sndpl[num].dat)) == -1)
		return 0;
	else
		return 1;
}

/* sound_stop:
 */
int sound_stop(int num)
{
	if (num > 31 || num < 0) {
		fprintf(stderr,
			"Sample stop from slot higher than 31 or lower than 0 requested!\n");
		return 0;
	}
	if (!sndpl[num].active) {
		fprintf(stderr,
			"Sample stop from unallocated slot requested!\n");
		return 0;
	}
	return FSOUND_StopSound(sndpl[num].channel);
}


/* sound_attr_set
 */
int sound_attr_set(int num, int frequency, int volume, int panning)
{
	if (num > 31 || num < 0) {
		fprintf(stderr,
			"Sample attributes set in slot higher than 31 or lower than 0 requested!\n");
		return 0;
	}

	if (!sndpl[num].active) {
		fprintf(stderr,
			"Sample attributes set in unallocated slot requested!\n");
		return 0;
	}
	FSOUND_Sample_SetDefaults(sndpl[num].dat, frequency, volume,
				  panning, -1);
}


/* sound_freq_get:
 */
int sound_freq_get(int num)
{
	int i = 0;
	if (num > 31 || num < 0) {
		fprintf(stderr,
			"Sample attributes get from slot higher than 31 or lower than 0 requested!\n");
		return 0;
	}
	if (!sndpl[num].active) {
		fprintf(stderr,
			"Sample attributes get from unallocated slot requested!\n");
		return 0;
	}
	FSOUND_Sample_GetDefaults(sndpl[num].dat, &i, NULL, NULL, NULL);

	return i;
}


/* sound_vol_get:
 */
int sound_vol_get(int num)
{
	int i;
	if (num > 31 || num < 0) {
		fprintf(stderr,
			"Sample attributes get from slot higher than 31 or lower than 0 requested!\n");
		return 0;
	}
	if (!sndpl[num].active) {
		fprintf(stderr,
			"Sample attributes get from unallocated slot requested!\n");
		return 0;
	}
	FSOUND_Sample_GetDefaults(sndpl[num].dat, NULL, &i, NULL, NULL);
	return i;
}


/* sound_pan_get:
 */
int sound_pan_get(int num)
{
	int i;
	if (num > 31 || num < 0) {
		fprintf(stderr,
			"Sample attributes get from slot higher than 31 or lower than 0 requested!\n");
		return 0;
	}
	if (!sndpl[num].active) {
		fprintf(stderr,
			"Sample attributes get from unallocated slot requested!\n");
		return 0;
	}
	FSOUND_Sample_GetDefaults(sndpl[num].dat, NULL, NULL, &i, NULL);
	return i;
}
