/*
 *  Packdude game engine
 *  Copyright (C) 2003  Jaan Pullerits
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------
 * Simple Interface Creation functions
 *
 * 	Created by JP on 28th February 2003
 */

#include <allegro.h>
#include "sic.h"
#include "main.h"
#include "jpgalleg.h"
#include "Python.h"

static struct {
	int active;
	int x1, y1;
	int x2, y2;
	char mover[33];
	char mout[33];
	char click[33];
} sic_ztab[32];

static struct {
	int active;
	BITMAP *bmp;
} sic_ipool[32];

static int sic_active = 0;
static int in_zone = -1;
static int nozone = 0;

static BITMAP *sic_surf;

/* sic_clear:
 *
 * 	Clears all tables and image pool.
 */
void sic_clear(void)
{
	int i;
	for (i = 0; i < 32; i++) {
		sic_ztab[i].active = 0;
		if (sic_ipool[i].active)
			sic_image_free(i);
	}
}

/* sic_image_load:
 *
 *		Loads an image to sic pool.
 *
 */
int sic_image_load(int num, char *fname)
{
	if (num > 32 || num < 0) {
		fprintf(stderr,
			"Request to load picture (%s) to invalid slot (%i).\n",
			fname, num);
		return 0;
	}
	if (!exists(fname)) {
		fprintf(stderr,
			"Request to load picture (%s) what doesn't exist.\n",
			fname);
		return 0;
	}

	if (sic_ipool[num].active)
		sic_image_free(num);

	sic_ipool[num].active = 1;
	sic_ipool[num].bmp = load_jpg(fname, NULL);
	return 1;
}

/* sic_image_free:
 *
 * 	Free the slot in sic pool.
 */
int sic_image_free(int num)
{
	if (num > 32 || num < 0) {
		fprintf(stderr,
			"Request to free picture from invalid slot (%i).\n",
			num);
		return 0;
	}
	if (!sic_ipool[num].active)
		return 1;

	destroy_bitmap(sic_ipool[num].bmp);
	sic_ipool[num].active = 0;

	return 1;
}

/* sic_image_put:
 * 	
 * 	Puts a image from pool to screen
 */
int sic_image_put(int num, int x, int y)
{
	if (num > 32 || num < 0) {
		fprintf(stderr,
			"Request to put picture from invalid slot (%i).\n",
			num);
		return 0;
	}
	if (!sic_ipool[num].active) {
		fprintf(stderr,
			"Request to put picture from slot (%i) what is not allocated.\n",
			num);
		return 0;
	}
	if (!sic_active) {
		fprintf(stderr,
			"Request to put picture from slot (%i) was made while SIC not active.\n",
			num);
		return 0;
	}

	scare_mouse();

	blit(sic_ipool[num].bmp, sic_surf, 0, 0, x, y,
	     sic_ipool[num].bmp->w, sic_ipool[num].bmp->h);

	unscare_mouse();

	return 1;
}

// --------------------- ZONE STUFF

/* sic_zone_clear:
 *
 * 	Clear zones
 */
void sic_zone_clear(void)
{
	int i;
	for (i = 0; i < 32; i++)
		sic_ztab[i].active = 0;
}

/* sic_zone_add:
 *
 * 	Adds a zone to sic zone pool
 *
 */
int sic_zone_add(int num, int x1, int y1, int x2, int y2, char *mover,
		 char *mout, char *click)
{
	nozone = 0;

	if (num > 32 || num < 0) {
		fprintf(stderr,
			"Request to add zone to invalid slot (%i).\n",
			num);
		return 0;
	}
	if (strlen(mover) > 32 || strlen(mout) > 32 || strlen(click) > 32) {
		fprintf(stderr,
			"Request to add command longer than 32 character to slot %i.\n",
			num);
		return 0;
	}

	sic_ztab[num].active = 1;

	if (x1 > x2) {
		sic_ztab[num].x1 = x2;
		sic_ztab[num].x2 = x1;
	} else {
		sic_ztab[num].x1 = x1;
		sic_ztab[num].x2 = x2;
	}

	if (y1 > y2) {
		sic_ztab[num].y1 = y2;
		sic_ztab[num].y2 = y1;
	} else {
		sic_ztab[num].y1 = y1;
		sic_ztab[num].y2 = y2;
	}

	if (!(x1 - x2) || !(y1 - y2))
		nozone = 1;

	strcpy(sic_ztab[num].mover, mover);
	strcpy(sic_ztab[num].mout, mout);
	strcpy(sic_ztab[num].click, click);

	return 1;
}

/* sic_zone_remove:
 *
 * 	Remove zone from pool
 */
int sic_zone_remove(int num)
{
	if (num > 32 || num < 0) {
		fprintf(stderr,
			"Request to remove zone from invalid slot (%i).\n",
			num);
		return 0;
	}

	sic_ztab[num].active = 0;

	return 1;
}

/* static run_command:
 *
 * 	Runs a python string
 */
static void run_command(int i, int cmd)
{
	if (cmd == 0)
		sprintf(txtbuf, sic_ztab[i].mover, i, sic_ztab[i].x1,
			sic_ztab[i].y1);
	else if (cmd == 1)
		sprintf(txtbuf, sic_ztab[i].mout, i, sic_ztab[i].x1,
			sic_ztab[i].y1);
	else
		sprintf(txtbuf, sic_ztab[i].click, i, sic_ztab[i].x1,
			sic_ztab[i].y1);

	if (txtbuf)
		PyRun_SimpleString(txtbuf);
}

/* static check_zone:
 *
 * 	Monitors mouse cursor on zone. If mouse exits the zone returns
 * 	false. If mouse is clicked in the zone returns true. Also runs
 * 	the neccesary strings.
 */
static int check_zone(int i)
{
	int x, y, run = 0;

	while ((x = mouse_x) >= sic_ztab[i].x1 && x <= sic_ztab[i].x2 &&
	       (y = mouse_y) >= sic_ztab[i].y1 && y <= sic_ztab[i].y2) {

		if (mouse_b & 1) {
			if (!strcmp(sic_ztab[i].click, "")) {
				// Wait mouse button release.
				while (mouse_b & 1 || mouse_b & 2);
				return 1;
			} else {
				run_command(i, 2);
				run = 0;
				while (mouse_b & 1);
			}
		}

		if (!run) {
			run_command(i, 0);
			run = 1;
		}

	}
	if (run) {
		run_command(i, 1);
	}
	return 0;
}

/* static scan_zones:
 *
 * 	Scan zones for activity
 *
 */
static int scan_zones(void)
{
	int i, ret = -1;
	int exit_anykey = cfg_get("sic_exit_any_key");
	int exit_esc = cfg_get("sic_exit_esc");

	clear_keybuf();
	while (1) {
		for (i = 0; i < 32; i++) {
			if (sic_ztab[i].active && check_zone(i))
				return i;
		}

		if ((exit_anykey && keypressed())
		    || (exit_esc && key[KEY_ESC]))
			return 0;
	}
}

/* sic_do:
 *
 * 	Starts the sic main loop.
 */
int sic_do(int background)
{
	int i, ret;

	if (sic_active) {
		fprintf(stderr,
			"SIC allready active, you cannot nest sic_do() calls!!!.\n");
		return 0;
	}
	if (background > 32 || background < 0) {
		fprintf(stderr,
			"Request to load background from invalid slot (%i).\n",
			background);
		return 0;
	}
	if (!sic_ipool[background].active) {
		fprintf(stderr,
			"Request to load background from slot (%i) what is not allocated.\n",
			background);
		return 0;
	}
	// Activate SIC
	if (cur_depth < 16) {
		if (!set_video(16))
			return 0;
	}

	sic_active = 1;

	sic_surf = scrbuf;
	clear_bitmap(sic_surf);

	// Blit the background
	sic_image_put(background, 0, 0);

	// Do mouseouts
	for (i = 0; i < 32; i++) {
		if (sic_ztab[i].active) {
			run_command(i, 1);
		}
	}

	// TODO: Add fancier screen update
	bltscr(sic_surf);

	sic_surf = screen;

	if (nozone) {		// Evil hack to create zoneless screens.
		for (i = 0; i < 32; i++) {
			if (sic_ztab[i].active) {
				run_command(i, 2);
				break;
			}
		}
		sic_active = 0;
		return 0;
	}

	show_mouse(screen);
	ret = scan_zones();
	show_mouse(NULL);

	// Wait mouse button release.
	while (mouse_b & 1 || mouse_b & 2);

	sic_active = 0;

	return ret;
}

/* sic_low_clear:
 *
 * 	Low level clear for SIC structures.
 * 	Called by init_system()
 */
void sic_low_clear(void)
{
	int i;
	for (i = 0; i < 32; i++) {
		sic_ztab[i].active = 0;
		sic_ipool[i].active = 0;
	}
}
