/*
 *  Packdude game engine
 *  Copyright (C) 2003  Jaan Pullerits
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------
 * MAPFORMAT routines
 *
 *  Created on 22 May 2002 by JP
 *
 * Look carefully, as you see the format of mapfile isn't what
 * you might expect.
 *
 */

#include <allegro.h>
#include <stdio.h>
#include "mapform.h"
#include "tileset.h"
#include "main.h"
#include "animators.h"
#include "refresh.h"
#include <string.h>

#define MAPFORM_VERSION 2


/*****************************************************************************

    Function:    read_to_null(char data[], int maxlenght, FILE *fp)

    Description: Reads bytes from FP to DATA[] until EOF is returned,
                 MAXLENGHT - 1 bytes is read or NULL is returned. After that
                 function terminates string with NULL;
    Parameters:  Data buffer, maximum bytes to read and file pointer
    Return:      Bytes read

*****************************************************************************/

int read_to_null(char data[], int maxlenght, FILE * fp)
{

	int i;
	int c;

	for (i = 0; i < maxlenght - 1; i++) {

		c = getc(fp);

		if (c == EOF || c == 0)
			break;

		data[i] = (char) c;

	}

	data[i] = 0;

	return i;

}


/*****************************************************************************

    Function:    save_map(char *filename)

    Description: Saves the map
    Parameters:  filename of the file
    Return:      -* RETURN TABLE *-
    
                 0 - Successful save
                 1 - unused
                 2 - unused
                 3 - Disk write error

*****************************************************************************/

int save_map(char *filename)
{

	FILE *fp;
	int x, y;
	char version = MAPFORM_VERSION;

	if ((fp = fopen(filename, "wb"))) {
		fputs("PDMAP", fp);	// Header
		fputc(version, fp);	// Version

		for (x = 0; x < 20; x++) {
			for (y = 0; y < 15; y++) {
				fputc(tile_data[x][y], fp); // Tile data
				fputc(bg_data[x][y], fp);   // Background data
				fputc(map_data[x][y], fp);	 // Map data
				if (map_data[x][y] >= 253) {
					fputs(map_parm.cmd[x][y], fp);
					fputc(0, fp);
				}

			}
		}

		fputc(map_parm.numbg, fp);	// Number of bg colors
		for (x = 0; x < 5; x++) {
			fputc(map_parm.bg[x], fp);	// bg colors
		}

		fclose(fp);

	} else
		return 3;

	return 0;

}

/*****************************************************************************

    Function:    load_map(char *filename)

    Description: Loads the map
    Parameters:  filename of the map
    
    Return:      -* RETURN TABLE *-
    
                 0 - Successful load
                 1 - Not valid map file
                 2 - Unsupported map version
                 3 - File not found

*****************************************************************************/

int load_map(char *filename)
{

	FILE *fp;
	int x, y;
	char header[6];
	char version;

	if ((fp = fopen(filename, "rb"))) {
		fgets(header, 6, fp);	// Header

		if (strcmp(header, "PDMAP"))
			return 1;	// Not valid map file

		version = (char) fgetc(fp);	// Version;

		if (1 > version > 2)
			return 2;	// Unknown version

		for (x = 0; x < 20; x++) {
			for (y = 0; y < 15; y++) {
				tile_data[x][y] = fgetc(fp);	// Tile data
				if(version == 2)
					bg_data[x][y] = fgetc(fp);	// Background data
				map_data[x][y] = fgetc(fp);	// Map data
				if (map_data[x][y] >= 253)
					read_to_null(map_parm.cmd[x][y], 33, fp);	// Commands
				else
					strncpy(map_parm.cmd[x][y], "",
						33);
			}
		}

		map_parm.numbg = (unsigned char) fgetc(fp);	// Number of bg colors
		for (x = 0; x < 5; x++) {
			map_parm.bg[x] = (unsigned char) fgetc(fp);	// bg colors
		}

		fclose(fp);
		strncpy(map_parm.path, filename, 1024);

	} else
		return 3;
	return 0;
}

/*****************************************************************************

    Function:    report_map_err(int code)

    Description: Displays map loader error messages
    Parameters:  Error code
    Return:      True if no error, FALSE otherwise

*****************************************************************************/

int report_map_err(int code)
{

	switch (code) {
	case 0:
		return TRUE;
		break;
	case 1:
		alert("MAPFORMAT ERROR", NULL, "Not valid packdude file!",
		      "OK", NULL, KEY_ENTER, 0);
		break;
	case 2:
		alert("MAPFORMAT ERROR", NULL, "Unsupported file version!",
		      "OK", NULL, KEY_ENTER, 0);
		break;
	case 3:
		alert("MAPFORMAT ERROR", NULL,
		      "File not found or disk read/write error!", "OK",
		      NULL, KEY_ENTER, 0);
		break;
	default:
		alert("MAPFORMAT ERROR", NULL, "Unknown error code!", "OK",
		      NULL, KEY_ENTER, 0);
		break;
	}

	return FALSE;
}

/* block_set:
 * 	Sets the block ;)
 */
int block_set(int x, int y, int tag, int tile, char *cmd)
{
	if (x >= 20 || y >= 15) {
		fprintf(stderr, "Request to set block out of grid.\n");
		return 0;
	}
	if (tag > 255) {
		fprintf(stderr, "Request to set block tag > 255.\n");
		return 0;
	}
	if (tile > 255) {
		fprintf(stderr, "Request to set block tile > 255.\n");
		return 0;
	}
	if (strlen(cmd) > 32) {
		fprintf(stderr,
			"Request to set block command longer than 32 characters.\n");
		return 0;
	}

	if (tag >= 0)
		map_parm.block[x][y] = tag;

	if (tile >= 0)
		map_parm.tile[x][y] = tile;

	if (strcmp(cmd, "0"))
		strcpy(map_parm.cmd[x][y], cmd);

	dirty_block[x][y] = 1;

	return 1;
}

/* block_get_tile:
 * 	Get tile num of preticular block
 */
int block_get_tile(int x, int y)
{
	if (x >= 20 || y >= 15) {
		fprintf(stderr,
			"Request to get tile from block out of grid.\n");
		return 0;
	}
	return map_parm.tile[x][y];
}

/* block_get_tag:
 * 	Get tag num of preticular block
 */
int block_get_tag(int x, int y)
{
	if (x >= 20 || y >= 15) {
		fprintf(stderr,
			"Request to get tag from block out of grid.\n");
		return 0;
	}
	return map_parm.block[x][y];
}

/* block_get_cmd:
 * 	Get command of preticular block
 */
char *block_get_cmd(int x, int y)
{
	if (x >= 20 || y >= 15) {
		fprintf(stderr,
			"Request to get command from block out of grid.\n");
		return 0;
	}
	return map_parm.cmd[x][y];
}
