/*
 *  Packdude game engine
 *  Copyright (C) 2003  Jaan Pullerits
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------
 * GRAPHICS EDITOR
 *
 *  Created on 19 May 2002 by JP
 *
 */

#include <allegro.h>		// Allegro header files
#include "tileset.h"		// Tileset routines
#include "main.h"
#include <stdio.h>
#include "paletteed.h"

#define NUM_UNDOS  10
#define BRUSH_STEP 0.25
static int current_color = 0;	// Drawing color
static int current_tool = 0;	// tool
static BITMAP *copybuffer;	// Copybuffer
static BITMAP *image;		// Memory for bitmap
static BITMAP *undo;		// Undo buffer
static signed char tile_changed = 0, tileset_changed = 0;
static float brush_size = 3;

#define NUM_TOOLS 7
static unsigned char *tool_list[NUM_TOOLS] =
    { "Pencil", "Paintbrush", "Line", "Rectangle",
	"Filled rectangle", "Circle", "Filled circle"
};

/*****************************************************************************

    Function:    redraw_toolkit()

    Description: Draws the tool list.
    Parameters:  none
    Return:      none

*****************************************************************************/
void static redraw_toolkit()
{
	int color;
	int i, y = 10;
	scare_mouse();
	for (i = 0; i < NUM_TOOLS; i++) {
		if (i == current_tool)
			color = 15;

		else
			color = 7;
		textprintf(screen, font, 190, y, color, "%s",
			   tool_list[i]);
		y += 8;
	}
	unscare_mouse();
}


/*****************************************************************************

    Function:    redraw_brush_sizer()

    Description: Draws the brush size_window
    Parameters:  none
    Return:      none

*****************************************************************************/
static void redraw_brush_sizer()
{
	BITMAP *buf;
	buf = create_bitmap(128, 128);
	clear_bitmap(buf);
	rect(buf, 0, 0, 127, 127, 7);
	circlefill(buf, 63, 63, brush_size * 4, 7);
	scare_mouse();
	blit(buf, screen, 0, 0, 360, 10, 128, 128);
	unscare_mouse();
	destroy_bitmap(buf);
}

/*****************************************************************************

    Function: redraw_edit_zone()

    Description: Draws the edit screen
    Parameters:  none
    Return:      none

*****************************************************************************/
void static redraw_edit_zone()
{
	int x, y;
	scare_mouse();		// Go away, this isn't for you, mousy
	rect(screen, 9, 9, 138, 138, 7);	// A box
	for (y = 0; y < 32; y++) {	// Draw "pixels"
		for (x = 0; x < 32; x++) {
			rectfill(screen, x * 4 + 10, y * 4 + 10,
				 x * 4 + 13, y * 4 + 13, getpixel(image,
								  x, y));
		}
	}
	unscare_mouse();	// Ok, you may come back now :)
}


/*
 * Draw ruler
 *
 */
void static draw_ruler(int x, int y)
{
	int static old_x = -1, old_y = -1;
	if (x == old_x && y == old_y)
		return;
	if ((x > 9 && x < 138) && (y > 9 && y < 138)) {
		textprintf(screen, font, 10, 139, 15, "%i : %i        ",
			   M2C((x - 10)), M2C((y - 10)));
	}
}


/*****************************************************************************

    Function: redraw_edit_zone_from_bitmap(BITMAP *buf)

    Description: Draws the edit screen
    Parameters:  Bitmap to draw from
    Return:      none

*****************************************************************************/
void static redraw_edit_zone_from_bitmap(BITMAP *buf)
{
	int x, y;
	scare_mouse();		// Go away, this isn't for you, mousy
	rect(screen, 9, 9, 138, 138, 7);	// A box
	for (y = 0; y < 32; y++) {	// Draw "pixels"
		for (x = 0; x < 32; x++) {
			rectfill(screen, x * 4 + 10, y * 4 + 10,
				 x * 4 + 13, y * 4 + 13, getpixel(buf, x,
								  y));
		}
	}
	unscare_mouse();	// Ok, you may come back now :)
}


/*****************************************************************************

    Function: redraw_palette()

    Description: Redraws the palette window
    Parameters:  none
    Return:      none

*****************************************************************************/
void static redraw_palette()
{
	int x, y, color;
	color = 0;
	scare_mouse();
	rect(screen, 9, 147, 138, 276, 7);	// Draw box
	for (y = 0; y < 16; y++) {
		for (x = 0; x < 16; x++) {
			rectfill(screen, x * 8 + 10, y * 8 + 148,
				 x * 8 + 17, y * 8 + 155, color);
			color++;
		}
	}
	unscare_mouse();
}


/*****************************************************************************

    Function: redraw_sprite

    Description: Draws the sprite image
    Parameters:  none
    Return:      none

*****************************************************************************/
void static redraw_sprite()
{
	scare_mouse();
	rect(screen, 147, 9, 180, 42, 7);	// Draw box
	blit(image, screen, 0, 0, 148, 10, 32, 32);
	unscare_mouse();
}

/*****************************************************************************

    Function: redraw_color_box

    Description: Draws the edit color box
    Parameters:  none
    Return:      none

*****************************************************************************/
void static redraw_color_box()
{
	rect(screen, 147, 52, 180, 62, 7);
	rectfill(screen, 148, 53, 179, 61, current_color);
}

/*****************************************************************************

    Function: redraw_screen()

    Description: Redraws whole screen by callin neccesary functions
    Parameters:  none
    Return:      none

*****************************************************************************/
void static redraw_screen()
{
	scare_mouse();
	clear_bitmap(screen);	// clear screen
	unscare_mouse();	// other functions call their own scare_mouse()
	redraw_palette();	// Redraw palette,
	redraw_sprite();	// sprite,
	redraw_color_box();	// color box,
	redraw_edit_zone();	// edit zone,
	redraw_toolkit();	// toolkit,
	redraw_brush_sizer();	// brush sizer
}

/*****************************************************************************

    Function:    register_undo()

    Description: Register undo
    Parameters:  none
    Return:      none

*****************************************************************************/
static void register_undo()
{
	blit(image, undo, 0, 0, 0, 0, 32, 32);
}

/*****************************************************************************

    Function:    do_undo

    Description: Restores bitmap from undo buffer
    Parameters:  none
    Return:      none

*****************************************************************************/
static void do_undo()
{
	BITMAP *buf;
	buf = create_bitmap(32, 32);
	blit(undo, buf, 0, 0, 0, 0, 32, 32);
	blit(image, undo, 0, 0, 0, 0, 32, 32);
	blit(buf, image, 0, 0, 0, 0, 32, 32);
	destroy_bitmap(buf);
	redraw_edit_zone();
	redraw_sprite();
}

/*****************************************************************************

    Function: change_color(x, y);

    Description: Changes drawing color to color of the pixel on x, y coords 
    Parameters:  x and y coords of pixel
    Return:      none

*****************************************************************************/
void static change_color(int x, int y)
{
	if (x < 0 || x >= 640 || y < 0 || y >= 480)	// Just in case
		return;
	scare_mouse();
	current_color = getpixel(screen, x, y);
	redraw_color_box();
	unscare_mouse();
}


/*****************************************************************************

    Function:    make_draw(int x, int y)

    Description: Creates drawing using current_tool
    Parameters:  Starting point coords
    Return:      none

*****************************************************************************/
void static make_draw(int x, int y)
{
	BITMAP *buf;
	int prv_x = -1, prv_y = -1;
	buf = create_bitmap(32, 32);
	blit(image, buf, 0, 0, 0, 0, 32, 32);
	register_undo();
	while (mouse_b & 1) {
		if (M2C(mouse_x - 10) != prv_x
		    || M2C(mouse_y - 10) != prv_y) {
			if (current_tool > 1)
				blit(image, buf, 0, 0, 0, 0, 32, 32);
			if (current_tool == 0)	//Pencil
				putpixel(buf, M2C(mouse_x - 10),
					 M2C(mouse_y - 10), current_color);

			else if (current_tool == 1)	//Paintbrush
				circlefill(buf, M2C(mouse_x - 10),
					   M2C(mouse_y - 10), brush_size,
					   current_color);

			else if (current_tool == 2)
				line(buf, x, y, M2C((mouse_x - 10)),
				     M2C((mouse_y - 10)), current_color);

			else if (current_tool == 3)	//Rectangle
				rect(buf, x, y, M2C((mouse_x - 10)),
				     M2C((mouse_y - 10)), current_color);

			else if (current_tool == 4)	//Filled rectangle
				rectfill(buf, x, y, M2C((mouse_x - 10)),
					 M2C((mouse_y - 10)),
					 current_color);

			else if (current_tool == 5)	//Ellipse
				ellipse(buf, x, y, M2C((mouse_x - 10)) - x,
					M2C((mouse_y - 10)) - y,
					current_color);

			else if (current_tool == 6)	//Filled ellipse
				ellipsefill(buf, x, y,
					    M2C((mouse_x - 10)) - x,
					    M2C((mouse_y - 10)) - y,
					    current_color);
			redraw_edit_zone_from_bitmap(buf);
			prv_x = M2C(mouse_x - 10);
			prv_y = M2C(mouse_y - 10);
		}
	}
	blit(buf, image, 0, 0, 0, 0, 32, 32);
	destroy_bitmap(buf);
	redraw_edit_zone();
	redraw_sprite();
	tile_changed = TRUE;
}


/********
 * This function traces the outline of tile.
 *********************************************/
void static trace_outline(BITMAP *tile)
{
	int x, y;
	BITMAP *tmp;

	tmp = create_bitmap(32, 32);
	
	bmpcpy(tile, tmp);
	
	register_undo();
	for(y = 0; y < 32; y++){;

		for(x = 0; x < 32; x++){
			if(
					getpixel(tile, x, y) == 0 &&
					
					(getpixel(tile, x + 1, y) > 0 ||
					 getpixel(tile, x - 1, y) > 0 ||
					 getpixel(tile, x, y + 1) > 0 ||
					 getpixel(tile, x, y - 1) > 0)
					
				)
				putpixel(tmp, x, y, current_color);
		}
	}
	
	bmpcpy(tmp, tile);

	destroy_bitmap(tmp);
	
	tile_changed = TRUE;
	redraw_edit_zone();
	redraw_sprite();
}


/*****************************************************************************

    Function: clear_image()

    Description: Clears image, and redraws neccesary windows
    Parameters:  none
    Return:      none

*****************************************************************************/
static void clear_image()
{
	clear_bitmap(image);
	redraw_sprite();
	redraw_edit_zone();
}

/* M A I N *******************************************************************

    Function: int edit_graphics()

    Description: Graphics editor main function
    Parameters:  none
    Return:      0 on success
                 1 on GFX init error

*****************************************************************************/
int edit_graphics()
{
	int inkey;
	signed char aborted = FALSE;
	int temp_col, x, y;
	RGB col;
	BITMAP *temp;
	image = tilebuf;
	copybuffer = create_bitmap(32, 32);	// Copybuffer
	undo = create_bitmap(32, 32);	// Undobuffer
	clear_bitmap(image);	// let's clear it just in case
	load_tile(image, current_tile);	// and load a tile in it
	redraw_screen();
	show_mouse(screen);
	while (!aborted) {	// Main loop of GRAPHED

		/* Keyboard handler */
		while (keypressed()) {
			inkey = readkey();
			if (ISKEY(KEY_ESC)) {	// Exit GRAPHED
				if (tileset_changed || tile_changed)
					if (prompt("Save tileset?")) {
						if (tile_changed)
							save_tile(image);
						save_tileset(ts.fname);
					}
				aborted = TRUE;
			}

			else if (ISKEY(KEY_C)) {	// Clear image
				if (prompt("Clear image?")) {
					clear_image();
					tile_changed = TRUE;
				}
				redraw_screen();
			}

			else if (ISKEY(KEY_S)) {	// Save tileset
				if (tileset_changed || tile_changed)
					if (prompt("Save tileset?")) {
						if (tile_changed) {
							save_tile(image);
							tile_changed =
							    FALSE;
						}
						save_tileset(ts.fname);
						tileset_changed = FALSE;
					}
			}

			else if (ISKEY(KEY_SPACE)) {	// Select tile to edit
				if (tile_changed)
					if (prompt("Save tile?")) {
						save_tile(image);
						tileset_changed = TRUE;
					}
				tile_changed = FALSE;
				select_tile(image);
				redraw_screen();
				while (mouse_b & 1 || mouse_b & 2);	// Wait until user releases
				// mouse buttons...
			} else if (ISKEY(KEY_F)) {
				register_undo();
				scare_mouse();	// Floodfill
				floodfill(image, M2C(mouse_x - 10),
					  M2C(mouse_y - 10),
					  current_color);
				unscare_mouse();
				redraw_edit_zone();
				redraw_sprite();
				tile_changed = TRUE;
			} else if (ISKEY(KEY_G)) {	// Copy image to buffer
				blit(image, copybuffer, 0, 0, 0, 0, 32,
				     32);
			} else if (ISKEY(KEY_P)) {	// Paste image from buffer
				blit(copybuffer, image, 0, 0, 0, 0, 32,
				     32);
				tile_changed = TRUE;
				redraw_edit_zone();
				redraw_sprite();
			} else if (ISKEY(KEY_D)) {	// DARKER
				temp_col = current_color;
				get_palette(pal);
				col = pal[current_color];
				for (x = 1;
				     (temp_col == current_color
				      && x < (col.r << 2)
				      && x < (col.g << 2)
				      && x < (col.b << 2)); x++) {
					temp_col =
					    makecol((col.r << 2) - x,
						    (col.g << 2) - x,
						    (col.b << 2) - x);
				}
				current_color = temp_col;
				redraw_color_box();
				while (key[KEY_D]);
				clear_keybuf();
			} else if (ISKEY(KEY_B)) {	// BRIGHTER
				temp_col = current_color;
				get_palette(pal);
				col = pal[current_color];
				for (x = 1; (temp_col == current_color);
				     x++) {
					temp_col =
					    makecol((col.r << 2) + x,
						    (col.g << 2) + x,
						    (col.b << 2) + x);
				}
				current_color = temp_col;
				redraw_color_box();
				while (key[KEY_B]);
				clear_keybuf();
			} else if (ISKEY(KEY_PLUS_PAD)) {
				if (brush_size < 15) {
					brush_size += BRUSH_STEP;
					redraw_brush_sizer();
				}
			} else if (ISKEY(KEY_MINUS_PAD)) {
				if (brush_size > 1) {
					brush_size -= BRUSH_STEP;
					redraw_brush_sizer();
				}
			} else if (ISKEY(KEY_Z)) {
				if (key[KEY_LCONTROL]
				    || key[KEY_RCONTROL])
					do_undo();
			} else if (ISKEY(KEY_R)) {	// Replace color
				temp_col =
				    getpixel(image, M2C(mouse_x - 10),
					     M2C(mouse_y - 10));
				if (temp_col != -1) {
					scare_mouse();
					register_undo();
					for (x = 0; x < 32; x++) {
						for (y = 0; y < 32; y++) {
							if (getpixel
							    (image, x,
							     y) ==
							    temp_col)
								putpixel
								    (image,
								     x, y,
								     current_color);
						}
					}
					unscare_mouse();
					redraw_edit_zone();
					redraw_sprite();
					tile_changed = TRUE;
				}
			} else if (ISKEY(KEY_T)) {	// Turn tile
				register_undo();
				temp = create_bitmap(32, 32);
				clear_bitmap(temp);
				rotate_sprite(temp, image, 0, 0,
					      itofix(64));
				blit(temp, image, 0, 0, 0, 0, 32, 32);
				destroy_bitmap(temp);
				redraw_edit_zone();
				redraw_sprite();
				tile_changed = TRUE;
			} else if (ISKEY(KEY_V)) {	// Flip tile vertical
				register_undo();
				temp = create_bitmap(32, 32);
				clear_bitmap(temp);
				rotate_sprite_v_flip(temp, image, 0, 0,
						     itofix(0));
				blit(temp, image, 0, 0, 0, 0, 32, 32);
				destroy_bitmap(temp);
				redraw_edit_zone();
				redraw_sprite();
				tile_changed = TRUE;
			} else if (ISKEY(KEY_H)) {	// Flip tile horizontal
				register_undo();
				temp = create_bitmap(32, 32);
				clear_bitmap(temp);
				rotate_sprite_v_flip(temp, image, 0, 0,
						     itofix(128));
				blit(temp, image, 0, 0, 0, 0, 32, 32);
				destroy_bitmap(temp);
				redraw_edit_zone();
				redraw_sprite();
				tile_changed = TRUE;
			} else if (ISKEY(KEY_RIGHT)) {	// Translate tile to right
				register_undo();
				temp = create_bitmap(32, 32);
				clear_bitmap(temp);
				blit(image, temp, 0, 0, 1, 0, 32, 32);
				blit(image, temp, 0, 0, -31, 0, 32, 32);
				blit(temp, image, 0, 0, 0, 0, 32, 32);
				destroy_bitmap(temp);
				redraw_edit_zone();
				redraw_sprite();
				tile_changed = TRUE;
			} else if (ISKEY(KEY_LEFT)) {	// Translate tile to left
				register_undo();
				temp = create_bitmap(32, 32);
				clear_bitmap(temp);
				blit(image, temp, 0, 0, 31, 0, 32, 32);
				blit(image, temp, 0, 0, -1, 0, 32, 32);
				blit(temp, image, 0, 0, 0, 0, 32, 32);
				destroy_bitmap(temp);
				redraw_edit_zone();
				redraw_sprite();
				tile_changed = TRUE;
			} else if (ISKEY(KEY_UP)) {	// Translate tile up
				register_undo();
				temp = create_bitmap(32, 32);
				clear_bitmap(temp);
				blit(image, temp, 0, 0, 0, 31, 32, 32);
				blit(image, temp, 0, 0, 0, -1, 32, 32);
				blit(temp, image, 0, 0, 0, 0, 32, 32);
				destroy_bitmap(temp);
				redraw_edit_zone();
				redraw_sprite();
				tile_changed = TRUE;
			} else if (ISKEY(KEY_DOWN)) {	// Translate tile down
				register_undo();
				temp = create_bitmap(32, 32);
				clear_bitmap(temp);
				blit(image, temp, 0, 0, 0, 1, 32, 32);
				blit(image, temp, 0, 0, 0, -31, 32, 32);
				blit(temp, image, 0, 0, 0, 0, 32, 32);
				destroy_bitmap(temp);
				redraw_edit_zone();
				redraw_sprite();
				tile_changed = TRUE;
			} else if (ISKEY(KEY_M)) {
				draw_ruler(mouse_x, mouse_y);
			}
			else if (ISKEY(KEY_F2)) {
				edit_palette(current_color);
				redraw_screen();
			} else if (ISKEY(KEY_O)){ // trace Outline
				trace_outline(image);
			}
			
		}

		/* Keyboard handler */
		if ((mouse_x > 9 && mouse_x < 138) && 
				(mouse_y > 9 && mouse_y < 138)) {	// Mouse is in edit zone         
			if (mouse_b & 1)
				make_draw(M2C(mouse_x - 10),
					  M2C(mouse_y - 10));
			if (mouse_b & 2)
				change_color(mouse_x, mouse_y);
		}

		else if ((mouse_x > 9 && mouse_x < 139) &&
				(mouse_y > 148 && mouse_y < 276)) {	// Mouse is in palette area
			if (mouse_b & 2 || mouse_b & 1)
				change_color(mouse_x, mouse_y);
		}

		else if ((mouse_x > 190 && mouse_x < 350) && (mouse_y > 10 &&
					mouse_y < (10 + (NUM_TOOLS << 3)))) {	// Mouse is in toolpick area
			if (mouse_b & 2 || mouse_b & 1) {
				current_tool = (mouse_y - 10) >> 3;
				redraw_toolkit();
			}
		}

		else if ((mouse_x > 350 && mouse_x < 488) && 
				(mouse_y > 10 && mouse_y < 138)) {	// Mouse is in brush sizer area
			if (mouse_b & 1) {
				if (brush_size < 15) {
					brush_size += BRUSH_STEP;
					redraw_brush_sizer();
				}
			}

			else if (mouse_b & 2) {
				if (brush_size > 1) {
					brush_size -= BRUSH_STEP;
					redraw_brush_sizer();
				}
			}
			while ((mouse_b & 1) || (mouse_b & 2));
		}
	}
	destroy_bitmap(copybuffer);
	destroy_bitmap(undo);
	return 0;
}
