/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include "dialog.h"
#include "script.h"
#include "map.h"
#include "unit.h"
#include "unitprog.h"

/* Unit control function implementations take the unit structure as their */
/*  argument, do something with the unit, then return */

static CPOPCODE opcode_implementation[256];

/* Duplicate the command list */
SCRIPT_STACK *duplicate_commands(SCRIPT_STACK *p)
{
   SCRIPT_STACK *mp;

   if (!p)
      return NULL;

   mp = malloc(sizeof(SCRIPT_STACK));
   
   ASSERT(mp);
   
   mp->progress = p->progress;
   mp->flags = p->flags;
   mp->prog = p->prog;
   mp->next = duplicate_commands(p->next);
   mp->prev = NULL;
   if (mp->next)
      mp->next->prev = mp;
   return mp;
}

/* Find first program in a list of mini-progs that matches flags */
SCRIPT_STACK *match_prog_flags(SCRIPT_STACK *p, int flags)
{
   if (!p)
      return NULL;
   if (p->flags & flags) {
      return p;
   }
   return match_prog_flags(p->next, flags);
}

/* Push a command on the UNIT's control list if possible */
void push_command(UNITDATA *c, char *minip, int flag)
{
   SCRIPT_STACK *p = c->cprog;

   /* if program is NULL, don't add it */
   if (!minip)
      return;

   /* Don't stack up to infinity... */
   if (c->stack_size >= 255)
      return;

   /* Find the last node in the list */
   /* At the end of this loop, p will point to the new node */
   if (p) {                     // already nodes in the list
      while (p->next && (p->next != c->cprog))
         p = p->next;
      p->next = malloc(sizeof(SCRIPT_STACK));
      p->next->prev = p;
      p->next->next = NULL;
      p = p->next;
   } else {                     // new list
      c->cprog = p = malloc(sizeof(SCRIPT_STACK));
      p->prev = NULL;
      p->next = NULL;
   }
   if (p) {
      p->argc = 0;
      p->prog = minip;
      p->progress = 0;
      p->flags = flag;
      c->stack_size++;
   }
}

/* Passes an argument to the LAST program in the array */
void pass_script_argument(UNITDATA *c, int arg)
{
   SCRIPT_STACK *p;
   
   ASSERT(c);
   ASSERT(c->cprog->argc<3);
   
   p = c->cprog;
   if (p) {
      /* Find last node */
      while (p->next && (p->next != c->cprog))
         p = p->next;
      p->argv[p->argc] = arg;
      p->argc++;
   }
}

/* Insert a command after the current command */
/*  If `del' is non-zero, the trailing commands will be stripped  */
/*  (expand later to allow some commands to remain fixed!) */
void insert_command(UNITDATA *s, char *minip, int flag, int del)
{
   SCRIPT_STACK *p;
   SCRIPT_STACK *newp;

   /* if program is NULL, don't add it */
   if (!minip)
      return;

   if (del) {
      rmv_command_tail(s);
   }

   /* Don't stack up to infinity... */
   if (s->stack_size >= 255)
      return;

   p = s->cprog;
   /* already nodes in the list */
   if (p) {
      newp = malloc(sizeof(SCRIPT_STACK));
      if (newp) {
         newp->prog = minip;
         newp->flags = flag;

         newp->next = p->next;
         newp->prev = p;

         if (newp->next) {
            newp->next->prev = newp;
         }
         if (newp->prev) {
            newp->prev->next = newp;
         }
         s->stack_size++;
      }
      /* new list; in this case, insert_command equals push_command */
   } else {
      push_command(s, minip, flag);
   }

}

/* Release the head of the program list */
void release_prog_head(UNITDATA *c)
{
   SCRIPT_STACK *nextp = NULL;

   if (c->cprog) {
      nextp = c->cprog->next;
      if (!(c->cprog->flags & SCRIPT_SAVE)) {
         /* Remove the program */
         if (c->cprog->prev) {
            c->cprog->prev->next = c->cprog->next;
         }
         if (c->cprog->next) {
            c->cprog->next->prev = c->cprog->prev;
         }
         free(c->cprog);
         c->stack_size--;
      }
      c->cprog = nextp;
      c->ip = 0;
   }
}

/* Removes a command tail */
void rmv_command_tail(UNITDATA *c)
{
   SCRIPT_STACK *p = c->cprog;
   SCRIPT_STACK *pn;

   if (c->stack_size <= 1)
      return;

   if (p) {
      p = p->next;
      while ((p && p->next) && (p != c->cprog)) {
         pn = p->next;
         free(p);
         p = pn;
      }
      c->cprog->flags &= ~SCRIPT_LOOP;
      c->cprog->next = NULL;
      c->cprog->prev = NULL;
      c->stack_size = 1;
   }
}

/* Get the number of positions in the queue that are full; this is either */
/*  the queuecount, or the queuecount+1, depending on wether or not a command */
/*  is active. If the command is flagged to be canceled, it won't be counted. */
int get_queuecount(UNITDATA *c)
{
   if (c->cprog && !(c->cprog->flags&SCRIPT_CANCEL))
      return c->queuecount+1;
   else
      return c->queuecount;
}

/* Pushes a command onto the unit's control queue, if there is room there */
/* Returns TRUE on success, FALSE on failure */
int push_queue(UNITDATA *c, TARGET target, int command_id)
{
   int n;

   if (!c->unit_command[command_id])
      return TRUE;

   /* The running program counts as part of the queue */      
   n = get_queuecount(c);

   if (n >= c->queuesize)
      return FALSE;
      
   n = (c->queuestart + c->queuestart) % c->queuesize;
      
   c->cqueue[n].prog = c->unit_command[command_id]->script;
   c->cqueue[n].flags = c->unit_command[command_id]->flags;
   c->cqueue[n].icon = c->unit_command[command_id]->icon;
   c->cqueue[n].argc = c->unit_command[command_id]->argc;
   c->cqueue[n].argv[0] = c->unit_command[command_id]->argv[0];
   c->cqueue[n].argv[1] = c->unit_command[command_id]->argv[1];
   c->cqueue[n].argv[2] = c->unit_command[command_id]->argv[2];
   c->cqueue[n].target.unit = target.unit;
   c->cqueue[n].target.x = target.x;
   c->cqueue[n].target.y = target.y;
   
   c->queuecount++;   
   return TRUE;
}

/* Pops a command off the queue into the cprog state */
void pop_queue(UNITDATA *c)
{
   int n;
   int k;

   if (c->queuecount) {
      ASSERT(c->cqueue);
      
      n = c->queuestart%c->queuesize;
      //n = 0;
      push_command(c, c->cqueue[n].prog, c->cqueue[n].flags);
      
      /* Pass script arguements */
      for (k=0; k<c->cqueue[n].argc; k++)
         pass_script_argument(c, c->cqueue[n].argv[k]);

      c->queuecount--;
      c->queuestart = (c->queuestart+1)%c->queuesize;
   }
   
   
}

/* Removes the first item from the command queue; returns TRUE if there are */
/*  still commands in the queue. */
int clear_queue_head(UNITDATA *c)
{
   int n;
   
   ASSERT(c->cqueue);
   
   if (c->queuecount) {
      c->queuecount--;      
      for (n=0; n<=c->queuecount; n++) {
         c->cqueue[n].flags = c->cqueue[n+1].flags;
         c->cqueue[n].prog = c->cqueue[n+1].prog;
         c->cqueue[n].icon = c->cqueue[n+1].icon;
      }
   }

   return c->queuecount;
}

/* Removes command tail and destroys current program, if ip==0 */
void remove_commands(UNITDATA *c)
{
   rmv_command_tail(c);
   if ((c->cprog) && (c->ip)) {
      free(c->cprog);
      c->cprog = NULL;
      c->ip = 0;
      c->stack_size = 0;
   }
}

/* Loop the current command cycle */
void loop_command(UNITDATA *c)
{
   SCRIPT_STACK *p = c->cprog;

   if (!p)
      return;

   if (p->flags & SCRIPT_LOOP) {
      if (p) {
         while (p->next) {
            p = p->next;
            p->flags |= SCRIPT_LOOP;
         }
         p->next = match_prog_flags(c->cprog, SCRIPT_SAVE);
      }
   }
}

/* Stop all UNIT commands */
void stop_all_scripts(void)
{
   UNIT *c;

   c = active;
   while (c) {
      rmv_command_tail(c->data);
      free(c->data->cprog);
      c->data->cprog = NULL;
      c->data->stack_size = 0;
      c->data->ip = 0;
      c = c->next;
   }
   c = mapobj;
   while (c) {
      rmv_command_tail(c->data);
      free(c->data->cprog);
      c->data->cprog = NULL;
      c->data->stack_size = 0;
      c->data->ip = 0;
      c = c->next;
   }
}

static int unknown_opcode(UNITDATA *ud, int *loop)
{
   popup_message("Unknown byte code encountered: 0x%02x - program corrupt!",
                 ud->cprog->prog[ud->ip]);
   return 0;
}

/* Initialize mini-program command processor */
void script_cp_init(void)
{
   int c;

   for (c = 0; c < 256; c++)
      opcode_implementation[c] = unknown_opcode;
}

void register_opcode(const int opcode, CPOPCODE func)
{
   opcode_implementation[opcode] = func;
}

int script_exec(UNITDATA *c)
{
   SCRIPT_STACK *prog = c->cprog;
   int script_loop;
   int res = 0;

   /* check wait cycle */

   script_loop = 0;
   do {
      if (c->wait_counter) {
         c->wait_counter--;
         return 2;
      }

      if (prog == NULL) {
         return 3;
      }

      if (prog->prog == NULL) {
         return 4;
      }


      res = opcode_implementation[prog->prog[c->ip]] (c, &script_loop);

      script_loop++;
   } while (script_loop < 1);
   return res;
}
