/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <stdio.h>
#include "playgame.h"
#include "upkeep.h"
#include "map.h"
#include "maptile.h"
#include "unit.h"
#include "udraw.h"
#include "ugrp.h"
#include "path.h"
#include "dialog.h"
#include "unitprog.h"
#include "icpanel.h"
#include "script.h"

typedef struct PLAYER {
   UNIT *units;                 /* List of all units belonging to player */
   UNIT *groups[10];            /* List of `quick' groups (cf # in StarCraft) */
   UNITDATA *last_clicked;      /* Last unit clicked */
   UNIT *agroup;                /* Group of currently selected units */
   char *name;                  /* Player name */
   int flags;                   /* Player flags, eg, Human, Computer, Neutral */
   int last_button;             /* Last interface button selected */
   int last_button_pressed;     /* Last interface button pressed */
   TARGET target;               /* Target selected by the player */
   int resource[8];             /* Amount of resources in stock */
} PLAYER;

/* maximum of 8 players and one neutral nation (0) */
static PLAYER player[9];

static void clear_active_list(const int player_id)
{
   UNIT *u;

   u = player[player_id].agroup;
   while (u) {
      player[player_id].agroup = player[player_id].agroup->next;

      u->data->flags &= ~CF_SELECTED;
      register_unit(u->data);
      mark_unit(u->data);

      free_unit(u);
      u = player[player_id].agroup;
   }
}

/* Returns TRUE if the unit can be added to the active player list */
static int can_add_to_selection(const int player_id, UNITDATA *ud)
{
   if (!can_be_selected(ud))
      return FALSE;
   if (!player[player_id].agroup)
      return TRUE;
   if (get_num_selected_units(player_id) > 11)
      return FALSE;
   if ((player[player_id].agroup->data->player != player_id) ||
       (ud->player != player_id))
      return FALSE;
   if ((player[player_id].agroup->data->flags & CF_SEL_MUL) &&
       (ud->flags & CF_SEL_MUL))
      return TRUE;

   return FALSE;
}

static void add_active_list(const int player_id, UNITDATA *ud)
{
   UNIT *u;

   /* Don't add same unit twice! */
   u = player[player_id].agroup;
   while (u) {
      if (u->data == ud)
         return;
      u = u->next;
   }

   /* Add to the end of the list */
   if (player[player_id].agroup) {
      u = player[player_id].agroup;
      while (u->next)
         u = u->next;
      u->next = alloc_unit();
      u->next->prev = u;
      u->next->next = NULL;
      u = u->next;
   } else {
      u = player[player_id].agroup = alloc_unit();
      u->next = u->prev = NULL;
   }

   u->data = ud;
   ud->flags |= CF_SELECTED;
}

static void rmv_active_list(const int player_id, UNITDATA *ud)
{
   UNIT *u = player[player_id].agroup;

   while (u) {
      if (u->data == ud) {
         if (u == player[player_id].agroup)
            player[player_id].agroup = u->next;

         if (u->next)
            u->next->prev = u->prev;
         if (u->prev)
            u->prev->next = u->next;
         free_unit(u);
         ud->flags &= ~CF_SELECTED;
         return;
      }
      u = u->next;
   }
}

static void toggle_active_list(const int player_id, UNITDATA *ud)
{
   UNIT *u;

   u = player[player_id].agroup;
   while (u) {
      if (u->data == ud) {
         rmv_active_list(player_id, ud);
         return;
      }
      u = u->next;
   }
   if (can_add_to_selection(player_id, ud))
      add_active_list(player_id, ud);
}

/* Re-initialize all player data */
/* Note that this currently has a memory leak! All pointers are set to NULL, */
/*  but the groups should be freed as well. */
/* If the global unit datastructures are reset at the same time, then this is */
/*  not nescessary */
void reset_players(void)
{
   int c;
   int b;

   for (c = 0; c < 9; c++) {
      player[c].units = NULL;
      for (b = 0; b < 9; b++)
         player[c].groups[b] = NULL;
      for (b = 0; b < 8; b++)
         player[c].resource[b] = 0;
      player[c].name = "Computer";
      player[c].last_clicked = NULL;
      player[c].agroup = NULL;
      player[c].flags = PLAYER_LOCAL | PLAYER_PASSIVE;
      player[c].last_button = -1;
      player[c].last_button_pressed = -1;
      player[c].target.x = -1;
      player[c].target.y = -1;
      player[c].target.unit = NULL;
   }
}

/* returns the ID of the player running on the local machine */
int get_local_player(void)
{
   return 1;
//   return 2;
//   return 6;
}

int get_resource(const int player_id, const int res)
{
   return player[player_id].resource[res];
}

int inline get_player_upkeep(const int player, const int resource)
{
   return get_unitlist_upkeep(get_player_units(player), resource);
}

int inline get_player_provision(const int player, const int resource)
{
   return get_unitlist_provision(get_player_units(player), resource);
}

void add_resource(const int player_id, const int res, const int amount)
{
   player[player_id].resource[res] += amount;
}

/* Gives a specific unit to a player */
void give_to_player(const int player_num, UNIT *unit)
{
   unit->next = player[player_num].units;
   if (player[player_num].units)
      player[player_num].units->prev = unit;
   player[player_num].units = unit;
   unit->data->player = player_num;
}

/* Takes a unit from a player; returns a pointer to the unit data structure */
/*  that is now free */
UNIT *take_from_player(const int player_num, UNITDATA *udta)
{
   UNIT *u;

   for (u = player[player_num].units; u; u = u->next) {
      if (u->data == udta) {
         if (u == player[player_num].units)
            player[player_num].units = player[player_num].units->next;
         if (u->next)
            u->next->prev = u->prev;
         if (u->prev)
            u->prev->next = u->next;
         u->next = u->prev = NULL;
         return u;
      }
   }

   return NULL;
}


/* Removes a unit from all selections */
void remove_from_all_selections(UNITDATA *udta)
{
   int c;
   int n;

   for (c = 0; c < 9; c++) {
      if (player[c].last_clicked == udta)
         player[c].last_clicked = NULL;
      rmv_active_list(c, udta);
      for (n = 0; n < 10; n++)
         //if (in_list(player[c].groups[n], udta))
         player[c].groups[n] = remove_from_list(udta, player[c].groups[n]);

   }
}

/* Removes a unit from active selections */
void remove_from_active_selections(UNITDATA *udta)
{
   int c;

   for (c = 0; c < 9; c++) {
      if (player[c].last_clicked == udta)
         player[c].last_clicked = NULL;
      rmv_active_list(c, udta);
   }
}

/* Count number of units in numbered group */
int get_group_num_units(const int player_id, const int group_nr)
{
   UNIT *u = player[player_id].groups[group_nr];
   int c = 0;

   while (u) {
      c++;
      u = u->next;
   }
   return c;
}

/* Sets the active group to a numbered group (if it has members) */
void retrieve_unit_groupnr(const int player_id, const int groupnr)
{
   UNIT *u;
   UNIT *ulst;
   UNIT *ut;
   int n = 0;

   ASSERT(groupnr < 10);

   if (player[player_id].groups[groupnr]) {

      /* New select the numbered group */
      ulst = duplicate_list(player[player_id].groups[groupnr]);

      u = ulst;
      while (u) {
         if (can_be_selected(u->data)) {
            n++;
         }
         u = u->next;
      }

      /* If any units at all can be selected */
      if (n) {
         /* First deselect the current group */
         while (player[player_id].agroup) {
            mark_unit(player[player_id].agroup->data);
            rmv_active_list(player_id, player[player_id].agroup->data);
         }

         u = ulst;
         while (u) {
            if (can_be_selected(u->data)) {
               u->data->flags |= CF_SELECTED;
               mark_unit(u->data);
               u = u->next;
               n++;
            } else {
               ut = u;
               u = u->next;
               ulst = remove_from_list(ut->data, ulst);
            }
         }

         player[player_id].agroup = ulst;
      } else {
         destroy_list(ulst);
      }
   }
}

/* Stores the active group under a group number */
void store_unit_groupnr(const int player_id, const int groupnr)
{
   UNIT *u;

   ASSERT(groupnr < 10);

   if (!player[player_id].agroup)
      return;

   if (player[player_id].agroup->data->player != player_id)
      return;



   /* First free the old group structure */
   u = player[player_id].groups[groupnr];
   while (u) {
      u->data->flags &= ~CF_SELECTED;
      u = u->next;
      free_unit(player[player_id].groups[groupnr]);
      player[player_id].groups[groupnr] = u;
   }

   /* Now store the active group */
   player[player_id].groups[groupnr] =
      duplicate_list(player[player_id].agroup);
   for (u = player[player_id].groups[groupnr]; u; u = u->next) {
      u->data->flags |= CF_SELECTED;
      mark_unit(u->data);
   }
}

/* Check if the selected group equals a numbered group */
int group_is_active_group(const int player_id, const int groupnr)
{
   UNIT *u;
   UNIT *ulst = player[player_id].groups[groupnr];
   
   for (u=player[player_id].agroup; u; u=u->next) {
      if (can_be_selected(u->data))
         if (!in_list(ulst, u->data))
            return FALSE;
   }

   /* All units in one group are in the other, so if the groups have the same */
   /*  size, they're actually the same. */
   if (get_num_selected_units(player_id) == get_group_num_units(player_id, groupnr))
      return TRUE;

   /* Must also check the other way round to catch units that cannot be */
   /*  selected for the time being */
   ulst = player[player_id].agroup;
   for (u=player[player_id].groups[groupnr]; u; u=u->next) {
      if (can_be_selected(u->data))
         if (!in_list(ulst, u->data))
            return FALSE;
   }
   return TRUE;
}

/* returns a linked list with all units belonging to some player */
UNIT *get_player_units(const int player_id)
{
   return player[player_id].units;
}

/* get list of units player currently has selected */
UNIT *get_selected_units(const int player_id)
{
   return player[player_id].agroup;
}

/* Get the leader of the currently selected group */
UNITDATA *get_captain(const int player_id)
{
   if (player[player_id].agroup) {
      return player[player_id].agroup->data;
   } else {
      return NULL;
   }
}

/* Count number of units in current group */
int get_num_selected_units(const int player_id)
{
   UNIT *u = player[player_id].agroup;
   int c = 0;

   while (u) {
      c++;
      u = u->next;
   }
   return c;
}

/* Shift the selected units */
void shift_selection(const int player_id)
{
   UNITDATA *u;

   if (!player[player_id].agroup)
      return;

   u = player[player_id].agroup->data;

   rmv_active_list(player_id, u);
   add_active_list(player_id, u);
   //set_screen_changed();
}

/* registered the last clicked unit for player player_id */
void set_last_clicked_unit(const int player_id, UNITDATA *u)
{
   player[player_id].last_clicked = u;
}

UNITDATA *get_last_clicked_unit(const int player_id)
{
   return player[player_id].last_clicked;
}

void set_last_touched_button(const int player_id, int u)
{
   player[player_id].last_button = u;
}

int get_last_touched_button(const int player_id)
{
   return player[player_id].last_button;
}

void set_last_clicked_button(const int player_id, int u)
{
   player[player_id].last_button_pressed = u;
}

int get_last_clicked_button(const int player_id)
{
   return player[player_id].last_button_pressed;
}

void set_command_panel(void)
{
   UNIT *u = player[get_local_player()].agroup;
   int n;

   clear_command_panel_buttons();
   
//   if (get_queuecount(u->data)>=u->data->queuesize)
//      return;
   
   if (u && !(u->data->flags & CF_NOORDERS)) {
      for (n = 0; n < 12; n++)
         if (u->data->unit_command[n]) {
            add_command_panel_button(n, u->data->unit_command[n],
                                     u->data->unit_command[n]->name,
                                     u->data->unit_command[n]->flags);
         }
   }
}

/* Take appropriate action if a unit was actually selected */
/* Flags determines what to do: add/remove unit from the current list */
void register_click(const int player_id, const int flags)
{
   switch (flags) {
      case CLICK_MAKE_SELECTION:
         clear_active_list(player_id);
      case CLICK_ADD_SELECTION:
         if (can_add_to_selection(player_id, player[player_id].last_clicked)) {
            add_active_list(player_id, player[player_id].last_clicked);

            register_unit(player[player_id].last_clicked);
            mark_unit(player[player_id].last_clicked);
            set_screen_changed();
            set_last_clicked_unit(player_id, NULL);
         }
         break;
      case CLICK_RMV_SELECTION:
         rmv_active_list(player_id, player[player_id].last_clicked);

         mark_unit(player[player_id].last_clicked);
         set_screen_changed();
         set_last_clicked_unit(player_id, NULL);
         break;
      case CLICK_TOGGLE_SELECTION:
         toggle_active_list(player_id, player[player_id].last_clicked);

         register_unit(player[player_id].last_clicked);
         mark_unit(player[player_id].last_clicked);
         set_screen_changed();
         set_last_clicked_unit(player_id, NULL);
         break;
      default:
         game_message("Internal error:\nInvalid selection flag");
         break;
   }                            /* end of switch */
}

/* Send a targeted command to the active group (provided that the units in */
/*  the active group belong to the player) */
void give_group_targeted_command(const int player_id, const int command,
                                 const TARGET target, const int script_flags)
{
   UNIT *u;
   int tx, ty;
   TARGET t;

   u = player[player_id].agroup;
   while (u) {
      if ((u->data->player == player_id) && !(u->data->flags & CF_NOORDERS)) {
      
         if (u->data->unit_command[command]) {      
            /* Clear command queue */
            if (u->data->unit_command[command]->flags & UCMD_CLR_QUEUE) {
               while (clear_queue_head(u->data));
            }
         
            /* Clear path if requested, or a new path must be set */
            if (u->data->unit_command[command]->flags & UCMD_CLR_TARGET) {

               /* Clear old path */
               if (u->data->path) {
                  clear_path(u->data);
               }

               u->data->ai_target.x = u->data->x/TILE_WIDTH;
               u->data->ai_target.y = u->data->y/TILE_HEIGHT;
               u->data->ai_target.unit = u->data;
               u->data->path_tries = 0;
            }
         }
      
         /* Set target location (if needed) */
         if (u->data->unit_command[command] &&
             u->data->unit_command[command]->flags & UCMD_TARGET) {

            t.x = target.x;
            t.y = target.y;
            t.unit = target.unit;

            /* Set unit target */
//            u->data->ai_target.x = target.x;
//            u->data->ai_target.y = target.y;
//            u->data->ai_target.unit = target.unit;
            u->data->path_tries = 0;

            if (path_exists
                (u->data->x / TILE_WIDTH, u->data->y / TILE_HEIGHT, target.x,
                 target.y, u->data->layer)) {
               tx = target.x;
               ty = target.y;
            } else if (!(u->data->unit_command[command]->flags & UCMD_PLACE)) {
               find_closest_target(u->data->x / TILE_WIDTH,
                                   u->data->y / TILE_HEIGHT,
                                   target.x, target.y, &tx, &ty);
            } else {
               tx = -1;
               ty = -1;
            }

//            set_path(u->data,
//                     create_path(u->data->x / TILE_WIDTH,
//                                 u->data->y / TILE_HEIGHT,
//                                 tx, ty, u->data->layer));
            t.x = tx;
            t.y = ty;
         } else {
            t.unit = u->data;
         }
         /* Push command */
         if (u->data->unit_command[command]) {
            if (u->data->unit_command[command]->flags & UCMD_CANCEL)
               if (u->data->cprog)
                  u->data->cprog->flags |= SCRIPT_CANCEL;

            push_queue(u->data, t, command);
         }
      }
      /* Break if this command is exclusive */
      if (u->data->unit_command[command] &&
          u->data->unit_command[command]->flags & UCMD_EXCLUSIVE) {
         break;
      } else {
         u = u->next;
      }
   }
}

/* Set the group to execute the specified script (provided that the units */
/*  in the active group belong to the player) */
/*
void push_group_script(const int player_id, char *script, int cmd_flags,
                       int script_flags)
{
   UNIT *u;

   u = player[player_id].agroup;

   if (cmd_flags & UCMD_EXCLUSIVE) {
      if (u->data->player == player_id) {
         if (cmd_flags & UCMD_CLR_QUEUE)
            while (clear_queue_head(u->data));
         push_queue(u->data, u->data->ai_target, script, script_flags, NULL);
      }
   } else {
      while (u) {
         if (cmd_flags & UCMD_CLR_QUEUE)
            while (clear_queue_head(u->data));
         if (u->data->player == player_id) {
            push_queue(u->data, u->data->ai_target, script, script_flags, NULL);
         }
         u = u->next;
      }
   }
}
*/

/* Set a player's target location */
/* The unit, if not NULL, overrides whatever x and y may be specified. */
void set_player_target(const int player_id, const int x, const int y,
                       UNITDATA *u)
{
   player[player_id].target.unit = u;
   player[player_id].target.x = x;
   player[player_id].target.y = y;
}

TARGET get_player_target(const int player_id)
{
   return player[player_id].target;
}

/* Functions used for debugging purposes */
void dump_player_units(const int player_id)
{
   UNIT *u;

   fprintf(stderr, "Units for player #%d:\n", player_id);
   for (u = player[player_id].units; u; u = u->next) {
      fprintf(stderr, "%s at (%d, %d)\n", u->data->name, u->data->x,
              u->data->y);
   }
   fprintf(stderr, "\n");
}

