/**********************************************/
/* Rise of the Tribes                         */
/* C version, Take 2                          */
/* Evert Glebbeek 1998, 2002                  */
/* eglebbk@phys.uva.nl                        */
/**********************************************/
#include <allegro.h>
#include <string.h>
#include <stdio.h>
#include "gfx.h"
#include "maptile.h"
#include "ugfx.h"
#include "sprites.h"
#include "dialog.h"
#include "global.h"

#define NUM_CHARS_GFX   ZZZ_THEEND

typedef struct {
   int c1;
   int c2;
   int c3;
   int c4;
} COLOUR_SCHEME;

/* Palette lookup table for colour schemes */
/* This is the palette in sprites.dat */
static PALETTE colour_scheme_palette;
static PALETTE icon_palette;

/* Player specific colours are the colours 208, 209, 210, 211 in each */
/*  graphic. These are replaced by the following set of colours from the */
/*  colour_scheme_palette */
/* The palette entries that are modified are just those of player 1 */
static COLOUR_SCHEME colour_schemes[9] = {
   {200, 204, 196, 129},
   {208, 209, 210, 211},
   {212, 213, 214, 215},
   {216, 217, 218, 219},
   {220, 221, 222, 223},
   {224, 225, 226, 227},
   {228, 229, 230, 231},
   {232, 233, 234, 235},
   {236, 237, 238, 239},
};

/* Access character graphics by id-name (from datafile) */
char **namekey = NULL;

static UNIT_GFX *cgfx = NULL;
static DATAFILE *icondat = NULL;
static int num_icons;

/* Convert a datafile to a list of unit graphics */
static void df_to_ugfx(DATAFILE * df, UNIT_GFX * gfx_list)
{
   int x = 0, y = 0, w = 0, h = 0;
   int gw = 0, gh = 0;
   DATAFILE *eodf;
   int c = 0;
   int l;

   gfx_list->numgfx = 0;
   gfx_list->gfx = NULL;
   eodf = find_datafile_object(df, "ZZZ_NUMSPRITES");

   /* Get number of graphics */
   /* The DF contains sprites+palette */
   l = (int)(eodf - df);

   sscanf(get_datafile_property(eodf, DAT_ID('G', 'F', 'X', 'W')), "%d", &w);
   sscanf(get_datafile_property(eodf, DAT_ID('G', 'F', 'X', 'H')), "%d", &h);
   if (!w || !h) {
      popup_message("Graphic error:\n" "width or height not set correctly");
   }

   gfx_list->pal = malloc(sizeof(PALETTE));
   memcpy(gfx_list->pal, eodf->dat, sizeof(PALETTE));
   select_palette(*gfx_list->pal);

   gfx_list->x_adjust = realloc(gfx_list->x_adjust, l * sizeof(int));
   gfx_list->y_adjust = realloc(gfx_list->y_adjust, l * sizeof(int));
   gfx_list->gfx = realloc(gfx_list->gfx, l * sizeof(BITMAP *));

   for (c = 0; c < l; c++) {
      sscanf(get_datafile_property(&(df[c]), DAT_ID('X', 'S', 'I', 'Z')),
             "%d", &gw);
      sscanf(get_datafile_property(&(df[c]), DAT_ID('Y', 'S', 'I', 'Z')),
             "%d", &gh);

      gfx_list->gfx[c] =
         create_bitmap_ex(bitmap_color_depth(df[c].dat), gw, gh);

      if (gfx_list->gfx[c] == NULL) {
         set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
         allegro_message
            ("Cannot create unit graphic structure (Memory problem)");
         exit(EXIT_FAILURE);
      }

      clear_bitmap(gfx_list->gfx[c]);
      blit(df[c].dat, (gfx_list->gfx[c]), 0, 0, 0, 0, gw, gh);

      sscanf(get_datafile_property(&(df[c]), DAT_ID('X', 'P', 'O', 'S')),
             "%d", &x);
      sscanf(get_datafile_property(&(df[c]), DAT_ID('Y', 'P', 'O', 'S')),
             "%d", &y);

      gfx_list->x_adjust[c] = (w - TILE_WIDTH) / 2 - x;
      gfx_list->y_adjust[c] = (2 * c + 1) * (h) / 2 - y - TILE_HEIGHT / 2;

      gfx_list->numgfx++;
   }
   unselect_palette();
}

/* Called once at startup */
void cgfx_init(void)
{
   int c = NUM_CHARS_GFX;
   
   cgfx = realloc(cgfx, c * sizeof(UNIT_GFX));
   namekey = realloc(namekey, c * sizeof(char *));
   for (c = 0; c < NUM_CHARS_GFX; c++) {
      cgfx[c].numgfx = 0;
      cgfx[c].gfx = NULL;
      cgfx[c].pal = NULL;
      cgfx[c].x_adjust = cgfx[c].y_adjust = NULL;
      namekey[c] = NULL;
   }
}

void cgfx_load(void)
{
   DATAFILE *df;
   DATAFILE *pd;
   int c;

   set_color_conversion(COLORCONV_NONE);

   /* Load character graphics */
   df = load_datafile(get_game_path("sprites.dat"));
   for (c = 0; df[c].type == DAT_FILE; c++) {
      /* Each entry is by itself a data file */
      df_to_ugfx(df[c].dat, &(cgfx[c]));
      namekey[c] =
         strdup(get_datafile_property(&(df[c]), DAT_ID('N', 'A', 'M', 'E')));
   }

   /* Get the palette holding the player colours */
   pd = find_datafile_object(df, "ZZZ_THEEND");
   if (pd) {
      memcpy(colour_scheme_palette, pd->dat, sizeof(PALETTE));
   }
   unload_datafile(df);

   /* Load icons (eg. objects) */
   if (icondat)
      unload_datafile(icondat);
   icondat = load_datafile(get_game_path("icons.dat"));

   pd = find_datafile_object(icondat, "ZZZ_NUMICONS");
   if (pd) {
      memcpy(icon_palette, pd->dat, sizeof(PALETTE));
   }
   num_icons = pd - icondat;

   set_color_conversion(COLORCONV_TOTAL + COLORCONV_DITHER);
}

BITMAP *get_icon_gfx(int icon_id)
{
//   if (icon_id>=ZZZ_NUMICONS)
//      return NULL;
   return icondat[icon_id].dat;
}

PALETTE *get_icon_palette(const int cs)
{
   PALETTE *pal = &icon_palette;

   /* Change the appropriate colours */
   (*pal)[colour_schemes[1].c1].r =
      colour_scheme_palette[colour_schemes[cs].c1].r;
   (*pal)[colour_schemes[1].c1].g =
      colour_scheme_palette[colour_schemes[cs].c1].g;
   (*pal)[colour_schemes[1].c1].b =
      colour_scheme_palette[colour_schemes[cs].c1].b;

   (*pal)[colour_schemes[1].c2].r =
      colour_scheme_palette[colour_schemes[cs].c2].r;
   (*pal)[colour_schemes[1].c2].g =
      colour_scheme_palette[colour_schemes[cs].c2].g;
   (*pal)[colour_schemes[1].c2].b =
      colour_scheme_palette[colour_schemes[cs].c2].b;

   (*pal)[colour_schemes[1].c3].r =
      colour_scheme_palette[colour_schemes[cs].c3].r;
   (*pal)[colour_schemes[1].c3].g =
      colour_scheme_palette[colour_schemes[cs].c3].g;
   (*pal)[colour_schemes[1].c3].b =
      colour_scheme_palette[colour_schemes[cs].c3].b;

   (*pal)[colour_schemes[1].c4].r =
      colour_scheme_palette[colour_schemes[cs].c4].r;
   (*pal)[colour_schemes[1].c4].g =
      colour_scheme_palette[colour_schemes[cs].c4].g;
   (*pal)[colour_schemes[1].c4].b =
      colour_scheme_palette[colour_schemes[cs].c4].b;
   return pal;
}


BITMAP *get_char_gfx(int char_id, int frame)
{
   return cgfx[char_id].gfx[frame];
}

PALETTE *get_char_palette(const int char_id, const int cs)
{
   PALETTE *pal = cgfx[char_id].pal;

   /* Change the appropriate colours */
   (*pal)[colour_schemes[1].c1].r =
      colour_scheme_palette[colour_schemes[cs].c1].r;
   (*pal)[colour_schemes[1].c1].g =
      colour_scheme_palette[colour_schemes[cs].c1].g;
   (*pal)[colour_schemes[1].c1].b =
      colour_scheme_palette[colour_schemes[cs].c1].b;

   (*pal)[colour_schemes[1].c2].r =
      colour_scheme_palette[colour_schemes[cs].c2].r;
   (*pal)[colour_schemes[1].c2].g =
      colour_scheme_palette[colour_schemes[cs].c2].g;
   (*pal)[colour_schemes[1].c2].b =
      colour_scheme_palette[colour_schemes[cs].c2].b;

   (*pal)[colour_schemes[1].c3].r =
      colour_scheme_palette[colour_schemes[cs].c3].r;
   (*pal)[colour_schemes[1].c3].g =
      colour_scheme_palette[colour_schemes[cs].c3].g;
   (*pal)[colour_schemes[1].c3].b =
      colour_scheme_palette[colour_schemes[cs].c3].b;

   (*pal)[colour_schemes[1].c4].r =
      colour_scheme_palette[colour_schemes[cs].c4].r;
   (*pal)[colour_schemes[1].c4].g =
      colour_scheme_palette[colour_schemes[cs].c4].g;
   (*pal)[colour_schemes[1].c4].b =
      colour_scheme_palette[colour_schemes[cs].c4].b;
   return pal;
}

int get_gfx_dx(int char_id, int frame)
{
   return cgfx[char_id].x_adjust[frame];
}

int get_gfx_dy(int char_id, int frame)
{
   return cgfx[char_id].y_adjust[frame];
}

/* Find the cgfx index number associated with the char name */
/* Returns -1 on failure */
int find_cgfx_name(const char *s)
{
   int c;

   for (c = 0; c < NUM_CHARS_GFX; c++) {
      if (!strcmp(s, namekey[c]))
         return c;
   }

   return -1;
}

/* Find the icon number associated with the name s */
/* Returns -1 on failure */
int find_icon_name(const char *s)
{
   int c;

   for (c = 0; c < num_icons; c++) {
      if (!strcmp
          (s,
           get_datafile_property(&(icondat[c]), DAT_ID('N', 'A', 'M', 'E'))))
         return c;
   }

   return -1;
}

/* Get the number of frames available or this character */
int get_num_gfx(const int char_id)
{
   return cgfx[char_id].numgfx;
}

/* Get player colour; intensity can be 0, 1, 2 or 3 */
int get_player_colour(const int player_id, const int intensity)
{
   switch (intensity) {
      case 0:
         return colour_schemes[player_id].c1;
      case 1:
         return colour_schemes[player_id].c2;
      case 2:
         return colour_schemes[player_id].c3;
      case 3:
         return colour_schemes[player_id].c4;
   }
   return 0;
}
